# -*- coding: utf-8 -*-
import copy
import numbers


from mpfs.metastorage.postgres.queries import SqlTemplatedQuery


class PostgresCursor(object):
    """
    Курсор над постгресом, который предоставляет интерфейс, похожий на монговый - по нему можно итерироваться,
    есть функция count и проч.

    Не используй этот класс для нового кода, это только для поддержки работы старого.
    """
    def __init__(self, session, query_with_params, dao_item_cls):
        self.session = session
        self.dao_item_cls = dao_item_cls
        self._pg_cursor = None
        self.query_with_params = query_with_params
        self._query_with_params_and_offset = copy.copy(query_with_params)  # local copy
        self._has_limit = False

    def count(self):
        if isinstance(self.query_with_params.query, SqlTemplatedQuery):
            return self._calculate_total_rows()
        return self.get_cursor().rowcount

    def _calculate_total_rows(self):
        query = copy.deepcopy(self.query_with_params)
        query.query = query.query.get_replaced_select_with_count()
        query.query = query.query.get_removed_order_by()
        cursor, = self.session.execute_queries([query])
        return cursor.fetchone()[0]

    def _count_without_cursor(self):
        if isinstance(self.query_with_params.query, SqlTemplatedQuery):
            return self._calculate_total_rows()
        cursor, = self.session.execute_queries([self._query_with_params_and_offset])
        return cursor.rowcount

    def apply_offset(self, skip):
        if not isinstance(skip, numbers.Number):
            raise TypeError()

        if skip:
            if 'OFFSET ' in self.query_with_params.query:
                raise RuntimeError('Cannot apply OFFSET because OFFSET has already been found in query: `%s`' %
                                   self.query_with_params.query)

            rowcount = self._count_without_cursor()
            if rowcount <= skip:
                return rowcount

            self._query_with_params_and_offset.query = self.query_with_params.query + ' OFFSET %d' % skip
        return -1

    def is_limit_applied(self):
        return self._has_limit

    def apply_limit(self, limit):
        if self._has_limit:
            return

        if not isinstance(limit, numbers.Number):
            raise TypeError()

        if 'LIMIT ' not in self._query_with_params_and_offset.query:
            self._query_with_params_and_offset.query += ' LIMIT %d' % limit
        self._has_limit = True

    def __iter__(self):
        return self

    def next(self):
        item_data = self.get_cursor().fetchone()
        if item_data is None:
            raise StopIteration()
        return self._convert_item_data_to_mongo_dict(item_data)

    def _create_dao_item(self, item_data):
        return self.dao_item_cls.create_from_pg_data(item_data)

    def _convert_item_data_to_mongo_dict(self, item_data):
        item = self._create_dao_item(item_data)
        return item.get_mongo_representation(skip_missing_fields=True)

    def __getitem__(self, index):
        if isinstance(index, (int, long)) and index == 0:
            return next(self, None)
        else:
            raise NotImplementedError()

    def get_cursor(self):
        if self._pg_cursor is None:
            self._pg_cursor, = self.session.execute_queries([self._query_with_params_and_offset])
        return self._pg_cursor


class PostgresCursorChain(object):
    """
    Класс, позволяющий работать с несколькими курсорами, как с одним, при этом умеет правильно обрабатывать общий
    limit и offset для всех курсоров
    """
    def __init__(self, cursors, limit=0, offset=0):
        assert isinstance(cursors, (list, tuple))
        assert cursors

        self.cursors = cursors
        self.limit = limit
        self._cursors_iter = iter(self.cursors)
        self._current_cursor = self._cursors_iter.next()
        self._number = 0
        self._skip = offset

    def count(self, with_limit_and_skip=False):
        total_sum = sum(cursor.count() for cursor in self.cursors)
        if not with_limit_and_skip:
            return total_sum

        result = total_sum
        if self._skip:
            result = max(total_sum - self._skip, 0)
        if self.limit and self.limit < result:
            return self.limit
        return result

    def __iter__(self):
        return self

    def next(self):
        if self.limit and self.limit <= self._number:
            raise StopIteration()

        if self._skip:
            skipped_items = self._current_cursor.apply_offset(self._skip)
            while skipped_items >= 0:
                self._current_cursor = self._cursors_iter.next()
                self._skip -= skipped_items
                skipped_items = self._current_cursor.apply_offset(self._skip)
            self._skip = 0

        while True:
            try:
                if self.limit and not self._current_cursor.is_limit_applied():
                    self._current_cursor.apply_limit(self.limit - self._number)

                item = self._current_cursor.next()

                if self.limit:
                    self._number += 1

                return item
            except StopIteration:
                self._current_cursor = self._cursors_iter.next()  # will break infinite loop with StopIteration

    def __getitem__(self, index):
        if isinstance(index, (int, long)) and index == 0:
            return next(self, None)
        else:
            raise NotImplementedError()


class PostgresEmptyCursor(object):
    def count(self):
        return 0

    def __iter__(self):
        return self

    def next(self):
        raise StopIteration()

    def __getitem__(self, index):
        return None
