# -*- coding: utf-8 -*-
import binascii
import datetime
import re
import string
import time
import uuid
from types import NoneType

from enum import Enum
from bson import ObjectId
from psycopg2.tz import FixedOffsetTimezone

from mpfs.common.util import to_json, from_json
from mpfs.common.util.datetime_util import SafeTimestampDateTime
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.mongo.util import decompress_data, compress_data
from mpfs.metastorage.postgres.schema import (
    AlbumItemType,
    AlbumLayout,
    AntiVirusScanStatus,
    DiscountProvidableLinesType,
    ChangelogOpType,
    ChangelogSharedType,
    MediaType,
    PromoCodeArchiveStatusType,
    ResourceType,
)


class FieldValidationError(Exception):
    pass


class DAOItemField(object):
    """Базовый класс для аттрибута DAO-класса.

    Предназначен в первую очередь, чтобы приводить данные из разных БД (из монги и из постгреса) к одному виду + для
    обратной операции (чтобы из общего вида приводить к виду, который моджно положить в базу).
    Также содержит в себе ключ для получения данных из словаря, который возвращает монга, и из словаря, который
    возвращает постгрес.

    Пример использования:
        class StringField(DAOItemField):
            ...

        class FileDAOItem(BaseDAOItem):
            path = StringField(mongo_path='key', pg_path='path')
    """

    class NotSpecifiedDefault(object):
        pass

    not_specified_default = NotSpecifiedDefault()

    def __init__(self, mongo_path, pg_path, mongo_item_parser=None, default_value=not_specified_default):
        self.mongo_path = mongo_path
        self.pg_path = pg_path
        self.mongo_item_parser = mongo_item_parser
        self.default_value = default_value

    def from_mongo(self, value):
        """Метод получения общего значения из монги, с учетом валидации
        """
        self.validate_mongo_value(value)
        return self._convert_from_mongo(value)

    def to_mongo(self, value):
        """Метод получения монгового значения их общего значения, с учетом валидации
        """
        self.validate_common_value(value)
        return self._convert_to_mongo(value)

    def from_postgres(self, value):
        """Метод получения общего значения из постгреса, с учетом валидации
        """
        self.validate_postgres_value(value)
        return self._convert_from_postgres(value)

    def to_postgres(self, value):
        """Метод получения постгресового значения их общего значения, с учетом валидации
        """
        self.validate_common_value(value)
        return self._convert_to_postgres(value)

    def _convert_from_mongo(self, value):
        """Конвертирует value из монгового представления в питоновое
        """
        return value

    def _convert_to_mongo(self, value):
        """Конвертирует value из питонового в монговое представление
        """
        return value

    def _convert_from_postgres(self, value):
        """Конвертирует value из постгресового представления в питоновое
        """
        return value

    def _convert_to_postgres(self, value):
        """Конвертирует value из питонового в постгресовое представление
        """
        return value

    def validate_common_value(self, value):
        """Валидирует значение поля, сконвертированного в общее представление
        """
        pass

    def validate_mongo_value(self, value):
        """Валидирует значение поля, полученное из монги
        """
        pass

    def validate_postgres_value(self, value):
        """Валидирует значение поля, полученное из постгреса
        """
        pass


class EnumField(DAOItemField):
    def __init__(self, mongo_path, pg_path, enum_class, default_value=None):
        if default_value is not None and not isinstance(default_value, Enum):
            raise TypeError("Enum required as default value. Got: %r" % default_value)
        super(EnumField, self).__init__(mongo_path, pg_path, default_value=default_value)
        self._enum_class = enum_class

    def validate_common_value(self, value):
        if not isinstance(value, Enum):
            raise TypeError("Enum required as default value. Got: %r" % value)

    def _convert_from_postgres(self, value):
        return self._enum_class(value)

    def _convert_to_postgres(self, value):
        return value.value

    def _convert_from_mongo(self, value):
        return self._convert_from_postgres(value)

    def _convert_to_mongo(self, value):
        return self._convert_to_postgres(value)


class BoolField(DAOItemField):
    def _convert_from_mongo(self, value):
        if not isinstance(value, bool):
            raise TypeError('Value of type %s instead of bool' % type(value))
        return value


class IntegerAsBoolField(BoolField):
    def _convert_from_mongo(self, value):
        """Конвертирует `integer` в `bool`
        """
        if isinstance(value, int):
            return bool(value)
        elif isinstance(value, basestring):
            if value == '0':
                return False
            elif value == '1':
                return True
            else:
                raise TypeError('Value of type string allows only "0" or "1" value (value is `%s`)' % value)
        raise TypeError('Value of type %s instead of int or string value of "0" or "1"' % type(value))

    def _convert_to_mongo(self, value):
        """Конвертирует `bool` в `int`
        """
        return int(value)


class DateTimeStructureField(DAOItemField):
    pass


class DateTimeField(DAOItemField):
    def _convert_from_mongo(self, value):
        """Конвертирует `int` (timestamp) в `datetime.datetime`
        """
        return SafeTimestampDateTime.fromtimestamp(value)

    def _convert_to_mongo(self, value):
        """Конвертирует `datetime.datetime` в `int` (если нет дробных секунд) или `float` (timestamp)
        """
        if isinstance(value, SafeTimestampDateTime):
            seconds = value.get_timestamp()
        else:
            seconds = time.mktime(value.timetuple())
        if value.microsecond:
            return seconds + value.microsecond / 1e6
        else:
            return int(seconds)


class MicrosecondsTimestampField(DAOItemField):

    def _convert_to_postgres(self, value):
        return SafeTimestampDateTime.fromtimestamp(value / 1e6)

    def _convert_from_postgres(self, value):
        if isinstance(value, SafeTimestampDateTime):
            seconds = value.get_timestamp()
        else:
            seconds = time.mktime(value.timetuple())
        seconds = int(seconds) * 10 ** 6
        if value.microsecond:
            seconds += value.microsecond
        return seconds


class DateTimeDeltaField(DAOItemField):
    def _convert_from_mongo(self, value):
        """конвертирует `int` (timestamp в секундах) в `datetime.timedelta`"""
        return datetime.timedelta(seconds=value)

    def _convert_to_mongo(self, value):
        """конвертирует `datetime.timedelta` в `int`"""
        return value.total_seconds()


class DateTimeWithTimezoneField(DAOItemField):
    TIMEZONE_OFFSET = None

    def _convert_from_postgres(self, value):
        if self.TIMEZONE_OFFSET is None:
            raise RuntimeError('Usage of DateTimeWithTimezoneField without specifying TIMEZONE_OFFSET')

        if isinstance(value, datetime.datetime) and value.tzinfo is not None:
            value = value.astimezone(FixedOffsetTimezone(self.TIMEZONE_OFFSET))
            value = value.replace(tzinfo=None)
        else:
            raise NotImplementedError()
        return value

    def _convert_to_postgres(self, value):
        if self.TIMEZONE_OFFSET is None:
            raise RuntimeError('Usage of DateTimeWithTimezoneField without specifying TIMEZONE_OFFSET')

        if isinstance(value, datetime.datetime):
            value = value.replace(tzinfo=FixedOffsetTimezone(self.TIMEZONE_OFFSET))
        else:
            raise NotImplementedError()
        return value


class DateField(DAOItemField):

    def _convert_from_mongo(self, value):
        if isinstance(value, int):
            return datetime.datetime.strptime(str(value), "%Y%m%d").date()
        return value

    def _convert_to_mongo(self, value):
        if not isinstance(value, int):
            return int(value.strftime("%Y%m%d"))
        return value

    def _convert_from_postgres(self, value):
        if not isinstance(value, datetime.date):
            raise NotImplementedError()
        return value

    def _convert_to_postgres(self, value):
        if not isinstance(value, datetime.date):
            raise NotImplementedError()
        return value


class MSKDateTimeField(DateTimeWithTimezoneField):
    TIMEZONE_OFFSET = 180


class UTCDateTimeField(DateTimeWithTimezoneField):
    TIMEZONE_OFFSET = 0


class IntegerField(DAOItemField):
    def _convert_from_mongo(self, value):
        return int(value)

    def _convert_to_mongo(self, value):
        return int(value)


class RealField(DAOItemField):
    def _convert_from_mongo(self, value):
        return float(value)

    def _convert_to_mongo(self, value):
        return float(value)


class MediaTypeField(DAOItemField):
    def _convert_from_mongo(self, value):
        """Преобразует монговые builtin_extensions int'ы в питоновый enum MediaType (определен в постгресовой коллекции)
        """
        for k, v in builtin_extensions.iteritems():
            if v == value:
                return MediaType(k)
        raise LookupError()

    def _convert_to_mongo(self, value):
        """Преобразует питоновый enum MediaType в монговые builtin_extensions int'ы
        """
        return builtin_extensions[value.value]

    def _convert_from_postgres(self, value):
        return MediaType(value)

    def _convert_to_postgres(self, value):
        return value.value


class StringField(DAOItemField):
    def __init__(self, mongo_path, pg_path, mongo_item_parser=None, default_value=DAOItemField.not_specified_default,
                 length=None):
        super(StringField, self).__init__(mongo_path, pg_path,
                                          mongo_item_parser=mongo_item_parser, default_value=default_value)
        self.length = length

    def validate_common_value(self, value):
        if not isinstance(value, (basestring, NoneType)):
            raise FieldValidationError('Value of type %s instead of str, field %s' % (type(value), self.pg_path))
        if self.length is not None and self.length != len(value):
            raise FieldValidationError('Value of length %d, expected %d' % (len(value), self.length))

    def validate_mongo_value(self, value):
        self.validate_common_value(value)

    def validate_postgres_value(self, value):
        self.validate_common_value(value)


class StidField(StringField):
    stid_re = re.compile(
        r'('
        r'[0-9]+\.(yadisk:([0-9]*|preview|uploader|share_(production|test)|digest)\.[0-9E:]+|[0-9]+\.[A-Za-z0-9/+-_])'
        r'|'
        r'ava:disk:[0-9]+:[A-Za-z0-9]{10,50}'
        r')'
    )

    def validate_common_value(self, value):
        if self.default_value is None and value is None:
            return

        if self.stid_re.match(value) is None:
            raise FieldValidationError('Stid `%s` doesn\'t match regular expression in StidField.stid_re' % value)


class PathField(StringField):
    def validate_common_value(self, value):
        if not value.startswith('/'):
            raise FieldValidationError('Path `%s` doesn\'t start with /' % value)


class NullableStringField(StringField):
    def validate_common_value(self, value):
        if value is None:
            return
        super(NullableStringField, self).validate_common_value(value)


class UuidField(StringField):
    def validate_postgres_value(self, value):
        if not isinstance(value, uuid.UUID):
            raise FieldValidationError('UUID value `%s` is not type of uuid.UUID (real type is `%s`)' %
                                       (value, type(value)))

    def _convert_from_postgres(self, value):
        return value.hex

    def _convert_to_postgres(self, value):
        return uuid.UUID(hex=value)


class HidField(StringField):
    hid_re = re.compile(r'[0-9a-fA-F]{32}')

    def __init__(self, mongo_path, pg_path, mongo_item_parser=None, default_value=DAOItemField.not_specified_default):
        super(HidField, self).__init__(mongo_path, pg_path, mongo_item_parser, default_value, length=32)

    def validate_common_value(self, value):
        super(HidField, self).validate_common_value(value)
        if self.hid_re.match(value) is None:
            raise FieldValidationError('Hid `%s` has strange symbols' % value)

    def validate_postgres_value(self, value):
        if not isinstance(value, uuid.UUID):
            raise FieldValidationError('UUID value `%s` is not type of uuid.UUID (real type is `%s`)' %
                                       (value, type(value)))

    def _convert_from_mongo(self, value):
        """Преобразует монговый Binary в питоновую строку
        """
        return str(value)

    def _convert_to_mongo(self, value):
        """Преобразует монговый Binary в питоновую строку
        """
        return Binary(str(value))

    def _convert_from_postgres(self, value):
        """Преобразует постгресовый uuid в питоновую строку
        """
        return value.hex

    def _convert_to_postgres(self, value):
        """Преобразует питоновую строку в постгресовый uuid
        """
        return uuid.UUID(hex=value)


class FidField(StringField):
    def validate_postgres_value(self, value):
        if value is not None and not isinstance(value, uuid.UUID):
            raise FieldValidationError('UUID value `%s` is not type of uuid.UUID (real type is `%s`)' %
                                       (value, type(value)))

    def _convert_from_postgres(self, value):
        if value is None:
            return None
        if isinstance(value, uuid.UUID):
            return value.hex

    def _convert_to_postgres(self, value):
        if value is None:
            return None
        return uuid.UUID(hex=value)


class Md5Field(StringField):
    hid_re = re.compile(r'[0-9a-fA-F]{32}')

    def __init__(self, mongo_path, pg_path, mongo_item_parser=None, default_value=DAOItemField.not_specified_default):
        super(Md5Field, self).__init__(mongo_path, pg_path, mongo_item_parser, default_value, length=32)

    def validate_common_value(self, value):
        super(Md5Field, self).validate_common_value(value)
        if self.hid_re.match(value) is None:
            raise FieldValidationError('MD5 `%s` has strange symbols' % value)

    def validate_postgres_value(self, value):
        if not isinstance(value, uuid.UUID):
            raise FieldValidationError('UUID value `%s` is not type of uuid.UUID (real type is `%s`)' %
                                       (value, type(value)))

    def _convert_from_postgres(self, value):
        return value.hex

    def _convert_to_postgres(self, value):
        return uuid.UUID(hex=value)


class UuidField(StringField):
    def from_postgres(self, value):
        if isinstance(value, uuid.UUID):
            return value.hex
        raise NotImplementedError()

    def to_postgres(self, value):
        return uuid.UUID(hex=value)


class UidField(DAOItemField):
    def _convert_from_postgres(self, value):
        """Постгресовый интовый uid возвращается как строка
        """
        return str(value)

    def _convert_to_postgres(self, value):
        return int(value)


class AntiVirusStatusField(DAOItemField):
    # mongo_map = {
    #     'healthy': 1,
    #     'infected': 2,
    #     'unknown': 4
    # }

    @staticmethod
    def convert_num_to_av_scan_status(value):
        """Преобразует монговые drweb int'ы в питоновый enum AntiVirusScanStatus (определен в постгресовой коллекции)
        """
        if value in (0, 1):
            return AntiVirusScanStatus.clean
        if value == 2:
            return AntiVirusScanStatus.infected
        if value == 4:
            return AntiVirusScanStatus.error
        else:
            raise ValueError('Value 1, 2 or 4 of drweb expected, found %d' % value)

    @staticmethod
    def convert_av_scan_status_to_num(value):
        """Преобразует питоновый enum AntiVirusScanStatus в монговые drweb int'ы
        """
        if value == AntiVirusScanStatus.clean:
            return 1
        elif value == AntiVirusScanStatus.infected:
            return 2
        else:
            return 4

    def _convert_from_mongo(self, value):
        return self.convert_num_to_av_scan_status(value)

    def _convert_to_mongo(self, value):
        return self.convert_av_scan_status_to_num(value)

    def _convert_from_postgres(self, value):
        return AntiVirusScanStatus(value)

    def _convert_to_postgres(self, value):
        return value.value


class Sha256Field(StringField):
    sha256_re = re.compile(r'[0-9a-fA-F]{64}')

    def __init__(self, mongo_path, pg_path, mongo_item_parser=None, default_value=DAOItemField.not_specified_default):
        super(Sha256Field, self).__init__(mongo_path, pg_path, mongo_item_parser, default_value, length=64)

    def validate_common_value(self, value):
        super(Sha256Field, self).validate_common_value(value)
        if self.sha256_re.match(value) is None:
            raise FieldValidationError('SHA256 `%s` has strange symbols' % value)

    def _convert_to_postgres(self, value):
        return '\\x' + value

    def validate_postgres_value(self, value):
        if not isinstance(value, basestring) and not isinstance(value, buffer):
            raise FieldValidationError('Hex value of type buffer or string expected but `%s` received' % type(value))

    def _convert_from_postgres(self, value):
        if isinstance(value, basestring):
            if value.startswith('\\x'):
                return value[2:]
            return value

        hex_value = binascii.hexlify(value)
        if len(hex_value) != 64:
            raise ValueError('hex value of length 64 expected')

        return hex_value


class ByteStringField(StringField):
    def validate_postgres_value(self, value):
        if not isinstance(value, basestring) and not isinstance(value, buffer):
            raise FieldValidationError('Hex value of type buffer or string expected but `%s` received' % type(value))

    def _convert_from_postgres(self, value):
        if isinstance(value, basestring):
            return value
        return str(value)


class ByteArrayField(DAOItemField):
    def _convert_from_mongo(self, value):
        return str(value)

    def _convert_to_mongo(self, value):
        return Binary(str(value))

    def _convert_to_postgres(self, value):
        return '\\x' + binascii.hexlify(value)

    def _convert_from_postgres(self, value):
        return value


class FileIdField(Sha256Field):
    pass


class JsonField(DAOItemField):
    """Поле, преобразующее словарь.

    Использутеся для хранения словарей в `jsonb`.
    """
    def _convert_from_postgres(self, value):
        return from_json(value)

    def _convert_to_postgres(self, value):
        return to_json(value)


class CompressedJsonField(DAOItemField):
    def _convert_from_mongo(self, value):
        return decompress_data(value)

    def _convert_to_mongo(self, value):
        return compress_data(value)

    def _convert_to_postgres(self, value):
        return to_json(value)

    def _convert_from_postgres(self, value):
        return from_json(value)


class ObjectIdField(DAOItemField):
    def _convert_from_mongo(self, value):
        return value

    def _convert_to_mongo(self, value):
        return value

    def _convert_from_postgres(self, value):
        return ObjectId(binascii.hexlify(value))

    def _convert_to_postgres(self, value):
        return '\\x' + str(value)


class StringArrayField(DAOItemField):
    """Поле, преобразующее список значений типа `str`.
    """
    def validate_common_value(self, value):
        if not isinstance(value, list):
            raise FieldValidationError('`list` is required, got `%s` instead.' % type(value))

    def validate_mongo_value(self, value):
        self.validate_common_value(value)

    def validate_postgres_value(self, value):
        self.validate_common_value(value)

    def _convert_from_mongo(self, value):
        return map(str, value)

    def _convert_to_mongo(self, value):
        return map(str, value)

    def _convert_from_postgres(self, value):
        return map(str, value)

    def _convert_to_postgres(self, value):
        return map(str, value)


class UidArrayField(DAOItemField):
    """Поле, преобразующее список значений типа `uid`.
    """
    def validate_common_value(self, value):
        if not isinstance(value, list):
            raise FieldValidationError('`list` is required, got `%s` instead.' % type(value))

    def validate_mongo_value(self, value):
        self.validate_common_value(value)

    def validate_postgres_value(self, value):
        self.validate_common_value(value)

    def _convert_from_mongo(self, value):
        return map(str, value)

    def _convert_to_mongo(self, value):
        return map(str, value)

    def _convert_from_postgres(self, value):
        return map(str, value)

    def _convert_to_postgres(self, value):
        return map(int, value)


class IntArrayField(DAOItemField):
    """Поле, преобразующее список значений типа `int`.
    """
    def validate_common_value(self, value):
        if not isinstance(value, list):
            raise FieldValidationError('`list` is required, got `%s` instead.' % type(value))

    def validate_mongo_value(self, value):
        self.validate_common_value(value)

    def validate_postgres_value(self, value):
        self.validate_common_value(value)

    def _convert_from_mongo(self, value):
        return map(int, value)

    def _convert_to_mongo(self, value):
        return map(int, value)

    def _convert_from_postgres(self, value):
        return map(int, value)

    def _convert_to_postgres(self, value):
        return map(int, value)


class ResourceTypeField(DAOItemField):
    def _convert_from_mongo(self, value):
        return ResourceType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return ResourceType(value)

    def _convert_to_postgres(self, value):
        return value.value


class AlbumLayoutField(DAOItemField):
    def _convert_from_mongo(self, value):
        return AlbumLayout(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return AlbumLayout(value)

    def _convert_to_postgres(self, value):
        return value.value


class AlbumItemTypeField(DAOItemField):
    def _convert_from_mongo(self, value):
        return AlbumItemType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return AlbumItemType(value)

    def _convert_to_postgres(self, value):
        return value.value


class ChangelogOpTypeField(DAOItemField):
    def _convert_from_mongo(self, value):
        return ChangelogOpType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return ChangelogOpType(value)

    def _convert_to_postgres(self, value):
        return value.value


class ChangelogSharedTypeField(DAOItemField):
    def _convert_from_mongo(self, value):
        return ChangelogSharedType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return ChangelogSharedType(value)

    def _convert_to_postgres(self, value):
        return value.value


class PromoCodeArchiveStatusTypeField(DAOItemField):

    def _convert_from_mongo(self, value):
        return PromoCodeArchiveStatusType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return PromoCodeArchiveStatusType(value)

    def _convert_to_postgres(self, value):
        return value.value


class DiscountProvidableLinesTypeField(DAOItemField):

    def _convert_from_mongo(self, value):
        return DiscountProvidableLinesType(value)

    def _convert_to_mongo(self, value):
        return value.value

    def _convert_from_postgres(self, value):
        return DiscountProvidableLinesType(value)

    def _convert_to_postgres(self, value):
        return value.value
