# -*- coding: utf-8 -*-

import enum


class ShardType(enum.Enum):
    MONGO = "mongo"
    POSTGRES = "postgres"


class ShardEndpoint(object):
    DELIMITER = ':'

    def __init__(self, shard_type, shard_name):
        if not isinstance(shard_type, ShardType):
            raise ValueError('Wrong shard type: `%s`' % shard_type)
        self._shard_type = shard_type
        self._shard_name = shard_name

    @classmethod
    def parse(cls, raw_endpoint):
        parts = raw_endpoint.split(cls.DELIMITER)
        if len(parts) != 2:
            raise ValueError('Wrong endpoint: `%s`' % raw_endpoint)
        if parts[0] == ShardType.MONGO.value:
            shard_type = ShardType.MONGO
        elif parts[0] == ShardType.POSTGRES.value:
            shard_type = ShardType.POSTGRES
        else:
            raise ValueError("Wrong shard type: `%s`" % parts[0])
        return ShardEndpoint(shard_type, parts[1])

    def serialize(self):
        return self.DELIMITER.join([self._shard_type.value, self._shard_name])

    def get_type(self):
        return self._shard_type

    def get_name(self):
        return self._shard_name

    def is_pg(self):
        return self._shard_type == ShardType.POSTGRES

    def is_mongo(self):
        return self._shard_type == ShardType.MONGO

    def __eq__(self, other):
        if not isinstance(other, type(self)):
            raise TypeError('Shard endpoint comparison is possible for another shard endpoint')

        if self._shard_type != other._shard_type:
            return False
        return self._shard_name == other._shard_name

    def __ne__(self, other):
        return not (self == other)

    def __repr__(self):
        return '<%s %s>' % (self.__class__.__name__, self.serialize())
