# -*- coding: utf-8 -*-
from copy import deepcopy

_operation_to_cls_map = {}


def get_operation_cls_by_key(mongo_key):
    global _operation_to_cls_map
    if mongo_key in _operation_to_cls_map:
        return _operation_to_cls_map[mongo_key]
    raise LookupError('No class for operation %s' % mongo_key)


class OperationNodeMetaclass(type):
    """Вспомогательный класс для автозаполнения `_operation_to_cls_map`.
     `_operation_to_cls_map` представляет из себя отображение типа операции к классу, представляющему эту операцию и
     являющемуся наследником от OperationNode (для которого этот метакласс и написан).

     Пример заполнения `_operation_to_cls_map`:
      {
        '$and': AndOperation,
        '$or': OrOperation,
        ...
      }
    """
    def __init__(cls, name, bases, class_dict):
        global _operation_to_cls_map
        super(OperationNodeMetaclass, cls).__init__(name, bases, class_dict)
        mongo_key = class_dict.get('mongo_key')
        if mongo_key:
            _operation_to_cls_map[mongo_key] = cls


class SpecTreeNode(object):
    def __init__(self):
        self.children = None

    def convert_to_sql(self):
        raise NotImplementedError()


class OperationNode(SpecTreeNode):
    __metaclass__ = OperationNodeMetaclass
    mongo_key = None


class ValueNode(SpecTreeNode):
    def __init__(self):
        super(ValueNode, self).__init__()
        self.value = None
        self.postgres_value_replacement = None

    def __nonzero__(self):
        return bool(self.value)

    def convert_to_sql(self):
        assert self.postgres_value_replacement is not None
        return ':%s' % self.postgres_value_replacement


class KeyNode(OperationNode):
    def __init__(self):
        super(KeyNode, self).__init__()
        self.mongo_key = None
        self.postgres_key = None

    def convert_to_sql(self):
        assert len(self.children) == 1
        assert self.postgres_key is not None
        return '%s%s' % (self.postgres_key, self.children[0].convert_to_sql())


class JoinOperation(OperationNode):
    postgres_key = None

    def convert_to_sql(self):
        converted_children = [c.convert_to_sql() for c in self.children]
        postgres_operation = ' %s ' % self.postgres_key
        return '(%s)' % postgres_operation.join(converted_children)


class CompareOperation(OperationNode):
    postgres_key = None

    def convert_to_sql(self):
        assert len(self.children) == 1
        child = self.children[0].convert_to_sql()
        return '%s%s' % (self.postgres_key, child)


class AndOperation(JoinOperation):
    mongo_key = '$and'
    postgres_key = 'AND'


class OrOperation(JoinOperation):
    mongo_key = '$or'
    postgres_key = 'OR'


class LteOperation(CompareOperation):
    mongo_key = '$lte'
    postgres_key = '<='


class LtOperation(CompareOperation):
    mongo_key = '$lt'
    postgres_key = '<'


class GtOperation(CompareOperation):
    mongo_key = '$gt'
    postgres_key = '>'


class GteOperation(CompareOperation):
    mongo_key = '$gte'
    postgres_key = '>='


class NeOperation(CompareOperation):
    mongo_key = '$ne'
    postgres_key = ' IS DISTINCT FROM '


class EqOperation(CompareOperation):
    mongo_key = '$eq'
    postgres_key = '='


class ExistsOperation(OperationNode):
    mongo_key = '$exists'

    def convert_to_sql(self):
        assert len(self.children) == 1
        is_exists = bool(self.children[0])
        if is_exists:
            return ' IS NOT NULL'
        else:
            return ' IS NULL'


class InOperation(OperationNode):
    mongo_key = '$in'

    def convert_to_sql(self):
        converted_children = [c.convert_to_sql() for c in self.children]
        return ' IN (%s)' % ','.join(converted_children)


class SpecAST(object):
    def __init__(self, root_node):
        self.root = root_node

    @classmethod
    def build_tree_from_spec(cls, spec):
        assert isinstance(spec, dict)
        spec = deepcopy(spec)
        node = cls._process_expression(spec)
        return cls(node)

    def convert_to_sql(self):
        return self.root.convert_to_sql()

    def convert_mongo_key_value_nodes(self, convert_key, convert_key_value, param_key='param'):
        all_nodes = self._get_all_nodes()
        key_nodes = filter(lambda x: isinstance(x, KeyNode), all_nodes)

        counter = 0
        params = {}

        for node in key_nodes:
            node.postgres_key = convert_key(node.mongo_key)

            all_child_nodes = self._get_all_child_nodes(node, skip_exists_node_children=True)
            all_child_value_nodes = filter(lambda x: isinstance(x, ValueNode), all_child_nodes)
            for n in all_child_value_nodes:
                replacement = '%s%d' % (param_key, counter)
                counter += 1
                n.postgres_value_replacement = replacement
                _, params[replacement] = convert_key_value(node.mongo_key, n.value)

        return params

    @classmethod
    def _process_expression(cls, item):
        if not isinstance(item, dict):
            node = ValueNode()  # если это не похоже на словарь, то это наверное лист дерева, просто заполняем value
            node.value = item
            return node

        mongo_key, arguments = cls._convert_implicit_and_expression(item)
        mongo_key, arguments = cls._convert_in_expression_with_none(mongo_key, arguments)

        if mongo_key.startswith('$'):
            node = get_operation_cls_by_key(mongo_key)()
            if isinstance(arguments, (list, tuple)):
                node.children = [cls._process_expression(arg) for arg in arguments]
            else:
                node.children = [cls._process_expression(arguments)]
            return node
        else:
            mongo_key, arguments = cls._convert_implicit_eq_expression(mongo_key, arguments)
            node = KeyNode()
            node.mongo_key = mongo_key
            node.children = [cls._process_expression(arguments)]
            return node

    @staticmethod
    def _convert_implicit_and_expression(item):
        """Преобразуем два типа сокращенных записи с выражением $and, а именно:
         1. {'field1': 1, 'field2': 2} к {'$and': [{'field1': 1}, {'field2': 2}]}
         2. {'field1': {'$lte': 10, '$gt': 2}} к {'$and': [{'field1': {'$gt': 2}}, {'field1': {'$lte': 10}}]}
        """
        if len(item) > 1:
            mongo_key = '$and'
            arguments = [{k: v} for k, v in item.iteritems()]
        else:
            mongo_key, arguments = item.items()[0]

        if not mongo_key.startswith('$') and isinstance(arguments, dict) and len(arguments) > 1:
            arguments = [{mongo_key: {k: v}} for k, v in arguments.iteritems()]
            mongo_key = '$and'

        return mongo_key, arguments

    @staticmethod
    def _convert_implicit_eq_expression(mongo_key, arguments):
        """Преобразуем сокращенную запись {'field': 1} к полной {'field': {'$eq': 1}}
        """
        if not isinstance(arguments, dict):
            arguments = {'$eq': arguments}
        return mongo_key, arguments

    @staticmethod
    def _convert_in_expression_with_none(mongo_key, arguments):
        """Преобразуем запись вида {'value': {'$in': [1, 2, 3, None, 4]}} к
        записи вида {'$or': [{'value': {'$in': [1, 2, 3, 4]}}, {'value': {'$exists': 0}}]}
        потому что если не преобразовать, то None в sql преобразуется к 'value IN (1, 2, 3, NULL, 4)', что работать не
        будет, а должно преобразоваться к 'value IS NULL OR value IN (1, 2, 3, 4)'
        """
        if not mongo_key.startswith('$') and isinstance(arguments, dict) and len(arguments) == 1 and arguments.keys()[0] == '$in':
            in_arguments = arguments.values()[0]
            if isinstance(in_arguments, list) and None in in_arguments:
                in_arguments.remove(None)
                arguments = [{mongo_key: {'$exists': 0}}, {mongo_key: {'$in': in_arguments}}]
                mongo_key = '$or'
        return mongo_key, arguments

    def _get_all_nodes(self, skip_exists_node_children=False):
        return self._get_all_child_nodes(self.root, skip_exists_node_children)

    def _get_all_child_nodes(self, node, skip_exists_node_children=False):
        if node.children:
            nodes = [node]
            if skip_exists_node_children and isinstance(node, ExistsOperation):
                # это нужно, чтобы не пытаться преобразовать значения в ValueNode'ах, родительские к которым
                # являются ExistsOperation, так как 0 или 1 бессмыссленно преобразовывать в терминологии постгреса
                return nodes
            for n in node.children:
                nodes.extend(self._get_all_child_nodes(n, skip_exists_node_children))
            return nodes
        else:
            return [node]


def convert_spec_to_sql(spec, convert_key=None, convert_key_value=None, param_key='param'):
    tree = SpecAST.build_tree_from_spec(spec)
    params = tree.convert_mongo_key_value_nodes(convert_key, convert_key_value, param_key)
    return tree.convert_to_sql(), params
