# -*- coding: utf-8 -*-
"""

Общее окружение процесса

"""
__error_log__ = None
__access_log__ = None
__default_log__ = None
__system_user__ = None
__share_user__ = None
__storage__ = None
__hostname__ = None
__trace_mpfs_errors__ = None
__cloud_request_id__ = None
__tvm_ticket__ = None
__tvm_2_0_tickets__ = {}
__use_shared_folders__ = None
__hostip__ = None
__dbctl__ = None
__usrctl__ = None
__host__ = None
__event_dispatcher__ = None
__is_uwsgi_process__ = False
__rabbitmq_hosts__ = None
__async_task_uwsgi_host_submitter__ = None
__app_name__ = None
__process_creation_timestamp__ = None
__register_after_fork__impl__ = None
__tvm_clients__ = None
__tvm_2_0_clients__ = None
__read_preference__ = None
__authorization_networks__ = None
__handlers_groups__ = None


import collections
import inspect
import logging
import multiprocessing
import traceback
import os
import sys
import pkgutil
import time
import threading
import random
import uuid

from copy import copy
from pwd import getpwnam
from inspect import isclass

import click
import flask

try:
    # pymongo3
    from pymongo.periodic_executor import _shutdown_executors
    shutdown_pymongo_threads = _shutdown_executors
except ImportError:
    # pymongo2
    from pymongo.mongo_replica_set_client import shutdown_monitors
    shutdown_pymongo_threads = shutdown_monitors

from pymongo.read_preferences import ReadPreference

from mpfs.core.client import MPFSClient
from mpfs.common.errors import StorageInitUser
from mpfs.common.static.tags.conf_sections import TVM2_CLIENT_IDS
from mpfs.common.util import loader, logger
from mpfs.core.event_dispatcher.dispatcher import EventDispatcher
from mpfs.core.event_dispatcher.events import Event
from mpfs.engine.request_globals import (
    get_request_global_variable,
    set_request_global_variable
)

sys.tracebacklimit = 10000
sys.setrecursionlimit(2048)


try:
    import uwsgi
    __is_uwsgi_process__ = True
except ImportError:
    __is_uwsgi_process__ = False


def setup_logs(default_log_in=None, error_log_in="mpfs.error", access_log_in="mpfs.access"):
    global __error_log__
    global __access_log__
    global __default_log__
    # Устанавливаем логи
    __default_log__ = default_log_in
    __error_log__   = error_log_in
    __access_log__  = access_log_in


def setup_host_details():
    global __hostname__
    global __hostip__
    global __host__

     # Устанавливаем хостнейм и адрес
    from socket import gethostname, gethostbyname, getaddrinfo
    host = gethostname()

    chunks = filter(None, host.split('.'))
    machine = chunks.pop(0)

    __hostname__ = host
    try:
        __hostip__ = gethostbyname(__hostname__)
    except:
        __hostip__ = getaddrinfo(__hostname__, 'http')[0][4][0]
    __host__ = machine


def setup_sys_users():
    # Загружаем системных пользователей
    global __system_user__
    global __share_user__

    from mpfs.config import settings
    import mpfs.common.util.dbnaming
    sys_uid = mpfs.common.util.dbnaming.dbuid(settings.system['system']['uid'])
    share_uid = mpfs.common.util.dbnaming.dbuid(settings.system['system']['share_uid'])

    __system_user__ = sys_uid
    __share_user__ = share_uid

    return sys_uid, share_uid


def setup_event_dispatcher():
    global __event_dispatcher__

    def get_event_module_names(path, result=None, prefix='mpfs.core', module_name_pattern='events'):
        if result is None:
            result = list()
        for module_loader, name, ispkg in pkgutil.walk_packages([path]):
            if ispkg:
                get_event_module_names("%s/%s" % (path, name), result, "%s.%s" % (prefix, name), module_name_pattern)
            else:
                if name == module_name_pattern:
                    result.append("%s.%s" % (prefix, name))
        return result

    def event_classes(event_module_names):
        result = []
        for module_name in event_module_names:
            mod = loader.import_module(module_name)
            for _, val in mod.__dict__.iteritems():
                if (isclass(val) and issubclass(val, Event)):
                    result.append(val)
        return result

    events_root_package = 'mpfs.core'
    events_root_path = sys.modules[events_root_package].__path__[0]
    event_module_names = get_event_module_names(events_root_path)
    event_classes = event_classes(event_module_names)
    __event_dispatcher__ = EventDispatcher(event_classes)
    subscription_module_names = get_event_module_names(events_root_path, module_name_pattern='event_subscribtions')
    for module_name in subscription_module_names:
        __import__(module_name)


def setup_main_thread():
    threading.current_thread().setName('main')


def seteuid():
    """Установить effective user id.
    """
    os.setuid(getpwnam('nginx')[2])


def setup(default_log_in=None, error_log_in="mpfs.error", access_log_in="mpfs.access",
          trace_mpfs=True, setuid=True, need_create_sys_users_and_repo=False):
    setup_logs(default_log_in, error_log_in, access_log_in)
    setup_host_details()
    setup_sys_users()
    setup_main_thread()

    set_process_ctime(time.time())

    global __storage__
    global __trace_mpfs_errors__
    global __use_shared_folders__
    global __dbctl__
    global __usrctl__

    # Устанавливаем нужного юзера для процесса
    from mpfs.config import settings
    if setuid:
        seteuid()

    setup_rabbitmq_hosts()

    try:
        if settings.system['system']['storage'] == 'mongo':
            import mpfs.metastorage.mongo.source
            __dbctl__ = mpfs.metastorage.mongo.source.MongoSourceController()
            __dbctl__.setup()

            import mpfs.metastorage.mongo.user
            __usrctl__ = mpfs.metastorage.mongo.user.MongoUserController()
        else:
            raise NotImplementedError()

        __use_shared_folders__ = settings.feature_toggles['use_shared_folders']

        if need_create_sys_users_and_repo:
            create_sys_users_and_repo()

    except Exception, e:
        get_error_log().error(traceback.format_exc())
        raise

    setup_event_dispatcher()

    # Настраиваем правила обработки
    __trace_mpfs_errors__ = trace_mpfs


def create_sys_users_and_repo():
    """Создать системного пользователя и шаренного.
    Также создать репозиторий и дистрибутивы

    ..notes:: Этот метод нужно вызывать для первоначальной настройки базы
        По идее такое нужно только в деве, единожды.
    """
    # Цикл. импорт
    import mpfs.core.services.stock_service

    if dbctl():
        uid = share_user()
        try:
            need_init = not usrctl().check(uid)
        except StorageInitUser:
            need_init = True

        if need_init:
            get_default_log().debug('creating repository for %s' % uid)
            usrctl().create(uid)

        mpfs.core.services.stock_service._setup()


def rabbitmq_hosts(shuffle):
    global __rabbitmq_hosts__

    if __rabbitmq_hosts__ and shuffle:
        hosts = copy(__rabbitmq_hosts__)
        random.shuffle(hosts)
        return hosts

    return __rabbitmq_hosts__


def _get_script_name():
    try:
        return click.get_current_context().command_path.replace(' ', '-')
    except RuntimeError:
        script_name = os.path.basename(sys.argv[0])
        script_name = os.path.splitext(script_name)[0]
        return script_name


def setup_admin_script():
    """Настроить админский (привелегированный) скрипт МПФС.

    Настраивает:
        * подсистему логирования по-умолчанию
        * устанавливает имя приложения :const:`__app_name__` как имя скрипта.
        * коннекты к СУБД
        * меняет euid на `nginx`

    """
    from mpfs.config import settings
    set_cloud_req_id('admin_script-%s' % uuid.uuid4().hex)
    logger.configure_logging(settings.logger['dict_config'], app_name=_get_script_name())

    set_register_after_fork_impl(multiprocessing.util.register_after_fork)
    from mpfs.core.zookeeper.shortcuts import prepare_zookeeper_settings
    prepare_zookeeper_settings(use_cache=False)
    setup(need_create_sys_users_and_repo=False)
    get_default_log().info('%s === SCRIPT: %s ===' % (time.strftime("%Y-%m-%d %X"), sys.argv))

    if settings.services['tvm_2_0']['enabled'] and not settings.feature_toggles['tvm_daemon_usage_enabled']:
        from mpfs.core.services.tvm_2_0_service import tvm2
        tvm2.update_public_keys()
        tvm2.update_service_tickets()


def setup_anyone_script():
    """Настроить неадминский скрипт МПФС.

    Настраивает:
        * подсисетму логирования так, чтобы логи не писались вообще.
        (лог-файлы не будут создаваться)
        * коннекты к СУБД
        * не меняет euid на `nginx`

    """
    from mpfs.config import settings
    SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']

    logging.basicConfig(level=logging.CRITICAL)
    logger.configure_logging({'version': 1})
    # Подавляет сообщения о том, что не найдены логгеры
    logging.raiseExceptions = False
    set_register_after_fork_impl(multiprocessing.util.register_after_fork)
    from mpfs.core.zookeeper.shortcuts import prepare_zookeeper_settings
    prepare_zookeeper_settings(use_cache=False)
    setup(setuid=False)

    if SERVICES_TVM_2_0_ENABLED:
        from mpfs.core.services.tvm_2_0_service import tvm2
        tvm2.update_public_keys()
        tvm2.update_service_tickets()


def get_process_ctime():
    global __process_creation_timestamp__
    return __process_creation_timestamp__


def set_process_ctime(ctime):
    global __process_creation_timestamp__
    __process_creation_timestamp__ = ctime


def set_async_task_uwsgi_submitter(host):
    global __async_task_uwsgi_host_submitter__
    __async_task_uwsgi_host_submitter__ = host


def get_async_task_uwsgi_submitter():
    global __async_task_uwsgi_host_submitter__
    return __async_task_uwsgi_host_submitter__


def set_read_preference(read_preference):
    global __read_preference__
    __read_preference__ = read_preference


def get_read_preference():
    return __read_preference__


def usrctl():
    return __usrctl__


def dbctl():
    """
    :rtype: mpfs.metastorage.mongo.source.CommonMongoSourceController
    """
    return __dbctl__


def get_default_log():
    """Получить логгер уровня модулей.

    Если переопределено имя лога по-умолчанию :const:`__default_log__`,
    то возвращается логгер с указанным именем.

    Иначе, используется способ, альтернативный стандартному:
    >>> logging.getLogger(__name__)
    https://docs.python.org/2/howto/logging.html#advanced-logging-tutorial

    .. doctest::
    >>> __default_log__ = None
    >>> l1 = get_default_log()
    >>> l2 = logging.getLogger(__name__)
    >>> l1 is l2
    True

    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """

    name = __default_log__
    if name is None:
        stack = inspect.stack()
        frame = stack[1][0]
        module = inspect.getmodule(frame)
        name = module.__name__
    return logger.get(name)


def get_error_log():
    """Получтиь логгер необработанных ошибок.

    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.get(__error_log__)


def get_access_log():
    """Получтиь логгер входных запросов.

    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.get(__access_log__)


def get_requests_log():
    """Получтиь логгер запросов в СУБД и сторонние сервисы.

    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.get("mpfs.requests")


def get_requests_postgres_log():
    """Получтиь логгер запросов в постгрес.

    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.get('mpfs.requests.postgres')


def get_service_log(name):
    """Получтиь логгер ответов сервиса `name`.

    :type: str
    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.get("mpfs.service.%s" % name)


def get_stat_log(postfix=''):
    """Получтиь логгер статистики для отчета `ydisk-mpfs-stat-log-{postfix}`

    Значение `postfix` добавляется как постфикс к `tskv_format`
    Служит для отделения отчетов внутри таблицы ydisk-mpfs-stat-log.

    :type postfix: str
    :rtype: :class:`~mpfs.common.util.logger.MPFSLogger`
    """
    return logger.StatLoggerAdapter(logger.get("mpfs.stat"), postfix)


def get_event_history_log():
    return logger.get('event-history')


def system_user():
    return __system_user__


def share_user():
    return __share_user__


def storage():
    return __storage__


def hostname():
    return __hostname__


def host():
    return __host__


def hostip():
    return __hostip__


def trace_mpfs_errors():
    return __trace_mpfs_errors__


def event_dispatcher():
    global __event_dispatcher__
    return __event_dispatcher__


def is_uwsgi_process():
    global __is_uwsgi_process__
    return __is_uwsgi_process__


def setup_rabbitmq_hosts():
    global __rabbitmq_hosts__
    from mpfs.engine.queue2.utils import get_broker_urls
    try:
        __rabbitmq_hosts__ = get_broker_urls()
    except Exception as e:
        get_error_log().exception('Error on getting rabbitmq hosts: %s: %s' % (type(e), e))
        raise


def set_req_id(rid):
    """
    MPFS Request ID: [PID]_[RANDINT]
    :param string rid: request_id
    :return: None
    """
    set_request_global_variable('rid', rid)


def get_req_id():
    return get_request_global_variable('rid')


def set_cloud_req_id(crid):
    """
        Yandex-Cloud-Request-ID
    :param crid:
    :return:
    """
    set_request_global_variable('crid', crid)


def get_cloud_req_id():
    return get_request_global_variable('crid')


def set_external_tvm_ticket(tvm_ticket):
    set_request_global_variable('external_tvm_ticket', tvm_ticket)


def get_external_tvm_ticket():
    return get_request_global_variable('external_tvm_ticket')


def setup_authorization_networks():
    global __authorization_networks__
    from mpfs.config import settings

    if settings.auth['network_authorization']['enabled']:
        from mpfs.frontend.api.auth import NetworkAuthorization

        __authorization_networks__ = NetworkAuthorization.collect_authorization_network()


def get_authorization_networks():
    return __authorization_networks__


def setup_handlers_groups():
    global __handlers_groups__
    from mpfs.config import settings

    __handlers_groups__ = {}
    for group_name, handlers_names in settings.auth['handlers_groups'].items():
        for handler_name in handlers_names:
            __handlers_groups__[handler_name] = group_name


def get_handlers_groups():
    return __handlers_groups__


def set_tvm_ticket(tvm_ticket):
    global __tvm_ticket__
    if not tvm_ticket:
        return
    __tvm_ticket__ = tvm_ticket


def get_tvm_ticket():
    from mpfs.core.services.tvm_service import tvm

    global __tvm_ticket__
    if __tvm_ticket__ is not None and not __tvm_ticket__.is_expired():
        return __tvm_ticket__
    # TODO: небезопасно: если не получим тикет, будет исключение.
    # TODO: можно использовать старый, если было исключение, либо обновлять в фоне
    __tvm_ticket__ = tvm.get_new_ticket()
    return __tvm_ticket__


def set_tvm_clients(client_ids):
    global __tvm_clients__
    __tvm_clients__ = client_ids


def get_tvm_clients():
    return __tvm_clients__


def setup_tvm_2_0_clients(auth_conf):
    """Составляет список клиентов с TVM 2.0 авторизацией.

    :param auth_conf: auth-секция конфига с клиентами MPFS
    """
    global __tvm_2_0_clients__

    clients = [MPFSClient(service_name, service_conf)
               for service_name, service_conf in auth_conf.items()]
    client_ids = {}
    for client in clients:
        if not client.has_tvm_2_0():
            continue

        if TVM2_CLIENT_IDS not in client.tvm_2_0:
            get_error_log().error('Service "%s" is missing %s' % (client.name, TVM2_CLIENT_IDS))
            continue

        for client_id in client.tvm_2_0[TVM2_CLIENT_IDS]:
            client_ids[client_id] = client

    __tvm_2_0_clients__ = client_ids


def get_tvm_2_0_clients():
    return __tvm_2_0_clients__


def set_tvm_2_0_service_ticket_for_client(client_id, tvm_ticket):
    global __tvm_2_0_tickets__
    __tvm_2_0_tickets__[client_id] = tvm_ticket


def get_tvm_2_0_service_ticket_for_client(client_id):
    """Возвращаем tvm2 тикет для клиента

    Если тикета нет, пробует переполучить; если не получится, кидает исключение
    Если тикет протух, пробует переполучить; если не получится, отдаёт старый
    """
    global __tvm_2_0_tickets__
    from mpfs.core.services.tvm_2_0_service import tvm2

    service_ticket = __tvm_2_0_tickets__.get(client_id)
    if service_ticket is None or service_ticket.is_expired():
        try:
            service_ticket = tvm2.get_new_service_ticket(client_id)
            __tvm_2_0_tickets__[client_id] = service_ticket
        except Exception:
            if client_id in __tvm_2_0_tickets__:
                get_error_log().error('Couldn\'t get service ticket for %s, using old ticket' % client_id)
            else:
                get_error_log().error('Couldn\'t get service ticket for %s and no old ticket, failing' % client_id)
                raise
    return service_ticket


def set_tvm_2_0_user_ticket(tvm_ticket):
    set_request_global_variable('tvm_2_0_user_ticket', tvm_ticket)


def get_tvm_2_0_user_ticket():
    return get_request_global_variable('tvm_2_0_user_ticket')


def set_app_name(app_name):
    global __app_name__
    __app_name__ = app_name


def get_app_name():
    return __app_name__


def get_global_request():
    return get_request_global_variable('request')


def set_global_request(request):
    return set_request_global_variable('request', request)


def get_global_tld():
    """Получить глобальное значние top level domain.

    Если мы под uWSGI, то это значение всегда доступно в глобальном запросе.
    Иначе деаем фолбек на значение по-умолчанию.
    """
    if is_uwsgi_process():
        return get_global_request().tld
    from mpfs.config import settings  # цикл. импорт
    return settings.user['default_tld']


def get_global_real_tld():
    """Возвращает оригинальный tld из запроса без изменений"""
    return get_global_request().real_tld


def set_show_nda(show_nda):
    global_request = get_global_request()
    if not global_request:
        return
    global_request.show_nda = show_nda


def get_show_nda():
    return get_global_request().show_nda or None


def use_shared_folders():
    return __use_shared_folders__


def reset_cached():
    if use_shared_folders():
        from mpfs.core.social.share import Group, LinkToGroup
        Group.reset()
        LinkToGroup.reset()

    from mpfs.core.services.passport_service import Passport
    from mpfs.core.services.uaas_service import uaas, new_uaas
    from mpfs.core.metastorage.control import DiskInfoCollection, UserIndexCollection
    from mpfs.core.filesystem.symlinks import Symlink

    set_show_nda(None)
    services_with_cache = [Passport, uaas, new_uaas]
    for service in services_with_cache:
        service.reset()
    UserIndexCollection.reset()
    DiskInfoCollection.reset()
    Symlink.reset()

    from mpfs.metastorage.mongo.mapper import MPFSMongoReplicaSetMapper
    MPFSMongoReplicaSetMapper.reset()

    from mpfs.metastorage.postgres.query_executer import PGQueryExecuter
    PGQueryExecuter.reset_shapei_cache()

    from mpfs.core.user.standart import StandartUser
    StandartUser.cache_reset()

    from mpfs.core.zookeeper.shortcuts import push_new_settings_to_queue
    push_new_settings_to_queue(block=False)


def reset_connections():
    from mpfs.dao.session import Session
    Session.clear_cache()


def pre_fork():
    """
    Выполнить действия перед форком.

    Корректно завершить треды-мониторы pymongo и
    проверить, что живых тредов нет.

    .. warning:: Должен быть вызван в родительском процесе после
        всех обращений к БД, т.е. после метода :meth:`~mpfs.engine.process.setup`
    """
    shutdown_pymongo_threads()

    current = threading.current_thread()
    for thread in threading.enumerate():
        if thread is not current and thread.is_alive():
            raise RuntimeError("Threads are not allowed before fork: %r" % thread)


def uwsgi_after_fork(obj, func):
    """Добавить `obj`, `func` в список коллбеков, которые вызываются после форка.

    Эта реалиация для uwsgi.
    Под uWSGI стандартный механизм из модуля :mod:`multiprocessing.util` не работает.
    Поэтому заставляем uWSGI использовать этот механизм напрямую, через uWSGI API.

    :type obj: Any
    :type func: :class:`collections.Callable`
    :return:
    """
    import uwsgi
    uwsgi.post_fork_hook = multiprocessing.util._run_after_forkers
    multiprocessing.util.register_after_fork(obj, func)


def get_register_after_fork_impl():
    """Вернуть обработчик регистрации коллбеков, которые вызываются после форка.

    :rtype: :class:`collections.Callable` | None
    """

    return __register_after_fork__impl__


def set_register_after_fork_impl(impl):
    """Установить обработчик регистрации коллбеков, которые вызываются после форка.

    :type impl: :class:`collections.Callable`
    """
    global __register_after_fork__impl__

    if not isinstance(impl, collections.Callable):
        raise TypeError("`Callable` is required, got `%s` instead" % type(impl))

    if __register_after_fork__impl__ is not None:
        raise RuntimeError("Already set")

    __register_after_fork__impl__ = impl


def register_after_fork(obj, func):
    """Добавить коллбек в список тех, которые вызываются после форка.

    :type obj: Any
    :type func: :class:`collections.Callable`
    """
    impl = get_register_after_fork_impl()
    if impl is None:
        raise RuntimeError('Set `register_after_fork` implementation first')

    get_default_log().info('[register_after_fork] New item is registered: func=%s, obj=%s' % (func, obj))
    impl(obj, func)


class Signal(object):
    """Перечисление типов сигналов

    Типом является целочисленное значение 0 <= i <= 255
    """
    # TODO: Распилить сигналы по сервисам когда отделим Платформу

    DISCOVERY = 1
    CONDUCTOR_CACHE = 2
    BROKERS_RECONNECT = 3
    TVM_KEYS = 4
    TVM_2_0_KEYS = 5
    TVM_2_0_SERVICE_TICKETS = 6
    HBF_CACHE = 7
