# -*- coding: utf-8 -*-

import os
import random
import socket
import time

import mpfs.engine.process

from mpfs.common.errors import RabbitMQCacheFileNotFoundError, RabbitMQCacheBrokenDataError
from mpfs.common.util.persistent_dict import PersistentDict
from mpfs.config import settings
from mpfs.core.services.conductor_service import ConductorService


QUEUE2_HOST_DATACENTER = settings.queue2['host_datacenter']
QUEUE2_RABBITMQ_HOSTS = settings.queue2['rabbitmq']['hosts']
QUEUE2_RABBITMQ_CACHE = settings.queue2['rabbitmq']['cache']
QUEUE2_RABBITMQ_RELOAD_CACHE_FROM_CONDUCTOR_ON_START = \
    settings.queue2['rabbitmq']['reload_cache_from_conductor_on_start']
QUEUE2_RABBITMQ_CONDUCTOR_QUERY_MAX_DELAY_MS = \
    settings.queue2['rabbitmq']['conductor_query_max_delay_ms']


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class BrokersConductorCache(object):
    def __init__(self, path=QUEUE2_RABBITMQ_CACHE, fmt='json'):
        self._path = path
        self._format = fmt
        self._current_hostname = socket.gethostname()
        self._broker_hosts = []
        self._current_datacenter = None
        self._conductor_service = ConductorService()

    def _get_brokers(self):
        brokers = []
        for item in QUEUE2_RABBITMQ_HOSTS:
            if item == 'localhost':
                brokers.append((item, self._current_datacenter))  # только для development'а
            elif item.startswith('%'):
                group_name = item[1:]
                result = self._conductor_service.get_group(group_name, ['fqdn', 'root_datacenter_name'])
                brokers.extend([(i['fqdn'], i['root_datacenter_name']) for i in result])
            else:
                info = self._conductor_service.get_host_by_fqdn(item)
                brokers.append((info['fqdn'], info['root_datacenter']))
        return brokers

    def _get_current_host_datacenter_name(self):
        if QUEUE2_HOST_DATACENTER:
            return QUEUE2_HOST_DATACENTER

        info = self._conductor_service.get_host_by_fqdn(self._current_hostname)
        return info['root_datacenter']

    def _read_from_file(self):
        if not os.path.exists(self._path):
            error_log.error('Cache file not found: %s' % self._path)
            raise RabbitMQCacheFileNotFoundError()

        try:
            persistent_dict = PersistentDict(self._path, flag='r',
                                             mode=None, format=self._format)
            return ([(item['fqdn'], item['dc']) for item in persistent_dict['rabbitmq_hosts']],
                    persistent_dict['current_dc'])
        except Exception:
            error_log.exception('Invalid cache data')
            raise RabbitMQCacheBrokenDataError()

    def _dump_to_file(self, rabbitmq_hosts, current_host_datacenter_name):
        try:
            persistent_dict = PersistentDict(self._path, flag='c', mode=None, format=self._format)
        except ValueError:
            error_log.error('Cache file format error, rewriting it: %s' % self._path)
            persistent_dict = PersistentDict(self._path, flag='n', mode=None, format=self._format)

        persistent_dict['rabbitmq_hosts'] = [
            {'fqdn': fqdn, 'dc': datacenter} for fqdn, datacenter in rabbitmq_hosts
        ]
        persistent_dict['current_dc'] = current_host_datacenter_name

        try:
            persistent_dict.close()
        except IOError, e:
            error_log.error('Cannot create file %s in cache folder (%s), cache was not updated.' % (self._path, e))

    def load(self):
        if QUEUE2_RABBITMQ_RELOAD_CACHE_FROM_CONDUCTOR_ON_START:
            delay = random.randint(0, QUEUE2_RABBITMQ_CONDUCTOR_QUERY_MAX_DELAY_MS)
            time.sleep(delay / 1000.0)

            try:
                self._current_datacenter = self._get_current_host_datacenter_name()
                self._broker_hosts = self._get_brokers()
            except Exception as e:
                error_log.exception('Error while fetching hosts from conductor: %s' % e)
                self._broker_hosts, self._current_datacenter = self._read_from_file()
            else:
                self._dump_to_file(self._broker_hosts, self._current_datacenter)
        else:
            self._broker_hosts, self._current_datacenter = self._read_from_file()

    def get_rabbitmq_hosts(self, filter_by_current_datacenter=True):
        if filter_by_current_datacenter:
            filtered_hosts = [host for host, dc in self._broker_hosts if dc == self._current_datacenter]
            if len(filtered_hosts):
                return filtered_hosts

        return [host for host, dc in self._broker_hosts]
