# -*- coding: utf-8 -*-

from __future__ import absolute_import

import urlparse

import mpfs.engine.process

from celery import current_app
from celery.worker.control import Panel


@Panel.register
def reset_connection(state, **kwargs):
    from mpfs.engine.queue2.utils import get_amqp_url_by_fqdn

    worker_host = state.hostname
    broker_host = None
    if state.consumer.connection is not None:
        broker_host = state.consumer.connection.hostname

    new_broker_host = kwargs.get('new_broker', None)
    if not new_broker_host:
        return {'error': 'no broker specified'}

    if broker_host == new_broker_host:
        return {'ok': 'already connected'}

    log = mpfs.engine.process.get_default_log()
    log.info('Worker %s received signal to restart connection with args %s (now connected to %s)' %
             (worker_host, kwargs, broker_host))

    broker_urls = state.app.conf.BROKER_URL
    new_broker_url = get_amqp_url_by_fqdn(new_broker_host)

    if broker_urls is not None:
        if isinstance(broker_urls, list):
            if new_broker_url in broker_urls:
                broker_urls.remove(new_broker_url)
            broker_urls.insert(0, new_broker_url)
        elif isinstance(broker_urls, basestring):
            if new_broker_url != broker_urls:
                broker_urls = [new_broker_url, broker_urls]
        else:
            raise TypeError('BROKER_URL is not list and string, %s received' % type(broker_urls))
    else:
        raise RuntimeError('BROKER_URL is None, but reset_connection received')

    # сохраняем полученный хост в список допустимых хостов на случай обрыва соединения с новым брокером
    # или если мы не сможем к нему подключиться именно сейчас
    state.app.conf.BROKER_URL = broker_urls

    # а теперь переключаемся на него и сбрасываем соединение (новое переустановится автоматически)
    try:
        state.consumer.connection.switch(new_broker_url)
        state.consumer.connection.close()
    except Exception as e:
        log.exception('Switch to new connection failed (%s)' % e)

    return {'ok': 'connection reset'}


@Panel.register
def set_pool(state, pool_size=1, **kwargs):
    delta = pool_size - state.consumer.pool._pool.target_processes
    state.consumer.pool.set_pool_size(pool_size)
    if delta:
        state.consumer._update_prefetch_count(delta)
    return {'ok': 'pool size will be set to %d' % pool_size}


@Panel.register
def active_queues(state):
    """Return information about the queues a worker consumes from."""

    source = (state.consumer.task_consumer.queues
              if state.consumer.pool._pool.is_consuming else
              state.consumer.pool._pool.suspended_queues)
    if state.consumer.task_consumer:
        return [dict(queue.as_dict(recurse=True))
                for queue in source]
    return []


def reconnect_uwsgi_worker(broker_url, timeout):
    log = mpfs.engine.process.get_default_log()
    try:
        broker_connection_pool = current_app.pool
        with broker_connection_pool.acquire(block=True, timeout=timeout) as connection:
            parsed_url = urlparse.urlparse(broker_url)
            host = parsed_url.hostname

            log.info('Received request to reconnect to %s (connected to %s)' % (host, connection.hostname))

            if connection.hostname == host:
                log.info('Already connected to %s' % broker_url)
                return

            connection.switch(broker_url)
    except Exception as e:
        error_log = mpfs.engine.process.get_error_log()
        error_log.exception('Error while trying to switch worker connection: %s' % e)
