# -*- coding: utf-8 -*-
"""
Модуль для логирования метрик тасков
"""

import mpfs.common.errors
import mpfs.common.errors.share
import mpfs.engine.process
import mpfs.engine.queue2.celery
from mpfs.common.static import messages

logger = mpfs.engine.process.get_default_log()
LOG_TEMPLATE = (
    'Task %(task_id)s %(effective_status)s (try %(task_retries)s), '
    'name: %(task_type)s (%(oper_type)s, %(oper_subtype)s) '
    '(processed: %(process_time)s sec, lifetime: %(lifetime)s sec), task_status: %(task_status)s, '
    'oper_state: %(oper_state)s, oper_title: %(oper_title)s, oper_id: %(oper_id)s')


class OperationJobBinding(object):
    """
    Объект для связывания job-ы с выполняемой операцией

    Писалось для нужд логирования.
    Убрать, когда появится разделение job-type для разных типов операций.

    Работаем так:
        1. В начале работы таска делаем `set_job`
        2. Когда инициализируется операция делаем `bind_operation`
        3. В конце обработки в джобе получаем операцию `get_operation`
    """
    _job = None
    _operation = None

    @classmethod
    def _clear(cls):
        cls._job = None
        cls._operation = None

    @classmethod
    def set_job(cls, job):
        cls._clear()
        if isinstance(job, mpfs.engine.queue2.celery.BaseTask):
            if job.name.endswith('operation'):
                cls._job = job

    @classmethod
    def bind_operation(cls, operation):
        if cls._job is None:
            return
        cls._operation = operation

    @classmethod
    def get_operation(cls):
        if cls._job is not None:
            return cls._operation


class EffectiveStatus(object):
    """
    Содержит логику вычисления эффективного статуса
    """
    OK = 'OK'
    FAIL = 'FAIL'
    OPER_OK_TITLES = ('COMPLETED', 'DONE', 'EXECUTING')
    # Эти ошибки мы считаем корректными при завершении операции
    # В эффективный статус пишем OK, даже если сама операция FAILED
    white_list_oper_exceptions = (
        mpfs.common.errors.CopyNotFound,
        mpfs.common.errors.CopyParentNotFound,
        mpfs.common.errors.CopySame,
        mpfs.common.errors.CopyTargetExists,
        mpfs.common.errors.HidBlocked,
        mpfs.common.errors.MkdirFolderAlreadyExist,
        mpfs.common.errors.MkdirNotFound,
        mpfs.common.errors.MoveSame,
        mpfs.common.errors.NoFreeSpace,
        mpfs.common.errors.OwnerHasNoFreeSpace,
        mpfs.common.errors.NoFreeSpaceCopyToDisk,
        mpfs.common.errors.NoFreeSpaceMoveFromShared,
        mpfs.common.errors.ResourceLocked,
        mpfs.common.errors.ResourceLockFailed,
        mpfs.common.errors.ResourceNotFound,
        mpfs.common.errors.RmNotFound,
        mpfs.common.errors.share.GroupNoPermit,
    )

    @classmethod
    def _is_operation_exception_expected(cls, operation):
        try:
            # в `exception` лежит результат `sys.exc_info()`
            exception_class = operation.data['error']['exception'][0]
            return exception_class in cls.white_list_oper_exceptions
        except Exception:
            return False

    @classmethod
    def get(cls, task_status, operation, oper_title):
        if task_status != 'OK':
            return cls.FAIL

        if operation:
            if oper_title in cls.OPER_OK_TITLES:
                return cls.OK
            elif cls._is_operation_exception_expected(operation):
                return cls.OK
            else:
                return cls.FAIL
        return cls.OK


def log_metrics(task_id=None, task_type=None, task_status=None, task_retries=None, process_time=None, lifetime=None):
    log_data = {
        'task_id': task_id,
        'task_type': task_type,
        'task_status': task_status,
        'task_retries': task_retries,
        'process_time': '%.2f' % process_time if isinstance(process_time, float) else '%s' % process_time,
        'lifetime': '%.2f' % lifetime if isinstance(lifetime, float) else '%s' % lifetime,
    }
    operation = OperationJobBinding.get_operation()
    if operation:
        log_data['oper_id'] = operation.id
        log_data['oper_uid'] = operation.uid
        log_data['oper_type'] = getattr(operation, 'type', None)
        log_data['oper_subtype'] = getattr(operation, 'subtype', None)
        log_data['oper_state'] = getattr(operation, 'state', None)
        log_data['oper_title'] = messages.true_operation_titles.get(log_data['oper_state'])
    else:
        oper_keys = (
            'oper_id',
            'oper_uid',
            'oper_type',
            'oper_subtype',
            'oper_state',
            'oper_title',
        )
        for oper_key in oper_keys:
            log_data[oper_key] = None

    log_data['effective_status'] = EffectiveStatus.get(task_status, operation, log_data['oper_title'])

    for key, value in log_data.items():
        if value is None:
            log_data[key] = '-'

    logger.info(LOG_TEMPLATE % log_data)
