# -*- coding: utf-8 -*-
import mpfs.engine.process
from mpfs.common.static.tags import ROOT_DATACENTER

from mpfs.engine.process import Signal
from mpfs.engine.queue2.brokers_cache import BrokersConductorCache
from mpfs.config import settings
from mpfs.core.services.conductor_service import ConductorService
from mpfs.core.signals import register_signal, add_timer, Target
from mpfs.metastorage.postgres.services import Sharpei


QUEUE2_MPFS_BALANCER_HOSTNAME = settings.queue2['mpfs_balancer_hostname']
QUEUE2_RABBITMQ = settings.queue2['rabbitmq']
QUEUE2_PUT_TASKS_IN_NEW_QUEUE = settings.queue2['put_tasks_in_new_queue']
QUEUE2_MPFS_AUTORECONNECTOR_PERIOD = settings.queue2['mpfs']['autoreconnector']['period']
QUEUE2_MPFS_AUTORECONNECTOR_TIMEOUT = settings.queue2['mpfs']['autoreconnector']['timeout']


_broker_reconnector = None

try:
    # автоматически добавляется мастер процессом
    import uwsgi
    UWSGI = True
except ImportError:
    UWSGI = False


def get_rabbitmq_hosts():
    # celery по умолчанию берет первый хост из списка и если он недоступен,
    # подключается в соответствии с BROKER_FAILOVER_STRATEGY,
    # поэтому чтобы у нас подключались к разным хостам, перемешиваем этот список предварительно
    # список стратегий можно посмотреть тут:
    # http://docs.celeryproject.org/projects/kombu/en/latest/reference/kombu.connection.html#kombu.connection.Connection.failover_strategy
    cache = BrokersConductorCache()
    cache.load()
    hosts = cache.get_rabbitmq_hosts()
    return hosts


def get_callback_host():
    current_hostname = mpfs.engine.process.hostname()
    if UWSGI:
        if QUEUE2_MPFS_BALANCER_HOSTNAME == 'localhost' or QUEUE2_MPFS_BALANCER_HOSTNAME is None:
            return current_hostname
        return QUEUE2_MPFS_BALANCER_HOSTNAME
    else:
        if QUEUE2_MPFS_BALANCER_HOSTNAME is None:
            async_task_uwsgi_submitter = mpfs.engine.process.get_async_task_uwsgi_submitter()
            if async_task_uwsgi_submitter:
                return async_task_uwsgi_submitter

        if QUEUE2_MPFS_BALANCER_HOSTNAME == 'localhost':
            return current_hostname
        return QUEUE2_MPFS_BALANCER_HOSTNAME


def get_amqp_url_by_fqdn(fqdn):
    """
    Возвращает url для подключения celery к брокеру, на котором установлен rabbitmq
    :param fqdn: Имя хоста с rabbitmq
    :rtype: str
    :return: Строка вида amqp://user:password@host:port/vhost, где host - параметр fqdn, с которым позвали функцию.
    Имя пользователя, пароль, порт и vhost получает из настроек - секция queue2->rabbitmq в файле global_settings.yaml
    """
    # Celery URL должен быть такой:
    #   amqp://<rabbitmq_user_name>:<rabbitmq_user_password>@<rabbitmq_host>/<rabbitmq_vhost>
    # Например:
    #   amqp://mpfs_test:mpfs_test@queue01f.dst.yandex.net:5672/disk_test_queller
    broker_url_template = 'amqp://%(user)s:%(password)s@%%(host)s:%(port)s/%(vhost)s' % QUEUE2_RABBITMQ
    broker_url = broker_url_template % {'host': fqdn}
    return broker_url


def get_broker_urls():
    if not QUEUE2_PUT_TASKS_IN_NEW_QUEUE:
        return None
    return [get_amqp_url_by_fqdn(host) for host in get_rabbitmq_hosts()]


def reconnect_uwsgi_worker_to_default_broker():
    from mpfs.engine.queue2.control import reconnect_uwsgi_worker  # циклический импорт

    broker_urls = mpfs.engine.process.rabbitmq_hosts(shuffle=False)
    if not broker_urls:
        mpfs.engine.process.get_error_log().error('Empty broker urls list')
        return

    uwsgi_worker_id = int(uwsgi.worker_id())  # вообще оно int, но если вдруг перестанет быть, то лучше пусть сломается
    # количество брокеров меньше количества воркеров, поэтому имитируем тут равномерное распределение воркеров по кроликам:
    uwsgi_worker_broker_url = broker_urls[uwsgi_worker_id % len(broker_urls)]

    reconnect_uwsgi_worker(uwsgi_worker_broker_url, QUEUE2_MPFS_AUTORECONNECTOR_TIMEOUT)


def setup_brokers_reconnect_timer():
    global _broker_reconnector

    if not mpfs.engine.process.is_uwsgi_process():
        raise RuntimeError('Setup_brokers_reconnect_timer should be called only from uwsgi process')

    _broker_reconnector = BrokerReconnectWrapper()
    register_after_fork = mpfs.engine.process.get_register_after_fork_impl()
    register_after_fork(_broker_reconnector, BrokerReconnectWrapper.reconnect)  # устанавливаем соединение к своему кролику в каждом uwsgi воркере после fork'а

    register_signal(
        Signal.BROKERS_RECONNECT,
        lambda signum: reconnect_uwsgi_worker_to_default_broker(),
        target=Target.ALL_WORKERS
    )
    add_timer(Signal.BROKERS_RECONNECT, QUEUE2_MPFS_AUTORECONNECTOR_PERIOD)  # устанавливаем периодический обработчик


class BrokerReconnectWrapper(object):
    """
    Класс для того, чтобы воспользоваться функцией register_after_fork,
    она требует долгоживущий инстанс этого класса, потому что держит weakref на него
    """
    def reconnect(self):
        reconnect_uwsgi_worker_to_default_broker()


def get_master_dc(uid):
    if uid is None:
        return

    try:
        master = Sharpei().get_shard(uid).get_master()
        dc = ConductorService().get_host_by_fqdn(master.host)[ROOT_DATACENTER]
    except:
        return

    return dc


def get_task_func_name(full_name):
    return full_name.rsplit('.', 1)[-1]
