# -*- coding: utf-8 -*-
"""

MPFS
API

"""
import traceback
import math

from time import time

import mpfs.engine.process

from mpfs.common.util.credential_sanitizer import CredentialSanitizer
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.common.util.ycrid_parser import YcridPlatformPrefix, YcridParser
from mpfs.core.degradations import check_degradation_by_path, check_degradation_by_client
from mpfs.core.zookeeper.hooks import update_settings_in_current_worker
from mpfs.engine.process import (
    get_global_request,
    set_global_request,
    get_access_log,
    get_default_log,
    get_error_log,
    set_external_tvm_ticket,
    share_user
)

from mpfs.config import settings
from mpfs.common.util import generator, filter_uid_by_percentage, parameters_checker
from mpfs.common import errors
from mpfs.core.services.tvm_service import TVMTicket
from mpfs.core.social.share import Group
from mpfs.core.user.base import User
from mpfs.core.social.publicator import Publicator
from mpfs.engine.process import set_tvm_2_0_user_ticket
from mpfs.frontend import request
from mpfs.frontend.api.auth import NetworkAuthorization, Auth
from mpfs.frontend.formatter import default_formatter
from mpfs.core.services.rate_limiter_service import rate_limiter

log = get_access_log()
dlog = get_default_log()
error_log = get_error_log()

usrctl = mpfs.engine.process.usrctl()

USER_DEFAULT_TLD = settings.user['default_tld']
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']
SYSTEM_VALID_TOP_LEVEL_DOMAINS = settings.system['valid_top_level_domains']
SYSTEM_SYSTEM_OLD_SHARE_UID = settings.system['system']['old_share_uid']
RATE_LIMITER_LIMIT_BY_UID = settings.rate_limiter['limit_by_uid']
RATE_LIMITER_LIMIT_BY_UID_UIDS = set(RATE_LIMITER_LIMIT_BY_UID['uids'])
AUTH_NETWORK_AUTHORIZATION_ENABLED = settings.auth['network_authorization']['enabled']
FEATURE_TOGGLES_DISABLE_API_FOR_OVERDRAFT_PERCENTAGE = settings.feature_toggles['disable_api_for_overdraft_percentage']
PLATFORM_DISABLE_FOR_OVERDRAFT_ALWAYS_ALLOW_HANDLERS = settings.platform['disable_for_overdraft'][
    'always_allow_handlers']


class API(object):
    req_class = request.UserRequest

    client = None

    _uid_field_names = ('uid', 'owner_uid', 'user_uid')
    _special_uids = {'0', mpfs.engine.process.share_user()}

    def __init__(self):
        self.formatter_obj = self.formatter()

    @property
    def req(self):
        return get_global_request()

    @req.setter
    def req(self, value):
        set_global_request(value)

    def setup(self, params):
        self.req = self.req_class(params)
        self.params = params

        self.req.set_id()
        # Адский костыль для стораджа уровня реквеста
        Group.reset_storage()

        cloud_req_id = self.req.request_headers.get('Yandex-Cloud-Request-ID') or generator.cloud_request_id()

        if self.req.request_headers.get('Ticket'):
            raw_tvm_ticket = self.req.request_headers.get('Ticket')
            tvm_ticket = TVMTicket.build_tvm_ticket(raw_tvm_ticket)
            set_external_tvm_ticket(tvm_ticket)
        else:
            set_external_tvm_ticket(None)

        self.req.set_cloud_id(cloud_req_id)
        self.user_ip = self.req.request_headers.get('X-Real-IP')

        self._set_meta()
        self._set_tld()
        self._set_show_nda()
        self._set_skip_overdraft_check()

        # Костыль, чтобы не разломать листинги. Подробности тут:
        # https://st.yandex-team.ru/CHEMODAN-29351#1459849438000
        # Убрать эту строчку когда:
        #  1. Уберем из Платформы запросы с этим параметром
        #  2. Попросим Верстку (@vitkarpov) не передавать нам этот параметр.
        self.params.pop('decode_custom_properties', None)

        for p in ('path', 'src', 'dst'):
            if p in self.params and not isinstance(self.params[p], unicode):
                self.params[p] = self.params[p].decode('utf-8')

    def get_timeout(self):
        timeouts = settings.fcgi['response_timeout']
        return timeouts.get(self.current_method, timeouts['default'])

    def process(self, method_name):
        start_time = time()
        result = None

        formatter = self.formatter_obj
        formatter.setup(request=self.req)
        self.current_method = method_name
        self.req.method_name = method_name
        update_settings_in_current_worker()
        try:
            # Устанавливаем юзера здесь, а не в setup, чтобы ошибки при походе в базу обрабатывались в общем except-блок
            self._set_user()
            check_degradation_by_path(self.req, self.params)
            self._check_user_rate_limit()
            self._check_overdraft_restrictions(method_name)

            # устанавливаем tvm тикет здесь, а не в setup, чтобы исключения, кидающиеся при проверке, обработались в
            # общем except-блоке
            set_tvm_2_0_user_ticket(None)

            if AUTH_NETWORK_AUTHORIZATION_ENABLED:
                NetworkAuthorization.authorize(self.req, self.params)

            Auth.authorize(self.req, method_name, self.params)
            check_degradation_by_client(self.req, self.params)
            self._check_arguments()

            try:
                apply_method = getattr(self, method_name)
            except AttributeError:
                raise errors.MPFSNotImplemented()

            apply_method()
            result = formatter.process(method_name)
        except Exception as e:
            self.req.set_error(e)
            result = formatter.error(e)
            if not isinstance(e, errors.MPFSRootException) or e.log:
                error_log.error(traceback.format_exc())
            else:
                message = " ".join(traceback.format_exc().splitlines()[-3:])
                dlog.error(message)
                error_log.debug(traceback.format_exc())
        finally:
            try:
                mpfs.engine.process.reset_connections()
            except Exception as e:
                error_log.error(traceback.format_exc())

        uid = None
        if hasattr(self.req, 'user') and self.req.user and self.req.user.uid:
            uid = self.req.user.uid
        headers_list = CredentialSanitizer.get_headers_list_with_sanitized_credentials(self.req.http_req.headers)
        s = ' '.join(['"%s: %s"' % (k, v) for k, v in headers_list])
        log.info('', extra={
            'method': self.req.http_req.environ.get('REQUEST_METHOD'),
            'uri': self.req.http_req.environ.get('REQUEST_URI'),
            'uid': uid,
            'proto': self.req.http_req.environ.get('SERVER_PROTOCOL'),
            'status': self.req.http_resp.status or 200,  # bullshit
            'request_time': time() - start_time,
            'headers': s
        })
        return result

    def compose_args(self, keys):
        args = {}
        for k in keys:
            try:
                if k[2]:
                    x = self.params[k[1]]
                    if isinstance(x, unicode):
                        x = x.encode('utf-8')
                    val = k[2](x)

                    if k[2] is float and (math.isnan(val) or math.isinf(val)):
                        raise ValueError()
                else:
                    val = self.params[k[1]]
            except KeyError:
                try:
                    val = k[3]
                except IndexError:
                    raise ValueError("parameter '%s' must be defined" % k[0])
            args[k[0]] = val
        return args

    def _process(self, obj, method_name, args=None, keys=()):
        args = args or {}
        if keys:
            args.update(self.compose_args(keys))
        self.req.set_args(args)

        from mpfs.common.util import timeout
        try:
            obj_method = getattr(obj, method_name)
        except AttributeError:
            raise errors.MPFSNotImplemented()
        else:

            if not self._user_init_in_progress_allowed_for_method(obj_method) and self._is_users_init_in_progress():
                raise errors.StorageInitUser()

            self.req.method_name = self.current_method
            if settings.feature_toggles['use_threaded_timeout']:
                data = timeout(
                    obj_method, args=(self.req,),
                    timeout_duration=self.get_timeout()
                )
            else:
                data = obj_method(self.req)
            self.req.set_result(data)

    def _check_arguments(self):
        for param in ['path', 'src', 'dst']:
            if param not in self.params:
                continue

            try:
                parameters_checker.check_path_forbidden_symbols(self.params[param])
            except ValueError as ve:
                error_message = ve.message + ' request: uid=%(uid)s, parameter=%(p_name)s'\
                                % {'uid': self.params['uid'],
                                   'p_name': param}
                raise errors.UrlPathError(error_message)

    def _set_meta(self):
        """Установить опциональный параметр meta.
        """
        meta = self.params.pop('meta', None)
        if meta is not None:
            meta = filter(None, meta.split(','))
        self.req.meta = meta

    def _set_tld(self):
        """Установить опциональный параметр tld.
        """
        tld = self.params.pop('tld', USER_DEFAULT_TLD) or USER_DEFAULT_TLD
        # если указан и пустой, то все равно брать из USER_DEFAULT_TLD

        self.req.real_tld = tld  # оригиальный переданный tld из запроса без изменений, если передан
        if tld not in SYSTEM_VALID_TOP_LEVEL_DOMAINS:
            tld = USER_DEFAULT_TLD
        self.req.tld = tld

    def _set_show_nda(self):
        """Установить опциональный праметр show_nda, фильтрующий YaTeam'ную папку.
        Возможные значения:
            1 - deprecated. Приравниваем к web.
            web - запрос из вёрстки
            kladun - запрос из кладуна
        """
        show_nda = self.params.pop('show_nda', None)
        if show_nda == '1' or show_nda == 1:
            self.req.show_nda = 'web'
        elif show_nda == 'web' or show_nda == 'kladun':
            self.req.show_nda = show_nda
        else:
            self.req.show_nda = None

    def _set_skip_overdraft_check(self):
        skip_overdraft_check = self.params.pop('skip_overdraft_check', None)
        if skip_overdraft_check == '1' or skip_overdraft_check == 1:
            self.req.skip_overdraft_check = True
        else:
            self.req.skip_overdraft_check = False

    def _set_user(self):
        """
        Установить юзера из базы в req, если в ручку передан uid.
        Если пользователя в базе не нашли или он есть, но недоинициализирован, то установить None.
        """
        uid = self.params.get('uid', None)
        # чтобы не переделывать клиентов, которые приходят с uid=share_production, мы подставляем правильный uid тут
        if uid and uid == SYSTEM_SYSTEM_OLD_SHARE_UID:
            uid = share_user()
            self.params['uid'] = share_user()
        private_hash = self.params.get('private_hash', None)
        user = None
        symlink_addr = None
        experiment_manager.update_context(uid=uid)

        if uid:
            try:
                user = User(uid)
            except (errors.StorageInitUser, errors.StorageEmptyDiskInfo):
                user = None
        if private_hash:
            try:
                _, symlink_addr = Publicator.parse_private_hash(private_hash)
            except Exception:
                pass

        self.req.user = user
        self.req.symlink_addr = symlink_addr

    def _check_user_rate_limit(self):
        if not RATE_LIMITER_LIMIT_BY_UID['enable']:
            return
        if not RATE_LIMITER_LIMIT_BY_UID_UIDS:
            return

        uid = None
        group_name = None
        try:
            uid = self.req.user.uid
        except AttributeError:
            try:
                uid = self.req.symlink_addr.uid
            except AttributeError:
                pass
            else:
                group_name = RATE_LIMITER_LIMIT_BY_UID['public_group_name']
        else:
            group_name = RATE_LIMITER_LIMIT_BY_UID['common_group_name']
        if uid is None:
            return

        uid = str(uid)
        if uid not in RATE_LIMITER_LIMIT_BY_UID_UIDS:
            return

        rate_limiter.check_limit_exceeded(group_name, uid)

    def _check_overdraft_restrictions(self, handler_name):
        if self.client in ('billing', 'service'):
            return

        platform_prefix = YcridParser.get_platform(mpfs.engine.process.get_cloud_req_id())
        if platform_prefix not in (YcridPlatformPrefix.DAV,
                                   YcridPlatformPrefix.REST):
            return

        user_agent = self.req.request_headers.get('user-agent', None)
        if (UserAgentParser.is_yandex_search_mobile(user_agent) or
            UserAgentParser.is_yandex_mail_mobile(user_agent) or
            UserAgentParser.is_yandex_notes(user_agent)):
            return

        if handler_name in PLATFORM_DISABLE_FOR_OVERDRAFT_ALWAYS_ALLOW_HANDLERS:
            return

        if self.req.skip_overdraft_check:
            return

        if (self.req.user and
            self.req.user.uid and
            filter_uid_by_percentage(self.req.user.uid, FEATURE_TOGGLES_DISABLE_API_FOR_OVERDRAFT_PERCENTAGE) and
            self.req.user.is_in_overdraft_for_restrictions()):
            raise errors.APIRestrictedForOverdraftUserError()

    @staticmethod
    def _user_init_in_progress_allowed_for_method(method_obj):
        return getattr(method_obj, '_allow_user_init_in_progress', False)

    def _is_users_init_in_progress(self):
        for uid_field_name in self._uid_field_names:
            uid = getattr(self.req, uid_field_name, None)
            if uid and uid not in self._special_uids and usrctl.user_init_in_progress(uid):
                return True
        return False


class Default(API):
    '''
    Дефолтное клиентское API
    Выдает результаты в структурах питона

    Общая схема работы:
    - принять и проверить параметры,
    - вызвать нужный объект,
    - получить результат
    - преобразовать результат в нужный вид
    '''

    client = 'default'
    formatter = default_formatter.Default

    def process_core_method(self, method_name, args=None, keys=()):
        args = args or {}
        self._process(self.core, method_name, args, keys)

    def process_invite_method(self, method_name, args=None, keys=()):
        args = args or {}
        self._process(self.invite, method_name, args, keys)
