# -*- coding: utf-8 -*-
"""
Механизмы авторизации MPFS.
"""

from collections import defaultdict
from ticket_parser2.api.v1.exceptions import TvmException

from mpfs.common import errors
from mpfs.common.errors import MPFSError, AuthorizationError
from mpfs.common.static.tags import LOCALHOST_IPV4, LOCALHOST_IPV6
from mpfs.common.static.tags.conf_sections import USER_TICKET_POLICY, NO_CHECKS, STRICT
from mpfs.common.static.tags.tvm import TVM2_SILENT_MODE_LOG_PREFIX, HandlersGroupsNames
from mpfs.common.util.iptools import IPRangeList
from mpfs.config import settings
from mpfs.core.services.hbf_service import HbfService
from mpfs.engine.process import (
    set_tvm_2_0_user_ticket,
    get_authorization_networks,
    get_error_log,
    get_tvm_2_0_clients,
    get_default_log,
    get_handlers_groups,
)
from mpfs.core.services.tvm_2_0_service import (
    TVM2Ticket,
    tvm2,
)


error_log = get_error_log()
default_log = get_default_log()

SERVICES_TVM_2_0_ENABLED = None
AUTH_BYPASS_AUTH_HANDLERS = None
AUTH_TVM_2_0_SILENT_MODE = None
AUTH_TVM_2_0_SILENT_MODE_ADDRESSES = None
AUTH_TVM_2_0_SILENT_MODE_NETWORKS = None
AUTH_TVM_2_0_SILENT_MODE_ADDRESSES_IP_RANGE = None


def reload_settings():
    global SERVICES_TVM_2_0_ENABLED
    global AUTH_BYPASS_AUTH_HANDLERS
    global AUTH_TVM_2_0_SILENT_MODE
    global AUTH_TVM_2_0_SILENT_MODE_ADDRESSES
    global AUTH_TVM_2_0_SILENT_MODE_NETWORKS
    global AUTH_TVM_2_0_SILENT_MODE_ADDRESSES_IP_RANGE

    SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']
    AUTH_BYPASS_AUTH_HANDLERS = settings.auth['bypass_auth_handlers']
    AUTH_TVM_2_0_SILENT_MODE = settings.auth['tvm_2_0']['silent_mode']['global']
    AUTH_TVM_2_0_SILENT_MODE_ADDRESSES = IPRangeList(*settings.auth['tvm_2_0']['silent_mode']['addresses'])
    AUTH_TVM_2_0_SILENT_MODE_NETWORKS = list(settings.auth['tvm_2_0']['silent_mode']['networks'])


reload_settings()


def _raise_if_not_tvm_checks_silent_mode(exc, ip=None, handler=None):
    silent_mode = AUTH_TVM_2_0_SILENT_MODE

    if not silent_mode and ip:
        silent_mode = ip in AUTH_TVM_2_0_SILENT_MODE_ADDRESSES

        i = 0
        while not silent_mode and i < len(AUTH_TVM_2_0_SILENT_MODE_NETWORKS):
            silent_mode = ip in HbfService().networks.get(AUTH_TVM_2_0_SILENT_MODE_NETWORKS[i], [])
            i += 1

    if silent_mode:
        error_log.error('%s: %s' % (TVM2_SILENT_MODE_LOG_PREFIX, exc.message))
    else:
        raise exc


class Auth(object):
    BYPASS_AUTH_FOR_ADDRS = [LOCALHOST_IPV4, LOCALHOST_IPV6]

    @staticmethod
    def authorize(request, method, url_params=None):
        """Авторизация в MPFS.

        Без авторизации пропускаем:
          1. Запросы с localhost
          2. Запросы в ручки version, ping
        """
        if request.http_req.remote_addr in Auth.BYPASS_AUTH_FOR_ADDRS:
            return

        if method in AUTH_BYPASS_AUTH_HANDLERS:
            return

        handlers_groups = get_handlers_groups()
        group = handlers_groups.get(method, HandlersGroupsNames.GENERAL)

        try:
            TVM2.authorize(request, restricted_handlers_group=group, url_params=url_params)
        except MPFSError as e:
            error_log.error(e.message)
            raise AuthorizationError()


class TVM2(object):
    @staticmethod
    def authorize(request, url_params=None, restricted_handlers_group=None):
        if not SERVICES_TVM_2_0_ENABLED:
            return
        tvm_service_ticket = TVM2._check_tvm_2_0_service_ticket(request)
        tvm_user_ticket = TVM2._check_and_set_tvm_2_0_user_ticket(request)

        tvm_src_client_id = tvm_service_ticket.src if tvm_service_ticket is not None else None

        if restricted_handlers_group is not None:
            TVM2._check_tvm_2_0_restricted_handlers(tvm_src_client_id, restricted_handlers_group)

        client = get_tvm_2_0_clients().get(tvm_src_client_id)
        request.tvm_2_0_service_client = client
        if client is None or not hasattr(client, 'tvm_2_0'):
            return
        user_ticket_policy = client.tvm_2_0.get(USER_TICKET_POLICY, NO_CHECKS)
        if user_ticket_policy == NO_CHECKS:
            return
        if tvm_user_ticket is None:
            default_log.info('User ticket\'s required but it\'s missing or wasn\'t validated')
            if user_ticket_policy == STRICT:
                _raise_if_not_tvm_checks_silent_mode(errors.AuthorizationError(),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
            return
        uid = None
        if hasattr(request, 'user') and request.user and request.user.uid:
            uid = request.user.uid

        # user can be legally not initialized in case it will be initialized in handler method.
        # For example see /billing/service_create
        if not uid and url_params and 'uid' in url_params:
            uid = url_params['uid']

        user_ticket_uids = {str(uid)
                            for uid in tvm_user_ticket.uids}
        if str(uid) not in user_ticket_uids:
            default_log.info('Uids from ticket and request are different: ticket_uid=%s req_uid=%s'
                             % (tvm_user_ticket.default_uid, uid))
            if user_ticket_policy == STRICT:
                _raise_if_not_tvm_checks_silent_mode(errors.AuthorizationError(),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
            return

    @staticmethod
    def set_tvm_2_0_user_ticket_from_request(request):
        """Пробрасывает юзер-тикет во внешнии сервисы, если юзере-тикет был передан."""
        raw_tvm_user_ticket = request.request_headers.get('X-Ya-User-Ticket')
        if not raw_tvm_user_ticket:
            return

        # Передаем дальше ровто то, что получили
        # (без проверки; для проверки и пробрасывания использовать: _check_and_set_tvm_2_0_user_ticket)
        tvm_user_ticket = TVM2Ticket.build_tvm_ticket(raw_tvm_user_ticket)
        set_tvm_2_0_user_ticket(tvm_user_ticket)

    @staticmethod
    def _check_tvm_2_0_service_ticket(request):
        raw_tvm_service_ticket = request.request_headers.get('X-Ya-Service-Ticket')
        if not raw_tvm_service_ticket:
            _raise_if_not_tvm_checks_silent_mode(errors.TVM2InvalidTicketError('No ticket provided'),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
            return

        tvm_service_ticket = TVM2Ticket.build_tvm_ticket(raw_tvm_service_ticket)
        try:
            validated_ticket = tvm2.validate_service_ticket(tvm_service_ticket)
        except TvmException as exc:
            _raise_if_not_tvm_checks_silent_mode(errors.TVM2InvalidTicketError(extra_msg=exc.message),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
        else:
            # Проверяем, что клиент зарегистрирован среди списка клиентов MPFS
            registered_clients = get_tvm_2_0_clients()
            if validated_ticket.src not in registered_clients:
                _raise_if_not_tvm_checks_silent_mode(
                    errors.TVM2ClientNotRegisteredError(extra_msg=str(validated_ticket.src),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
                )
            return validated_ticket

    @staticmethod
    def _check_and_set_tvm_2_0_user_ticket(request):
        raw_tvm_user_ticket = request.request_headers.get('X-Ya-User-Ticket')
        if not raw_tvm_user_ticket:
            return

        tvm_user_ticket = TVM2Ticket.build_tvm_ticket(raw_tvm_user_ticket)
        validated_user_ticket = None
        try:
            validated_user_ticket = tvm2.validate_user_ticket(tvm_user_ticket)
        except TvmException as exc:
            _raise_if_not_tvm_checks_silent_mode(errors.TVM2InvalidTicketError(extra_msg=exc.message),
                                                     handler=request.method_name, ip=request.http_req.remote_addr)
        set_tvm_2_0_user_ticket(tvm_user_ticket)
        return validated_user_ticket

    @staticmethod
    def _check_tvm_2_0_restricted_handlers(client_id, group):
        """Проверяем, что клиенту разрешено использовать указанную группу ручек"""
        registered_clients = get_tvm_2_0_clients()

        handlers_restrictions = settings.auth['tvm_2_0_restricted_handlers']
        disallow_general_handlers = settings.auth['tvm_2_0_disallow_general_handlers']

        if group == HandlersGroupsNames.GENERAL:
            if client_id not in registered_clients or registered_clients[client_id].name not in disallow_general_handlers:
                return

        elif group not in handlers_restrictions:
            return

        if client_id not in registered_clients:
            _raise_if_not_tvm_checks_silent_mode(
                errors.AuthorizationError(extra_msg='unknown client %s' % client_id))
            return

        if group == HandlersGroupsNames.GENERAL or registered_clients[client_id].name not in handlers_restrictions[group]:
            _raise_if_not_tvm_checks_silent_mode(
                errors.AuthorizationError(extra_msg=str(registered_clients[client_id].name)))


class SignAndTs(object):
    @staticmethod
    def authorize(request, url_params=None):
        # TODO: перенести сюда проверку подписи из nginx
        if not isinstance(url_params, dict):
            raise errors.SignAndTsAuthorizationError(extra_msg='no url_params')
        if 'sign' not in url_params:
            raise errors.SignAndTsAuthorizationError(extra_msg='no sign parameter')
        if 'ts' not in url_params:
            raise errors.SignAndTsAuthorizationError(extra_msg='no ts parameter')


class NetworkAuthorization(object):
    allowed_authorization_methods = {
        'tvm_2_0': TVM2,
        'sign_and_ts': SignAndTs,
    }

    @staticmethod
    def collect_authorization_network():
        networks = settings.auth['networks']
        result = defaultdict(dict)
        for name, network in networks.items():
            result[name]['addresses'] = IPRangeList(*network['addresses'])
            result[name]['auth_methods'] = [NetworkAuthorization.allowed_authorization_methods[method_name]
                                            for method_name in network['auth_methods']]
        return result

    @staticmethod
    def authorize(request, url_params=None):
        # Проверяем авторизацию по сети
        remote_addr = request.http_req.remote_addr
        authorization_networks = get_authorization_networks()
        for network in authorization_networks.values():
            if remote_addr in network['addresses']:
                exceptions = []
                for auth_method in network['auth_methods']:
                    try:
                        auth_method.authorize(request, url_params)
                    except errors.MPFSError as e:
                        exceptions.append(e)
                    else:
                        return
                for e in exceptions:
                    error_log.error(e.message)
                raise errors.AuthorizationError()
