# -*- coding: utf-8 -*-
"""

MPFS
API MPFS

"""
import sys, traceback

from time import localtime, strftime

import mpfs.engine.process
import mpfs.common.util.logger as logger
import mpfs.core.base as core
import mpfs.invite as invite
import mpfs.frontend.formatter.disk as disk_formatter
from mpfs.common.errors import DjfsApiProxyNotImplementedError, BadRequestError

from mpfs.config import settings
from mpfs.core.filesystem.resources.photounlim import PhotounlimLastModifiedIterationKey
from mpfs.core.services.djfs_albums import djfs_albums
from mpfs.frontend.api import Default
from mpfs.common.static import messages
from mpfs.common.static import codes
from mpfs.common.static import tags
from mpfs.frontend import request
from mpfs.common import errors
from mpfs.frontend.api import API
from mpfs.core.address import Address
from mpfs.common.util import type_to_bool, from_json, normalized_hash, filter_uid_by_percentage
from mpfs.common.util.filetypes import builtin_extensions
from mpfs.core.user.constants import SHARE_UID, PHOTOUNLIM_AREA_PATH
from mpfs.core.lenta.utils import LentaMediaType
from mpfs.core.services.djfs_api_service import djfs_api
from mpfs.metastorage.postgres.schema import AlbumType

JAVA_DJFS_API_PROXY_MKDIR_ENABLED = settings.java_djfs_api['proxy']['mkdir']['enabled']
JAVA_DJFS_API_PROXY_MKDIR_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['mkdir']['enabled_uids'])
JAVA_DJFS_API_PROXY_MOVE_ENABLED = settings.java_djfs_api['proxy']['move']['enabled']
JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['move']['enabled_uids'])
JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED = settings.java_djfs_api['proxy']['trash_append']['enabled']
JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['trash_append']['enabled_uids'])
JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED = settings.java_djfs_api['proxy']['bulk_info']['enabled']
JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['bulk_info']['enabled_uids'])
JAVA_DJFS_API_PROXY_BULK_INFO_DJFS_SUPPORTED_META = set(settings.java_djfs_api['proxy']['bulk_info']['djfs_supported_meta'])
STORE_CTIME_FROM_CLIENT_ENABLED = settings.feature_toggles['store_ctime_from_client']
JAVA_DJFS_API_PROXY_PUBLIC_LIST_ENABLED = settings.java_djfs_api['proxy']['public_list']['enabled']
JAVA_DJFS_API_PROXY_PUBLIC_LIST_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['public_list']['enabled_uids'])
JAVA_DJFS_API_PROXY_PUBLIC_INFO_ENABLED = settings.java_djfs_api['proxy']['public_info']['enabled']
JAVA_DJFS_API_PROXY_PUBLIC_INFO_ENABLED_UIDS = set(settings.java_djfs_api['proxy']['public_info']['enabled_uids'])
JAVA_DJFS_API_PROXY_INFO_ENABLED = settings.java_djfs_api['proxy']['info']['enabled']
JAVA_DJFS_API_PROXY_INFO_SUPPORTED_META = set(settings.java_djfs_api['proxy']['info']['supported_meta'])
JAVA_DJFS_API_PROXY_INFO_SUPPORTED_ROOT_PATHS = set(settings.java_djfs_api['proxy']['info']['supported_root_paths'])
JAVA_DJFS_API_PROXY_DEFAULT_FOLDERS_ENABLED = settings.java_djfs_api['proxy']['default_folders']['enabled']
JAVA_DJFS_API_PROXY_DEFAULT_FOLDERS_YCRID_PREFIXES = settings.java_djfs_api['proxy']['default_folders']['ycrid_prefixes']
JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS_PERCENT = settings.java_djfs_api['proxy']['move']['enabled_uids_percent']


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def _convert_to_address(uid, path):
    '''
    workaround для миграции на новую схему адресации
    '''
    try:
        address = Address(path)
        if str(address.uid) != str(uid):
            raise errors.PermissionDenied()
        return address.id
    except errors.AddressError:
        return Address.Make(uid, path).id


def _check_social_feature_toggle():
    if not settings.feature_toggles['social_photo']:
        raise errors.MPFSNotImplemented()


class DiskApi(Default):

    core = core
    invite = invite
    formatter = disk_formatter.Default

    def base_args(self, params={}):
        params = params or self.params
        path = self.params.pop('path', '/disk')
        uid = str(self.params.pop('uid'))
        path = _convert_to_address(uid, path)
        connection_id = self.params.pop('connection_id', '')
        return {'uid': uid, 'path': path, 'connection_id': connection_id}

    def modifying_request_parameters(self):
        args = self.base_args()
        args['callback'] = self.params.pop('callback', '')
        return args

    def base_copy_move_args(self, params={}):
        params = params or self.params

        uid = str(self.params.pop('uid'))
        src = self.params.pop('src')
        dst = self.params.pop('dst')
        connection_id = self.params.pop('connection_id', '')
        callback = self.params.pop('callback', '')
        force = int(self.params.pop('force', 0))
        force_djfs_albums_callback = self.params.pop('force_djfs_albums_callback', False)

        src = _convert_to_address(uid, src)
        dst = _convert_to_address(uid, dst)
        return {
            'uid': uid,
            'src': src,
            'dst': dst,
            'force': force,
            'callback': callback,
            'connection_id': connection_id,
            'force_djfs_albums_callback': force_djfs_albums_callback
        }

    def list_subresources(self):
        """Получение листинга директории"""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('mimetypes', 'mimetypes', str, ''),
            ('mediatypes', 'mediatypes', str, ''),
            ('sort', 'sort', str, 'name'),
            ('offset', 'offset', int, 0),
            ('amount', 'amount', int, 10),
            ('order', 'order', int, 1),
            ('extended_response', 'extended_response', int, 0),
        )
        self.process_core_method('list_subresources', keys=keys)

    def dir_size(self):
        """Размер и количество файлов в директории"""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
        )
        self.process_core_method('dir_size', keys=keys)

    def public_dir_size(self):
        """Размер и количество файлов в публичной директории"""
        keys = (
            ('private_hash', 'private_hash', str),
        )
        args = {
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('public_dir_size', keys=keys, args=args)

    def search_sizes_per_mediatype(self):
        """Получение информации о занимаемом дисковом пространстве ресурсами всего диска,
        сгруппированными по медиатипу, отсортированными по размеру."""
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('search_sizes_per_mediatype', keys=keys)

    def search_bulk_info(self):
        """Получение информации о ресурсах из поиска по file_id"""
        keys = (
            ('uid', 'uid', str),
            ('file_ids', 'file_ids', str),
            ('sort', 'sort', str, ''),
            ('order', 'order', int, 1),
            ('search_meta', 'search_meta', str, 'path,file_id'),
        )
        self.process_core_method('search_bulk_info', keys=keys)

    def public_search_bulk_info(self):
        """Получение информации о ресурсах из поиска по file_id и private_hash."""
        keys = (
            ('private_hash', 'private_hash', str),
            ('file_ids', 'file_ids', str),
            ('sort', 'sort', str, ''),
            ('order', 'order', int, 1),
            ('search_meta', 'search_meta', str, 'path,file_id'),
        )
        self.process_core_method('public_search_bulk_info', keys=keys)

    def _do_list(self, core_method_name):
        '''
        Получение листинга директории
        '''
        args = self.base_args(self.params)
        args.update({
            'args' : {
                'bounds' : {
                    'sort'  : self.params.pop('sort', '') or 'name',
                    'order' : int(self.params.pop('order', '') or 1),
                },
            },
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None)
        })
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['args']['bounds'][bound] = int(self.params.pop(bound))
        args['args']['filter'] = self.params
        self.process_core_method(core_method_name, args)
        self.req.form.apply_filters()
        self.req.form.apply_bounds()

    def dir_list(self):
        self._do_list('dir_list')

    def list(self):
        self._do_list('content')

    def tree(self):
        '''
        Получение дерева директорий
        '''
        args = self.base_args(self.params)
        args.update({
            'deep_level' : int(self.params.pop('deep_level', 1)),
            'parents' : bool(int(self.params.pop('parents', 0))),
            'args' : {
                'bounds' : {
                    'sort' : self.params.pop('sort', '') or 'name',
                    'order' : int(self.params.pop('order', '') or 1),
                },
            },
            tags.META_NAMES : self.params.pop(tags.META_NAMES, None)
        })
        args['args']['filter'] = self.params
        self.process_core_method('tree', args)

    def fulltree(self):
        '''
        Получение полного дерева ресурсов
        '''
        args = self.base_args(self.params)
        args.update({
            'deep_level'  : self.params.pop('deep_level', None),
            'relative'    : bool(int(self.params.pop('relative', 1))),
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            tags.META_NAMES : self.params.pop(tags.META_NAMES, None),
        })
        self.process_core_method('fulltree', args)

    def parents(self):
        '''
        Получение парентов по пути
        '''
        data = {'path' : self.params['path']}
        self.req.set_result(data)

    def services(self):
        '''
        Получение списка сервисов
        '''
        self.params['path'] = self.params.get('path', '/')
        args = self.base_args(self.params)
        args.update({
            'deep_level' : int(self.params.pop('deep_level', 1)),
            'args' : {
                'bounds' : {
                    'sort' : self.params.pop('sort', '') or 'name',
                    'order' : int(self.params.pop('order', '') or 1),
                },
            }
        })
        args['args']['filter'] = self.params
        self.process_core_method('services', args)

    def mkdir(self):
        '''
        Создание каталога
        '''
        has_extra_params = False
        if self.params.viewkeys() > {'uid', 'path', 'connection_id', 'force'} or self.params.get('force', '0') != '0':
            # пока что логируем запросы с дополнительными параметрами и обрабатываем в Питоне
            log.error('JAVAMKDIR unsupported extra parameters: %s' % self.params.viewkeys())
            has_extra_params = True

        if (JAVA_DJFS_API_PROXY_MKDIR_ENABLED or self.params.get('uid') in JAVA_DJFS_API_PROXY_MKDIR_ENABLED_UIDS)\
                and not has_extra_params:
            response = djfs_api.mkdir(self.params.get('uid'), self.params.get('path'), self.params.get('connection_id'))
            self.req.set_result(response)
            return

        args = self.modifying_request_parameters()
        args['changes'] = self.params
        self.process_core_method('mkdir', args)

    def fotki_mkfile(self):
        """Создать файл без колбэков."""
        keys = (
            ('uid', 'uid', str),
            ('file_mid', 'file_mid', str),
            ('digest_mid', 'digest_mid', str),
            ('md5', 'md5', normalized_hash),
            ('sha256', 'sha256', normalized_hash),
            ('size', 'size', int),
            ('mime_type', 'mime_type', str),
            ('ctime', 'ctime', int),
            ('mtime', 'mtime', int),
            ('path', 'path', str)
        )
        args = {}
        if 'preview_mid' in self.params:
            args['preview_mid'] = self.params.pop('preview_mid')

        if 'etime' in self.params:
            args['etime'] = self.params.pop('etime')

        if 'source' in self.params:
            args['source'] = self.params.pop('source')

        if 'type' in self.params:
            args['type'] = self.params.pop('type')

        uid = self.params['uid']
        args['uid'] = uid
        self.process_core_method('fotki_mkfile', keys=keys, args=args)

    def user_set_readonly(self):
        '''Перевести пользователя в режим "только для чтения"'''

        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('user_set_readonly', keys=keys)

    def user_unset_readonly(self):
        '''Отключить режим "только для чтения" для пользователя'''

        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('user_unset_readonly', keys=keys)

    def rm(self):
        '''
        Удаление ресурса
        '''
        args = self.modifying_request_parameters()
        args['md5'] = normalized_hash(self.params.get('md5', ''))
        self.process_core_method('rm', args)

    def rm_by_resource_id(self):
        args = {
            'uid': str(self.params['uid']),
            'resource_id': str(self.params['resource_id']),
            'callback': self.params.pop('callback', ''),
            'connection_id': self.params.pop('connection_id', ''),
            'md5': self.params.pop('md5', ''),
            'sha256': self.params.pop('sha256', ''),
            'size': self.params.pop('size', None),
            'rm_all': type_to_bool(self.params.pop('rm_all', False)),
            'files_only': type_to_bool(self.params.pop('files_only', False)),
        }
        self.process_core_method('rm_by_resource_id', args)

    def async_rm(self):
        args = self.modifying_request_parameters()
        args['md5'] = normalized_hash(self.params.get('md5', ''))
        self.process_core_method('async_rm', args)

    def async_rm_by_resource_id(self):
        args = {
            'uid': str(self.params['uid']),
            'resource_id': str(self.params['resource_id']),
            'callback': self.params.pop('callback', ''),
            'connection_id': self.params.pop('connection_id', ''),
            'md5': self.params.pop('md5', ''),
            'sha256': self.params.pop('sha256', ''),
            'size': self.params.pop('size', None),
            'rm_all': type_to_bool(self.params.pop('rm_all', False)),
            'files_only': type_to_bool(self.params.pop('files_only', False)),
        }
        self.process_core_method('async_rm_by_resource_id', args)

    def async_trash_drop_all(self):
        args = {
            'callback' : self.params.pop('callback', ''),
            'connection_id' : self.params.get('connection_id'),
            'uid' : str(self.params['uid']),
        }
        self.process_core_method('async_trash_drop_all', args)

    def async_trash_append(self):
        args = self.modifying_request_parameters()
        args['version'] = self.params.get('version', '')
        args['md5'] = self.params.get('md5', '')

        has_extra_params = False
        supported_java_params = {'uid', 'path', 'md5', 'version', 'callback', 'connection_id'}
        if args.viewkeys() > supported_java_params:
            log.error('JAVATRASH unsupported extra parameters: %s' % args.viewkeys())
            has_extra_params = True

        if (JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED or args['uid'] in JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED_UIDS) \
                and not has_extra_params:
            try:
                response = djfs_api.async_trash_append(**args)
                self.req.set_result(response)
                return
            except DjfsApiProxyNotImplementedError:
                pass

        self.process_core_method('async_trash_append', args)

    def async_trash_drop(self):
        args = self.modifying_request_parameters()
        args['version'] = self.params.get('version', '')
        self.process_core_method('async_trash_drop', args)

    def trash_drop(self):
        args = self.modifying_request_parameters()
        self.process_core_method('trash_drop', args)

    def async_trash_restore(self):
        args = self.modifying_request_parameters()
        args['name'] = self.params.get('name', '')
        args['force'] = bool(int(self.params.get('force', 0)))
        self.process_core_method('async_trash_restore', args)

    def async_social_contacts(self):
        keys = (
            ('uid', 'uid', str),
            ('groups', 'groups', int, 0),
        )
        self.process_core_method('async_social_contacts', keys=keys)

    def social_rights(self):
        keys = (
            ('uid', 'uid', str),
            ('scenario', 'scenario', str, 'invite'),
        )
        self.process_core_method('social_rights', keys=keys)

    def social_get_albums(self):
        '''
        Выдает список фотоальбомов пользователей из соц. сети
        '''
        _check_social_feature_toggle()
        args = {
            'uid'     : str(self.params.pop('uid')),
            'provider': self.params.pop('provider'),
        }
        self.process_core_method('social_get_albums', args)

    def social_create_album(self):
        '''
        Создает фотоальбом в соц. сети
        '''
        _check_social_feature_toggle()
        args = {
            'uid'     : str(self.params.pop('uid')),
            'provider': self.params.pop('provider'),
            'title'   : self.params.pop('title'),
            'privacy' : self.params.pop('privacy'),
        }
        self.process_core_method('social_create_album', args)

    def async_social_export_photos(self):
        _check_social_feature_toggle()
        args = {
            'uid': self.params.pop('uid'),
            'provider': self.params.pop('provider'),
            'albumid': self.params.pop('albumid'),
            'photos' : from_json(self.params.pop('photos')),
        }
        self.process_core_method('async_social_export_photos', args)

    def async_social_import_photos(self):
        _check_social_feature_toggle()
        args = {
            'uid': self.params.pop('uid'),
            'provider': self.params.pop('provider'),
            'type': self.params.pop('type'),
            'connection_id': self.params.pop('connection_id', ''),
        }
        self.process_core_method('async_social_import_photos', args)

    def url(self):
        '''
        Получение прямой ссылки на файл
        '''
        args = self.base_args(self.params)
        args['params'] = {
            'inline' : bool(int(self.params.get('inline') or 0)),
            'orig'   : bool(int(self.params.get('orig') or 0)),
        }
        self.process_core_method('url', args)

    def gdpr_takeout_electrichki(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('gdpr_takeout_electrichki', keys=keys)

    def video_streams(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('use_http', 'use_http', int, 0),
            ('user_ip', 'user_ip', str, ''),
            ('client_id', 'client_id', str, '')
        )
        self.process_core_method('video_streams', keys=keys)

    def public_video_streams(self):
        keys = (
            ('private_hash', 'private_hash', str),
            ('use_http', 'use_http', int, 0),
            ('user_ip', 'user_ip', str, ''),
            ('uid', 'uid', str, ''),
            ('client_id', 'client_id', str, '')
        )
        self.process_core_method('public_video_streams', keys=keys)

    def video_url(self):
        '''
        Получение ссылки на стриминг видео
        '''
        args = self.base_args(self.params)
        self.process_core_method('video_url', args)

    def public_video_url(self):
        '''
        Получение ссылки на публичное видео
        '''
        uid = self.params.get('uid')
        yandexuid = self.params.get('yandexuid')
        args = {
            'uid': str(uid) if uid else None,
            'yandexuid': str(yandexuid) if yandexuid else None,
            'private_hash': self.params['private_hash'],
        }
        self.process_core_method('public_video_url', args)

    def _prepare_store_args(self):
        self.params.pop('disk_client_file_id', None)

        args = self.modifying_request_parameters()
        args.update({
            'force'      : int(self.params.pop('force', 0)),
            'md5'        : normalized_hash(self.params.pop('md5', '')),
            'replace_md5': normalized_hash(self.params.pop('replace_md5', None)),
            'sha256'     : normalized_hash(self.params.pop('sha256', '')),
            'size'       : int(self.params.pop('size', 0)),
            'digest'     : self.params.pop('digest', ''),
            'client_type': self.client,
            'user_ip'   : self.user_ip,
            'skip_check_space': int(self.params.pop('skip_check_space', 0)),
            'skip_speed_limit': int(self.params.pop('skip_speed_limit', 0)),
            'live_photo_md5': normalized_hash(self.params.pop('live_photo_md5', None)),
            'live_photo_sha256': normalized_hash(self.params.pop('live_photo_sha256', None)),
            'live_photo_size': self.params.pop('live_photo_size', None),
            'user_agent': self.req.request_headers.get('user-agent', None),
            'live_photo_type': self.params.pop('live_photo_type', None),
            'live_photo_operation_id': self.params.pop('live_photo_operation_id', None),
            'fos_app_version': self.params.pop('fos_app_version', None),
            'fos_reply_email': self.params.pop('fos_reply_email', None),
            'fos_expire_seconds': self.params.pop('fos_expire_seconds', None),
            'fos_os_version': self.params.pop('fos_os_version', None),
            'fos_subject': self.params.pop('fos_subject', None),
            'fos_recipient_type': self.params.pop('fos_recipient_type', None),
            'source_id': self.params.pop('source_id', None),
            'force_deletion_log_deduplication': type_to_bool(self.params.pop('force_deletion_log_deduplication', False)),
            'device_original_path': self.params.pop('device_original_path', None),
            'device_collections': self.params.pop('device_collections', None),
            'photostream_destination': self.params.pop('photostream_destination', None),
        })
        ctime = int(self.params.pop('ctime', 0))
        if ctime:
            args['ctime'] = ctime
            if STORE_CTIME_FROM_CLIENT_ENABLED:
                self.params['ctime'] = ctime

        mtime = int(self.params.pop('mtime', 0))
        if mtime:
            self.params['mtime'] = mtime
            args['mtime'] = mtime

        etime = int(self.params.pop('etime', 0))
        if etime != 0:
            args['etime'] = etime

        try:
            args['use_https'] = bool(int(self.params.pop('use_https')))
        except (KeyError, ValueError):
            args['use_https'] = None
        args['changes'] = self.params

        return args

    def store(self):
        """Загрузка файла"""
        args = self._prepare_store_args()
        self.process_core_method('store', args)

    def attach_store(self):
        """Загрузка файла для аттачевых разделов"""
        args = self._prepare_store_args()
        self.process_core_method('attach_store', args)

    def restore_file(self):
        keys = (
            ('uid', 'uid', str),
            ('md5', 'md5', normalized_hash),
            ('sha256', 'sha256', normalized_hash),
            ('size', 'size', int),
        )
        self.process_core_method('restore_file', keys=keys)

    def extract_file_from_archive(self):
        """
        Извлечение файла из архива
        """
        keys = (
            ('uid', 'uid', str),
            ('src_archive', 'src_archive', str, ''),
            ('private_hash', 'private_hash', str, ''),
            ('src_file', 'src_file', str),
            ('dst', 'dst', str),
            ('connection_id', 'connection_id', str, ''),
        )
        self.process_core_method('extract_file_from_archive', keys=keys)

    def dstore(self):
        '''
        Дельта-обновление файла
        '''
        args = self.modifying_request_parameters()
        args['md5'] = normalized_hash(self.params.pop('md5'))
        try:
            args['use_https'] = bool(int(self.params.pop('use_https')))
        except (KeyError, ValueError):
            args['use_https'] = None
        args['changes'] = self.params
        self.process_core_method('dstore', args)

    def default_folders(self):
        '''
        Получение списка стандартных папок с учётом локализации
        '''
        if JAVA_DJFS_API_PROXY_DEFAULT_FOLDERS_ENABLED and any([prefix for prefix in JAVA_DJFS_API_PROXY_DEFAULT_FOLDERS_YCRID_PREFIXES if self.req.cloud_id and self.req.cloud_id.startswith(prefix)]):
            response = djfs_api.default_folders(self.params)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        args = self.base_args(self.params)
        args['check_exist'] = bool(int(self.params.get('check_exist', 0)))
        self.process_core_method('default_folders', args)

    def astore(self):
        '''
        Докачка файла
        '''
        args = self.base_args(self.params)
        args['md5'] = normalized_hash(self.params['md5'])
        args['callback'] = self.params.get('callback', '')
        args['user_agent'] = self.req.request_headers.get('user-agent', None)
        self.process_core_method('astore', args)

    def async_store_external(self):
        '''
        Сохранение файла из веба в Диск
        '''
        uid = self.params.pop('uid')
        args = {
            'uid': uid,
            'target': '%s:%s' % (uid, self.params.pop('path')),
            'external_url': self.params.pop('external_url'),
            'connection_id': self.params.pop('connection_id', ''),
            'disable_redirects': bool(int(self.params.pop('disable_redirects', 0))),
            'disable_retries': bool(int(self.params.pop('disable_retries', 0))),
        }

        self.process_core_method('async_store_external', args)

    def status(self):
        '''
        Проверка статуса операции
        '''
        args = {
            'uid': str(self.params['uid']),
            'oid': str(self.params['oid']),
            tags.META_NAMES : self.params.pop(tags.META_NAMES, None),
        }
        self.process_core_method('status', args)

    def info(self):
        """Получить информацию о ресурсе

        .. warning::
           Если установлен ``unzip_file_id``, то ресурс в базе пересохранится
           с раззипованным file_id.
        """

        DJFS_SUPPORTED_PARAMS = {'uid', 'path', 'preview_size', 'preview_crop', 'preview_quality',
                                 'preview_allow_big_size', 'preview_type', 'preview_animate',  'tld'}
        has_unsupported_params = self._has_unsupported_params(DJFS_SUPPORTED_PARAMS,
                                                              JAVA_DJFS_API_PROXY_INFO_SUPPORTED_META)

        path = self.params['path'] if 'path' in self.params.keys() else ''
        has_unsupported_root_paths = True
        for rootPath in JAVA_DJFS_API_PROXY_INFO_SUPPORTED_ROOT_PATHS:
            if path.startswith(rootPath):
                has_unsupported_root_paths = False
                break

        if JAVA_DJFS_API_PROXY_INFO_ENABLED and not has_unsupported_params and not has_unsupported_root_paths:
            response = djfs_api.info(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        args = self.base_args(self.params)
        args.update({
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            'preview_type': self.params.pop('preview_type', None),
            'preview_animate': bool(int(self.params.pop('preview_animate', 0))),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
            'unzip_file_id': bool(int(self.params.pop('unzip_file_id', 0))),
        })
        self.process_core_method('info', args)

    def info_by_file_id(self):
        """
        Найти информацию о файле по owner_uid и file_id.
        Вернуть представление от лица uid'а.
        """
        keys = (
            ('uid', 'uid', str),
            ('file_id', 'file_id', str),
            ('owner_uid', 'owner_uid', str, ''),
        )
        self.process_core_method('info_by_file_id', keys=keys)

    def image_dimensions_by_file_id(self):
        """
        Получить размер картинки по uid и file_id.
        """
        keys = (
            ('uid', 'uid', str),
            ('file_id', 'file_id', str),
        )
        self.process_core_method('image_dimensions_by_file_id', keys=keys)

    def image_metadata_by_file_id(self):
        """
        Получить размер картинки по uid и file_id c учетом поворота.
        """
        keys = (
            ('uid', 'uid', str),
            ('file_id', 'file_id', str),
            ('orientation', 'orientation', int),
        )
        self.process_core_method('image_metadata_by_file_id', keys=keys)

    def info_by_resource_id(self):
        """
        Найти информацию о файле по resource_id.
        Вернуть представление от лица uid'а.
        """
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
        )
        self.process_core_method('info_by_resource_id', keys=keys)

    def _has_unsupported_params(self, supported_params, supported_meta):
        if not set(self.params.keys()).issubset(supported_params):
            return True

        if self.req.meta is None:
            return False

        if len(self.req.meta) == 0:
            return True

        return not set(self.req.meta).issubset(supported_meta)

    def bulk_info_by_resource_ids(self):
        """
        Получение информации о пачке ресурсов
        """
        keys = (
            ('uid', 'uid', str),
            ('enable_service_ids', 'enable_service_ids', str,
             ','.join(['/disk', '/attach', '/trash', PHOTOUNLIM_AREA_PATH])),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )

        DJFS_SUPPORTED_PARAMS = {'uid', 'enable_service_ids', 'preview_size', 'preview_crop', 'preview_quality',
                                 'preview_allow_big_size'}
        has_unsupported_params = self._has_unsupported_params(DJFS_SUPPORTED_PARAMS,
                                                              JAVA_DJFS_API_PROXY_BULK_INFO_DJFS_SUPPORTED_META)
        if has_unsupported_params:
            log.error('JAVABULKINFO unsupported parameters: %s, meta: %s', self.params.keys(), self.req.meta)

        if (has_unsupported_params is False and (JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED or self.params.get('uid') in
                                                 JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED_UIDS)):
            response = djfs_api.bulk_info_by_resource_ids(self.params, self.req.meta, self.req.http_req.data)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        self.process_core_method('bulk_info_by_resource_ids', keys=keys)

    def bulk_info_by_office_online_sharing_urls(self):
        """
        Получение информации о пачке ресурсов по office_doc_ids
        """
        keys = (
            ('uid', 'uid', str),
        )

        self.process_core_method('bulk_info_by_office_online_sharing_urls', keys=keys)

    def bulk_info(self):
        """
        Получение информации о пачке ресурсов
        """
        keys = (
            ('uid', 'uid', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )

        DJFS_SUPPORTED_PARAMS = {'uid', 'preview_size', 'preview_crop', 'preview_quality', 'preview_allow_big_size'}
        has_unsupported_params = self._has_unsupported_params(DJFS_SUPPORTED_PARAMS,
                                                              JAVA_DJFS_API_PROXY_BULK_INFO_DJFS_SUPPORTED_META)
        if has_unsupported_params:
            log.error('JAVABULKINFO unsupported parameters: %s, meta: %s', self.params.keys(), self.req.meta)

        if (has_unsupported_params is False and (JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED or self.params.get('uid') in
                                                 JAVA_DJFS_API_PROXY_BULK_INFO_ENABLED_UIDS)):
            response = djfs_api.bulk_info(self.params, self.req.meta, self.req.http_req.data)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        self.process_core_method('bulk_info', keys=keys)

    def publish(self):
        '''
        Публикация файла
        '''
        args = self.base_args(self.params)
        self.req.set_args(args)
        self.req.set_result(core.set_public(self.req))

    def copy_disk(self):
        """
        Скопировать содержимое диска пользователя другому пользователю в диск
        uid - идентификатор пользователя, которому будет скопирован диск пользователя с идентификатором src_uid
        dst - путь до папки, куда будет скопирован диск
        """
        keys = (
            ('uid', 'uid', str),
            ('src_uid', 'src_uid', str),
            ('dst', 'dst', str),
            ('connection_id', 'connection_id', str, '')
        )
        self.process_core_method('copy_disk', keys=keys)

    def import_attach_to_disk(self):
        """
        Создать операцию перекладывания файла из почты по mid/hid в диск по пути path
        """
        keys = (
            ('uid', 'uid', str),
            ('mail_mid', 'mail_mid', str),
            ('mail_hid', 'mail_hid', str),
            ('dst', 'dst', str),
            ('overwrite', 'overwrite', type_to_bool, False),
            ('autosuffix', 'autosuffix', type_to_bool, False),
        )
        self.process_core_method('import_attach_to_disk', keys=keys)

    def import_attaches_to_disk(self):
        """
        Создать операцию перекладывания нескольких файлов из почты по mid/hid в диск по пути path
        """
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('overwrite', 'overwrite', type_to_bool, False),
            ('autosuffix', 'autosuffix', type_to_bool, False),
        )
        self.process_core_method('import_attaches_to_disk', keys=keys)

    def import_file_from_service(self):
        """
        Создать операцию перекладывания файла из внешнего сервиса в диск в папку загрузки
        """
        keys = (
            ('uid', 'uid', str),
            ('service_id', 'service_id', str),
            ('service_file_id', 'service_file_id', str),
            ('file_name', 'file_name', str),
            ('overwrite', 'overwrite', type_to_bool, False),
            ('autosuffix', 'autosuffix', type_to_bool, True),
        )
        self.process_core_method('import_file_from_service', keys=keys)

    def copy(self):
        args = self.base_copy_move_args(self.params)
        self.process_core_method('copy_resource', args)

    def async_copy(self):
        args = self.base_copy_move_args(self.params)
        self.process_core_method('async_copy_resource', args)

    def move(self):
        args = self.base_copy_move_args(self.params)
        args['get_lenta_block_id'] = int(self.params.pop('get_lenta_block_id', 0))
        args['return_status'] = int(self.params.pop('return_status', 0))
        args['check_hids_blockings'] = int(self.params.pop('check_hids_blockings', 1)) != 0

        has_extra_params = False
        supported_java_params = {
            'uid', 'src', 'dst', 'force', 'connection_id', 'callback', 'get_lenta_block_id', 'return_status',
            'check_hids_blockings', 'force_djfs_albums_callback'
        }
        if args.viewkeys() > supported_java_params:
            log.error('JAVAMOVE unsupported extra parameters: %s' % args.viewkeys())
            has_extra_params = True

        uid = args['uid']
        if ((JAVA_DJFS_API_PROXY_MOVE_ENABLED or uid in JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS or
                filter_uid_by_percentage(uid, JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS_PERCENT)) and
                not has_extra_params):
            try:
                response = djfs_api.move(**args)
                self.req.set_result(response)
                return
            except DjfsApiProxyNotImplementedError:
                pass

        self.process_core_method('move_resource', args)

    def async_move(self):
        args = self.base_copy_move_args(self.params)

        has_extra_params = False
        supported_java_params = {'uid', 'src', 'dst', 'force', 'connection_id', 'callback', 'force_djfs_albums_callback'}
        if args.viewkeys() > supported_java_params:
            log.error('JAVAMOVE unsupported extra parameters: %s' % args.viewkeys())
            has_extra_params = True

        uid = args['uid']
        if ((JAVA_DJFS_API_PROXY_MOVE_ENABLED or uid in JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS) or
                filter_uid_by_percentage(uid, JAVA_DJFS_API_PROXY_MOVE_ENABLED_UIDS_PERCENT) and
                not has_extra_params):
            try:
                response = djfs_api.async_move(**args)
                self.req.set_result(response)
                return
            except DjfsApiProxyNotImplementedError:
                pass

        self.process_core_method('async_move_resource', args)

    def setprop(self):
        args = self.base_args(self.params)
        deleted = self.params.pop('setprop_delete', '') or []
        if deleted:
            deleted = deleted.split(',')
        self.params.pop('_request', '')
        args['deleted'] = deleted
        args['changes'] = self.params
        self.process_core_method('setprop', args)

    def diff(self):
        args = self.base_args(self.params)
        args['version'] = self.params.get('version')
        args['allow_quick_move_deltas'] = type_to_bool(self.params.get('allow_quick_move_deltas', 0))
        self.process_core_method('diff', args)

    def snapshot(self):
        keys = (
            ('uid', 'uid', str),
            ('session_id', 'session_id', str, None),
        )
        self.process_core_method('snapshot', keys=keys)

    def indexer_snapshot(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('indexer_snapshot', keys=keys)

    def deltas(self):
        keys = (
            ('uid', 'uid', str),
            ('base_revision', 'base_revision', long),
            ('allow_quick_move_deltas', 'allow_quick_move_deltas', type_to_bool, False),
        )
        self.process_core_method('deltas', keys=keys)

    def search(self):
        '''
        Старая ручка поиска
        Поддердживаем временно, для обратной совместимости с версткой
        '''
        args = self.base_args(self.params)
        args['sort'] = self.params.pop('sort', '')
        args['query'] = self.params.get('query', '*')
        self.process_core_method('deprecated_search', args)

    def new_search(self):
        """
        Новая ручка поиска по Диску
        """
        args = self.base_args(self.params)
        args.update({
            'query': self.params.pop('query', '*'),
            'args': {
                'bounds': {
                    'sort': self.params.pop('sort', '') or 'name',
                    'order': int(self.params.pop('order', '') or 1),
                },
            },
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            'force': int(self.params.pop('force', 0)),
            'count_lost_results': bool(int(self.params.pop('count_lost_results', 0))),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
            'session_id': self.params.pop('sessionid', None)
        })
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['args']['bounds'][bound] = int(self.params.pop(bound))
        args['args']['filter'] = self.params
        self.process_core_method('search', args)
        self.req.form.apply_filters()
        self.req.form.apply_bounds()

    def geo_search(self):
        """
        Ручка поиска UCS по времени и координатам
        """
        args = self.base_args(self.params)
        latitude = self.params.pop('latitude', None)
        longitude = self.params.pop('longitude', None)
        distance = self.params.pop('distance', None)

        args.update({
            'query': self.params.pop('query', '*'),
            'args': {
                'bounds': {},
            },
            'latitude': None if latitude is None else float(latitude),
            'longitude': None if longitude is None else float(longitude),
            'distance': None if distance is None else int(distance),
            'end_date': long(self.params.pop('end_date', 0)),
            'start_date': long(self.params.pop('start_date', 0)),
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'count_lost_results': bool(int(self.params.pop('count_lost_results', 0))),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
            'session_id': self.params.pop('sessionid', None)
        })
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['args']['bounds'][bound] = int(self.params.pop(bound))
        args['args']['filter'] = {}
        self.process_core_method('geo_search', args)
        self.req.form.apply_filters()
        self.req.form.apply_bounds()

    def version(self):
        '''
        Выдача версии пакета
        '''
        self.req.set_result(core.version(self.req))

    def ping(self):
        '''
        Пинг для админов
        '''
        self.req.set_result(core.ping(self.req))

    def ping_slb(self):
        '''
        Пинг для балансера проверяющий базу system
        '''
        self.req.set_result(core.ping_slb(self.req))

    def space(self):
        args = {
            'uid': str(self.params['uid']),
            'count_default_folders': str(self.params.pop('count_default_folders', '')) or None,
        }
        self.process_core_method('space', args)

    def hardlink_copy(self):
        args = self.base_args(self.params)
        args.update({
            'md5'        : normalized_hash(self.params.pop('md5')),
            'size'       : self.params.pop('size'),
            'sha256'     : normalized_hash(self.params.pop('sha256')),
            'client_type': self.client,
            'replace_md5': normalized_hash(self.params.pop('replace_md5', None)),
        })
        args['changes'] = self.params
        self.process_core_method('hardlink_copy', args)

    def trash_append_file(self):
        args = self.base_args(self.params)
        args['version'] = self.params.get('version', '')
        self.process_core_method('trash_append', args)

    def trash_drop_all(self):
        args = {
               'uid' : str(self.params['uid']),
               'connection_id' : self.params.pop('connection_id', ''),
               }
        self.process_core_method('trash_drop_all', args)

    def trash_restore(self):
        args = self.base_args(self.params)
        args['name'] = self.params.get('name', '')
        args['force'] = bool(int(self.params.get('force', 0)))
        self.process_core_method('trash_restore', args)

    def restore_deleted(self):
        args = self.base_args(self.params)
        args['dest'] = self.params['dest']
        self.process_core_method('restore_deleted', args)

    def trash_append(self):
        args = self.base_args(self.params)
        args['version'] = self.params.get('version', '')
        args['md5'] = normalized_hash(self.params.get('md5', ''))

        has_extra_params = False
        supported_java_params = {'uid', 'path', 'md5', 'version', 'connection_id'}
        if args.viewkeys() > supported_java_params:
            log.error('JAVATRASH unsupported extra parameters: %s' % args.viewkeys())
            has_extra_params = True

        if (JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED or args['uid'] in JAVA_DJFS_API_PROXY_TRASH_APPEND_ENABLED_UIDS) \
                and not has_extra_params:
            try:
                response = djfs_api.trash_append(**args)
                self.req.set_result(response)
                return
            except DjfsApiProxyNotImplementedError:
                pass

        self.process_core_method('trash_append', args)

    def trash_append_by_resource_id(self):
        args = {
            'uid': str(self.params['uid']),
            'resource_id': str(self.params['resource_id']),
            'callback': self.params.pop('callback', ''),
            'connection_id': self.params.pop('connection_id', ''),
            'md5': self.params.pop('md5', ''),
            'sha256': self.params.pop('sha256', ''),
            'size': self.params.pop('size', None),
            'append_all': type_to_bool(self.params.pop('append_all', False)),
            'files_only': type_to_bool(self.params.pop('files_only', False)),
        }
        self.process_core_method('trash_append_by_resource_id', args)

    def async_trash_append_by_resource_id(self):
        args = {
            'uid': str(self.params['uid']),
            'resource_id': str(self.params['resource_id']),
            'callback': self.params.pop('callback', ''),
            'connection_id': self.params.pop('connection_id', ''),
            'md5': self.params.pop('md5', ''),
            'sha256': self.params.pop('sha256', ''),
            'size': self.params.pop('size', None),
            'append_all': type_to_bool(self.params.pop('append_all', False)),
            'files_only': type_to_bool(self.params.pop('files_only', False)),
        }
        self.process_core_method('async_trash_append_by_resource_id', args)

    def active_operations(self):
        args = {'uid' : self.params['uid'], 'show_hidden': type_to_bool(self.params.pop('show_hidden', False))}
        self.process_core_method('active_operations', args)

    def timeline(self):
        args = self.base_args(self.params)
        args.update({
            'deep_level' : int(self.params.pop('deep_level', -1)),
            'args' : {
                'bounds' : {
                    'sort'  : self.params.pop('sort', '') or 'ctime',
                    'order' : int(self.params.pop('order', '') or 1),
                },
            },
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            tags.META_NAMES : self.params.pop(tags.META_NAMES, None),
        })
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['args']['bounds'][bound] = int(self.params.pop(bound))
        args['args']['filter'] = self.params
        self.process_core_method('timeline', args)

    def get_last_files(self):
        # В текущих требованиях только для корня диска.
        self.params['path'] = '/disk'
        args = self.base_args(self.params)
        args.update({
            'args': {
                'bounds': {
                    'sort': 'mtime',
                    'order': 0,
                    'amount': int(self.params.pop('amount', 50))
                },
                'filter': {}
            },
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
        })
        self.process_core_method('set_last_files', args)

    def new_get_last_files(self):
        keys = (
            ('uid', 'uid', str),
            ('amount', 'amount', int, 10),
            ('offset', 'offset', int, 0),
        )
        self.process_core_method('new_get_last_files', keys=keys)

    def wake_up_push_start(self):
        keys = (
            ('uid', 'uid', str),
            ('device_id', 'device_id', str),
            ('interval', 'interval', int, settings.wake_up['push_interval']),
        )
        self.process_core_method('wake_up_push_start', keys=keys)


    def wake_up_push_stop(self):
        keys = (
            ('session_id', 'session_id', str),
        )
        self.process_core_method('wake_up_push_stop', keys=keys)

    def bulk_action(self):
        '''
        Групповое выполнение операций
        '''
        args = {
            'uid': str(self.params['uid']),
            'cmd': self.params['cmd'],
            'callback' : self.params.pop('callback', ''),
        }
        self.process_core_method('bulk_action', args)

    def set_public(self):
        '''
        Публикация файла
        '''
        args = self.modifying_request_parameters()
        args['user_ip'] = self.user_ip
        self.process_core_method('set_public', args)

    def set_private(self):
        '''
        Отмена публикации файла
        '''
        args = self.modifying_request_parameters()
        args['return_info'] = bool(self.params.get('return_info', False))
        self.process_core_method('set_private', args)

    def set_public_settings(self):
        '''
        Установка настроей публичной ссылки
        '''
        args = self.modifying_request_parameters()
        args['read_only'] = bool(self.params.get('read_only', False))
        args['password'] = self.params.get('password', None)
        args['available_until'] = self.params.get('available_until', None)
        self.process_core_method('set_public_settings', args)

    def get_public_settings(self):
        '''
        Получение настроек публичной ссылки
        '''
        args = self.modifying_request_parameters()
        self.process_core_method('get_public_settings', args)

    def get_public_settings_by_hash(self):
        '''
        Получение настроек публичной ссылки по шифрованному ключу
        '''
        args = self.modifying_request_parameters()
        args['private_hash'] = self.params['private_hash']
        self.process_core_method('get_public_settings_by_hash', args)

    def public_info(self):
        '''
        Получение info по шифрованному ключу
        '''
        DJFS_SUPPORTED_PARAMS = {'private_hash', 'uid', 'preview_size', 'preview_crop',
                                 'preview_quality', 'preview_allow_big_size', 'increment_views'}
        DJFS_SUPPORTED_META = {'drweb', 'mediatype', 'mimetype', 'short_url', 'size', 'sizes', 'file_id', 'comment_ids',
                               'views_counter', 'download_counter', 'blockings', 'page_blocked_items_num'}
        has_unsupported_params = self._has_unsupported_params(DJFS_SUPPORTED_PARAMS, DJFS_SUPPORTED_META)

        if has_unsupported_params:
            log.error('JAVAPUBLICINFO unsupported parameters: %s, meta: %s', self.params.keys(), self.req.meta)
        if (has_unsupported_params is False and (JAVA_DJFS_API_PROXY_PUBLIC_INFO_ENABLED or self.params.get('uid') in
                                                 JAVA_DJFS_API_PROXY_PUBLIC_INFO_ENABLED_UIDS)):
            response = djfs_api.public_info(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        args = {
            'private_hash': self.params['private_hash'],
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            'uid': str(self.params.pop('uid')) if 'uid' in self.params else None,
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('public_info', args)

    def public_list(self):
        '''
        Получение листинга по шифрованному ключу
        '''
        DJFS_SUPPORTED_PARAMS = {'private_hash', 'uid', 'sort', 'order', 'preview_size', 'preview_crop',
                                 'preview_quality', 'preview_allow_big_size', 'amount', 'offset'}
        DJFS_SUPPORTED_META = {'drweb', 'mediatype', 'mimetype', 'short_url', 'size', 'sizes', 'file_id', 'comment_ids',
                               'views_counter', 'download_counter', 'blockings', 'page_blocked_items_num'}
        has_unsupported_params = self._has_unsupported_params(DJFS_SUPPORTED_PARAMS, DJFS_SUPPORTED_META)

        if has_unsupported_params:
            log.error('JAVAPUBLICLIST unsupported parameters: %s, meta: %s', self.params.keys(), self.req.meta)
        if (has_unsupported_params is False and (JAVA_DJFS_API_PROXY_PUBLIC_LIST_ENABLED or self.params.get('uid') in
                                                 JAVA_DJFS_API_PROXY_PUBLIC_LIST_ENABLED_UIDS)):
            response = djfs_api.public_list(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return

        args = {
            'args' : {
                'bounds' : {
                    'sort'  : self.params.pop('sort', '') or 'name',
                    'order' : int(self.params.pop('order', '') or 1),
                },
            },
            'private_hash': self.params.pop('private_hash'),
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
            'uid': str(self.params.pop('uid')) if 'uid' in self.params else None,
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['args']['bounds'][bound] = int(self.params.pop(bound))
        args['args']['filter'] = self.params
        self.process_core_method('public_content', args)
        self.req.form.apply_filters()
        self.req.form.apply_bounds()

    def search_public_list(self):
        '''
        Получение листинга из поиска по шифрованному ключу
        '''
        keys = (
            ('private_hash', 'private_hash', str),
            ('search_meta', 'search_meta', str, 'path,name,type,id'),
            ('uid', 'uid', str, None),
        )
        self.process_core_method('search_public_list', keys=keys)

    def public_fulltree(self):
        '''
        Получение полного дерева по шифрованному ключу
        '''
        args = {
            'private_hash': self.params.pop('private_hash'),
            'deep_level'  : self.params.pop('deep_level', None),
            'preview_size': self.params.pop('preview_size', None),
            'preview_crop': int(self.params.pop('preview_crop', 0)),
            'preview_quality': int(self.params.pop('preview_quality', 0)),
            'preview_allow_big_size': type_to_bool(self.params.pop('preview_allow_big_size', False)),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
            'uid': str(self.params.pop('uid')) if 'uid' in self.params else None,
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('public_fulltree', args)

    def public_url(self):
        '''
        Получение url по шифрованному ключу
        '''
        args = {
            'private_hash': self.params['private_hash'],
            'inline': bool(int(self.params.get('inline') or 0)),
            'uid': str(self.params.pop('uid')) if 'uid' in self.params else None,
            'check_blockings': bool(int(self.params.get('check_blockings', 1))),
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('public_url', args)

    def public_notification(self):
        args = {
            'uid'     : str(self.params['uid']),
            'path'    : _convert_to_address(self.params['uid'], self.params['path']),
            'emails'  : self.params['emails'],
            'message' : self.params.get('message', ''),
            'locale'  : self.params.get('locale', None)
        }
        self.process_core_method('public_notification', args)

    def block(self):
        args = {
            'private_hash': self.params['private_hash'],
        }
        self.process_core_method('block', args)

    def public_copy(self):
        args = {
            'uid': str(self.params['uid']),
            'private_hash': self.params['private_hash'],
            'name': self.params.get('name', ''),
            'save_path': self.params.get('save_path'),
            'connection_id': self.params.get('connection_id', ''),
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('public_copy', args)

    def lenta_public_log_visit(self):
        keys = (
            ('uid', 'uid', str),
            ('short_url', 'short_url', str),
        )
        self.process_core_method('lenta_public_log_visit', keys=keys)

    def async_public_copy(self):
        args = {
            'uid': str(self.params['uid']),
            'private_hash': self.params['private_hash'],
            'name': self.params.get('name', ''),
            'save_path': self.params.get('save_path'),
            'connection_id': self.params.get('connection_id', ''),
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }
        self.process_core_method('async_public_copy', args)

    def async_copy_default(self):
        args = self.base_args()
        args['name'] = self.params.get('name', '')
        args['force'] = self.params.get('force', 0)
        args['public'] = False
        args['force_djfs_albums_callback'] = self.params.get('force_djfs_albums_callback', False)
        if self.params.get('public') in (1, '1'):
            args['public'] = True
        self.process_core_method('async_copy_default', args)

    def setting_set(self):
        self._set_user_var('settings')

    def setting_remove(self):
        self._remove_user_var('settings')

    def state_set(self):
        self._set_user_var('states')

    def state_remove(self):
        self._remove_user_var('states')


    def user_install_device(self):
        '''
        Регистрация нового устройства пользователя в системе
        '''
        args = {
            'uid'     : str(self.params.pop('uid')),
            'id'      : str(self.params.pop('id')),
            'type'    : self.params.pop('type'),
            'project' : self.params.pop('project', 'disk'),
        }
        args['info'] = self.params
        self.process_core_method('user_install_device', args)

    def user_uninstall_device(self):
        '''
        Удаление устройства пользователя в системе
        '''
        args = {
            'uid'     : str(self.params.pop('uid')),
            'id'      : str(self.params.pop('id')),
            'type'    : self.params.pop('type'),
            'project' : self.params.pop('project', 'disk'),
        }
        self.process_core_method('user_uninstall_device', args)


    def _set_user_var(self, type):
        '''
        Установка настройки/состояния
        '''
        args = {
            'uid'      : str(self.params['uid']),
            'project'  : self.params.get('project', 'disk'),
            'key'      :  str(self.params['key']),
            'value'    : str(self.params['value']),
            'namespace': self.params.get('namespace', None),
            'type'     : type,
        }
        self.process_core_method('set_user_var', args)

    def _remove_user_var(self, type):
        '''
        Удаление настройки/состояния
        '''
        args = {
            'uid'      : str(self.params['uid']),
            'project'  : self.params.get('project', 'disk'),
            'key'      : str(self.params['key']),
            'namespace': self.params.get('namespace', None),
            'type'     : type,
        }
        self.process_core_method('remove_user_var', args)

    def user_info(self):
        '''
        Выдача полной юзерской информации
        '''
        args = {
            'uid': str(self.params['uid']),
            'project': self.params.get('project', 'disk'),
        }
        self.process_core_method('user_info', args)

    def user_feature_toogles(self):
        '''
        Название ручки написано с ошибкой. Это специально, это нужно для сохранения совметимости
        См: https://st.yandex-team.ru/CHEMODAN-43113
        '''
        self.user_feature_toggles()

    def user_feature_toggles(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('user_feature_toggles', keys=keys)

    def awaps_user_info(self):
        '''
        Выдача юзерской информации для awaps
        '''
        args = {
            'uid': str(self.params['uid']),
        }
        self.process_core_method('awaps_user_info', args)

    def user_init(self):
        '''
        Инициализация пользователя
        '''
        keys = (
            ('uid', 'uid', str),
            ('locale', 'locale', str, None),
            ('source', 'source', str, None),
            ('shard', 'shard', str, None),
            ('noemail', 'noemail', int, 0),
            ('init10gb', 'init10gb', int, 1),   # initialize user with initial_10gb bonus
            ('b2b_key', 'b2b_key', str, None),  # признак того, что инициализиуемый пользователь является b2b'шным. Может быть идентификатором организации
            ('add_services', 'add_services', str, None)
        )
        self.process_core_method('user_init', keys=keys)

    def can_init_user(self):
        """
        Проверить возможность инициализации
        """

        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('can_init_user', keys=keys)

    def user_make_b2b(self):
        '''
        Сделать обычного пользователя b2bшным
        '''
        keys = (
            ('uid', 'uid', str),
            ('b2b_key', 'b2b_key', str, None),
        )
        self.process_core_method('user_make_b2b', keys=keys)

    def user_reset_b2b(self):
        '''
        Сделать b2bшного пользователя обычным
        '''
        keys = (
            ('uid', 'uid', str),
            ('noemail', 'noemail', int, 0),
        )
        self.process_core_method('user_reset_b2b', keys=keys)

    def user_check(self):
        '''
        Проверка - нужно ли инициировать пользователя
        Должно уйти в замен user_info
        '''
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('user_check', args)

    def user_remove(self):
        '''
        DEPRECATED: Старая тестовая ручка, не для продакшена

        Удаление пользователя
        '''
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('user_remove', args)

    def async_user_remove(self):
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('async_user_remove', args)

    def user_invite_hash(self):
        '''
        Выдать персональный инвайт-код пользователя
        '''
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('user_invite_hash', args)

    def user_invite_info(self):
        '''
        Выдать сводку по инвайтам
        '''
        args = {
            'uid': str(self.params.get('uid')),
            'hash': self.params.get('hash'),
        }
        self.process_core_method('user_invite_info', args)

    def user_info_attributes(self):
        """
        Выдать аттрибуты пользователя из паспорта
        """
        args = {
            'uid': str(self.params.get('uid')),
        }
        self.process_core_method('user_info_attributes', args)

    def user_activity_info(self):
        """
        Выдать данные по активности пользователя
        """
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('user_activity_info', keys=keys)

    def async_user_invite_friend(self):
        '''
        Отправка инвайта другу
        '''
        args = {
            'uid'     : str(self.params.pop('uid')),
            'provider': self.params.pop('provider'),
            'address' : self.params.pop('address'),
        }
        args['info'] = self.params
        self.process_core_method('async_user_invite_friend', args)

    def async_invite_contacts(self):
        '''
        Получение информации о контактах со статусами отправленности
        '''
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('async_invite_contacts', keys=keys)

    def user_invite_activated(self):
        '''
        Получение информации о моих инвайтах: кто активировал
        '''
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('user_invite_activated', args)

    def user_invite_sent(self):
        '''
        Получение информации об отправленных инвайтах
        '''
        args = {'uid': str(self.params['uid'])}
        self.process_core_method('user_invite_sent', args)

    def do(self):
        self.bulk_action()

    def iddqd(self):
        self.bulk_action()

    def share_create_group(self):
        args = self.base_args(self.params)
        self.process_core_method('share_create_group', args)

    def share_invite_user(self):
        uid = str(self.params.pop('uid', '')) or None
        path = self.params.pop('path', None)
        args = {'uid' : uid}
        if path:
            args['group_path'] = _convert_to_address(uid, path)
        else:
            args['group_path'] = None
        keys = (
            ('gid', 'gid', str, None),
            ('rights', 'rights', int, 640),
            ('locale', 'locale', str, None),
            ('universe_login', 'universe_login', str),
            ('universe_service', 'universe_service', str),
            ('user_avatar', 'avatar', str, ''),
            ('user_name', 'name', str, ''),
            ('connection_id', 'connection_id', str, ''),
            ('auto_accept', 'auto_accept', int, 0),
            ('ip', 'ip', str, ''),
        )
        self.process_core_method('share_invite_user', args=args, keys=keys)

    def share_invite_info(self):
        keys = (
            ('hsh', 'hash', str),
            ('user_uid', 'uid', str, None),
            ('include_files_count', 'include_files_count', int, 0),
        )
        self.process_core_method('share_invite_info', keys=keys)

    def share_folder_info(self):
        keys = (
                ('gid', 'gid', str),
                ('uid', 'uid', str),
                )
        self.process_core_method('share_folder_info', keys=keys)

    def share_activate_invite(self):
        keys = (
                ('hsh', 'hash', str),
                ('uid', 'uid', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_activate_invite', keys=keys)

    def share_change_rights(self):
        keys = (
                ('uid', 'uid', str),
                ('gid', 'gid', str),
                ('user_uid', 'user_uid', str, None),
                ('rights', 'rights', int),
                ('universe_login', 'universe_login', str, None),
                ('universe_service', 'universe_service', str, None),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_change_rights', keys=keys)

    def share_kick_from_group(self):
        keys = (
                ('uid', 'uid', str),
                ('user_uid', 'user_uid', str),
                ('gid', 'gid', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_kick_user', keys=keys)

    def share_leave_group(self):
        keys = (
                ('uid', 'uid', str),
                ('gid', 'gid', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_leave_group', keys=keys)

    def share_users_in_group(self):
        keys = (
                ('uid', 'uid', str),
                ('gid', 'gid', str),
                ('exclude_b2b', 'exclude_b2b', type_to_bool, False),
                ('iteration_key', 'iteration_key', str, None),
                )
        self.process_core_method('share_users_in_group', keys=keys)

    def lenta_list_group_invites(self):
        keys = (
            ('uid', 'uid', str),
            ('gid', 'gid', str),
            ('limit', 'limit', int, 20),
            ('iteration_key', 'iteration_key', str, ''),
        )
        self.process_core_method('lenta_list_group_invites', keys=keys)

    def share_uids_in_group(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str, ''),
            ('file_id', 'file_id', str, ''),
            ('owner_uid', 'owner_uid', str, ''),
            ('offset', 'offset', int, 0),
            ('amount', 'amount', int, 100000),
        )
        self.process_core_method('share_uids_in_group', keys=keys)

    def share_change_group_owner(self):
        keys = (
                ('gid', 'gid', str),
                ('user_uid', 'uid', str),
                ('uid', 'owner', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_change_group_owner', keys=keys)

    def share_list_all_folders(self):
        keys = (
                ('uid', 'uid', str),
                )
        self.process_core_method('share_list_all_folders', keys=keys)

    def share_list_joined_folders(self):
        keys = (
                ('uid', 'uid', str),
                )
        self.process_core_method('share_list_joined_folders', keys=keys)

    def share_list_owned_folders(self):
        keys = (
                ('uid', 'uid', str),
                )
        self.process_core_method('share_list_owned_folders', keys=keys)

    def share_list_not_approved_folders(self):
        keys = (
                ('uid', 'uid', str),
                )
        self.process_core_method('share_list_not_approved_folders', keys=keys)

    def share_unshare_folder(self):
        keys = (
                ('uid', 'uid', str),
                ('gid', 'gid', str, None),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_unshare_folder', keys=keys)

    def share_reject_invite(self):
        keys = (
                ('uid', 'uid', str),
                ('hsh', 'hash', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_reject_invite', keys=keys)

    def share_remove_invite(self):
        keys = (
                ('uid', 'uid', str),
                ('gid', 'gid', str),
                ('universe_login', 'universe_login', str),
                ('universe_service', 'universe_service', str),
                ('connection_id', 'connection_id', str, ''),
                )
        self.process_core_method('share_remove_invite', keys=keys)

    def share_b2b_synchronize_access(self):
        keys = (
            ('uid', 'uid', str),
            ('gid', 'gid', str),
        )
        self.process_core_method('share_b2b_synchronize_access', keys=keys)

    def invite_activate(self):
        '''
        Активирование инвайтного кода
        '''
        args = {
            'code'    : str(self.params.pop('code')),
            'uid'     : self.params.pop('uid'),
            'project' : self.params.pop('project', 'mpfs'),
        }
        args['info'] = self.params
        self.process_invite_method('activate', args)


    def mksysdir(self):
        '''
        Создание системного каталога (Загрузки/Фотострим)
        '''
        keys = (
            ('uid', 'uid', str),
            ('type', 'type', str),
        )
        self.process_core_method('mksysdir', keys=keys)


    def pushthelimitsto_10gb(self):
        """Добивка размера диска пользователя до 10ГБ."""
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('pushthelimitsto_10gb', keys=keys)


    def aviary_render(self):
        """Редактирование изображения через Aviary"""
        args = self.modifying_request_parameters()
        args['actionlist'] = self.params['actionlist']
        args['time_suffix'] = self.params.get('time_suffix', strftime("_%H-%M-%S", localtime()))
        args['override_url'] = self.params.get('override_url')
        self.process_core_method('aviary_render', args)


    def aviary_preview_url(self):
        """Получение ссылки на превью для авиари - по ней превью можно скачать без авторизации"""
        args = self.base_args(self.params)
        self.process_core_method('aviary_preview_url', args)

    def aviary3_image_info(self):
        """
        Возвращает данные изображения для рендеринга в Aviary:
          * ссылку на превью;
          * ссылку на оригинал;
          * (не реализовано) размеры оригинала.
        """
        args = self.base_args()
        self.process_core_method('aviary3_image_info', args)

    def aviary3_save_image(self):
        """
        Создает операцию сохранения изображения, находящегося по переданной ссылке.
        """
        args = self.modifying_request_parameters()
        args['filename_suffix'] = self.params.get('filename_suffix', strftime("_%H-%M-%S", localtime()))
        args['url'] = self.params.get('url')
        self.process_core_method('aviary3_save_image', args)

    def albums_create(self):
        """Создаёт пустой альбом."""
        keys = (
            ('uid', 'uid', str),
            ('title', 'title', str, None),
            ('cover', 'cover', str, None),
            ('layout', 'layout', str, None),
            ('flags', 'flags', str, None),
            ('description', 'description', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('album_type', 'album_type', str, AlbumType.PERSONAL.value),
            ('is_public', 'is_public', type_to_bool, True),
        )
        self.process_core_method('albums_create', keys=keys)

    def albums_create_from_folder(self):
        """Создаёт альбом из папки."""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('media_type', 'media_type', str, None),
            ('title', 'title', str, None),
            ('cover', 'cover', str, None),
            ('layout', 'layout', str, None),
            ('flags', 'flags', str, None),
            ('description', 'description', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('album_type', 'album_type', str, AlbumType.PERSONAL.value),
            ('is_public', 'is_public', type_to_bool, True),
        )
        self.process_core_method('albums_create_from_folder', keys=keys)

    def _albums_create_with_items_prepare_keys(self):
        return (
            ('uid', 'uid', str),
            ('no_items', 'no_items', type_to_bool, False),
            ('if_not_exists', 'if_not_exists', int, 0),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('album_type', 'album_type', str, AlbumType.PERSONAL.value),
            ('is_public', 'is_public', type_to_bool, True),
        )

    def albums_create_with_items(self):
        """Создаёт альбом с элементами."""
        keys = self._albums_create_with_items_prepare_keys()
        self.process_core_method('albums_create_with_items', keys=keys)

    def albums_exclude_from_generated(self):
        """Исключает файл из автосгенерированного альбома."""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('album_type', 'album_type', str),
        )
        self.process_core_method('albums_exclude_from_generated', keys=keys)

    def albums_list(self):
        """Возвращает список альбомов."""
        keys = (
            ('uid', 'uid', str),
            ('count_items', 'count_items', int, 0),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('amount', 'amount', int, 0),
            ('offset', 'offset', int, 0),
            ('album_type', 'album_type', str, AlbumType.PERSONAL.value),
        )
        self.process_core_method('albums_list', keys=keys)

    def album_get(self):
        """Возвращает альбом вместе со всем содержимым."""
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('amount', 'amount', int, None),
            ('last_item_id', 'last_item_id', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('album_get', keys=keys)

    def album_item_info(self):
        """Возвращает один элемент альбома."""
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str, None),
            ('item_id', 'item_id', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('album_item_info', keys=keys)

    def album_item_check(self):
        """Проверяет наличие ресурса в одном из альбомов пользователя"""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('album_id', 'album_id', str, None),
        )
        self.process_core_method('album_item_check', keys=keys)

    def album_set_attr(self):
        """Устанавливает значения атрибутов альбома."""
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('title', 'title', str, None),
            ('cover', 'cover', str, None),
            ('cover_offset_y', 'cover_offset_y', float, None),
            ('layout', 'layout', str, None),
            ('flags', 'flags', str, None),
            ('description', 'description', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('no_items', 'no_items', type_to_bool, False),
        )
        has_extra_params = False
        method_name = 'album_set_attr'
        set_attr_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = set_attr_settings['enabled']
        proxy_enabled_uids = set_attr_settings['enabled_uids']

        supported_java_params = set(set_attr_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            response = djfs_albums.album_set_attr(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_append_items(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('if_not_exists', 'if_not_exists', int, 0),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
        )
        has_extra_params = False
        method_name = 'album_append_items'
        album_append_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = album_append_settings['enabled']
        proxy_enabled_uids = album_append_settings['enabled_uids']

        supported_java_params = set(album_append_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            data = '[%s]' % (','.join(['"%s"' % x.get('path') for x in from_json(self.req.http_req.data).get('items')]))
            response = djfs_albums.album_append_items(self.params, self.req.meta, data)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_append_item(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('type', 'type', str),
            ('path', 'path', str, None),
            ('src_album_id', 'src_album_id', str, None),
            ('if_not_exists', 'if_not_exists', int, 0),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
        )
        has_extra_params = False
        method_name = 'album_append_items'
        album_append_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = album_append_settings['enabled']
        proxy_enabled_uids = album_append_settings['enabled_uids']

        supported_java_params = set(album_append_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            body = '["%s"]' % self.params.get('path')
            response = djfs_albums.album_append_items(self.params, self.req.meta, body)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return
        self.process_core_method('album_append_item', keys=keys)

    def album_copy(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('album_copy', keys=keys)

    def album_remove(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('connection_id', 'connection_id', str, ''),
        )
        has_extra_params = False
        method_name = 'album_remove'
        album_remove_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = album_remove_settings['enabled']
        proxy_enabled_uids = album_remove_settings['enabled_uids']

        supported_java_params = album_remove_settings['supported_params']
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            response = djfs_albums.album_remove(self.params)
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_item_move(self):
        keys = (
            ('uid', 'uid', str),
            ('item_id', 'item_id', str),
            ('album_id', 'album_id', str, None),
            ('to_index', 'to_index', int),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('album_item_move', keys=keys)

    def album_item_remove(self):
        keys = (
            ('uid', 'uid', str),
            ('item_id', 'item_id', str),
            ('album_id', 'album_id', str, None),
            ('connection_id', 'connection_id', str, ''),
        )
        has_extra_params = False
        method_name = 'album_item_remove'
        album_item_remove_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = album_item_remove_settings['enabled']
        proxy_enabled_uids = album_item_remove_settings['enabled_uids']

        supported_java_params = set(album_item_remove_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or
            self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            response = djfs_albums.album_item_remove(self.params)
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_item_set_attr(self):
        keys = (
            ('uid', 'uid', str),
            ('item_id', 'item_id', str),
            ('album_id', 'album_id', str, None),
            ('description', 'description', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
        )
        self.process_core_method('album_item_set_attr', keys=keys)

    def album_publish(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('no_items', 'no_items', type_to_bool, False),
        )
        has_extra_params = False
        method_name = 'album_publish'
        set_attr_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = set_attr_settings['enabled']
        proxy_enabled_uids = set_attr_settings['enabled_uids']

        supported_java_params = set(set_attr_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            response = djfs_albums.album_publish(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_unpublish(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
            ('connection_id', 'connection_id', str, ''),
            ('no_items', 'no_items', type_to_bool, False),
        )
        has_extra_params = False
        method_name = 'album_unpublish'
        set_attr_settings = settings.java_djfs_albums['proxy'][method_name]
        proxy_enabled = set_attr_settings['enabled']
        proxy_enabled_uids = set_attr_settings['enabled_uids']

        supported_java_params = set(set_attr_settings['supported_params'])
        if self.params.viewkeys() > supported_java_params:
            log.error('JAVAALBUM-%s unsupported extra parameters: %s' % (method_name, self.params.viewkeys()))
            has_extra_params = True

        if (proxy_enabled or self.params.get('uid') in proxy_enabled_uids) and not has_extra_params:
            response = djfs_albums.album_unpublish(self.params, self.req.meta)
            self.req.enforce_default_formatter()
            self.req.set_result(response)
            return
        self.process_core_method(method_name, keys=keys)

    def album_video_streams(self):
        keys = (
            ('uid', 'uid', str),
            ('album_id', 'album_id', str),
            ('item_id', 'item_id', str),
            ('user_ip', 'user_ip', str, ''),
            ('use_http', 'use_http', int, 0),
            ('client_id', 'client_id', str, '')
        )
        self.process_core_method('album_video_streams', keys=keys)

    def album_find_in_favorites(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('album_find_in_favorites', keys=keys)

    def public_album_block(self):
        keys = (
            ('public_key', 'public_key', str),
            ('reason', 'reason', str),
        )
        self.process_core_method('public_album_block', keys=keys)

    def public_album_unblock(self):
        keys = (
            ('public_key', 'public_key', str),
        )
        self.process_core_method('public_album_unblock', keys=keys)

    def public_album_get(self):
        keys = (
            ('public_key', 'public_key', str),
            ('uid', 'uid', str, None),
            ('amount', 'amount', int, None),
            ('last_item_id', 'last_item_id', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('public_album_get', keys=keys)

    def public_album_resources_list(self):
        keys = (
            ('public_key', 'public_key', str),
            ('uid', 'uid', str, None),
            ('amount', 'amount', int, None),
            ('last_item_id', 'last_item_id', str, None),
        )
        self.process_core_method('public_album_get', keys=keys)

    def public_album_items_list(self):
        keys = (
            ('public_key', 'public_key', str),
            ('uid', 'uid', str, None),
            ('amount', 'amount', int, None),
            ('last_item_id', 'last_item_id', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('public_album_items_list', keys=keys)

    def public_album_check(self):
        keys = (
            ('public_key', 'public_key', str),
            ('uid', 'uid', str, None),
        )
        self.process_core_method('public_album_check', keys=keys)

    def public_album_save(self):
        keys = (
            ('uid', 'uid', str),
            ('public_key', 'public_key', str),
            ('item_id', 'item_id', str, None),
            ('path', 'path', str, None),
            ('item_name', 'item_name', str, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('public_album_save', keys=keys)

    def async_public_album_save(self):
        keys = (
            ('uid', 'uid', str),
            ('public_key', 'public_key', str),
            ('item_id', 'item_id', str, None),
            ('path', 'path', str, None),
            ('item_name', 'item_name', str, None),
        )
        self.process_core_method('async_public_album_save', keys=keys)

    def public_album_download_url(self):
        keys = (
            ('uid', 'uid', str, None),
            ('public_key', 'public_key', str),
        )
        self.process_core_method('public_album_download_url', keys=keys)

    def public_album_item_download_url(self):
        keys = (
            ('uid', 'uid', str, None),
            ('public_key', 'public_key', str),
            ('item_id', 'item_id', str),
            ('inline', 'inline', int, 0),
        )
        self.process_core_method('public_album_item_download_url', keys=keys)

    def public_album_item_info(self):
        keys = (
            ('uid', 'uid', str, None),
            ('public_key', 'public_key', str),
            ('item_id', 'item_id', str),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', int, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        self.process_core_method('public_album_item_info', keys=keys)

    def public_album_item_video_url(self):
        keys = (
            ('uid', 'uid', str, None),
            ('public_key', 'public_key', str),
            ('item_id', 'item_id', str),
            ('yandexuid', 'yandexuid', str, None)
        )
        self.process_core_method('public_album_item_video_url', keys=keys)

    def public_album_social_wall_post(self):
        keys = (
            ('provider', 'provider', str),
            ('uid', 'uid', str),
            ('public_key', 'public_key', str),
        )
        self.process_core_method('public_album_social_wall_post', keys=keys)

    def async_public_album_social_wall_post(self):
        keys = (
            ('provider', 'provider', str),
            ('uid', 'uid', str),
            ('public_key', 'public_key', str, None),
            ('album_id', 'album_id', str, None),
        )
        self.process_core_method('async_public_album_social_wall_post', keys=keys)

    def public_album_search_bulk_info(self):
        keys = (
            ('public_key', 'public_key', str),
            ('file_ids', 'file_ids', str),
            ('sort', 'sort', str, ''),
            ('order', 'order', int, 1),
            ('search_meta', 'search_meta', str, 'file_id,width,height,cost_disk_aethetic_0'),
        )
        self.process_core_method('public_album_search_bulk_info', keys=keys)

    def fotki_album_public_url(self):
        keys = (
            ('uid', 'uid', str, '0'),
            ('owner_uid', 'owner_uid', str),
            ('fotki_album_id', 'fotki_album_id', int),  # int32
        )
        self.process_core_method('fotki_album_public_url', keys=keys)

    def fotki_album_item_public_url(self):
        keys = (
            ('uid', 'uid', str, '0'),
            ('owner_uid', 'owner_uid', str),
            ('path', 'path', str),
            ('fotki_album_id', 'fotki_album_id', int),  # int32
        )
        self.process_core_method('fotki_album_item_public_url', keys=keys)

    def list_public(self):
        keys = (
            ('uid', 'uid', str),
            ('type', 'type', str, None),
            ('offset', 'offset', int, 0),
            ('amount', 'amount', int, None),
            ('preview_size', 'preview_size', str, None),
            ('preview_crop', 'preview_crop', str, 0),
            ('preview_quality', 'preview_quality', int, 0),
            ('preview_allow_big_size', 'preview_allow_big_size', type_to_bool, False),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(core.list_public(self.req))

    def office_action_data(self):
        keys = (
            ('uid', 'uid', str),
            ('action', 'action', str),
            ('path', 'path', str, None),
            ('service_id', 'service_id', str, None),
            ('service_file_id', 'service_file_id', str, None),
            ('filename', 'filename', str, None),
            ('ext', 'ext', str, None),
            ('size', 'size', int, 0),
            ('locale', 'locale', str, None),
            ('post_message_origin', 'post_message_origin', str, ''),
            ('connection_id', 'connection_id', str, ''),
            ('oo_type', 'oo_type', str, 'desktop'),
            ('is_yandex_nets', 'is_yandex_nets', int, 0),
            ('set_office_selection_strategy', 'set_office_selection_strategy', str, None),
            ('source', 'source', str, None),
            ('region', 'region', str, None),
            ('disable_document_permissions', 'disable_document_permissions', str, ''),
        )
        self.process_core_method('office_action_data', keys=keys)

    def office_show_change_editor_buttons(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('office_show_change_editor_buttons', keys=keys)

    def office_only_office_callback(self):
        keys = (
            ('uid', 'uid', str),
            ('key', 'oo_key', str),
            ('token', 'token', str),
            ('subdomain', 'subdomain', str),
        )
        self.process_core_method('office_only_office_callback', keys=keys)

    def office_set_editor_type(self):
        keys = (
            ('uid', 'uid', str),
            ('office_online_editor_type', 'office_online_editor_type', str),
        )
        self.process_core_method('office_set_editor_type', keys=keys)

    def office_action_check(self):
        keys = (
            ('uid', 'uid', str),
            ('action', 'action', str, None),
            ('file_name', 'file_name', str, None),
            ('size', 'size', int, 0),
            ('service_id', 'service_id', str, 0),
            ('service_file_id', 'service_file_id', str, None),
            ('source', 'source', str, None),
            ('is_yandex_nets', 'is_yandex_nets', int, 0),
        )
        self.process_core_method('office_action_check', keys=keys)

    def office_switch_to_onlyoffice(self):
        keys = (
            ('uid', 'uid', str),
            ('is_yandex_nets', 'is_yandex_nets', int, 0),
        )
        self.process_core_method('office_switch_to_onlyoffice', keys=keys)

    def office_set_access_state(self):
        keys = (
            ('uid', 'uid', str),
            ('access_state', 'access_state', str),
            ('resource_id', 'resource_id', str),
            ('set_office_selection_strategy', 'set_office_selection_strategy', str, None),
            ('connection_id', 'connection_id', str, ''),
            ('region', 'region', str, None),
        )
        self.process_core_method('office_set_access_state', keys=keys)

    def office_get_access_state(self):
        keys = (
            ('uid', 'uid', str),
            ('office_doc_id', 'office_doc_id', str),
        )
        self.process_core_method('office_get_access_state', keys=keys)

    def office_set_selection_strategy(self):
        keys = (
            ('uid', 'uid', str),
            ('selection_strategy', 'selection_strategy', str),
            ('connection_id', 'connection_id', str, ''),
            ('region', 'region', str, None),
        )
        self.process_core_method('office_set_selection_strategy', keys=keys)

    def office_get_file_filters(self):
        self.process_core_method('office_get_file_filters')

    def office_get_file_urls(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
        )
        self.process_core_method('office_get_file_urls', keys=keys)

    def office_info_by_office_doc_short_id(self):
        """Получить информацию о файле по owner_uid и office_doc_short_id."""
        keys = (
            ('uid', 'uid', str),
            ('office_doc_short_id', 'office_doc_short_id', str),
            ('owner_uid', 'owner_uid', str, ''),
        )
        self.process_core_method('office_info_by_office_doc_short_id', keys=keys)

    def office_store(self):
        """Аплоад файла от office-online."""
        keys = (
            ('resource_id', 'resource_id', str),
            ('access_token', 'access_token', str),
            # ('access_token_ttl', 'access_token_ttl', str, None),
        )
        args = {
            'client_type': self.client
        }
        self.process_core_method('office_store', args, keys=keys)

    def office_hancom_lock(self):
        keys = (
            ('uid', 'uid', str),
            ('owner_uid', 'owner_uid', str),
            ('file_id', 'file_id', str),
        )
        self.process_core_method('office_hancom_lock', keys=keys)

    def office_hancom_unlock(self):
        keys = (
            ('uid', 'uid', str),
            ('oid', 'oid', str),
        )
        self.process_core_method('office_hancom_unlock', keys=keys)

    def office_hancom_store(self):
        keys = (
            ('uid', 'uid', str),
            ('oid', 'oid', str),
        )
        self.process_core_method('office_hancom_store', keys=keys)

    def office_download_redirect(self):
        """Редирект на заберун для office-online."""
        keys = (
            ('resource_id', 'resource_id', str),
            ('access_token', 'access_token', str),
        )
        self.process_core_method('office_download_redirect', keys=keys)

    def office_lock(self):
        """Установить или заменить лок."""
        keys = (
            ('uid', 'uid', str),
            ('owner_uid', 'owner_uid', str),
            ('file_id', 'file_id', str),
            ('office_lock_id', 'office_lock_id', str),
            ('old_office_lock_id', 'old_office_lock_id', str, ''),
        )
        self.process_core_method('office_lock', keys=keys)

    def office_unlock(self):
        """Снять лок."""
        keys = (
            ('uid', 'uid', str),
            ('owner_uid', 'owner_uid', str),
            ('file_id', 'file_id', str),
            ('office_lock_id', 'office_lock_id', str),
        )
        self.process_core_method('office_unlock', keys=keys)

    def office_rename(self):
        """Переименовать файл."""
        keys = (
            ('uid', 'uid', str),
            ('owner_uid', 'owner_uid', str),
            ('file_id', 'file_id', str),
            ('new_name', 'new_name', str),
            ('office_lock_id', 'office_lock_id', str)
        )
        self.process_core_method('office_rename', keys=keys)

    def office_generate_online_editor_url(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
        )
        self.process_core_method('office_generate_online_editor_url', keys=keys)

    def bulk_download_prepare(self):
        """
        Подготовка операции для скачивания группы файлов в архиве
        Принимает запрос с параметром uid и телом {"items": ["path1", "path2", "path3"]}
        Возвращает ссылку на заберун, содержащую uid и oid
        """
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('bulk_download_prepare', keys=keys)

    def public_bulk_download_prepare(self):
        """Подготовить операцию для скачивания группы публичных файлов/папок в архиве.

        Принимает запрос с параметром uid и телом
        {"items": ["public_hash_1:/relative", "public_hash_2", "public_hash_3:/relative"]}.
        Возвращает ссылку на заберун, содержащую uid и oid.
        """
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('public_bulk_download_prepare', keys=keys)

    def bulk_download_list(self):
        """
        Получение списка файлов из операции, созданной ручкой bulk_download_prepare
        Принимает запрос с параметром uid и oid, полученный из ручки bulk_download_prepare
        Формирует список файлов для архива и отдает в заберун
        """
        keys = (
            ('uid', 'uid', str),
            ('oid', 'oid', str),
        )
        self.process_core_method('bulk_download_list', keys=keys)

    def list_installers(self):
        """
        Получение списка инсталляторов с ссылками

        Ручка получает uid пользователя, если он залогинен и uid='0', если нет. Если пользователь залогинен, то можно
        передавать параметр al=1, тогда будут возвращаться ссылки на заберун с автологином, если al=0 или параметр
        не задан или пользователь не залогинен (uid=0), то возвращаем обычные ссылки.
        src - опциональный параметр, транслиуремый в заберун. Если пустая строка, то используется дефолтное значение
        """
        keys = (
            ('uid', 'uid', str, '0'),
            ('auto_login', 'al', type_to_bool, False),
            ('installer_source', 'src', str, None),
            ('open_url_after_install', 'open_url_after_install', str, None),
        )
        self.process_core_method('list_installers', keys=keys)

    def comments_owner(self):
        keys = (
            ('uid', 'uid', str),
            ('entity_type', 'entity_type', str),
            ('entity_id', 'entity_id', str),
        )
        self.process_core_method('comments_owner', keys=keys)

    def comments_permissions(self):
        keys = (
            ('uid', 'uid', str),
            ('entity_type', 'entity_type', str),
            ('entity_id', 'entity_id', str),
        )
        self.process_core_method('comments_permissions', keys=keys)

    def info_by_comment_id(self):
        args = {
            'uid': self.params.pop('uid'),
            'entity_type': self.params.pop('entity_type', None),
            'entity_id': self.params.pop('entity_id', None),
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None)
        }
        self.process_core_method('info_by_comment_id', args)

    def _lenta_block_list_prepare_args_and_keys(self):
        keys = (
            ('uid', 'uid', str),
            ('block_id', 'block_id', str, None),
            ('media_type', 'media_type', str, None),
            ('path', 'path', str, None),
            ('resource_id', 'resource_id', str, None),
            ('iteration_key', 'iteration_key', str, None),
            ('modify_uid', 'modify_uid', str, None),
            ('filter_photounlim', 'filter_photounlim', int, 0),
        )

        # Руками устанавливаем фильтры, чтоб они подхватились в
        # mpfs.core.services.disk_service.MPFSStorageService.get_request_parameters()
        filter_ = {
            'resource_type': 'file',
        }

        media_type = self.params.get('media_type', None)
        if media_type:
            media_types = media_type.split(',')
            mt_lst = LentaMediaType.clean_up_the_mess(media_types)
            mt_str = ','.join(mt_lst)
            self.params['media_type'] = mt_str
            filter_['media_type'] = mt_str

        mtime_gte = self.params.pop('mtime_gte', None)
        mtime_lte = self.params.pop('mtime_lte', None)

        if mtime_gte is None:
            raise ValueError('parameter "mtime_gte" must be defined')

        offset = int(self.params.get('offset') or 0)
        amount = int(self.params.get('amount') or 100)
        iteration_key = self.params.get('iteration_key', None)
        if iteration_key:
            # если передан iteration_key, то считаем его более приоритетным
            # и получаем mtime_lte и offset из него
            mtime_lte, offset = iteration_key.split('/')
            mtime_lte, offset = map(int, (mtime_lte, offset))

        if mtime_gte and mtime_lte:
            mtime_lte = int(mtime_lte)
            filter_['mtime_interval'] = '%s,%s' % (mtime_gte, mtime_lte)
        elif mtime_gte:
            filter_['mtime_gte'] = mtime_gte
        elif mtime_lte:
            filter_['mtime_lte'] = mtime_lte

        bounds = {
            'sort': 'mtime',
            'order': 0,  # от новых файлов к старым
            'offset': offset,
            'amount': amount,
        }

        args = {
            'args': {'filter': filter_, 'bounds': bounds},
        }
        return args, keys

    def lenta_block_list(self):
        args, keys = self._lenta_block_list_prepare_args_and_keys()
        args['args']['bounds']['max_sample_size'] = settings.lenta['max_results_count']
        self.process_core_method('lenta_block_list', args=args, keys=keys)

    def lenta_block_public_link(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('lenta_block_public_link', keys=keys)

    def lenta_create_album_from_block(self):
        args, keys = self._lenta_block_list_prepare_args_and_keys()
        keys += self._albums_create_with_items_prepare_keys()
        new_keys = (
            ('layout', 'layout', str, 'waterfall'),
            ('flags', 'flags', str, 'show_tags'),
            ('title', 'title', str)
        )
        keys += new_keys
        self.process_core_method('lenta_create_album_from_block', args=args, keys=keys)

    def versioning_get_checkpoints(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
            ('iteration_key', 'iteration_key', str, None),
        )
        self.process_core_method('versioning_get_checkpoints', keys=keys)

    def versioning_get_folded(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
            ('iteration_key', 'iteration_key', str),
        )
        self.process_core_method('versioning_get_folded', keys=keys)

    def versioning_restore(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
            ('version_id', 'version_id', str),
        )
        self.process_core_method('versioning_restore', keys=keys)

    def versioning_save(self):
        keys = (
            ('uid', 'uid', str),
            ('resource_id', 'resource_id', str),
            ('version_id', 'version_id', str),
        )
        self.process_core_method('versioning_save', keys=keys)

    def versioning_get_version(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str, None),
            ('resource_id', 'resource_id', str, None),
            ('version_id', 'version_id', str),
        )
        self.process_core_method('versioning_get_version', keys=keys)

    def locks_list(self):
        """Возвращает список lock'ов."""
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('locks_list', keys=keys)

    def locks_set(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
        )
        self.process_core_method('locks_set', keys=keys)

    def locks_delete(self):
        """Возвращает список lock'ов."""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
        )
        self.process_core_method('locks_delete', keys=keys)

    def staff_user_changed_callback(self):
        keys = (
            ('yateam_uid', 'yateam_uid', str),
        )
        self.process_core_method('staff_user_changed_callback', keys=keys)

    def staff_make_yateam_admin(self):
        keys = (
            ('yateam_uid', 'yateam_uid', str),
            ('uid', 'uid', str),
        )
        self.process_core_method('staff_make_yateam_admin', keys=keys)

    def staff_reset_yateam_admin(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('staff_reset_yateam_admin', keys=keys)

    def passport_user_2fa_changed(self):
        self.process_core_method('passport_user_2fa_changed')

    def passport_user_enabled_changed(self):
        self.process_core_method('passport_user_enabled_changed')

    def organization_event(self):
        self.process_core_method('organization_event')

    def office_enable_hancom(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('office_enable_hancom', keys=keys)

    def office_disable_hancom(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('office_disable_hancom', keys=keys)

    def enable_unlimited_autouploading(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('enable_unlimited_autouploading', keys=keys)

    def disable_unlimited_autouploading(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('disable_unlimited_autouploading', keys=keys)

    def set_unlimited_autouploading(self):
        keys = (
            ('uid', 'uid', str),
        )
        args = {
            # 'uid': self.params['uid'],
            'optional_params': {}
        }
        if self.params.get('unlimited_video_autoupload_enabled') is not None:
            args['optional_params']['video_status'] = bool(int(self.params['unlimited_video_autoupload_enabled']))
            if not self.params['unlimited_video_autoupload_reason'].lower() in core.VALID_UNLIMITED_AUTOUPLOAD_REASON:
                raise BadRequestError(extra_msg='Bad unlimited video autoupload reason')
            args['optional_params']['video_reason'] = self.params['unlimited_video_autoupload_reason']

        if self.params.get('unlimited_photo_autoupload_enabled') is not None:
            args['optional_params']['photo_status'] = bool(int(self.params['unlimited_photo_autoupload_enabled']))

        if self.params.get('is_asessor'):
            args['optional_params']['is_asessor'] = True

        self.process_core_method('set_unlimited_autouploading', args, keys)

    def is_user_in_unlim_experiment(self):
        keys = (
            ('uid', 'uid', str),
        )
        args = {
            'is_experiment': self.params.get('is_experiment'),
            'is_rkub_experiment': self.params.get('is_rkub_experiment'),
        }
        self.process_core_method('is_user_in_unlim_experiment', args, keys)

    def generate_zaberun_url(self):
        args = {
            'uid': self.params.pop('uid', '0'),
            'stid': self.params.pop('stid'),
            'file_name': self.params.pop('file_name'),
            'url_type': self.params.pop('url_type'),
            'optional_params': {}
        }
        optional_params = (
            ('md5', normalized_hash), ('content_type', str), ('hash', str), ('parser', str), ('hid', str), ('media_type', str),
            ('size', str),
            ('limit', int), ('fsize', int), ('expire_seconds', int), ('crop', int),
            ('inline', type_to_bool), ('eternal', type_to_bool), ('owner_uid', str),
        )
        for key, converter in optional_params:
            if key in self.params:
                args['optional_params'][key] = converter(self.params.get(key))
        self.process_core_method('generate_zaberun_url', args)

    def set_ps_billing_feature(self):
        args = {
            'uid': self.params.pop('uid',),
            'feature_name':  self.params.pop('feature_name',),
            'value':  self.params.pop('value',),
        }
        self.process_core_method('set_ps_billing_feature', args)

    def set_user_overdraft_date(self):
        keys = (
            ('uid', 'uid', str),
            ('overdraft_date', 'overdraft_date', str),
            ('force', 'force', int),
        )
        self.process_core_method('set_user_overdraft_date', keys=keys)

    def can_delete(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('can_delete', keys=keys)

    def promo_code_activate(self):
        keys = (
            ('uid', 'uid', str),
            ('promo_code', 'promo_code', str),
        )
        self.process_core_method('promo_code_activate', keys=keys)

    def photounlim_last_modified(self):
        """Недавно измененный файлы раздела /photounlim"""
        self.params['path'] = PHOTOUNLIM_AREA_PATH
        args = self.base_args(self.params)

        offset = 0
        iteration_key = self.params.get('iteration_key', None)
        # параметр итерирования приоритетен перед параметром начальной даты листинга
        if iteration_key:
            key = PhotounlimLastModifiedIterationKey.parse(iteration_key)
            mtime_gte = key.get_mtime_gte()
            offset = key.get_offset()
        else:
            mtime_gte = self.params.pop('start_date', None)

        filters = {}
        if mtime_gte:
            filters['mtime_gte'] = mtime_gte

        args.update({
            'args': {
                'bounds': {
                    'amount': int(self.params.pop('amount', 100)),
                    'offset': offset,
                    'order': 1,
                    'sort': 'mtime',
                },
                'filter': filters,
            },
            # Передаем прошлый ключ итерирования, чтобы вернуть его же, если новых элементов нет
            'prev_iteration_key': iteration_key,
            tags.META_NAMES: self.params.pop(tags.META_NAMES, None),
        })
        self.process_core_method('timeline', args)

    def rostelecom_activate(self):
        keys = (
            ('uid', 'uid', str),
            ('service_key', 'service_key', str),
            ('rostelecom_uid', 'rostelecom_uid', str, None),
        )
        self.process_core_method('rostelecom_activate', keys=keys)

    def rostelecom_deactivate(self):
        keys = (
            ('uid', 'uid', str),
            ('service_key', 'service_key', str),
            ('rostelecom_uid', 'rostelecom_uid', str, None),
        )
        self.process_core_method('rostelecom_deactivate', keys=keys)

    def rostelecom_unfreeze(self):
        keys = (
            ('uid', 'uid', str),
            ('service_key', 'service_key', str),
            ('rostelecom_uid', 'rostelecom_uid', str, None),
        )
        self.process_core_method('rostelecom_unfreeze', keys=keys)

    def rostelecom_freeze(self):
        keys = (
            ('uid', 'uid', str),
            ('service_key', 'service_key', str),
            ('rostelecom_uid', 'rostelecom_uid', str, None),
        )
        self.process_core_method('rostelecom_freeze', keys=keys)

    def rostelecom_list_services(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('rostelecom_list_services', keys=keys)

    def notes_init(self):
        keys = (
            ('uid', 'uid', str),
            ('src', 'src', str),
            ('dst', 'dst', str),
            ('note_revision_created', 'note_revision_created', str),
            ('note_attachment_mtime', 'note_attachment_mtime', str),
        )
        self.process_core_method('notes_init', keys=keys)

    def add_source_ids(self):
        keys = (
            ('uid', 'uid', str),
            ('md5', 'md5', str),
            ('sha256', 'sha256', str),
            ('size', 'size', int),
            ('is_live_photo', 'is_live_photo', type_to_bool, False),
        )
        self.process_core_method('add_source_ids', keys=keys)

    def check_source_id(self):
        keys = (
            ('uid', 'uid', str),
            ('source_id', 'source_id', str),
        )
        self.process_core_method('check_source_id', keys=keys)

    def check_source_ids(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('check_source_ids', keys=keys)

    def read_deletion_log(self):
        keys = (
            ('uid', 'uid', str),
            ('deletion_log_revision', 'deletion_log_revision', int, 0),
        )
        self.process_core_method('read_deletion_log', keys=keys)

    def check_folder_content_media_type(self):
        """Размер и количество файлов в директории"""
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
        )
        self.process_core_method('check_folder_content_media_type', keys=keys)

    def public_check_folder_content_media_type(self):
        """Размер и количество файлов в публичной директории"""
        keys = (
            ('private_hash', 'private_hash', str),
            ('uid', 'uid', str, None),
        )
        args = {
            'password': self.req.request_headers.get(core.PUBLIC_PASSWORD_HEADER),
            'password_token': self.req.request_headers.get(core.PUBLIC_PASSWORD_TOKEN_HEADER),
        }

        self.process_core_method('public_check_folder_content_media_type', keys=keys, args=args)

    def telemost_fos(self):
        keys = (
            ('app_version', 'app_version', str),
            ('reply_email', 'reply_email', str),
            ('os_version',  'os_version', str),
            ('subject', 'subject', str),
            ('recipient_type', 'recipient_type', str),
            ('uid', 'uid', str),
            ('ip', 'ip', str),
        )
        self.process_core_method('telemost_fos', keys=keys)
