# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.frontend.api.disk import DiskApi
from mpfs.frontend.request import ServiceRequest
from mpfs.frontend.formatter.disk.service import Service as service_formatter
from mpfs.common.static.tags.push import *
from mpfs.common.util import normalized_hash

log = mpfs.engine.process.get_default_log()


class Service(DiskApi):
    """
    API для сервисных функций
    Например, для приема коллбека от кладуна
    """
    client = 'service'
    formatter = service_formatter
    req_class = ServiceRequest

    def kladun_callback(self):
        args = {
            'uid': str(self.params['uid']),
            'oid': str(self.params['oid']),
            'body': self.params['status_xml'],
            'stage': self.params.get('type', None)
        }
        self.process_core_method('kladun_callback', args)

    def passport_callback(self):
        args = {
            'v': self.params['v'],
            'uid': self.params['uid'],
            'event': self.params['event'],
            'timestamp': self.params['timestamp'],
        }
        self.process_core_method('passport_callback', args)

    def kladun_download_counter_inc(self):
        """
        https://jira.yandex-team.ru/browse/CHEMODAN-10144

        Две следующие строчки нужны для того, чтобы преобразовать явно цифровой
        хеш (такие бывают) в строковый. Принудительное преобразование всех хешей
        в строку НЕ делаем, иначе получаем то, что в таске сверху.
        """
        hash = self.params['hash']
        hash = str(hash) if hash.isdigit() else hash

        args = {
            'hash': hash,
        }
        try:
            args['bytes_downloaded'] = int(self.params.pop('bytes'))
        except (KeyError, ValueError):
            args['bytes_downloaded'] = 0
        try:
            args['count'] = int(self.params.pop('count'))
        except (KeyError, ValueError):
            args['count'] = 1
        self.process_core_method('kladun_download_counter_inc', args)

    def direct_url(self):
        args = self.base_args()
        args['modified'] = self.req.request_headers.get('If-Modified-Since')
        self.process_core_method('direct_url', args)

    def public_direct_url(self):
        args = {
            'private_hash': self.params['private_hash'],
            'modified': self.req.request_headers.get('If-Modified-Since'),
            'uid': self.params.get('uid'),
        }
        self.process_core_method('public_direct_url', args)

    def dv_data(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('version_id', 'version_id', str, None),
        )
        self.process_core_method('dv_data', keys=keys)

    def public_dv_data(self):
        keys = (
            ('private_hash', 'private_hash', str),
            ('uid', 'uid', str, None),
        )
        self.process_core_method('public_dv_data', keys=keys)

    def _xiva_subscribe_args(self):
        args = {
            'uid': str(self.params.pop('uid')),
            'callback': self.params.pop('callback'),
            'service_callback': self.params.pop('service_callback', None),
            'return_id': bool(self.params.pop('return_id', 0)),
        }

        options = self.params
        try:
            channel = options.pop(CHANNEL)
            if channel:
                options[FORMAT] = channel
        except KeyError, e:
            pass

        args['options'] = options
        return args

    def xiva_subscribe(self):
        """
        Подписка на отсылку push уведомлений
        """
        args = self._xiva_subscribe_args()
        self.process_core_method('push_subscribe', args)

    def async_xiva_subscribe(self):
        """
        Асинхронная подписка на отсылку push уведомлений
        """
        args = self._xiva_subscribe_args()
        self.process_core_method('async_push_subscribe', args)

    def xiva_unsubscribe(self):
        """
        Отписка отсылки push уведомлений
        """
        args = {
            'uid': str(self.params['uid']),
            'callback': self.params.get('callback'),
            'subscription_id': self.params.get('subscription_id'),
        }
        self.process_core_method('push_unsubscribe', args)

    def mobile_subscribe(self):
        """Подписка на отсылку push уведомлений для моб. устройств"""
        keys = (
            ('uid', 'uid', str),
            ('token', 'token', str),
            ('allow', 'allow', str, ''),
            # берется из тела запроса
            ('resources', 'resources', str, ''),
        )
        self.process_core_method('mobile_subscribe', keys=keys)

    def mobile_unsubscribe(self):
        """Отписка на отсылку push уведомлений для моб. устройств"""
        keys = (
            ('uid', 'uid', str),
            ('token', 'token', str),
        )
        self.process_core_method('mobile_unsubscribe', keys=keys)

    def inspect(self):
        """
        Количество файлов и занимаемое пространство на сервисе
        """
        args = self.base_args(self.params)
        self.process_core_method('inspect', args)

    def recount(self):
        """
        Добавить пользователя в очередь на пересчёт счётчиков
        """
        args = self.base_args(self.params)
        self.process_core_method('recount', args)

    def echo(self):
        self.req.set_result(self.params)

    def restore_deleted(self):
        """
        Восстановление удалённого файла
        """
        args = {
            'uid': str(self.params['uid']),
            'path': self.params['path'],
            'dest': self.params['dest'],
            'force': self.params['force'],
        }
        self.process_core_method('restore_deleted', args)

    def reindex_search(self):
        keys = (
            ('uid', 'uid', str),
            ('index_type', 'index_type', str, None),
            ('index_body', 'index_body', int, 1),
            ('mediatype', 'mediatype', str, ''),
            ('force', 'force', int, 0)
        )
        self.process_core_method('reindex_search', keys=keys)

    def start_reindex_for_quick_move(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('start_reindex_for_quick_move', keys=keys)

    def check_reindexed_for_quick_move(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('check_reindexed_for_quick_move', keys=keys)

    def reindex_for_quick_move_callback(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('reindex_for_quick_move_callback', keys=keys)

    def mark_stid_deleted(self):
        args = {
            'stid': self.params['stid']
        }
        self.process_core_method('mark_stid_deleted', args)

    def add_sony_tablet_serial(self):
        args = {
            'serial': self.params['serial']
        }
        self.process_core_method('add_sony_tablet_serial', args)

    def remove_sony_tablet_serial(self):
        args = {
            'serial': self.params['serial']
        }
        self.process_core_method('remove_sony_tablet_serial', args)

    def hardlink(self):
        keys = (
            ('md5', 'md5', normalized_hash, ''),
            ('sha256', 'sha256', normalized_hash, ''),
            ('size', 'size', str, ''),
            ('hid', 'hid', str, None),
        )
        self.process_core_method('hardlink', keys=keys)

    def break_counters(self):
        """
        Сломать счётчики, ручка для тестирования
        """
        args = self.base_args(self.params)
        args['type'] = self.params['type']
        self.process_core_method('break_counters', args)

    def update_file_hash(self):
        """
            Обновление хэшей файла через кладун.
        """
        args = self.base_args(self.params)
        args['md5'] = normalized_hash(self.params.get('md5', None))
        args['sha256'] = normalized_hash(self.params.get('sha256', None))
        if 'size' in self.params:
            args['size'] = int(self.params['size'])
        else:
            args['size'] = None
        self.process_core_method('update_file_hash', args)

    def bulk_check_stids(self):
        keys = (
            ('service', 'service', str),
        )
        self.process_core_method('bulk_check_stids', keys=keys)

    def show_settings(self):
        """
        Получение текущих настроек системы.
        """
        self.req.set_result(self.core.show_settings(self.req))

    def user_set_pdd_domain(self):
        """Установить домен из Паспорта для ПДД-пользователя."""
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('user_set_pdd_domain', keys=keys)

    def force_snapshot(self):
        """Форсирование снепшота для пользователя"""
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('force_snapshot', keys=keys)

    def inactive_users_flow_add(self):
        keys = (
            ('uid', 'uid', str),
        )
        self.process_core_method('inactive_users_flow_add', keys=keys)

    def inactive_users_flow_add_from_yt(self):
        keys = (
            ('table_path', 'table_path', str),
            ('start_index', 'start_index', int, None),
            ('end_index', 'end_index', int, None),
        )
        self.process_core_method('inactive_users_flow_add_from_yt', keys=keys)

    def inactive_users_flow_update(self):
        keys = (
            ('uid', 'uid', str),
            ('state', 'state', str, None),
            ('start_time', 'start_time', str, None),
            ('delta_days', 'delta_days', int, None),
        )
        self.process_core_method('inactive_users_flow_update', keys=keys)
