# -*- coding: utf-8 -*-
"""
MPFS
API админки
"""
import mpfs.engine.process
import mpfs.core.support.interface as support

from mpfs.frontend.formatter import support_formatter
from mpfs.frontend.api.disk import DiskApi
from mpfs.core.metastorage.decorators import slave_read_ok
from mpfs.common.util import normalized_hash

log = mpfs.engine.process.get_access_log()
error_log = mpfs.engine.process.get_error_log()


class Support(DiskApi):
    """
    API админки
    Выдает результаты в json
    Описание формата: http://wiki.yandex-team.ru/pochta/ya.disk/MPFS/API/Support
    """
    client = 'support'
    formatter = support_formatter.Support

    def block_public_file(self):
        """
        Блокировка публичного файла
        """
        args = {
            'moderator': self.params.pop('moderator'),
            'comment': self.params.pop('comment'),
            'private_hash': self.params.pop('private_hash'),
            'type': self.params.pop('type', None),
            'notify': int(self.params.pop('notify', 0)),
            'view': self.params.pop('view', 'st'),
            'link': self.params.pop('link', ''),
            'hid_block_type': self.params.pop('hid_block_type', ''),
            'info': self.params,
        }
        self.req.set_args(args)
        self.req.set_result(support.block_public_file(self.req))

    def unblock_file(self):
        args = {
            'uid': str(self.params['uid']),
            'path': self.params['path'],
            'moderator': self.params.pop('moderator'),
            'unblock_hid': self.params.pop('unblock_hid', False),
            'info': self.params,
        }
        self.req.set_args(args)
        self.req.set_result(support.unblock_file(self.req))

    def get_comments(self):
        """
        Получение комментариев по пользователю
        """
        args = {'uid': self.params.pop('uid')}
        args['info'] = self.params
        self.req.set_args(args)
        self.req.set_result(support.get_comments(self.req))

    def find_by_hash(self):
        """
        Поиск заблокированного файла по хэшу
        """
        args = {'hash': self.params.pop('hash')}
        self.req.set_args(args)
        self.req.set_result(support.find_by_hash(self.req))

    def push_log(self):
        """
        Послать клиенту запрос на отсылку логов
        """
        args = {'uid': self.params.pop('uid')}
        self.req.set_args(args)
        self.req.set_result(support.push_log_notify(self.req))

    def block_user(self):
        """
        Заблокировать пользователя
        """
        args = {
            'uid': self.params.pop('uid'),
            'moderator': self.params.pop('moderator'),
            'comment': self.params.pop('comment'),
        }
        self.req.set_args(args)
        self.req.set_result(support.block_user(self.req))
        self.process_core_method('fail_operations', args)

    def unblock_user(self):
        """
        Разблокировать пользователя
        """
        args = {
            'uid': self.params.pop('uid'),
            'moderator': self.params.pop('moderator'),
            'comment': self.params.pop('comment'),
        }
        self.req.set_args(args)
        self.req.set_result(support.unblock_user(self.req))

    def get_public_file_address(self):
        """
        Получить адрес файла по публичному хэшу
        """
        args = {
            'private_hash': self.params.pop('private_hash'),
        }
        self.req.set_args(args)
        self.req.set_result(support.get_public_file_address(self.req))

    def download_url(self):
        """
        Получить url для скачивания файла без авторизации
        """
        args = self.base_args(self.params)
        self.req.set_args(args)
        self.req.set_result(support.download_url(self.req))

    def update_comment(self):
        """
        Обновить данные комметария #CHEMODAN-12175
        """
        address = self.params.pop('address', '')
        args = {
            'address': address,
            'data': self.params,
        }
        self.req.set_args(args)
        self.req.set_result(support.update_comment(self.req))

    def trash_restore_all(self):
        """
        Восстановить всё из корзины
        """
        args = self.base_args(self.params)
        self.req.set_args(args)
        self.req.set_result(support.trash_restore_all(self.req))

    def async_restore_deleted(self):
        """
        Восстановить удалённый ресурс из /hidden_data
        """
        args = self.base_args(self.params)
        args.update({
            'dest': self.params.pop('dest', ''),
            'force': self.params.pop('force', 0),
        })
        self.req.set_args(args)
        self.req.set_result(support.restore_deleted(self.req))

    def async_block_uids_by_hid(self):
        """
        Заблокировать все uid'ы, у которых был опубликован файл с заданным hid'ом.

        Дополнительно можно передать список UIDов, которые будут заблокированы с тем же комментарием.
        """
        additional_uids = self.params.pop('additional_uids', '')
        if additional_uids:
            additional_uids = additional_uids.split(',')
        else:
            additional_uids = []
        keys = (
            ('hid', 'hid', str),
            ('moderator', 'moderator', str),
            ('comment_text', 'comment_text', str),
            ('type', 'type', str, None),
        )
        args = self.compose_args(keys)
        args['additional_uids'] = additional_uids
        self.req.set_args(args)
        self.req.set_result(support.async_block_uids_by_hid(self.req))

    def is_single_owner(self):
        """
        Проверка есть ли у пользователя файл и является ли он едиственным владельцем
        """
        args = {
            'uid': self.params.pop('uid'),
            'size': self.params.pop('size'),
            'md5': normalized_hash(self.params.pop('md5')),
            'sha256': normalized_hash(self.params.pop('sha256')),
        }
        self.req.set_args(args)
        self.req.set_result(support.is_single_owner(self.req))

    def fail_operations(self):
        """
        Удалить операции пользователя
        """
        args = self.base_args(self.params)
        self.req.set_args(args)
        self.process_core_method('fail_operations', args)

    def block_zaberun_url(self):
        """
        Заблокировать stid файла по ссылке на заберун
        """
        args = {
            'hsh': self.params.pop('hsh'),
            'uid': self.params.pop('uid', ''),
            'moderator': self.params.pop('moderator', ''),
            'comment_type': self.params.pop('comment_type', ''),
        }
        self.req.set_args(args)
        self.req.set_result(support.block_zaberun_url(self.req))

    def get_moderation_list(self):
        """
        Получить список для модерации
        """
        args = {
            'sort': self.params.pop('sort', 'created'),
        }

        order = int(self.params.pop('order', 1))
        if order in (-1, 1):
            args['order'] = order
        else:
            args['order'] = 1

        args['bounds'] = dict()
        for bound in ('amount', 'offset'):
            if bound in self.params:
                args['bounds'][bound] = int(self.params.pop(bound))

        args['filters'] = self.params
        self.req.set_args(args)
        self.req.set_result(support.get_moderation_list(self.req))

    def set_status(self):
        """
        Поставить статус
        """
        args = {
            '_id': self.params.pop('_id'),
            'moderator': self.params.pop('moderator'),
            'status': self.params.pop('status'),
        }
        self.req.set_args(args)
        self.req.set_result(support.set_status(self.req))

    def set_limit_by_services(self):
        """
        Выставить лимит по услугам из billing_services
        https://st.yandex-team.ru/CHEMODAN-19881
        """
        args = {
            'uid': self.params.pop('uid'),
            'moderator': self.params.pop('moderator'),
        }
        self.req.set_args(args)
        self.req.set_result(support.set_limit_by_services(self.req))

    def uids_with_public_hid(self):
        """
        Получить uid'ы владельцев опубликованного файла с заданным hid'ом
        """
        keys = (
            ('hid', 'hid', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.uids_with_public_hid(self.req))

    def batch_unblock(self):
        """
        Произвести массовую разблокировку пользователей, хидов и ссылок
        """
        keys = (
            ('id', 'id', str),  # уникальный индентификатор операции массовой блокировки
            ('moderator', 'moderator', str),
            ('comment_text', 'comment_text', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.batch_unblock(self.req))

    def batch_block(self):
        """
        Произвести массовую блокировку пользователей, хидов и ссылок
        """
        keys = (
            ('hids', 'hids', str),
            ('uids', 'uids', str),
            ('public_hashes', 'public_hashes', str),
            ('moderator', 'moderator', str),
            ('comment_text', 'comment_text', str),
            ('comment_type', 'comment_type', str, None),
            ('notify', 'notify', int, 0),
            ('view', 'view', str, 'st'),
            ('link', 'link', str, ''),
            ('hid_block_type', 'hid_block_type', str, 'only_view_delete'),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.batch_block(self.req))

    @slave_read_ok
    def get_block_history(self):
        """
        Получить историю массовых блокировок
        """
        keys = (
            ('offset', 'offset', int, 0),
            ('amount', 'amount', int, 100),
            ('sort', 'sort', str, 'ctime'),
            ('order', 'order', int, -1),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.get_block_history(self.req))

    @slave_read_ok
    def list(self):
        super(Support, self).list()

    @slave_read_ok
    def get_moderation_queue_count(self):
        """
        Получить количество ресурсов с нужными атрибутами в очереди модерации
        """
        keys = (
            ('status', 'status', str, 'not-moderated'),
            ('created', 'created', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.get_moderation_queue_count(self.req))

    @slave_read_ok
    def file_search(self):
        """
        Строго для использования в саппорте!!! Не годится для продовой нагрузки, т.к. нету индексов по имени!!!
        Ручка поиска файлов по uid'у + имени файла в юзер дате, хидден дате и корзине.
        """
        keys = (
            ('uid', 'uid', str),
            ('query', 'query', str),
            ('amount', 'amount', int, 10),
            ('offset', 'offset', int, 0),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.search(self.req))

    def resolve_public_short_url(self):
        keys = (
            ('short_url', 'short_url', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.resolve_public_short_url(self.req))

    def add_user_to_prohibited_cleaning(self):
        """Добавить пользователя в список пользователей с отключённой чисткой корзины и хидден даты."""
        keys = (
            ('uid', 'uid', str),
            ('comment', 'comment', str),
            ('moderator', 'moderator', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.add_user_to_prohibited_cleaning(self.req))

    def remove_user_from_prohibited_cleaning(self):
        """Удалить пользователя из списка пользователей с отключённой чисткой корзины и хидден даты."""
        keys = (
            ('uid', 'uid', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.remove_user_from_prohibited_cleaning(self.req))

    def list_prohibited_cleaning_users(self):
        """Возвратить список пользователей для которых выключена чистка корзины и хидден даты."""
        self.req.set_result(support.list_prohibited_cleaning_users(self.req))

    def get_file_checksums(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.get_file_checksums(self.req))

    def force_remove_folder(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.force_remove_folder(self.req))

    def force_drop_trash(self):
        keys = (
            ('uid', 'uid', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.force_drop_trash(self.req))

    def force_restore_trash_folder(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.force_restore_trash_folder(self.req))

    def force_restore_hidden_folder(self):
        keys = (
            ('uid', 'uid', str),
            ('path', 'path', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.force_restore_hidden_folder(self.req))

    def reindex_faces_album(self):
        keys = (
            ('uid', 'uid', str),
            ('startrek_issue', 'startrek_issue', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.reindex_faces_album(self.req))

    def overdraft_allow_unblock(self):
        keys = [
            ('uid', 'uid', str)
        ]
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.overdraft_allow_unblock(self.req))

    def overdraft_unblock(self):
        keys = [
            ('uid', 'uid', str)
        ]
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.overdraft_unblock(self.req))


    def inactive_users_flow_allow_unblock(self):
        keys = [
            ('uid', 'uid', str)
        ]
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.inactive_users_flow_allow_unblock(self.req))

    def inactive_users_flow_unblock(self):
        keys = [
            ('uid', 'uid', str)
        ]
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.inactive_users_flow_unblock(self.req))

    def upload_traffic_counters(self):
        keys = (
            ('uid', 'uid', str),
        )
        args = self.compose_args(keys)
        self.req.set_args(args)
        self.req.set_result(support.upload_traffic_counters(self.req))
