# -*- coding: utf-8 -*-
'''
MPFS
Дефолтный форматтер
'''
import os
import mpfs.engine.process
from mpfs.common.util.urls import quote

from mpfs.config import settings
from mpfs.frontend.formatter.default_formatter import Default
from mpfs.common.static.messages import *
from mpfs.common.static.tags import *

SUPPORTED_PREVIEW_TYPES = set(settings.system['preview_types'])

HEADER_VALUES_MAPPER = {
    'path': 'X-Disk-Path',
    'resource_id': 'X-Disk-Resource-Id',
}


class MPFSFormatter(Default):

    def _postprocess_preview_sizes(self, metadata):
        if SIZES in metadata and PREVIEW_SIZES in metadata:
            sizes, preview_sizes = metadata[SIZES], metadata[PREVIEW_SIZES]
            for item in sizes:
                size_name = item.get(NAME)
                width  = preview_sizes.get(size_name, {}).get(WIDTH)
                height = preview_sizes.get(size_name, {}).get(HEIGHT)
                if width and height:
                    item[WIDTH], item[HEIGHT] = int(width), int(height)

        sizes = metadata.get(SIZES, [])

        if settings.feature_toggles['dynamicpreviews'] and len(sizes) > 1:
            try:
                custom_size = self.request.preview_size
            except AttributeError:
                custom_size = None
            if custom_size:
                custom_crop = self.request.preview_crop
                url = sizes[-1][URL]
                url = url.replace(CUSTOM_SIZE, custom_size)
                url = url.replace(CUSTOM_CROP, str(custom_crop))
                preview_type = getattr(self.request, 'preview_type', None)
                if preview_type and preview_type in SUPPORTED_PREVIEW_TYPES:
                    url += '&preview_type=%s' % preview_type
                preview_animate = getattr(self.request, 'preview_animate', False)
                if preview_animate:
                    url += '&animate=true'
                sizes[-1][URL] = url
                metadata['custom_preview'] = url  # https://st.yandex-team.ru/CHEMODAN-20547
            elif len(sizes) > 0 and sizes[-1]['name'] == 'C':
                # тут все странно, потому что всегда последней лежит превьюшка с именем C и плейсхолдерами
                # CUSTOM_SIZE и CUSTOM_CROP, которые заменяются выше, если мы запросили превьюшку определенного
                # размера, а если нет, то мы удаляем ее, потому что она не нужна. Логично, но если вызвать эту
                # функцию несколько раз для одного ресурса, то она удалит не только эту кастомную превьюшку, но
                # и последнюю "правильную" XXXL тоже
                sizes.pop(-1)

        preview_quality = getattr(self.request, 'preview_quality', None)
        preview_allow_big_size = getattr(self.request, 'preview_allow_big_size', None)

        for preview_field in ('custom_preview', 'preview', 'thumbnail'):
            if preview_field not in metadata:
                continue
            if preview_quality:
                metadata[preview_field] += '&quality=%s' % preview_quality
            if preview_allow_big_size and preview_field == 'custom_preview':
                metadata[preview_field] += '&allow_big_size=1'

        for size in sizes:
            if preview_quality:
                size[URL] += '&quality=%s' % preview_quality
            if preview_allow_big_size and size[NAME] == 'DEFAULT':
                size[URL] += '&allow_big_size=1'

    def folder_id(self, rid):
        if rid[-2:] == ":/":
            return rid
        elif not filter(None, rid.split('/')):
            return '/'
        else:
            return rid + '/'

    def _process_meta(self, form, resource):
        metainfo = dict(filter(lambda (x, y): x not in resource, form.meta.iteritems()))
        metainfo.update(dict(filter(lambda (x, y): x not in resource, form.dict().iteritems())))
        metainfo[VISIBLE] = int(metainfo.get(VISIBLE, 1))
        metainfo.pop('yarovaya_mark', None)

        if not settings.feature_toggles['broken'] and 'broken' in metainfo:
            metainfo['broken'] = 0

        self._postprocess_preview_sizes(metainfo)

        meta = self.request.meta
        if meta is not None:
            if meta:
                metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
            resource[META] = metainfo

            try:
                resource[META][ETIME] = form.etime
            except (AttributeError, KeyError):
                pass

        try:
            meta_names = self.request.meta_names
        except AttributeError:
            pass
        else:
            if meta_names is not None:
                resource[META_NAMES] = metainfo.keys()

        return

    def _base_element(self, form):
        if form.is_file:
            try:
                resource_id = form.key or form.id
            except AttributeError:
                resource_id = form.id
            resource_type = FILE
        else:
            resource_id = self.folder_id(form.id)
            resource_type = DIR

        result = {
                  ID    : resource_id,
                  TYPE  : resource_type,
                  PATH  : form.id,
                  NAME  : form.name,
                  CTIME : form.ctime,
                  MTIME : form.mtime,
                  UTIME : form.utime,
                  }
        try:
            result[ETIME] = form.etime
        except AttributeError:
            pass
        return result

    def _info(self, *args, **kwargs):
        form = self.request.form
        result = self._base_element(form)
        self._process_meta(form, result)
        return result

    def _transform_relative(self, data, *args, **kwargs):
        resource = data[THIS]
        self.request.form = data[THIS].form

        _this = self._info(*args, **kwargs)
        _this[ID]   = resource.id
        _this[PATH] = resource.path

        folders = []
        files   = []
        for item in data[LIST]:
            transformed = self._transform_relative(item)

            if item[THIS].is_folder:
                folders.append(transformed)
            else:
                files.append(transformed)

        return {THIS: _this, LIST: folders + files }

    def fulltree(self, *args, **kwargs):
        return self.to_json(
            self._transform_relative(
                self.data[RESULT], *args, **kwargs
            )
        )

    def public_fulltree(self, *args, **kwargs):
        return self.fulltree(*args, **kwargs)

    def _add_header_params_to_response(self, header_values_to_add):
        for k, v in header_values_to_add.iteritems():
            if k not in HEADER_VALUES_MAPPER:
                continue
            if k == 'path':
                v_to_put = quote(v, '/')
            elif isinstance(v, unicode):
                v_to_put = v.encode('utf-8')
            else:
                v_to_put = v
            self.request.http_resp.headers[HEADER_VALUES_MAPPER[k]] = v_to_put

    def _parse_and_pull_out_header_params(self, result):
        self._add_header_params_to_response(result.get('_headers', {}))
        result.pop('_headers', None)
        return result
