# -*- coding: utf-8 -*-
import demjson

from copy import deepcopy

import mpfs.engine.process
from mpfs.frontend.formatter.disk import MPFSFormatter
from mpfs.frontend.formatter.util import split_file, split_path
from mpfs.common.static.messages import *
from mpfs.common.static.tags import *
from mpfs.common.errors import *
from mpfs.common.static import messages

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
JSON = deepcopy(demjson.JSON)


class JSONDisordered(JSON):
    def __init__(self, *args, **kwargs):
        super(JSONDisordered, self).__init__(*args, **kwargs)
        self._sort_dictionary_keys = False


demjson.JSON = JSONDisordered


class Desktop(MPFSFormatter):
    
    def default_handler(self, *args, **kwargs):
        return self.to_json(self.request.data[RESULT])

    def list(self, **kw):
        meta = self.request.meta
        data = self.data[RESULT]
        result = []
        file_data = data['this']
        
        if not (file_data['type'] == 'dir' or data.get('list')):
            return self.info(**kw)
            
        root = {
            ID : self.folder_id(file_data[ID]),
            TYPE : "dir",
            PATH : file_data.get(PATH, ""),
            NAME : file_data[NAME],
            }

        for k in (CTIME, MTIME, UTIME):
            root[k] = file_data[k]
    
        metadata = file_data.pop(META, {})
        metainfo = dict(filter(lambda (x, y): x not in root, metadata.iteritems()))
        metainfo.update(dict(filter(lambda (x, y): x not in root, file_data.iteritems())))
        metainfo[VISIBLE] = int(metainfo.get(VISIBLE, 1))
                
        if self.request.meta_names is not None:
            root[META_NAMES] = metainfo.keys()

        if meta is not None:
            if meta:
                metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
            root[META] = metainfo
        
        result.append(root)
        
        for element in data['list']:
            folder_dict = {
                ID : element[ID],
                TYPE : element[TYPE],
                PATH : split_path(element[ID]).pop(),
                NAME : element[NAME],
            }
            for k in (CTIME, MTIME, UTIME):
                folder_dict[k] = element.get(k)
            if element[TYPE] == FILE:
                folder_dict.update({
                    SOURCE : element[SOURCE],
                    MIMETYPE : element.get(MIMETYPE, ''),
                    SIZE : element.get(SIZE)
                })
                try:
                    folder_dict[ETIME] = element[ETIME]
                except KeyError:
                    pass
            else:
                folder_dict[ID] = self.folder_id(folder_dict[ID])
            
            metadata = element.pop(META, {})
            self._postprocess_preview_sizes(metadata)
            
            metadata[VISIBLE] = int(metadata.get(VISIBLE, 1)) 
            metainfo = dict(filter(lambda (x, y): x not in folder_dict, metadata.iteritems()))
            metainfo.update(dict(filter(lambda (x, y): x not in folder_dict, element.iteritems())))
            
            if self.request.meta_names is not None:
                folder_dict[META_NAMES] = metainfo.keys()
            
            if meta is not None:
                if meta:
                    metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
                folder_dict[META] = metainfo
            
            result.append(folder_dict)
            
        return self.to_json(result)
        
    def public_notification(self, **kw):
        return ''

    def public_info(self, **kw):
        self.data[RESULT] = self.data[RESULT]['resource']
        return self.info(**kw)

    def public_copy(self, **kw):
        return ''
        
    def mkdir(self, *args, **kwargs):
        return ''
        
    def mksysdir(self, *args, **kwargs):
        self.data[RESULT]['this'] = self.data[RESULT]
        return self.info(*args, **kwargs)


    def rm(self, **kw):
        return ''

    def user_check(self, **kw):
        return self.to_json({NEED_INIT: self.data[RESULT]})
        
    def user_init(self, **kw):
        return ''

    def copy(self, **kw):
        return ''

    def move(self, **kw):
        return ''

    def setprop(self, **kw):
        return ''

    def hardlink_copy(self, **kw):
        return ''

    def one_element(self, form):
        keys = ('type', 'id', 'sha256', 'md5', 'size')

        key_map = {
            'id' : 'key',
        }
        result = {}
        form_dict = form.dict()
        for key in keys:
            if key in form_dict:
                val = form_dict.pop(key)
                result[key_map.get(key, key)] = val
        if self.request.meta is not None:
            result['meta'] = {}
            if self.request.meta:
                for key in self.request.meta:
                    if key in form_dict:
                        result['meta'][key_map.get(key, key)] = form_dict.get(key)
            else:
                result['meta'] = form_dict
        result['op'] = form.model__op
        return result

    def diff(self, **kw):
        if self.request.meta is None:
            meta_fields = set()
        elif self.request.meta:
            meta_fields = set(self.request.meta)
        else:
            meta_fields = None

        if meta_fields:
            required_fields = set((
                'type', 'uid', 'key', 'op', 'sha256', 'size',
                'md5', 'drweb', 'broken', 'media_type', 'mimetype',
                'etime', 'mtime', 'has_preview'))
            required_fields.update(meta_fields)

            def _filter_element(element):
                return dict([(k, v) for k, v in element.iteritems() if k in required_fields])

            self.request.data['result']['result'] = map(_filter_element, self.request.data['result']['result'])
            
        self.request.http_resp.headers['Item-Count'] = self.request.data['result']['amount']
        self.request.http_resp.headers['Version'] = self.request.data['result']['version']
        return self.to_json(self.request.data['result'])

    def error(self, err):
        response = self.request.http_resp
        if isinstance(err, MPFSRootException):
            response.status = getattr(err, 'response', default_response)
            _resp = {
                'code'  : getattr(err, 'code', ''),
                'title' : getattr(err, 'message', ''),
            }
            data = getattr(err, 'data', '')
            if data:
                _resp['data'] = data
            if getattr(err, 'headers', None):
                self._add_header_params_to_response(err.headers)
            return self.to_json(_resp)
        else:
            try:
                response.status = responses.get(int(err))
            except Exception:
                response.status = default_response

    def info(self, **kw):
        meta = self.request.meta
        data = self.data[RESULT]['this']

        if data[TYPE] == FILE:
            parts, fid = split_file(data[ID], data[NAME])
            id = data[ID]
        else:
            parts = split_path(data[ID])
            fid = id = self.folder_id(data[ID])
            
        result = {
            ID : id,
            TYPE : data[TYPE],
            PATH : fid,
            NAME : data[NAME],
            }
        for k in (CTIME, MTIME, UTIME):
            result[k] = data.get(k) or 0

        try:
            result[ETIME] = data[ETIME]
        except KeyError:
            pass

        if data[TYPE] == FILE:
            result.update({
                SOURCE : data.get(SOURCE),
                MIMETYPE : data.get(MIMETYPE),
                SIZE : data.get(SIZE)
            })
        
        metadata = data.pop(META, {})
        metadata[VISIBLE] = int(metadata.get(VISIBLE, 1)) 
        self._postprocess_preview_sizes(metadata)
                
        metainfo      = dict(filter(lambda (x, y): x not in result, metadata.iteritems()))
        metainfo.update(dict(filter(lambda (x, y): x not in result, data.iteritems())))


        try:
            meta_names = self.request.meta_names
            if meta_names is not None:
                result[META_NAMES] = metainfo.keys()
        except AttributeError:
            pass
            
    
        if meta is not None:
            if meta:
                metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
            result[META] = metainfo
              
        return self.to_json(result)

    def status(self, **kw):
        data = self.data[RESULT]

        # см. mpfs.common.static.messages для объяснения
        status = data[STATUS]
        data[STATUS] = messages.old_operation_titles[status]
        data[STATE] = messages.true_operation_titles[status]
        data[AT_VERSION] = data[AT_VERSION]

        progress_stage = filter(lambda x: x.get(NAME) == 'kladun_upload', data.get(STAGES, []))
        if progress_stage:
            data[PROGRESS] = progress_stage[0].get(PROGRESS, 0)
        
        if PROTOCOL in data:
            for i in xrange(len(data[PROTOCOL])):
                # см. mpfs.common.static.messages для объяснения
                op_status = data[PROTOCOL][i][STATUS]
                data[PROTOCOL][i][STATUS] = messages.old_operation_titles[op_status]
                data[PROTOCOL][i][STATE] = messages.true_operation_titles[op_status]

                del data[PROTOCOL][i][RESULT]
                if FORM in data[PROTOCOL][i]:
                    del data[PROTOCOL][i][FORM]
                    
        return self.to_json(data)


