# -*- coding: utf-8 -*-
import re
from itertools import ifilter

import mpfs.engine.process
from mpfs.config import settings
import mpfs.common.static.messages as messages
import mpfs.common.static.tags as tags
from mpfs.core.albums.logic.common import get_album_item_face_coordinates
from mpfs.core.filesystem.resources.photounlim import PhotounlimLastModifiedIterationKey
from mpfs.core.versioning.dao.version_data import VersionType

from mpfs.frontend.formatter.disk.desktop import Desktop
from mpfs.common.util import from_json, AutoSuffixator, filetypes
from mpfs.core.filesystem.resources.disk import MPFSResource, DiskResource
from mpfs.core.social.publicator import Publicator
from mpfs.core.yateam import logic as yateam_logic

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
SANITIZE_NAME = re.compile(u'[\x00-\x08\x0B-\x0C\x0E-\x1F\x7F]')

class NotSpecified(object):
    pass

FEATURE_TOGGLES_SAVE_REPLACED_YADISK_CC_TO_DATABASE = settings.feature_toggles['save_replaced_yadisk_cc_to_database']


class JSON(Desktop):
    content_type = 'application/json; charset=utf-8'
    LAZY_MPFS_RESOURCE_META_FIELDS = {
        'comment_ids': None,
        'resource_id': lambda x: str(x) if x else NotSpecified(),
    }
    """Доступные lazy meta-поля MPFSResource. `<field_name>: <serializer_func>` """

    @classmethod
    def lazy_meta(cls, mpfs_resource, fields=None):
        """
        Получение словаря с ленивыми meta-полями ресурса.

        :param mpfs_resource: MPFSResource instance.
        :param fields: set. Нужные lazy поля. None - все, empty set - ничего
        """
        if not isinstance(mpfs_resource, MPFSResource):
            return {}

        if fields is None:
            fields = cls.LAZY_MPFS_RESOURCE_META_FIELDS.viewkeys()
        else:
            fields = fields & cls.LAZY_MPFS_RESOURCE_META_FIELDS.viewkeys()

        if not fields:
            return {}

        result = dict()
        for field in fields:
            serializer = cls.LAZY_MPFS_RESOURCE_META_FIELDS.get(field)
            try:
                value = getattr(mpfs_resource, field)
                if serializer is not None:
                    value = serializer(value)
            except Exception:
                error_log.exception('Error during getting or serializing lazy field from resource')
                continue
            if not isinstance(value, NotSpecified):
                result[field] = value
        return result

    def default_success(self):
        return self.to_json({})

    def _process_meta(self, form, resource):
        metainfo = dict(ifilter(lambda (x, y): x not in resource, form.meta.iteritems()))
        metainfo.update(ifilter(lambda (x, y): x not in resource, form.dict().iteritems()))

        visible = metainfo.get(tags.VISIBLE, 1)
        metainfo[tags.VISIBLE] = int(visible) if visible is not None else 1
        metainfo.pop('yarovaya_mark', None)

        if not settings.feature_toggles['broken'] and 'broken' in metainfo:
            metainfo['broken'] = 0

        self._postprocess_preview_sizes(metainfo)

        meta = self.request.meta
        if meta is not None:
            if meta:
                # нужные конкретные meta-поля
                metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
                lazy_fields = set(meta)
            else:
                # нужны все meta-поля
                lazy_fields = None
            lazy_meta = self.lazy_meta(form.model, fields=lazy_fields)
            metainfo.update(lazy_meta)
            resource[tags.META] = metainfo

            try:
                resource[tags.META][tags.ETIME] = form.etime
            except (AttributeError, KeyError):
                pass
            try:
                if resource[tags.META][tags.ETIME] is None:
                    resource[tags.META].pop(tags.ETIME)
                if resource[tags.META].get('live_video_id'):
                    resource[tags.META].pop('live_video_id')
            except KeyError:
                pass

            if isinstance(form.model, DiskResource) and yateam_logic.is_yateam_subtree(form.model.address.path):
                if 'short_url' in resource[tags.META]:
                    resource[tags.META]['short_url'] = Publicator.get_short_url(form.model)
                if 'short_url_named' in resource[tags.META]:
                    resource[tags.META]['short_url_named'] = Publicator.get_short_url_named(form.model)
            elif isinstance(form.model, DiskResource):
                # https://st.yandex-team.ru/CHEMODAN-36162
                # if для https://st.yandex-team.ru/CHEMODAN-37231: только у DiskResource есть get_short_url,
                #     который зовётся в Publicator.get_short_url
                mapping = {
                    'short_url': Publicator.get_short_url,
                    'short_url_named': Publicator.get_short_url_named
                }
                for short_url_meta_key in ('short_url', 'short_url_named'):
                    if short_url_meta_key in resource[tags.META]:
                        short_url = mapping[short_url_meta_key](form.model)
                        if short_url:
                            replaced, short_url = Publicator.replace_yadisk_cc_url(short_url)
                            if replaced and FEATURE_TOGGLES_SAVE_REPLACED_YADISK_CC_TO_DATABASE:
                                form.model.meta[short_url_meta_key] = short_url
                                form.model.save()
                        resource[tags.META][short_url_meta_key] = short_url

        try:
            meta_names = self.request.meta_names
        except AttributeError:
            pass
        else:
            if meta_names is not None:
                resource[tags.META_NAMES] = metainfo.keys()

        try:
            resource[tags.META]['custom_properties'] = from_json(resource[tags.META]['custom_properties'])
        except KeyError:
            pass

    def _base_element(self, form):
        try:
            id = form.key or form.id
        except AttributeError:
            id = form.id

        if form.is_file:
            type = tags.FILE
        else:
            # https://jira.yandex-team.ru/browse/CHEMODAN-10336
            #if form.model.address.storage_name != 'mail':
            #    id = self.folder_id(id)
            #print form.model.address
            #print id
            id = self.folder_id(id)
            type = tags.DIR
        result = {
            tags.ID: id,
            tags.TYPE: type,
            tags.PATH: id[:-1] + id[-1].replace('/', ''),
            tags.NAME: SANITIZE_NAME.sub(' ', form.name),
            tags.CTIME: form.ctime,
            tags.MTIME: form.mtime,
            tags.UTIME: form.utime,
        }

        try:
            if form.etime is not None:
                result[tags.ETIME] = form.etime
        except AttributeError:
            pass
        return result

    def _format_mpfs_resource_without_form(self, mpfs_resource):
        """Отформатировать mpfs-ый ресурс без форм
        """
        setattr(mpfs_resource, 'model', mpfs_resource)
        try:
            if mpfs_resource.is_file:
                base_doc = self._file_element(mpfs_resource)
            else:
                base_doc = self._folder_element(mpfs_resource)
            self._process_meta(mpfs_resource, base_doc)
        finally:
            delattr(mpfs_resource, 'model')
        return base_doc

    def _list(self, *args, **kw):
        form = self.request.form

        if form.is_file:
            return self._info(**kw)
        else:
            root = self._base_element(form)
            self._process_meta(form, root)

            content = self._folder_content(
                all_elements=kw.get('all_elements'),
                timeline=kw.get('timeline')
            )

            return [root] + content

    def list(self, *args, **kw):
        return self.to_json(self._list(**kw))

    def dir_list(self, **kw):
        return self.list(**kw)

    def list_installers(self, **kw):
        form = self.request.form
        self.request.meta = ['file_url']

        root = self._base_element(form)
        self._process_meta(form, root)

        content = self._folder_content(
            all_elements=kw.get('all_elements'),
            timeline=kw.get('timeline')
        )

        return self.to_json(content)

    def timeline(self, *args, **kwargs):
        return self.list(all_elements=True, timeline=True)

    def photounlim_last_modified(self, *args, **kwargs):
        items = self._list(all_elements=True, timeline=True)
        result = self.to_json({'items': items,
                               'iteration_key':
                                   PhotounlimLastModifiedIterationKey.generate_next(
                                       # убираем корневой элемент, т.к. он не нужен для генерации
                                       items[1:],
                                       self.request.args.data['filter'].get('mtime_gte', ''),
                                       self.request.args.data['bounds']['offset'],
                                       self.request.prev_iteration_key
                                   ).serialize()})
        return result

    def get_last_files(self):
        return self.list(all_elements=True, timeline=True)

    def new_get_last_files(self, *args, **kwargs):
        result = []
        for resource in self.data['result']:
            result.append(self._format_mpfs_resource_without_form(resource))
        return self.to_json(result)

    def _folder_content(self, all_elements=False, timeline=False):
        form = self.request.form
        meta = self.request.meta

        if all_elements:
            all_folders = form.all_folders(root=True, cut=True)
            all_files = form.all_files(root=True, cut=True)
        else:
            all_folders = form.folders()
            all_files = form.files()

        result = []

        for element in all_folders:
            result.append(self._folder_element(element))

        for element in all_files:
            result.append(self._file_element(element))

        return result

    def _folder_element(self, element):
        folder_dict = self._base_element(element)

        try:
            element.meta[tags.HASFOLDERS] = element.form__hasfolders
        except AttributeError:
            pass
        element.meta[tags.VISIBLE] = int(element.visible) if element.visible is not None else 1

        try:
            element.meta[tags.STATUS] = element.status_dict.pop('value')
        except AttributeError:
            pass

        self._process_meta(element, folder_dict)
        return folder_dict

    def _file_element(self, element):
        file_dict = self._base_element(element)
        element.meta[tags.MEDIATYPE] = element.media_type
        element.meta[tags.VISIBLE] = int(element.visible) if element.visible is not None else 1
        element.meta[tags.SIZE] = element.size

        self._process_meta(element, file_dict)
        return file_dict

    def info(self, *args, **kwargs):
        return self.to_json(self._info(*args, **kwargs))

    def info_by_file_id(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def info_by_resource_id(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def info_by_comment_id(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def office_info_by_office_doc_short_id(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def bulk_info(self, *args, **kwargs):
        result = []
        for resource in self.data['result']:
            form = resource.form
            res_dict = self._base_element(form)
            self._process_meta(form, res_dict)
            result.append(res_dict)
        return self.to_json(result)

    def bulk_info_by_resource_ids(self, *args, **kwargs):
        return self.bulk_info(*args, **kwargs)

    def bulk_info_by_office_online_sharing_urls(self, *args, **kwargs):
        return self.bulk_info(*args, **kwargs)

    def share_remove_invite(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def share_kick_from_group(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def share_change_rights(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def share_unshare_folder(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def share_leave_group(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def set_private(self, *args, **kwargs):
        return self.to_json(self.data['result'])

    def state_set(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def state_remove(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def user_set_readonly(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def user_unset_readonly(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def share_reject_invite(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def user_init(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def __share_list_folders__(self, data, result):
        if not isinstance(data, (list, tuple)):
            data = [data, ]

        for form in data:
            resource = self._base_element(form)
            self._process_meta(form, resource)
            result.append(resource)

    def share_list_all_folders(self, **kw):
        result = []
        data = self.request.data['result']
        for subdata in data.itervalues():
            self.__share_list_folders__(subdata, result)
        return self.to_json(result)

    def share_list_owned_folders(self, **kw):
        result = []
        self.__share_list_folders__(self.request.data['result'], result)
        return self.to_json(result)

    def share_list_joined_folders(self, **kw):
        result = []
        self.__share_list_folders__(self.request.data['result'], result)
        return self.to_json(result)

    def tree(self, **kw):
        data = self.request.data['result']

        result = {'resource': []}
        def process_item(level_data, result):
            level_result = level_data['this']

            # Оставим на будущее
            # Пока верстка не поддерживает meta в tree
            meta = self.request.meta
            metainfo = level_result.pop(tags.META)
            if meta is not None:
                if meta:
                    metainfo = dict(filter(lambda (x, y): x in meta, metainfo.iteritems()))
                level_result[tags.META] = metainfo

            if level_result['type'] == 'dir':
                level_result['id'] += '/'
            level_result['resource'] = []
            for sibling in level_data['list']:
                process_item(sibling, level_result)
            result['resource'].append(level_result)
        process_item(data, result)
        return self.to_json(result)

    def mkdir(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def fotki_mkfile(self, *args, **kwargs):
        return self.default_success()

    def mksysdir(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def share_activate_invite(self):
        return self.info()

    def status(self, *args, **kwargs):
        data = self.data[tags.RESULT]

        # см. mpfs.common.static.messages для объяснения
        status = data[tags.STATUS]
        data[tags.STATUS] = messages.old_operation_titles[status]
        data[tags.STATE] = messages.true_operation_titles[status]

        progress_stage = filter(lambda x: x.get(tags.NAME) == 'kladun_upload', data.get(tags.STAGES, []))
        if progress_stage:
            data[tags.PROGRESS] = progress_stage[0].get(tags.PROGRESS, 0)

        if tags.PROTOCOL in data:
            for i in xrange(len(data[tags.PROTOCOL])):
                # см. mpfs.common.static.messages для объяснения
                op_status = data[tags.PROTOCOL][i][tags.STATUS]
                data[tags.PROTOCOL][i][tags.STATUS] = messages.old_operation_titles.get(op_status, op_status)
                data[tags.PROTOCOL][i][tags.STATE] = messages.true_operation_titles.get(op_status, op_status)
                del data[tags.PROTOCOL][i][tags.RESULT]
                if tags.FORM in data[tags.PROTOCOL][i]:
                    del data[tags.PROTOCOL][i][tags.FORM]

        if (hasattr(self, 'request') and
                self.request and hasattr(self.request, 'form') and
                self.request.form):
            data['resource'] = self._info(**kwargs)

        return self.to_json(data)

    def search(self, *args, **kwargs):
        data = self.data[tags.RESULT]
        this = data[tags.THIS]
        this[tags.ID] = self.folder_id(this[tags.ID])
        return self.to_json(data)

    def new_search(self, *args, **kwargs):
        return self.search_formatter()


    def geo_search(self):
        return self.search_formatter()

    def search_formatter(self):
        data = self.data[tags.RESULT]
        squery = data.pop('search-query')
        form = self.request.form
        root = self._base_element(form)
        self._process_meta(form, root)
        content = []
        for element in form.resources():
            if element.is_folder:
                content.append(self._folder_element(element))
            else:
                content.append(self._file_element(element))
        result = {
            'results': [root] + content,
            'query': squery
        }
        if 'lost_results_count' in data:
            result['lost_results_count'] = data['lost_results_count']
        return self.to_json(result)

    def search_sizes_per_mediatype(self):
        data = self.data[tags.RESULT]
        for item in data['items']:
            item['mediatype'] = filetypes.getGroupByNumber(item['mediatype'])
        return self.to_json(data)

    def public_info(self, *args, **kwargs):
        resource = self._info(*args, **kwargs)
        user_info = self.data[tags.RESULT]['user']
        public_pwd_token = self.data[tags.RESULT].get(tags.PUBLIC_PASSWORD_TOKEN, None)

        for field in (tags.ID, tags.UID, tags.HID, tags.PATH):
            try:
                del(resource[field])
            except KeyError:
                pass

        return self.to_json({'resource': resource, 'user': user_info, tags.PUBLIC_PASSWORD_TOKEN: public_pwd_token})

    def public_list(self, *args, **kwargs):
        return self.to_json(self._list(*args, **kwargs))

    def public_copy(self, *args, **kwargs):
        return self.info(*args, **kwargs)

    def tags_photo_timeline(self, *args, **kwargs):
        data = self.data[tags.RESULT]
        result = []
        for year, months_in_year in data.iteritems():
            per_year = []
            for month, count in months_in_year.iteritems():
                per_year.append({'month': str(month), 'count': count})
            result.append({'year': str(year), 'months': per_year})
        return self.to_json(result)

    def list_public(self, *args, **kwargs):
        result = []
        for resource in self.data['result']:
            form = resource.form
            res_dict = self._base_element(form)
            self._process_meta(form, res_dict)

            result.append(res_dict)
        return self.to_json(result)

    def list_subresources(self, *args, **kwargs):
        result = self.data['result']
        for i, item in enumerate(result['items']):
            if isinstance(item, dict):
                continue
            res = self._base_element(item.form)
            self._process_meta(item.form, res)
            result['items'][i] = res
        return self.to_json(result)

    def _format_album_item(self, item):
        self._album_recursion_fuse = set()
        return self.__format_nested_album_item(item)

    def __format_nested_album_item(self, item):
        result = item.as_json_dict()
        if item.object:
            from mpfs.core.albums.models import AlbumItem
            if item.obj_type in (AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE):
                res = self._base_element(item.object.form)
                self._process_meta(item.object.form, res)
                result['object'] = res
            elif item.obj_type == 'album':
                result['object'] = self.__format_nested_album(item.object)
            face_coord = get_album_item_face_coordinates(item)
            if face_coord:
                result.update(face_coord)
            result.pop('face_info', None)
        return result

    def _format_album(self, album, include_items=True, load_info_from_counters=True):
        self._album_recursion_fuse = set()
        return self.__format_nested_album(album,
                                          include_items=include_items,
                                          load_info_from_counters=load_info_from_counters)

    def __format_nested_album(self, album, include_items=True, load_info_from_counters=True):
        if album.id in self._album_recursion_fuse:
            return
        self._album_recursion_fuse.add(album.id)
        result = album.as_json_dict(load_info_from_counters=load_info_from_counters)
        if album.cover and album.cover.object:
            result['cover'] = self.__format_nested_album_item(album.cover)
            result['is_empty'] = False
        else:
            result['is_empty'] = True

        if not getattr(self.request, 'no_items', False) and include_items:
            # если элементы предварительно отобраны и сохранены в альбом, то форматируем их тоже
            result['items'] = [self.__format_nested_album_item(item) for item in album.items]
        return result

    def albums_create(self, *args, **kwargs):
        result = self._format_album(self.data['result'])
        return self.to_json(result)

    def albums_exclude_from_generated(self, *args, **kwargs):
        result = self.to_json({'items': self.data['result']})
        return result

    def albums_create_from_folder(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def albums_create_with_items(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def albums_list(self, *args, **kwargs):
        result = [self._format_album(album, include_items=False, load_info_from_counters=False)
                  for album in self.data['result']]
        return self.to_json(result)

    def album_get(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def album_item_info(self, *args, **kwargs):
        return self.album_item_move(*args, **kwargs)

    def album_item_check(self, *args, **kwargs):
        return self.album_item_move(*args, **kwargs)

    def album_set_attr(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def album_append_items(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def album_append_item(self, *args, **kwargs):
        return self.album_item_move(*args, **kwargs)

    def album_copy(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def album_item_move(self, *args, **kwargs):
        result = self._format_album_item(self.data['result'])
        return self.to_json(result)

    def album_item_set_attr(self, *args, **kwargs):
        return self.album_item_move(*args, **kwargs)

    def public_album_item_info(self, *args, **kwargs):
        return self.album_item_move(*args, **kwargs)

    def album_publish(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def album_unpublish(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def public_album_block(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def public_album_unblock(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def public_album_get(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def _album2fulltree(self, album, result=None, path='/', processed_album_ids=None):
        """
        Рекурсивное построение содержимого альбома

        https://st.yandex-team.ru/CHEMODAN-22036
        """
        if result is None:
            result = {'list': []}
        if processed_album_ids is None:
            processed_album_ids = set()
        result['this'] = {
            'type': 'dir',
            'path': path,
            'name': album.title,
            'id': album.id,
            'ctime': album.ctime,
            'mtime': album.mtime
        }
        processed_album_ids.add(album.id)
        auto_suffixator = AutoSuffixator()
        for item in album.items:
            path = path.rstrip('/')
            if item.obj_type == 'album':
                if item.object.id in processed_album_ids:
                    continue
                result['list'].append({'list': []})
                path = '%s/%s' % (path, item.object.title)
                self._album2fulltree(item.object,
                                     result=result['list'][-1],
                                     path=path,
                                     processed_album_ids=processed_album_ids)
            else:
                item_info = self._format_album_item(item)
                item_object = item_info.get('object')
                if item_object:
                    formated_item = item_object.copy()
                    formated_item['name'] = auto_suffixator(formated_item['name'])
                    formated_item['path'] = "%s/%s" % (path, formated_item['name'])
                    result['list'].append({'this': formated_item, 'list': []})
        return result

    def public_album_resources_list(self, *args, **kwargs):
        album = self.data['result']
        return self.to_json(self._album2fulltree(album))

    def public_album_items_list(self, *args, **kwargs):
        result = []
        for item in self.data['result']:
            result.append(self._format_album_item(item))
        return self.to_json(result)

    def public_album_check(self, *args, **kwargs):
        full_result = self._format_album(self.data['result'], include_items=False)
        result = {k: v for k, v in full_result.iteritems() if k in ('cover', 'title')}
        return self.to_json(result)

    def public_album_save(self, *args, **kwargs):
        result = []
        for resource in self.data['result']:
            if resource.is_folder:
                result.append(self._folder_element(resource))
            else:
                result.append(self._file_element(resource))
        return self.to_json(result)

    def bulk_download_list(self, *args, **kwargs):
        data = self.data[tags.RESULT]
        operation = data['operation']
        resources = data['resources']

        result = {
            tags.THIS: {
                tags.TYPE: 'dir',
                tags.PATH: '/',
                tags.NAME: 'archive',  # folder name inside of archive
                tags.ID: operation.id,
                tags.CTIME: operation.ctime,
                tags.MTIME: operation.mtime,
            },
            tags.LIST: []
        }

        if isinstance(resources, (list, tuple)):
            auto_suffixator = AutoSuffixator()
            for resource in resources:
                form = resource.form
                element = self._base_element(form)
                self._process_meta(form, element)

                element['name'] = auto_suffixator(element['name'])
                element[tags.PATH] = '/' + element['name']

                result[tags.LIST].append({
                    tags.THIS: element,
                    tags.LIST: [],
                })

        return self.to_json(result)

    def lenta_block_list(self, *args, **kwargs):
        return self.list(**kwargs)

    def lenta_create_album_from_block(self, *args, **kwargs):
        return self.albums_create_with_items(*args, **kwargs)

    def lenta_block_public_link(self, *args, **kwargs):
        return self.albums_create(*args, **kwargs)

    def deltas(self, *args, **kwargs):
        result = self.data['result']
        for diff_item in result['items']:
            if 'resource' in diff_item:
                resource = diff_item['resource']
                diff_item['resource'] = self._format_mpfs_resource_without_form(resource)
        return self.to_json(result)

    def move(self, *args, **kwargs):
        return self.to_json(self.request.data['result'])

    @staticmethod
    def _format_version(version_and_url):
        version, file_url = version_and_url
        iteration_key = version.get_iteration_key_for_folded_items()
        result = {
            'id': version.id,
            'type': version.type.value,
            'platform_created':  version.platform_created,
            'uid_created':  version.uid_created,
            'date_created': version.date_created.isoformat(),
            'folded_items_iteration_key': iteration_key.serialize() if iteration_key else None,
            'can_be_restored': version.can_be_restored(),
        }
        if version.type in (VersionType.binary, VersionType.current):
            result.update(
                {
                    'size': version.size,
                    'md5': version.md5,
                    'sha256': version.sha256,
                    'file_stid': version.file_stid,
                    'file_url': file_url,
                }
            )
        return result

    def versioning_get_checkpoints(self, *args, **kwargs):
        iteration_key, versions_and_urls = self.data['result']
        result = {
            'versions': [self._format_version(item) for item in versions_and_urls],
            'iteration_key': iteration_key.serialize() if iteration_key else None,
        }
        return self.to_json(result)

    def versioning_get_folded(self, *args, **kwargs):
        return self.versioning_get_checkpoints(*args, **kwargs)

    def versioning_get_version(self, *args, **kwargs):
        version, file_url = self.data['result']
        return self.to_json(self._format_version((version, file_url)))

    def versioning_restore(self, *args, **kwargs):
        resource = self.data['result']
        return self.to_json(self._format_mpfs_resource_without_form(resource))

    def versioning_save(self, *args, **kwargs):
        resource = self.data['result']
        return self.to_json(self._format_mpfs_resource_without_form(resource))

    @staticmethod
    def _format_lock_info(lock_info):
        result = {
            tags.UID: lock_info[tags.UID],
            tags.DTIME: lock_info[tags.DTIME],
        }
        if 'data' in lock_info:
            result[tags.DATA] = lock_info[tags.DATA]
        return result

    def locks_list(self, *args, **kwargs):
        locks = self.data['result']
        return self.to_json({path: self._format_lock_info(lock_info)
                             for path, lock_info in locks.items()})

    def locks_delete(self, *args, **kwargs):
        return self.default_success(*args, **kwargs)

    def locks_set(self, *args, **kwargs):
        lock_info = self.data['result']
        return self.to_json(self._format_lock_info(lock_info))

    def store(self, *args, **kwargs):
        result = self._parse_and_pull_out_header_params(self.data[tags.RESULT])
        return self.to_json(result)

    def attach_store(self, *args, **kwargs):
        result = self._parse_and_pull_out_header_params(self.data[tags.RESULT])
        return self.to_json(result)
