# -*- coding: utf-8 -*-
'''
MPFS
Mail форматтер, для верстки
'''

import re
import traceback
from lxml import etree


import mpfs.engine.process
from mpfs.frontend.formatter.disk import MPFSFormatter
from mpfs.common.static.messages import *
from mpfs.common.static import messages
from mpfs.common.static.tags import *
from mpfs.common.errors import *
from mpfs.frontend.formatter.util import split_file, split_path
from mpfs.common.util.docviewer_types import get_ready, get_icon_val, find_mime
from mpfs.common.util.filetypes import getGroupByName, getGroupByMimetype
from mpfs.common.util import wise_to_str, to_json


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class Mail(MPFSFormatter):
    
    content_type = 'text/xml; charset=utf-8'
    hidden_meta_fields = ('file_id', 'file_mid', 'digest_mid', 'md5', 'sha256', 'wh_version', 'pmid', 'emid')

    def process(self, f, **kw):
        response = super(Mail, self).process(f, **kw)    
        return etree.tostring(response, pretty_print=True)

    def default_handler(self, *args, **kwargs):
        return etree.Element(RESPONSE, status="1")

    def add_nonempty_node(self, parent, tag, data):
        if data:
            self._set_text(etree.SubElement(parent, tag), data)

    def list(self, **kw):
        form = self.request.form
        if form.is_file:
            return self.info()
        else:
            folder_list = etree.Element(FOLDERLIST)
            self._folder_content(
                folder_list,
                all_elements=kw.get('all_elements'),
                timeline=kw.get('timeline')
            )
            response = etree.Element(RESPONSE, status="1")
            zeropath = etree.SubElement(response, FOLDERLIST)
            current_folder = etree.SubElement(zeropath, FOLDER)
            path = etree.SubElement(current_folder, PATH)
            for part in filter(None, form.id.split('/')):
                self._set_text(etree.SubElement(path, PART), part)
            if kw.get('timeline'):
                path_named = etree.SubElement(current_folder, PATH_NAMED)
                for part in form.name_tree[::-1]:
                    self._set_text(etree.SubElement(path_named, PART), part)
            self._set_text(etree.SubElement(current_folder, NAME), form.name)
            self._set_text(etree.SubElement(current_folder, ID), form.id)
            self._set_text(etree.SubElement(current_folder, CHUNK_NUMFILES), form.form__chunk_numfiles)
            self._set_text(etree.SubElement(current_folder, TOTALLY_FILTERED), form.form__totally_filtered)
            self._set_text(etree.SubElement(current_folder, VISIBLE), form.visible or 1)
            self._set_text(etree.SubElement(current_folder, HASFOLDERS), form.form__hasfolders)
            if form.model__meta:
                meta_element = etree.SubElement(current_folder, META)
                self._parse_meta(meta_element, form.model__meta, '')
            current_folder_date_element = etree.SubElement(current_folder, DATE)
            for tag in (CTIME, MTIME,):
                self.add_nonempty_node(current_folder_date_element, tag, getattr(form, tag))
            current_folder.append(folder_list)
            return response

    def _list(self, response, kw, **extra):
        data = self.data['result']

        folder_description = data['this']

        folders = filter(lambda item: item.get(TYPE) == 'dir', data['list'])
        files = filter(lambda item: item.get(TYPE) == 'file', data['list'])

        zeropath = etree.SubElement(response, FOLDERLIST)
        current_folder = etree.SubElement(zeropath, FOLDER)
        path = etree.SubElement(current_folder, PATH)

        for part in filter(None, folder_description[ID].split('/')):
            self._set_text(etree.SubElement(path, PART), part)

        self._set_text(etree.SubElement(current_folder, NAME), folder_description[NAME])
        root_id = folder_description.get(ID, '')
        root_id = root_id if root_id == '/' else root_id
        self._set_text(etree.SubElement(current_folder, ID), root_id)
        self._set_text(etree.SubElement(current_folder, CHUNK_NUMFILES), len(files))
        self._set_text(etree.SubElement(current_folder, 'visible'), folder_description.get('visible', 1))

        folder_meta_data = folder_description.get(META, {})
        has_folders = int(folder_meta_data.get('hasfolders', False))
        self._set_text(etree.SubElement(current_folder, HASFOLDERS), has_folders)
        if folder_meta_data:
            meta_element = etree.SubElement(current_folder, META)
            self._parse_meta(meta_element, folder_meta_data, '')

        current_folder_date_element = etree.SubElement(current_folder, DATE)

        for etag in (CTIME, MTIME, CDATE, MDATE):
            self.add_nonempty_node(current_folder_date_element, etag, folder_description.get(etag))

        for k, v in extra.iteritems():
            self._set_text(etree.SubElement(current_folder, k), v)

        self._parse_list(etree.SubElement(current_folder, FOLDERLIST), folders, files)

        return response

    def _parse_list(self, folderlist, folders, files):
        for f in folders:
            sub_folder_element = etree.SubElement(folderlist, FOLDER)
            sub_folder_path = etree.SubElement(sub_folder_element, PATH)
            for part in filter(None, f.get(ID, '').split('/')):
                self._set_text(etree.SubElement(sub_folder_path, PART), part)
            self._set_text(etree.SubElement(sub_folder_element, NAME), f.get(NAME, ''), NAME)
            status_dict = f.get('status_dict')
            if isinstance(status_dict, dict) and status_dict:
                status_value = status_dict.pop('value')
                self._set_text(etree.SubElement(sub_folder_element, 'status', **dict((str(k), str(v)) for k, v in status_dict.iteritems())), status_value)

            meta_data = f.get(META, {})
            has_folders = meta_data.get('hasfolders', False)
            self._set_text(etree.SubElement(sub_folder_element, HASFOLDERS), has_folders)
            self._set_text(etree.SubElement(sub_folder_element, 'visible'), f.get('visible', 1))
            f_id = f.get(ID)
            if f_id:
                self._set_text(etree.SubElement(sub_folder_element, ID), f_id)
            if meta_data:
                meta_element = etree.SubElement(sub_folder_element, META)
                self._parse_meta(meta_element, meta_data, '')
            folder_date_element = etree.SubElement(sub_folder_element, DATE)
            for etag in (CTIME, MTIME, CDATE, MDATE):
                self.add_nonempty_node(folder_date_element, etag, f.get(etag))

        for f in files:
            file_element = etree.SubElement(folderlist, FILE)
            parts, id = split_file(f.get(ID, ''), f.get(NAME, ''))

            file_path = etree.SubElement(file_element, PATH)
            self._set_text(etree.SubElement(file_element, ID), f.get(ID, ''), ID)

            for part in filter(None, f.get(ID, '').split('/')):
                self._set_text(etree.SubElement(file_path, PART), part)

            self._set_text(etree.SubElement(file_element, NAME), f.get(NAME, ''), NAME)
            self._set_text(etree.SubElement(file_element, SIZE), f.get(SIZE, ''), SIZE)
            self._set_text(etree.SubElement(file_element, SOURCE), f.get(SOURCE, ''), SOURCE)
            self._set_text(etree.SubElement(file_element, 'visible'), f.get('visible', 1))

            file_date_element = etree.SubElement(file_element, DATE)
            for etag in (CTIME, MTIME, CDATE, MDATE):
                self.add_nonempty_node(file_date_element, etag, f.get(etag))
            mtype = f.get(MIMETYPE)
            if mtype in ('application/octet-stream', 'multipart/form-data', 'application/x-www-form-urlencoded'):
                mtype = ''
            docv_query = find_mime(mime=mtype, ext=f.get(NAME))
            if f.get(SOURCE, '') in ('mail', 'narod'):
                self._set_text(
                    etree.SubElement(file_element, TYPE),
                    getGroupByName(f.get(NAME, ''), mtype=mtype),
                    TYPE
                )
            else:
                self._set_text(
                    etree.SubElement(file_element, TYPE),
                    getGroupByMimetype(mtype, f.get(NAME, '')),
                    TYPE
                )
#            mtype = mtype or f.get(MIMETYPE)
            self._set_text(etree.SubElement(file_element, MIMETYPE), mtype, MIMETYPE)
            dtype = docv_query.get('dtype') if get_ready(f.get(NAME)) else ''
            self._set_text(etree.SubElement(file_element, DOCVIEWER_TYPE), dtype, DOCVIEWER_TYPE)
            fileicon = get_icon_val(f.get(NAME), f.get(MIMETYPE), None)
            self._set_text(etree.SubElement(file_element, FILE_ICON), fileicon, FILE_ICON)

            meta_data = f.get(META, {})
            tags = meta_data.pop(TAGS, [])
            thumb = meta_data.pop(THUMBNAIL, '')
            preview = meta_data.pop(PREVIEW, '')            
            drweb = meta_data.pop(DRWEB, '0')
            meta_element = etree.SubElement(file_element, META)

            if meta_data:
                self._parse_meta(meta_element, meta_data, '')

            self._set_text(etree.SubElement(meta_element, THUMBNAIL), re.sub("^(http|https):", '', thumb))
            self._set_text(etree.SubElement(meta_element, PREVIEW), re.sub("^(http|https):", '', preview))
            self._set_text(etree.SubElement(meta_element, DRWEB), drweb)

            tags_element = etree.SubElement(file_element, TAGS)
            for tag in tags:
                self.add_nonempty_node(tags_element, TAG, tag)

    def _parse_meta(self, head, data, tag, all_meta={}, show_hidden=False):
        if not show_hidden and tag in self.hidden_meta_fields:
            return
        else:
            if not all_meta:
                all_meta = data
            try:
                if isinstance(data, dict):
                    if tag in ('flags-all', 'flags', 'to', 'from', 'cc', 'bcc'):
                        h1 = etree.SubElement(head, tag)
                        for fk, fv in data.iteritems():
                            self._parse_meta(h1, fv, fk, all_meta, show_hidden=show_hidden)
                    elif tag and tag in ('previews', 'preview_sizes'):
                        pass
                    else:
                        for k, v in data.iteritems():
                            if k in ('flags-all', 'flags', 'group',):
                                h1 = etree.SubElement(head, k)
                                for fk, fv in v.iteritems():
                                    if fk in ('owner', ):
                                        h_owner = etree.SubElement(h1, fk)
                                        for _k, _v in fv.iteritems():
                                            self._set_text(etree.SubElement(h_owner, _k), _v)
                                    else:
                                        self._parse_meta(h1, fv, fk, all_meta, show_hidden=show_hidden)
                            elif tag:
                                self._parse_meta(etree.SubElement(head, tag), v, k, all_meta, show_hidden=show_hidden)
                            else:
                                self._parse_meta(head, v, k, all_meta, show_hidden=show_hidden)
                elif isinstance(data, (list, tuple)):
                    if tag in ('sizes',):
                        tag_element = etree.SubElement(head, tag)
                        for sz in data:
                            sz['url'] = re.sub("^(http|https):", '', sz['url'])
                            sz.update(all_meta.get('preview_sizes', {}).get(sz['name'], {}))
                            etree.SubElement(tag_element, tag, **sz)
                    elif tag:
                        for v in data:
                            self._parse_meta(head, v, tag, all_meta, show_hidden=show_hidden)
                elif tag:
                    try:
                        self._set_text(etree.SubElement(head, tag), data, tag)
                    except ValueError:
                        # CHEMODAN-3086
                        # Обрабатываем теги со знаками типа : или |, отрезая все до знака
                        valid, rest = re.split('\||\:', tag, 1)
                        self._set_text(etree.SubElement(head, valid), data, valid)
            except Exception:
                error_log.debug(traceback.format_exc())

    def _folder_content(self, parent_node, all_elements=False, timeline=False):
        '''
            Parse folder content with form
        '''
        form = self.request.form
        all_folders = []
        if all_elements:
            if not timeline:
                all_folders = form.all_folders(root=True, cut=True)
            all_files = form.all_files(root=True, cut=True)
        else:
            if not timeline:
                all_folders = form.folders()
            all_files = form.files()
        for f in all_folders:
            sub_folder_element = etree.SubElement(parent_node, FOLDER)
            sub_folder_path = etree.SubElement(sub_folder_element, PATH)

            for part in filter(None, f.id.split('/')):
                self._set_text(etree.SubElement(sub_folder_path, PART), part)
            if timeline:
                path_named = etree.SubElement(sub_folder_path, PATH_NAMED)
                for part in f.name_tree[::-1]:
                    self._set_text(etree.SubElement(path_named, PART), part)

            self._set_text(etree.SubElement(sub_folder_element, NAME), f.name, NAME)

            try:
                status_value = f.status_dict.pop('value')
                self._set_text(etree.SubElement(sub_folder_element, 'status', **dict((str(k), str(v)) for k, v in f.status_dict.iteritems())), status_value)
            except AttributeError:
                pass

            meta_data = f.meta
            self._set_text(etree.SubElement(sub_folder_element, HASFOLDERS), f.form__hasfolders)
            self._set_text(etree.SubElement(sub_folder_element, VISIBLE), f.visible or 1)
            self._set_text(etree.SubElement(sub_folder_element, ID), f.id)

            if meta_data:
                meta_element = etree.SubElement(sub_folder_element, META)
                self._parse_meta(meta_element, meta_data, '')
            folder_date_element = etree.SubElement(sub_folder_element, DATE)
            for tag in (CTIME, MTIME,):
                self.add_nonempty_node(folder_date_element, tag, getattr(f, tag))

        for f in all_files:
            file_element = etree.SubElement(parent_node, FILE)
            file_path = etree.SubElement(file_element, PATH)

            try:
                key_id = f.key
            except Exception:
                key_id = f.id

            self._set_text(etree.SubElement(file_element, ID), key_id)
            for part in filter(None, key_id.split('/')):
                self._set_text(etree.SubElement(file_path, PART), part)
            if timeline:
                path_named = etree.SubElement(file_element, PATH_NAMED)
                for part in f.name_tree[::-1]:
                    self._set_text(etree.SubElement(path_named, PART), part)

            self._set_text(etree.SubElement(file_element, NAME), f.name)
            self._set_text(etree.SubElement(file_element, SIZE), f.size)
            self._set_text(etree.SubElement(file_element, SOURCE), f.source)
            self._set_text(etree.SubElement(file_element, VISIBLE), f.visible or 1)

            file_date_element = etree.SubElement(file_element, DATE)
            for tag in (CTIME, MTIME,):
                self.add_nonempty_node(file_date_element, tag, getattr(f, tag))
            self._set_text(etree.SubElement(file_element, TYPE), f.media_type)
            self._set_text(etree.SubElement(file_element, FILE_ICON), f.file_icon)
            self._set_text(etree.SubElement(file_element, MIMETYPE), f.mimetype)
            self._set_text(etree.SubElement(file_element, DOCVIEWER_TYPE), f.docviewer_type)

            meta_data = f.meta
            tags = meta_data.pop(TAGS, [])
            thumb = meta_data.pop(THUMBNAIL, '')
            preview = meta_data.pop(PREVIEW, '')
            drweb = meta_data.pop(DRWEB, '0')
            meta_element = etree.SubElement(file_element, META)

            if meta_data:
                self._parse_meta(meta_element, meta_data, '')

            self._set_text(etree.SubElement(meta_element, THUMBNAIL), re.sub("^(http|https):", '', thumb))
            self._set_text(etree.SubElement(meta_element, PREVIEW), re.sub("^(http|https):", '', preview))
            self._set_text(etree.SubElement(meta_element, DRWEB), drweb)

            tags_element = etree.SubElement(file_element, TAGS)
            for tag in tags:
                self.add_nonempty_node(tags_element, TAG, tag)

    def services(self, **kw):
        return self.tree(**kw)

    def parents(self, **kw):
        data = self.request.data['result']
        response = etree.Element(RESPONSE, status="1")
        path = data['path']
        name_split = filter(None, path.split('/'))

        def simple_tree(parent_element, i=0):
            foldertree_element = etree.SubElement(parent_element, FOLDERTREE)
            folder_element = etree.SubElement(foldertree_element, FOLDER)
            path_element = etree.SubElement(folder_element, PATH)
            for p in split_path(path)[:i+1]:
                self._set_text(etree.SubElement(path_element, PART), p)
            self._set_text(etree.SubElement(folder_element, NAME), name_split[i], NAME)
            self._set_text(etree.SubElement(folder_element, ID), '/' + '/'.join(name_split[:i+1]))
            if i + 1 < len(name_split):
                simple_tree(folder_element, i=i+1)
        simple_tree(response)
        return response

    def tree(self, **kw):
        data = self.request.data['result']
        response = etree.Element(RESPONSE, status="1")

        def _subtree(node, parent):
            node.apply_bounds()
            element = etree.SubElement(parent, FOLDER)
            path = etree.SubElement(element, PATH)
            for p in split_path(node.id):
                self._set_text(etree.SubElement(path, PART), p)
            self._set_text(etree.SubElement(element, NAME), node.name, NAME)
            root_id = node.id
            root_id = root_id if root_id == '/' else root_id
            self._set_text(etree.SubElement(element, ID), root_id)
            date_element = etree.SubElement(element, DATE)
            for etag in (CTIME, MTIME):
                self.add_nonempty_node(date_element, etag, getattr(node, etag))
            self._set_text(etree.SubElement(element, HASFOLDERS), node.form__hasfolders)
            self._set_text(etree.SubElement(element, 'visible'), node.visible or 1)
            meta_data = node.meta
            if meta_data:
                meta_element = etree.SubElement(element, META)
                self._parse_meta(meta_element, meta_data, '')
            if hasattr(node, 'status_dict') and isinstance(node.status_dict, dict) and node.status_dict:
                status_value = node.status_dict.pop('value')
                self._set_text(etree.SubElement(element, 'status', **dict((str(k), str(v)) for k, v in node.status_dict.iteritems())), status_value)
            child_tree = etree.SubElement(element, FOLDERTREE)
            for child in node.folders():
                child_tree = _subtree(child, child_tree)
            return parent
        
        
        #node = self.request.form
        #response = _subtree(node, etree.SubElement(response, FOLDERTREE))
        
        def _oldschool_subtree(elem, parent):
            node = elem['this']
            element = etree.SubElement(parent, FOLDER)
            path = etree.SubElement(element, PATH)
            root_id = node['id']
            for p in split_path(root_id):
                self._set_text(etree.SubElement(path, PART), p)
            self._set_text(etree.SubElement(element, NAME), node['name'], NAME)
            root_id = root_id if root_id == '/' else root_id
            self._set_text(etree.SubElement(element, ID), root_id)
            date_element = etree.SubElement(element, DATE)
            for etag in (CTIME, MTIME):
                self.add_nonempty_node(date_element, etag, node.get(etag))
            self._set_text(etree.SubElement(element, HASFOLDERS), int(bool(node.get('hasfolders', 0))))
            self._set_text(etree.SubElement(element, 'visible'), node.get('visible', 1))
            meta_data = node['meta']
            if meta_data:
                meta_element = etree.SubElement(element, META)
                self._parse_meta(meta_element, meta_data, '')
            if 'status_dict' in node and isinstance(node['status_dict'], dict) and node['status_dict']:
                status_value = node['status_dict'].pop('value')
                self._set_text(etree.SubElement(element, 'status', **dict((str(k), str(v)) for k, v in node['status_dict'].iteritems())), status_value)
            child_tree = etree.SubElement(element, FOLDERTREE)
            for child in elem['list']:
                child_tree = _oldschool_subtree(child, child_tree)
            return parent
        
        name_split = filter(None, data['this']['id'].split('/'))

        def parent_tree(parent_element, i=0):
            if len(name_split) < 2:
                return parent_element
            folder_element = etree.SubElement(parent_element, FOLDER)
            path_element = etree.SubElement(folder_element, PATH)
            for p in split_path(data['this']['id'])[:i+1]:
                self._set_text(etree.SubElement(path_element, PART), p)
            self._set_text(etree.SubElement(folder_element, NAME), name_split[i], NAME)
            self._set_text(etree.SubElement(folder_element, ID), '/' + '/'.join(name_split[:i+1]))
            if i + 2 < len(name_split):
                return parent_tree(etree.SubElement(folder_element, FOLDERTREE), i=i+1)
            else:
                return etree.SubElement(folder_element, FOLDERTREE)

        tree_element = etree.SubElement(response, FOLDERTREE)
        _oldschool_subtree(data, tree_element)
        return response

    def timeline(self, *args, **kwargs):
        return self.list(all_elements=True, timeline=True)

    def url(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        file_url = data[FILE]
        self._set_text(etree.SubElement(response, FILE), re.sub("^(http|https):", '', file_url))
        return response

    def mkdir(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        try:
            folder = etree.SubElement(response, FOLDER)
            path = etree.SubElement(folder, PATH)
            for p in split_path(data[ID]):
                self._set_text(etree.SubElement(path, PART), p)
            for tg in (ID, NAME):
                if tg in data:
                    self._set_text(etree.SubElement(folder, tg), data[tg])
            date = etree.SubElement(folder, DATE)
            for tg in (MTIME, CTIME):
                if tg in data:
                    self._set_text(etree.SubElement(date, tg), data[tg])
            meta_element = etree.SubElement(folder, 'meta')
            for k,v in data.get('meta', {}).iteritems():
                #===============================================================
                # ignore invalid values
                #===============================================================
                self._parse_meta(meta_element, v, k)
#                self._set_text(etree.SubElement(folder, k), v)
        except KeyError:
            return unicode(self.data)
        return response

    def user_check(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, NEED_INIT), self.data[RESULT])
        return response

    def user_info(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        
        for key, value in data.iteritems():
            if key in ('settings', 'states', 'devices'):
                elem = etree.Element(key)
                response.append(elem)
                for namespace_name, namespace_data in value.iteritems():
                    namespace_elem = etree.Element(namespace_name)
                    elem.append(namespace_elem)
                    for k, v in namespace_data.iteritems():
                        item_element = None
                        if isinstance(v, dict):
                            arguments = dict(map(lambda x: (x, wise_to_str(v[x])), v.keys()))
                            arguments['key'] = k
                            try:
                                item_element = etree.Element('item', arguments)    
                            except Exception:
                                pass
                        else:
                            item_element = etree.Element('item', key=k, value=str(v))
                        if item_element is not None:
                            namespace_elem.append(item_element)
            elif key in ('space'):
                elem = etree.Element(key)
                response.append(elem)
                for k in (USED, FREE, LIMIT, TRASH, FILESIZE_LIMIT):
                    self._set_text(etree.SubElement(elem, k), value[k])
            else:
                self._set_text(etree.SubElement(response, str(key)), str(value))
        return response

    def copy(self, **kw):
        return self.info(**kw)

    def move(self, **kw):
        return self.info(**kw)
        
    def public_copy(self, **kw):
        return self.info(**kw)

    def _async_operation(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, OID), data[OID])
        self._set_text(etree.SubElement(response, TYPE), data[TYPE])
        return response
    
    def async_user_invite_friend(self, **kw):
        return self._async_operation(**kw)

    def async_rm(self, **kw):
        return self._async_operation(**kw)

    def async_copy(self, **kw):
        return self._async_operation(**kw)

    def async_copy_default(self, **kw):
        return self._async_operation(**kw)

    def async_move(self, **kw):
        return self._async_operation(**kw)

    def async_trash_drop_all(self, **kw):
        return self._async_operation(**kw)

    def async_trash_append(self, **kw):
        return self._async_operation(**kw)

    def async_trash_drop(self, **kw):
        return self._async_operation(**kw)

    def async_trash_restore(self, **kw):
        return self._async_operation(**kw)

    def async_social_contacts(self, **kw):
        return self._async_operation(**kw)
        
    def async_invite_contacts(self, **kw):
        return self._async_operation(**kw)
        
    def store(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        if UPLOAD_URL in data:
            self._set_text(etree.SubElement(response, UPLOAD_URL), data[UPLOAD_URL])
        self._set_text(etree.SubElement(response, OID), data[OID])
        self._set_text(etree.SubElement(response, TYPE), data[TYPE])
        return response

    def attach_store(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        if UPLOAD_URL in data:
            self._set_text(etree.SubElement(response, UPLOAD_URL), data[UPLOAD_URL])
        self._set_text(etree.SubElement(response, OID), data[OID])
        self._set_text(etree.SubElement(response, TYPE), data[TYPE])
        return response

    def dstore(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, PATCH_URL), data[PATCH_URL])
        self._set_text(etree.SubElement(response, OID), data[OID])
        self._set_text(etree.SubElement(response, TYPE), data[TYPE])
        return response

    def astore(self, **kw):
        return self.store(**kw)

    def status(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        operation_tag = etree.SubElement(response, OPERATION)
        self._operation(operation_tag, data)
        if hasattr(self, 'request') and self.request and hasattr(self.request, 'form') and self.request.form:
            self._info(self.request.form, operation_tag, show_hidden=True)
        return response

    def _process_iterable(self, root, val, tag=None):
        if tag:
            block = etree.SubElement(root, tag)
        else:
            block = root
        if isinstance(val, dict):
            for k, v in val.iteritems():
                self._process_iterable(etree.SubElement(block, k), v)
        elif isinstance(val, (list, tuple)):
            for v in val:
                self._process_iterable(block, v)
        else:
            self._set_text(block, val)

    def _operation(self, operation_tag, data):
        # см. mpfs.common.static.messages для объяснения
        self._set_text(etree.SubElement(operation_tag, STATUS), messages.old_operation_titles[data[STATUS]])
        self._set_text(etree.SubElement(operation_tag, STATE), messages.true_operation_titles[data[STATUS]])

        self._set_text(etree.SubElement(operation_tag, TYPE), data[TYPE])
        self._set_text(etree.SubElement(operation_tag, AT_VERSION), data[AT_VERSION])

        stages_tag = etree.SubElement(operation_tag, STAGES)
        stages_data = data.get(STAGES)

        if isinstance(stages_data, (list, tuple)):
            for stage_data in stages_data:
                self._process_iterable(etree.SubElement(stages_tag, STAGE), stage_data)
        elif isinstance(stages_data, dict):
            for stage_tag, stage_data in stages_data.iteritems():
                self._process_iterable(stage_tag, stage_data)

        if len(data.get(PROTOCOL,[])):
            protocol_tag = etree.SubElement(operation_tag, PROTOCOL)
            
            for op_data in data.get(PROTOCOL, []):
                op_tag = etree.SubElement(protocol_tag, OPERATION)
                op_type_tag = etree.SubElement(op_tag, TYPE)
                self._set_text(op_type_tag, op_data[TYPE])

                # см. mpfs.common.static.messages для объяснения
                op_status_tag = etree.SubElement(op_tag, STATUS)
                self._set_text(op_status_tag, messages.old_operation_titles[op_data[STATUS]])
                op_state_tag = etree.SubElement(op_tag, STATE)
                self._set_text(op_state_tag, messages.true_operation_titles[op_data[STATUS]])

                if FORM in op_data:
                    op_result = op_data[RESULT]
                    op_result_tag = etree.SubElement(op_tag, RESULT)
                    self._info(op_data[FORM], op_result_tag)
            
        if ERROR in data:
            error_tag = etree.SubElement(operation_tag, ERROR)
            self._set_text(etree.SubElement(error_tag, CODE), data[ERROR].get(CODE, ''))
            self._set_text(etree.SubElement(error_tag, MESSAGE), (data[ERROR].get(MESSAGE, '')))
            self._set_text(etree.SubElement(error_tag, TRACE), (data[ERROR].get(TRACE, '')))

    def info(self, **kw):
        return self._info(
            self.request.form,
            etree.Element(RESPONSE, status=unicode(self.data[STATUS]))
        )

    def _info(self, form, root_node, show_hidden=False):
        if form.is_file:
            resource_element = etree.SubElement(root_node, FILE)
            root_id = form.id
        elif form.is_folder:
            resource_element = etree.SubElement(root_node, FOLDER)
            root_id = form.id if form.id == '/' else form.id
        resource_path = etree.SubElement(resource_element, PATH)
        self._set_text(etree.SubElement(resource_element, ID), root_id)

        if hasattr(form, 'id_old'):
            self._set_text(etree.SubElement(resource_element, 'id_old'), form.id_old)

        for part in filter(None, root_id.split('/')):
            self._set_text(etree.SubElement(resource_path, PART), part)

        self._set_text(etree.SubElement(resource_element, NAME), form.name)
        self._set_text(etree.SubElement(resource_element, 'visible'), form.visible)
        if form.is_file:
            self._set_text(etree.SubElement(resource_element, SIZE), form.size)
            self._set_text(etree.SubElement(resource_element, TYPE), form.media_type)
            self._set_text(etree.SubElement(resource_element, FILE_ICON), form.file_icon)
            self._set_text(etree.SubElement(resource_element, MIMETYPE), form.mimetype)
            self._set_text(etree.SubElement(resource_element, DOCVIEWER_TYPE), form.docviewer_type)
            self._set_text(etree.SubElement(resource_element, SOURCE), form.source)
        elif form.is_folder:
            self._set_text(etree.SubElement(resource_element, HASFOLDERS), form.form__hasfolders)

        date_element = etree.SubElement(resource_element, DATE)
        for etag in (CTIME, MTIME):
            self.add_nonempty_node(date_element, etag, getattr(form, etag))

        meta_data = form.meta
        tags = meta_data.pop(TAGS, [])
        thumb = meta_data.pop(THUMBNAIL, '')
        preview = meta_data.pop(PREVIEW, '')
        drweb = meta_data.pop(DRWEB, '0')
        meta_element = etree.SubElement(resource_element, META)
        if meta_data:
            self._parse_meta(meta_element, meta_data, '', show_hidden=show_hidden)
        if form.is_file:
            self._set_text(etree.SubElement(meta_element, THUMBNAIL), re.sub("^(http|https):", '', thumb))
            self._set_text(etree.SubElement(meta_element, PREVIEW), re.sub("^(http|https):", '', preview))
            self._set_text(etree.SubElement(meta_element, DRWEB), drweb)

        tags_element = etree.SubElement(resource_element, TAGS)
        for tag in tags:
            self.add_nonempty_node(tags_element, TAG, tag)
        return root_node

    def search(self, **kw):
        data = self.data[RESULT]
        squery = data.pop('search-query')
        response = etree.Element(RESPONSE, status="1")
        return self._list(response, kw, **{'search-query' : squery})

    def space(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        for k in (USED, FREE, LIMIT, TRASH, FILESIZE_LIMIT):
            if k in data:
                self._set_text(etree.SubElement(response, k), data[k])
        return response

    def trash_restore(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, 'restore_point'), data)
        self._info(self.request.form, response)
        return response

    def restore_deleted(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        self._info(self.request.form, response)
        return response

    def trash_append(self, **kw):
        return self.info(**kw)

    def active_operations(self, **kw):
        operations = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        operations_element = etree.SubElement(response, 'operations')
        for op in operations:
            op_element = etree.SubElement(operations_element, OPERATION)
            data = op.pop('data', {})
            for k, v in op.iteritems():
                if v:
                    self._set_text(etree.SubElement(op_element, k), v)
            data_element = etree.SubElement(op_element, 'data')
            for k, v in data.iteritems():
                if v:
                    self._set_text(etree.SubElement(data_element, k), v)

        return response

    def bulk_action(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, OID), data[OID])
        self._set_text(etree.SubElement(response, TYPE), data[TYPE])
        return response

    def set_public(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, HASH), data[HASH])
        self._set_text(etree.SubElement(response, SHORTURL), data[SHORTURL])
        self._set_text(etree.SubElement(response, URL), data[URL])
        return response

    def public_info(self, **kw):
        form = self.request.form
        user_info = self.data[RESULT]['user']
        root_node = etree.Element(RESPONSE, status=unicode(self.data[STATUS]))
        user_node = etree.SubElement(root_node, 'user')
        for k,v in user_info.iteritems():
            self._set_text(etree.SubElement(user_node, k), v)
        resource_element = etree.SubElement(root_node, FILE)
        root_id = form.id
       
        self._set_text(etree.SubElement(resource_element, NAME), form.name)
        self._set_text(etree.SubElement(resource_element, SIZE), form.size)
        self._set_text(etree.SubElement(resource_element, TYPE), form.media_type)
        self._set_text(etree.SubElement(resource_element, FILE_ICON), form.file_icon)
        self._set_text(etree.SubElement(resource_element, MIMETYPE), form.mimetype)
        self._set_text(etree.SubElement(resource_element, DOCVIEWER_TYPE), form.docviewer_type)
        self._set_text(etree.SubElement(resource_element, SPEED_LIMITED), form.speed_limited)
       
        date_element = etree.SubElement(resource_element, DATE)
        for etag in (CTIME, MTIME):
            self.add_nonempty_node(date_element, etag, getattr(form, etag))

        meta_data = form.meta
        tags = meta_data.pop(TAGS, [])
        thumb = meta_data.pop(THUMBNAIL, '')
        preview = meta_data.pop(PREVIEW, '')
        drweb = meta_data.pop(DRWEB, '0')  
        
        meta_element = etree.SubElement(resource_element, META)
        self._set_text(etree.SubElement(meta_element, THUMBNAIL), re.sub("^(http|https):", '', thumb))
        self._set_text(etree.SubElement(meta_element, PREVIEW), re.sub("^(http|https):", '', preview))
        self._set_text(etree.SubElement(meta_element, DRWEB), drweb)
        
        if 'sizes' in meta_data:
            self._parse_meta(meta_element, {'sizes': meta_data.get('sizes')}, '', all_meta=meta_data)

        return root_node

    def public_url(self, **kw):
        return self.url(**kw)

    def error(self, err):
        xmlObject, http_error_code = self.xml_error(err)
        # Для верстки отдаем только коды >= 500
        if http_error_code >= 500:
            self.request.http_resp.status = http_error_code
        return xmlObject

    def msviewer_wacurl(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, URL), data[URL])
        return response

    def msviewer_filemeta(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, META), to_json(data[META]))
        return response

    def _user_invite_info(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")

        for item in data:
            item = dict(map(lambda x: (x, wise_to_str(item[x])), item.keys()))
            item_element = etree.Element('item', attrib=item)
            response.append(item_element)
        return response

    def user_invite_hash(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, HASH), data)
        return response

    def user_invite_info(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        for key, value in data.iteritems():
            if isinstance(value, dict):
                v = dict(map(lambda x: (x, wise_to_str(value[x])), value.keys()))
                item_element = etree.Element(key, attrib=v)
                response.append(item_element)
            else:
                self._set_text(etree.SubElement(response, key), value)
        return response

    def user_invite_activated(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        for item in data:
            user = etree.Element(USER)
            for k, v in item.iteritems():
                self._set_text(etree.SubElement(user, k), v)
            response.append(user)
        return response

    def social_rights(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        services = etree.Element(SERVICES)
        response.append(services)

        for provider, status in data.iteritems():
            service = etree.Element(SERVICE)
            self._set_text(etree.SubElement(service, PROVIDER), provider)
            self._set_text(etree.SubElement(service, AUTH), int(status[AUTH]))
            self._set_text(etree.SubElement(service, LINK), status[LINK])
            services.append(service)

        return response

    def share_create_group(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, 'gid'), data['gid'])
        return response

    def share_invite_user(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        self._set_text(etree.SubElement(response, 'gid'), data['gid'])
        self._set_text(etree.SubElement(response, 'hash'), data['hash'])
        self._set_text(etree.SubElement(response, 'oid'), data['oid'])
        return response

    def share_users_in_group(self, **kw):
        data = self.data[RESULT]
        response = etree.Element(RESPONSE, status="1")
        users_tag = etree.SubElement(response, 'users')
        for user in data['users']:
            one_user = etree.SubElement(users_tag, 'user')
            for k,v in user.iteritems():
                self._set_text(etree.SubElement(one_user, k), v)
        return response

    def __share_list_folders__(self, root, data=None):
        if data is None:
            data = self.data[RESULT]
        if not isinstance(data, (list, tuple)):
                data = [data, ]
        for each in data:
            self._info(each, root)

    def share_list_all_folders(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        for v in self.data[RESULT].itervalues():
            self.__share_list_folders__(response, data=v)
        return response

    def share_list_owned_folders(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        self.__share_list_folders__(response)
        return response

    def share_list_joined_folders(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        self.__share_list_folders__(response)
        return response

    def share_list_not_approved_folders(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        for folder_data in self.data[RESULT]:
            folder = etree.SubElement(response, 'folder')
            for k,v in folder_data.iteritems():
                self._set_text(etree.SubElement(folder, k), v)
        return response

    def share_invite_info(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        invite = etree.SubElement(response, 'invite')
        for k, v in self.data[RESULT].iteritems():
            self._set_text(etree.SubElement(invite, k), v)
        return response

    def share_folder_info(self, **kw):
        response = etree.Element(RESPONSE, status="1")
        folder = etree.SubElement(response, 'folder')
        for k, v in self.data[RESULT].iteritems():
            self._set_text(etree.SubElement(folder, k), v)
        return response

    def share_activate_invite(self):
        return self._info(
            self.request.form,
            etree.Element(RESPONSE, status=unicode(self.data[STATUS]))
        )
