# -*- coding: utf-8 -*-
from __future__ import absolute_import
import json
import cjson
import urllib
from email.message import Message

import mpfs.engine.process
import mpfs.common.static.messages as messages
import mpfs.common.static.tags as tags
import mpfs.common.errors as errors
from mpfs.frontend.formatter.disk import MPFSFormatter

log = mpfs.engine.process.get_default_log()


class Service(MPFSFormatter):
    def process(self, f, **kw):
        self.data = self.request.data
        if hasattr(self, f):
            return getattr(self, f)(**kw)
        else:
            return cjson.encode(self.request.data)

    def error(self, err):
        response = self.request.http_resp
        if not response:
            return
        if isinstance(err, errors.MPFSRootException):
            response.status = getattr(err, 'response', messages.default_response)
            if getattr(err, 'headers', None):
                self._add_header_params_to_response(err.headers)
            return cjson.encode({'code': getattr(err, 'code', ''), 'title': getattr(err, 'message', '')})
        else:
            try:
                code = int(err)
            except TypeError:
                code = messages.default_response
            response.status = messages.responses.get(code, messages.default_response)

    def kladun_callback(self, **kw):
        result = self.data[tags.RESULT]
        if not isinstance(result, dict):
            return cjson.encode(result)
        if '_headers' in result:
            result = self._parse_and_pull_out_header_params(result)
        return self.to_json(result)

    def kladun_download_counter_inc(self, **kw):
        return ''

    def xiva_subscribe(self, **kw):
        return self.data[tags.RESULT] if self.request.return_id else ''

    def xiva_unsubscribe(self, **kw):
        return ''

    def direct_url(self, **kw):
        response = self.request.http_resp
        response.version = (1, 1)
        msg = Message()

        # https://jira.yandex-team.ru/browse/CHEMODAN-6835
        filename = self.data[tags.RESULT]['name']
        encoded_name = 'UTF-8' + "''" + urllib.quote(filename.encode('utf-8'))

        for h, v in self.request.request_headers.iteritems():
            response.headers[h] = v

        headers = (
            (('X-Target-Content-Type', self.data[tags.RESULT]['mimetype']), {}),
            (('X-Target-Content-Disposition', 'attachment; filename*=' + encoded_name), {}),
            (('Location', self.data[tags.RESULT]['url']), {}),
        )
        for _args, _kwargs in headers:
            msg.add_header(*_args, **_kwargs)
        for h, v in msg._headers:
            response.headers[h] = v
        response.status = 302

    def public_direct_url(self, **kw):
        return self.direct_url()

    def dv_data(self, **kw):
        result = self.data['result']
        return self.to_json(result)

    def public_dv_data(self, **kw):
        result = self.data['result']
        return self.to_json(result)

    def public_copy(self, **kw):
        return cjson.encode(True)

    def inspect(self, **kw):
        return cjson.encode(self.data[tags.RESULT])

    def break_counters(self, **kw):
        return cjson.encode(self.data[tags.RESULT])

    def echo(self, **kw):
        result = dict(self.data[tags.RESULT].iteritems())
        log.info('echo: %s' % result)
        return cjson.encode(result)

    def hardlink(self, **kw):
        return cjson.encode(self.data[tags.RESULT])

    def bulk_check_stids(self, **kw):
        return cjson.encode(self.data[tags.RESULT])

    def show_settings(self, **kw):
        # cjson не переваривает содержимое ответа
        return json.dumps(self.data[tags.RESULT], indent=2)
