# -*- coding: utf-8 -*-


import os
import traceback

from itertools import cycle
from copy import deepcopy


import mpfs.engine.process

from mpfs.common.static import codes
from mpfs.common.util import logger
from mpfs.common import errors


log = mpfs.engine.process.get_access_log()
error_log = mpfs.engine.process.get_error_log()

ids = cycle(xrange(100000, 9223372036854775807))


class BaseRequest(object):

    fields_map = {}
    nonempty_fields = ('src', 'dst')
    str_fields = ('uid', 'md5', 'sha256')

    def __init__(self, params):
        self.http_req = params.pop('_request', '')
        self.http_resp = params.pop('_response', '')

        if self.http_req:
            self.query_string = self.http_req.requestLine
            self.request_headers = self.http_req.headers
        else:
            self.query_string = ''
            self.request_headers = {}

        try:
            self.__request_type = self.query_string.split('/')[1]
        except Exception:
            self.__request_type = ''

        self.form = None
        self.args_data = RequestArguments()

        self.is_default_formatter_required = False
        self.method_name = None

    def get_type(self):
        return self.__request_type

    @property
    def args(self):
        return deepcopy(self.args_data)

    def set_args(self, args):
        self.data = {}

        def encode_args(_args):
            if isinstance(_args, str) and not isinstance(_args, unicode):
                _args = _args.decode('utf-8')
            elif isinstance(_args, dict):
                for _k, _v in deepcopy(_args).iteritems():
                    if _k in self.nonempty_fields and not _v:
                        raise errors.EmptyFields(extra_msg=_k)
                    if _k in self.fields_map:
                        if isinstance(_v, dict):
                            for __k, __v in _v.iteritems():
                                if __k in self.fields_map[_k]:
                                    _v1 = _args[_k].pop(__k)
                                    _args[_k][self.fields_map[_k][__k]] = encode_args(_v1)
                                else:
                                    _args[_k][__k] = encode_args(__v)
                        elif _v in self.fields_map[_k]:
                            _args[_k] = encode_args(self.fields_map[_k][_v])
                    else:
                        _args[_k] = encode_args(_v)
            elif isinstance(_args, list):
                for i, _v in enumerate(_args):
                    _args[i] = encode_args(_v)
            return _args
        args = encode_args(args)
        self.args_data = RequestArguments(args.pop('args', {}))
        for k, v in args.iteritems():
            setattr(self, k, v)

    def enforce_default_formatter(self):
        self.is_default_formatter_required = True

    def set_result(self, data=None):
        self.data = {'status': codes.OK, 'result': data }

    def set_error(self, data=None):
        self.data = {'status': codes.ERROR, 'result': data}
        self.exc_traceback = traceback.format_exc()


class UserRequest(BaseRequest):

    def set_id(self):
        self.id = '%s_%s' % (os.getpid(), ids.next())
        mpfs.engine.process.set_req_id(self.id)
        mpfs.engine.process.reset_cached()

    def set_cloud_id(self, crid):
        self.cloud_id = crid
        mpfs.engine.process.set_cloud_req_id(crid)

    def set_form(self, form, force=False):
        if self.form is None or force:
            self.form = form


class MailRequest(UserRequest):

    fields_map = {
        'filter' : {
            'type' : 'media_type',
        },
        'sort' : {
            'type' : 'media_type',
        },
    }



class JSONRequest(UserRequest):

    fields_map = {
        'filter' : {
            'type' : 'media_type',
        },
        'sort' : {
            'type' : 'media_type',
        },
    }


class ServiceRequest(UserRequest):
    pass

class BrowserRequest(UserRequest):

    def set_args(self, args):
        self.data = {}
        for k,v in args.iteritems():
            setattr(self, k, v)


class RequestArguments(object):

    def __init__(self, data={}):
        assert isinstance(data, dict)
        self.data = data
        self.forward = True

    def __getitem__(self, *args, **kwargs):
        return self.data[args[0]]

    def __setitem__(self, *args, **kwargs):
        self.data[args[0]] = args[1]

    def __nonzero__(self):
        return bool(self.data)

