# -*- coding: utf-8 -*-
"""

MPFS
INVITE

Код инвайтов

"""
import string
import time
import random
import traceback
import inspect
import sys

import mpfs.engine.process

from mpfs.core.metastorage.control import invite_mpfs_codes
from mpfs.invite import errors
from mpfs.common.util import straighten_dictionary

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


'''
Состояния кодов
FREE - свободен
RESERVED - отправлен на определенный адрес
ACTIVATED - активирован
BLOCKED - заблокирован
'''
states = (
    FREE,
    RESERVED,
    ACTIVATED,
    BLOCKED,    
) = xrange(4)

'''
Имена состояний - нужны для отображения
'''
states_names = ('FREE', 'RESERVED', 'ACTIVATED', 'BLOCKED')


def generate_hash(project):
    '''
    Сгенерировать код
    '''
    def hash_generator(size=8, chars=string.ascii_uppercase + string.digits):
        return ''.join(random.choice(chars) for x in range(size))

    while 1:
        hash = hash_generator()
        response = invite_mpfs_codes.show(hash)
        if response.value is None:
            return hash
    
    
def select(project, common_args=None, meta_args=None, sort=None, maxresults=None):
    '''
    Выборка кодов по условиям
    '''
    if not common_args: common_args = {}
    if not meta_args: meta_args = {}
    if not sort: sort = 'ctime'

    common_args = straighten_dictionary(common_args)
    meta_args   = straighten_dictionary(meta_args)
        
    result = invite_mpfs_codes.find(
        None,
        None,
        common_args,
        meta_args,
        sort,
        maxresults
    )
        
    return result


class Code(object):
    '''
    Фабрика кодов    
    Данные храним в invite_mpfs_codes
    '''
    type = None
    
    @classmethod
    def Load(classname, project, hash):
        response = invite_mpfs_codes.show(hash)

        if response.value is None:
            raise errors.InviteNotFound()
            
        # старые коды сохранялись без типа
        if 'type' not in response.value.data:
            code_type = 'expendable'
        else:
            code_type = response.value.data.pop('type')
        
        try:
            klass = CODE_CLASSES[code_type]
        except Exception:
            raise errors.InviteNotFound()
        
        return klass(project, **response.value.data)


    def __init__(self, project, **data):
        for key, value in self.dir().iteritems():
            setattr(self, key, value)

        for key, value in data.iteritems():
            setattr(self, key, value)
            
        self.project = project
        
    
    def dir(self):
        return {
            'hash' : None, 
            'state': None,
            'ctime': None, 
            'mtime': None,
            'type' : self.type,
            'meta' : {},
        }


    def dict(self):
        result = {}
        attributes = self.dir()
        for attr in attributes.keys():
            result[attr] = getattr(self, attr)
        return result

    
    @classmethod
    def Create(classname, project, **kw):
        '''
        Создание кода
        Сохранит все переданные параметры  в meta
        '''
        current_time = int(time.time())
        hash = generate_hash(project)

        data = {
            'hash' : hash,
            'state': FREE,
            'ctime': current_time,
            'mtime': current_time,
            'type' : classname.type,
            'meta' : kw
        }
        invite_mpfs_codes.put(hash, data) 
        obj = classname(project, **data)
        
        log.debug('invite code (%s) %s:%s created' % (classname.type, project, hash))
        return obj


    def update(self, data={}):
        '''
        Обновление метаданных 
        '''
        for k, v in data.iteritems():
            self.meta[k] = v
        self.mtime = int(time.time())
        
        invite_mpfs_codes.update(self.hash, self.dict())
        log.info('invite code (%s) %s:%s updated' % (self.type, self.project, self.hash))


    def delete(self):
        try:
            invite_mpfs_codes.remove(self.hash)    
            log.info('invite code (%s) %s:%s removed' % (self.type, self.project, self.hash))
        except Exception, e:
            error_log.error(traceback.format_exc())
            log.error('invite code (%s) %s:%s dropping failed' % (self.type, self.project, self.hash))


    def activate(self, **kw):
        raise errors.InviteError()
        
        
    def block(self, **kw):
        self.state = BLOCKED
        self.update({'blocked': kw})
        
         
class ExpendableCode(Code):
    '''
    Класс невечного кода
    '''
    type = 'expendable'
        
    def activate(self, **kw):
        '''
        Активация кода
        Переданные параметры запишутся в meta.activated
        '''
        if self.can_be_activated():
            self.state = ACTIVATED
            self.update({'activated': kw})
        elif self.state == ACTIVATED:
            raise errors.InviteActivated()
        elif self.state == BLOCKED:
            raise errors.InviteBlocked()
        else:
            raise errors.InviteError(self.state)
            
            
    def block(self, **kw):
        '''
        Заблокировать код
        Переданные параметры запишутся в meta.blocked
        '''
        if self.can_be_activated():
            self.state = BLOCKED
            self.update({'blocked': kw})
        elif self.state == ACTIVATED:
            raise errors.InviteActivated()
        elif self.state == BLOCKED:
            raise errors.InviteBlocked()
        else:
            raise errors.InviteError(self.state)
            
            
    def can_be_activated(self):
        '''
        Проверка, что код не активирован
        '''
        if self.state in (FREE, RESERVED):
            return True
        return False
            
            
    def reserve(self, address, **kw):
        '''
        Резервирование кода
        Переданные параметры запишутся в meta.reserved
        '''
        if self.state is FREE:
            self.state = RESERVED
            kw['email'] = address
            self.update({'reserved': kw})
        elif self.state == ACTIVATED:
            raise errors.InviteActivated()
        elif self.state == BLOCKED:
            raise errors.InviteBlocked()
        elif self.state == RESERVED:
            raise errors.InviteReserved()
        else:
            raise errors.InviteError(self.state)
        

class EternalCode(Code):
    '''
    Класс вечного кода
    Не дизейблится после активации
    Может быть заблокирован
    '''
    type = 'eternal'
    
    def activate(self, **kw):
        '''
        Активация кода
        Переданные параметры будут вносится в массив meta.activated
        '''
        if self.state == BLOCKED:
            raise errors.InviteBlocked()
            
        self.state = ACTIVATED
        activated_users = self.meta.get('activated', [])
        activated_users.append(kw)
        self.update({'activated': activated_users})
        
        
    def can_be_activated(self):
        '''
        Проверка, что код не активирован
        '''
        if self.state in (FREE, RESERVED, ACTIVATED):
            return True
        return False


CODE_CLASSES = dict(
    (cls.type, cls) for (clstype, cls) in
    filter(
        lambda (c_type, c): hasattr(c, 'type') and c.type,
        inspect.getmembers(sys.modules[__name__], inspect.isclass)
    )
)
