# -*- coding: utf-8 -*-
"""

MPFS
INVITE

Код контакта 

"""
import time
import random
import traceback
import inspect
import sys
from copy import deepcopy

import mpfs.engine.process

from mpfs.core.metastorage.control import invite_mpfs_contacts
from mpfs.invite import errors
from mpfs.common.util import email_valid, straighten_dictionary
from mpfs.core.services.passport_service import Passport

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport = Passport()

'''
Состояния имейлов
NEW - новый
PROCESSED - нормально отработан
BLOCKED - заблокирован
'''
states = (
    NEW,
    PROCESSED,
    BLOCKED,    
) = xrange(3)

'''
Имена состояний - нужны для отображения
'''
states_names = ('NEW', 'PROCESSED', 'BLOCKED')

'''
Яндексовые домены
Принадлежащие к ним почтовые адреса мы считаем яндексовыми
'''
YANDEX_DOMAINS = [
    'yandex.ru',
]


def get_db_id(address, type):
    '''
    Получить DB-идентификатор контакта
    '''
    return '%s:%s' % (type, address)
    

def select(project, table='new', common_args=None, meta_args=None, sort=None, desc=False, shuffle=False, maxresults=None):
    '''
    Выборка контактов по условиям
    '''
    if not common_args: common_args = {}
    if not meta_args: meta_args = {}
    if not sort: sort = 'ctime'
    
    if table == 'new':
        common_args['state'] = NEW
    elif table == 'archive':
        common_args['state'] = (PROCESSED, BLOCKED)
    
    common_args = straighten_dictionary(common_args)
    meta_args   = straighten_dictionary(meta_args)
    
    if desc:
        sort = '-%s' % sort
    
    result = invite_mpfs_contacts.find(
        None,
        None,
        common_args,
        meta_args,
        sort,
        maxresults
    )
        
    if shuffle:
        random.shuffle(result)
    
    return result

        
class Contact(object):
    '''
    Класс записи контакта в очереди
    Данные храним в invite_mpfs_contact
    '''    
    type = None

    @classmethod
    def Load(classname, project, address, type):
        '''
        Загрузка произвольного контакта
        Возвратит объект нужного класса
        '''
        try:
            klass = CONTACT_CLASSES[type]
        except Exception:
            raise errors.ContactNotFound()
        
        return klass(project, address)
        
    @classmethod
    def Create(classname, project, address, type, **kw):
        '''
        Сохранение контакта в очереди
        Сохранит все переданные параметры в meta
        Возвратит объект нужного класса
        '''
        try:
            klass = CONTACT_CLASSES[type]
        except Exception:
            raise errors.ContactNotFound()
            
        return klass.Create(project, address, **kw)

    def __init__(self, project, address):
        self.db_id = get_db_id(address, self.type)
        response = invite_mpfs_contacts.show(self.db_id)
        
        if response.value is None:
            raise errors.ContactNotFound()

        for key, value in self.dir().iteritems():
            setattr(self, key, value)

        for key, value in response.value.data.iteritems():
            setattr(self, key, value)
            
        self.project = project
        self.locale = self.meta.get('locale')

    def dir(self):
        return {
            'address': None,
            'state'  : None,
            'type'   : None, 
            'ctime'  : None, 
            'mtime'  : None,
            'meta'   : {},
        }

    def dict(self):
        result = {}
        attributes = self.dir()
        for attr in attributes.keys():
            result[attr] = deepcopy(getattr(self, attr))
        return result

    @classmethod
    def CommonCreate(classname, project, address, **kw):
        try:
            exists = classname(project, address)
            if exists:
                raise errors.ContactAlreadyExist()
        except errors.ContactNotFound:
            pass
        
        db_id = get_db_id(address, classname.type)
        current_time = int(time.time())
        
        data = {
            'address': address,
            'state' : NEW,
            'ctime' : current_time,
            'mtime' : current_time,
            'meta'  : kw,
            'type'  : classname.type,
        }
        invite_mpfs_contacts.put(db_id, data)
            
        obj = classname(project, address)
        log.debug('invite contact %s:%s:%s pushed' % (project, classname.type, address))
        
        return obj

    def update(self, data={}):
        '''
        Обновление метаданных 
        '''
        for k, v in data.iteritems():
            self.meta[k] = v
        self.mtime = int(time.time())
        
        invite_mpfs_contacts.update(self.db_id, self.dict())
        log.info('invite contact %s:%s:%s updated' % (self.project, self.address, self.type))

    def archivated_or_blocked(self):
        return self.state is not NEW

    def archivate(self, hash, **kw):
        '''
        Перенос записи в архив с установкой состояния и кода 
        '''
        kw['hash'] = hash
        self._move_to_archive(PROCESSED, **kw)

    def block(self, **kw):
        '''
        Блокировка записи
        '''
        self._move_to_archive(BLOCKED, **kw)

    def _move_to_archive(self, state, **kw):
        '''
        Перенос записи в архив - практическая часть
        '''
        if not self.archivated_or_blocked():
            self.state = state
            self.update(kw)

    def delete(self):
        try:
            invite_mpfs_contacts.remove(self.db_id)    
            log.info('invite contact %s:%s:%s removed' % (
                self.project, self.type, self.address))
        except Exception, e:
            error_log.error(traceback.format_exc())
            log.error('invite contact %s:%s:%s dropping failed' % (
                self.project, self.type, self.address))
            

class Email(Contact):
    '''
    Класс записи имейла в очереди
    '''
    type = 'email'
    
    def __init__(self, project, address):
        super(Email, self).__init__(project, address)
        self.is_yandex = self._is_yandex()

    @classmethod
    def Create(classname, project, address, **kw):
        '''
        Сохранение имейла в очереди
        Сохранит все переданные параметры в meta
        '''
        if not email_valid(address):
            raise errors.ContactInvalid()
        
        # Проверяем что email не из ПДД
        if passport.is_from_pdd(address):
            raise errors.ContactNotServed()
        
        return classname.CommonCreate(project, address, **kw)

    def _is_yandex(self):
        name, domain = self.address.split('@')
        return domain in YANDEX_DOMAINS


class Facebook(Contact):
    '''
    Класс записи фейсбук-контакта в очереди
    '''
    type = 'fb'
    
    @classmethod
    def Create(classname, project, address, **kw):
        return classname.CommonCreate(project, address, **kw)
    
    
CONTACT_CLASSES = dict(
    (cls.type, cls) for (clstype, cls) in
    filter(
        lambda (c_type, c): hasattr(c, 'type') and c.type,
        inspect.getmembers(sys.modules[__name__], inspect.isclass)
    )
)
