# -*- coding: utf-8 -*-
"""

MPFS
INVITE

"""
import copy
import events as invite_events

from mpfs.invite import errors
from mpfs.config import settings
from mpfs.common.util import mailer
from mpfs.invite.code import Code, ExpendableCode, EternalCode
from mpfs.invite.code import select as select_codes
from mpfs.invite.contact import Contact
from mpfs.invite.contact import select as select_emails
from mpfs.core.user.base import Check, Create
from mpfs.core.services.passport_service import Passport
from mpfs.core.metastorage.decorators import allow_empty_disk_info


def list_new_emails(project, common_params=None, meta_params=None, sort=None, desc=False, shuffle=False, maxresults=None):
    '''
    Выборка по необработанным имейлам
    '''
    return select_emails(project, 'new', common_params, meta_params, sort, desc, shuffle, maxresults)

def list_archived_emails(project, common_params=None, meta_params=None, sort=None, maxresults=None):
    '''
    Выборка по обработанным имейлам
    '''
    return select_emails(project, 'archive', common_params, meta_params, sort, maxresults)
    
def list_codes(project, common_params=None, meta_params=None, order=None, maxresults=None):
    '''
    Выборка по кодам
    '''
    return select_codes(project, common_params, meta_params, order, maxresults)

def generate_expendable_code(project, **kw):
    '''
    Сгенерировать одноразовый код без привязки к почте
    '''
    return ExpendableCode.Create(project, **kw)
    
def generate_eternal_code(project, **kw):
    '''
    Сгенерировать вечный код без привязки к почте
    '''
    return EternalCode.Create(project, **kw)
    
def get_code(project, hash):
    '''
    Попробовать получить код
    '''
    return Code.Load(project, hash)
    
def block_code(project, hash, **kw):
    '''
    Заблокировать код указав причины
    '''
    Code.Load(project, hash).block(**kw)
    return True

def block_email(project, address, **kw):
    '''
    Заблокировать email, указав причины
    '''
    Contact.Load(project, address, type='email').block(**kw)
    return True

def remove_code(project, hash, **kw):
    '''
    Удалить код 
    '''
    Code.Load(project, hash).delete()
    return True

def remove_email(project, address, **kw):
    '''
    Удалить email
    '''
    Contact.Load(project, address, type='email').delete()
    return True

def code_info(project, hash):
    '''
    Выдать информацию по владельцу кода
    '''
    code = Code.Load(project, hash)
    referral = code.meta.get('referral')
    if not referral: raise errors.InviteHasNoOwner()
    
    userinfo = Passport().userinfo(referral)
    return userinfo.get('public_name')

def push_email(project, address, **kw):
    '''
    Поставить email в очередь 
    '''
    email = Contact.Create(project, address, type='email', **kw)
    if not bool(kw.get('no_notify', False)):
        template = settings.invite['templates']['pushed']
        params = {
            'email'   : email,
            'locale'  : email.locale,
        }
        mailer.send(email.address, template, params)
    return email

def generate_and_send_expendable_code(project, address):
    '''
    Сгенерировать одноразовый код для почты и выслать
    '''
    try:
        email = Contact.Load(project, address, type='email')
    except errors.ContactNotFound, e:
        email = Contact.Create(project, address, type='email')
        
    if email.archivated_or_blocked():
        raise errors.ContactArchivatedOrBlocked()

    code = generate_expendable_code(email.project)
    code.reserve(email.address, **email.meta)
    email.archivate(code.hash)

    params = {
        'code'    : code,
        'email'   : email,
        'locale'  : email.locale,
    }
    template = settings.invite['templates']['got']
    mailer.send(email.address, template, params)
    
    return code

@allow_empty_disk_info
def activate_code(project, hash, uid, **kw):
    '''
    Активировать код и выполнить инициализацию проекта
    '''
    code = Code.Load(project, hash)
    
    if not code.can_be_activated():
        raise errors.InviteActivated()
      
    kw.update({'uid': uid})
    activation_params = copy.copy(kw)

    referral = code.meta.get('referral', None)
    if referral:
        activation_params.update({'referral': referral})
    
    try:
        Check(uid)
    except Exception:
        pass
    else:
        raise errors.UserAlreadyExists(uid)
    
    Create(**activation_params)
    code.activate(**kw)

    invite_events.InviteActivationEvent(uid=uid, project=project, owner_uid=referral).send()

    return code
