# -*- coding: utf-8 -*-
import pymongo

from mpfs.engine.process import get_read_preference

from mpfs.metastorage.mongo.logging import __set_request_message__
from mpfs.metastorage.mongo.util import autoretry, get_active_mode, wrap_if_hasattr, LazyDatabaseObject


class MPFSMongoCollection(pymongo.collection.Collection):
    MPFS_MESSAGE_TMPL = '%(conn_name)s.%(db_name)s.%(coll_name)s.%(method)s(%(query_args)s, %(query_kwargs)s)'
    MPFS_READ_MESSAGE_TMPL = '%(conn_name)s.%(db_name)s.%(coll_name)s.%(method)s(%(query_args)s, %(query_kwargs)s, mode=%(mode)s)'
    MPFS_READ_METHODS = {'find', 'aggregate', 'find_one'}
    __function_call__find_one = False

    @staticmethod
    def get_global_read_preference():
        return get_read_preference()

    def _set_global_read_preference_if_defined(self, kwargs):
        if self.get_global_read_preference() is not None:
            kwargs['read_preference'] = self.get_global_read_preference()

    def _set_mpfs_request_message(self, method, query_args, query_kwargs):
        """
        Помощник логирования запросов в монгу
        """
        # портированная логика, чтобы `find_one` в логах не писался как `find`
        if method == 'find_one':
            self.__function_call__find_one = True
        elif method == 'find' and self.__function_call__find_one:
            self.__function_call__find_one = False
            return

        message_data = {
            'conn_name': self.database.connection.name,
            'db_name': self.coll.database.name,
            'coll_name': self.name,
            'method': method,
            'query_args': str(query_args),
            'query_kwargs': str(query_kwargs),
        }
        if method in self.MPFS_READ_METHODS:
            message_data['mode'] = \
                get_active_mode(self.database.connection, query_kwargs)
            message = self.MPFS_READ_MESSAGE_TMPL % message_data
        else:
            message = self.MPFS_MESSAGE_TMPL % message_data
        __set_request_message__(message)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def update(self, *args, **kwargs):
        # одновременные запросы с одинаковым query с флагом upsert могут конфликтовать
        # официальный workaround: retry
        #
        # https://st.yandex-team.ru/CHEMODAN-33935
        # https://jira.mongodb.org/browse/SERVER-14322

        if kwargs.get('upsert'):
            try:
                self._set_mpfs_request_message('update', args, kwargs)
                return super(MPFSMongoCollection, self).update(*args, **kwargs)
            except pymongo.errors.DuplicateKeyError:
                kwargs.pop('upsert')

        self._set_mpfs_request_message('update', args, kwargs)
        return super(MPFSMongoCollection, self).update(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def insert(self, *args, **kwargs):
        self._set_mpfs_request_message('insert', args, kwargs)
        return super(MPFSMongoCollection, self).insert(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def find_and_modify(self, *args, **kwargs):
        self._set_mpfs_request_message('find_and_modify', args, kwargs)
        return super(MPFSMongoCollection, self).find_and_modify(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def remove(self, *args, **kwargs):
        self._set_mpfs_request_message('remove', args, kwargs)
        return super(MPFSMongoCollection, self).remove(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def find(self, *args, **kwargs):
        self._set_global_read_preference_if_defined(kwargs)
        self._set_mpfs_request_message('find', args, kwargs)
        return super(MPFSMongoCollection, self).find(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def find_one(self, *args, **kwargs):
        self._set_global_read_preference_if_defined(kwargs)
        self._set_mpfs_request_message('find_one', args, kwargs)
        return super(MPFSMongoCollection, self).find_one(*args, **kwargs)

    @wrap_if_hasattr(autoretry, 'turn_off_autoretry')
    def aggregate(self, *args, **kwargs):
        self._set_global_read_preference_if_defined(kwargs)
        self._set_mpfs_request_message('aggregate', args, kwargs)
        return super(MPFSMongoCollection, self).aggregate(*args, **kwargs)

    def get_indexes(self):
        return self.database.connection.get_indexes(self.database.name, self.name)


class MPFSMongoLazyCollection(LazyDatabaseObject):
    db = None

    def __init__(self, db, collection_name, *args, **kwargs):
        super(MPFSMongoLazyCollection, self).__init__(*args, **kwargs)
        self.db = db
        self.__name = collection_name

    def get_indexes(self):
        return self.db.connection.get_indexes(self.database.name, self.name)

    def get_instance(self):
        if isinstance(self.db, LazyDatabaseObject):
            self.db = self.db.get_actual()
        return self.db[self.name]

    def __getattr__(self, item):
        return getattr(self.get_actual(), item)

    @property
    def name(self):
        return self.__name
