# -*- coding: utf-8 -*-

import random
from collections import namedtuple

from mpfs.engine.process import dbctl, usrctl
from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter


DBResultRecord = namedtuple('DBResultRecord', [
    'shard_name',
    'collection_name',
    'record',
])


class AllUserDataCollection(object):
    """Псевдоколлекция позволяющая обойти все коллекции с пользовательскими данными

    Возвращает сырые данные "как в базе"
    """

    DATA_COLLECTIONS = {
        'user_data',
        'trash',
        'attach_data',
        'narod_data',
        'hidden_data',
        'misc_data',
        'photounlim_data',
        'notes_data',
        'additional_data',
        'client_data'
    }

    def __init__(self, enable_collections=None, filter_postgres_shards=False):
        """
        :param enable_collections: в каких коллекциях искать документы. Подмножество `DATA_COLLECTIONS`
        """
        self.db = CollectionRoutedDatabase()

        self._search_in_collection_names = self.DATA_COLLECTIONS.copy()
        self._filter_postgres_shards = filter_postgres_shards

        if enable_collections:
            enable_collections = set(enable_collections)
            if not enable_collections.issubset(self.DATA_COLLECTIONS):
                raise ValueError("Unsupported collection names: %s" % (enable_collections - self.DATA_COLLECTIONS))
            self._search_in_collection_names = enable_collections

    def count(self, *args, **kwargs):
        """Посчитать кол-во документов на всех шардах

        :rtype generator:
        """
        return self._count_on_shards(self._get_all_shards(), *args, **kwargs)

    def find(self, *args, **kwargs):
        """Поискать на всех шардах

        :rtype generator:
        """
        return self._find_on_shards(self._get_all_shards(), *args, **kwargs)

    def find_on_uid_shard(self, uid, *args, **kwargs):
        """Поискать на шарде, где живет пользователь

        :rtype generator:
        """
        return self._find_on_shards([self._get_uid_shard(uid)], *args, **kwargs)

    def find_one(self, *args, **kwargs):
        return next(self.find(*args, **kwargs), None)

    def find_one_on_uid_shard(self, *args, **kwargs):
        return next(self.find_on_uid_shard(*args, **kwargs), None)

    def _get_all_shards(self):
        """Отдает список всех шардов, где хранятся пользовательские данные"""
        mongo_shards = list(dbctl().mapper.rspool.get_all_shards_names())
        pg_shards = []
        if not self._filter_postgres_shards:
            pg_shards = list(PGQueryExecuter().get_all_shard_ids())
        return mongo_shards + pg_shards

    @staticmethod
    def _get_uid_shard(uid):
        """Отдает имя шарда, на котором живет пользователь"""
        if usrctl().is_collection_in_postgres(uid, 'user_data'):
            shard_name = PGQueryExecuter().get_shard_id(uid)
        else:
            shard_name = dbctl().mapper.get_rsname_for_uid(uid)
        return shard_name

    def _count_on_shards(self, shard_names, *args, **kwargs):
        for shard_name, collection_name, cursor in self._get_cursors(shard_names, *args, **kwargs):
            yield DBResultRecord(
                shard_name=shard_name,
                collection_name=collection_name,
                record=cursor.count()
            )

    def _find_on_shards(self, shard_names, *args, **kwargs):
        for shard_name, collection_name, cursor in self._get_cursors(shard_names, *args, **kwargs):
            for record in cursor:
                yield DBResultRecord(
                    shard_name=shard_name,
                    collection_name=collection_name,
                    record=record
                )

    def _get_cursors(self, shard_names, *args, **kwargs):
        for not_supported_keyword in ('sort', 'skip', 'limit'):
            if not_supported_keyword in kwargs:
                raise NotImplementedError('No %r support' % not_supported_keyword)

        if not self._search_in_collection_names:
            raise StopIteration()
        if not shard_names:
            raise StopIteration()

        if not isinstance(shard_names, list):
            raise TypeError()

        # размазываем нагрузку между шардами
        random.shuffle(shard_names)

        for shard_name in shard_names:
            for collection_name in self._search_in_collection_names:
                kwargs['shard_name'] = shard_name
                cursor = self.db[collection_name].find_on_shard(*args, **kwargs)
                yield shard_name, collection_name, cursor
