# -*- coding: utf-8 -*-
import copy
import operator
import sys
import pymongo

from itertools import imap, ifilter
from collections import defaultdict, deque
from functools import wraps

from pymongo.errors import ConnectionFailure, PyMongoError, ConfigurationError, AutoReconnect

import mpfs.engine.process
import mpfs.common.util.dbnaming

from mpfs.config import settings
from mpfs.core.filesystem.dao.legacy import is_collection_uses_dao, is_new_fs_spec_required, CollectionRoutedDatabase
from mpfs.core.filesystem.dao.resource import AdditionalDataDAO
from mpfs.core.filesystem.live_photo import LivePhotoFilesManager
from mpfs.metastorage.mongo.util import *
from mpfs.metastorage.mongo.binary import ObjectId
from mpfs.common import errors
from mpfs.common.errors.share import GroupNotFound
from mpfs.common.util import Singleton, Cached, method_caller
from mpfs.metastorage import MResponse, Document
from mpfs.core.address import Address
from mpfs.common.static.tags import SLAVE, MASTER
from mpfs.metastorage.postgres.exceptions import UniqueConstraintViolationError

error_log = mpfs.engine.process.get_error_log()

#===============================================================================
# Data save options
#===============================================================================
WITH_VERSION = settings.mongo['options']['with_version']
RESOURCE_LOCK_DELAY = settings.mongo['options']['resource_lock_delay']

OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_ENABLE_CURSOR_INVALIDATION = \
    settings.operations['shared_folder_async_move']['enable_cursor_invalidation']
OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_CURSOR_INVALIDATION_RETRY_COUNT = \
    settings.operations['shared_folder_async_move']['cursor_invalidation_retry_count']


#===============================================================================
# Show version cache
#===============================================================================
last_user_version = 0


# Используется в find при итерировании по большому количеству записей.
LIMIT = 1000
#===============================================================================


def retry_expired_cursor_failure(func):
    @wraps(func)
    def wrapper(*args, **kwds):
        if not OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_ENABLE_CURSOR_INVALIDATION:
            for i in func(*args, **kwds):
                yield i
        else:
            retry_counter = 0
            has_elements = True
            while has_elements and retry_counter <= OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_CURSOR_INVALIDATION_RETRY_COUNT:
                try:
                    for i in func(*args, **kwds):
                        yield i
                        retry_counter = 0
                    has_elements = False
                except OperationFailure as e:
                    err_message = str(e.message)
                    if not (err_message.startswith('cursor id') and err_message.endswith('not valid at server')):
                        raise

                    error_log.info(
                        'Cursor failed, attempt %d from %d (%s)' % (
                            retry_counter,
                            OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_CURSOR_INVALIDATION_RETRY_COUNT,
                            str(e)
                        )
                    )
                    if retry_counter >= OPERATIONS_SHARED_FOLDER_ASYNC_MOVE_CURSOR_INVALIDATION_RETRY_COUNT:
                        raise
                    retry_counter += 1
    return wrapper


def get_last_user_version():
    return last_user_version


def set_last_user_version(v):
    global last_user_version
    last_user_version = v


def fsync_safe_w():
    return mpfs.engine.process.dbctl().fsync_safe_w()


class ErrorHandler(method_caller):
    exceptions_map = {
        AutoReconnect: errors.StorageConnectionError,
        ConnectionFailure: errors.StorageConnectionError,
        ConfigurationError: errors.StorageConfigurationError,
        OperationFailure: errors.StorageOperationError,
    }

    def __call__(self, *args, **kwargs):
        try:
            result = method_caller.__call__(self, *args, **kwargs)
        except PyMongoError as pme:
            error_log.error(traceback.format_exc())
            target = self.exceptions_map.get(type(pme))
            if not target:
                target = errors.StorageError
            raise target, pme, sys.exc_info()[2]
        except Exception:
            raise
        else:
            return result

# Alias for backward compatibility
error_handler = ErrorHandler


class BaseCollection(object):
    """
    Базовый класс коллекции
    Надстройка над монговой коллекцией

    name -  имя коллекции, под которым она хранится в базе
    is_sharded - признак того, что коллекция шардирована по юзерам и хранится на отдельных шардах
    is_common - признак того, что коллекция "общая", т.е. не шардирована по uid и хранится в отдельном общем месте
    uid_field - имя ключа в запросах, которое используется для шардированных таблиц для вычленения индентификатора юзера

    Алгоритм выбора, в какой базе лежит коллекция, следующий:
     - если коллекция is_common, то она живет в misc
     - если коллекция is_sharded, то она живет на шарде
     - если не is_common и не is_sharded, то она живет в монгосе

    (см. mpfs.metastorage.mongo.source.CommonMongoSourceController.__init__:70)
    """

    name = None
    is_sharded = False
    is_common = False
    uid_field = 'uid'

    def __init__(self, read_preference=None):
        self.db = CollectionRoutedDatabase()
        mongo_read_preference = None

        if read_preference == SLAVE:
            mongo_read_preference = pymongo.ReadPreference.SECONDARY_PREFERRED
        elif read_preference == MASTER:
            mongo_read_preference = pymongo.ReadPreference.PRIMARY_PREFERRED

        if mongo_read_preference:
            setattr(self.db[self.name], 'read_preference', mongo_read_preference)

    def slave_okay(self, *args):
        return mpfs.engine.process.dbctl().slave_okay()

    @property
    def collection(self):
        return self.db[self.name]

    def __getattribute__(self, name):
        attribute = object.__getattribute__(self, name)
        if name[0] == '_' or name in ('db', 'name', 'collection', 'old_db',
                                      'get_version', 'query_for_show'):
            return attribute
        elif hasattr(attribute, '__call__'):
            parent_args = sys._getframe(1).f_locals
            if 'self' not in parent_args or type(parent_args['self']) != type(self):
                return error_handler(attribute)
        return attribute

    def _fsync_safe_w(self):
        return mpfs.engine.process.dbctl().fsync_safe_w()

    def get_one(self, **spec):
        for k, v in spec.iteritems():
            if isinstance(v, (list, tuple)):
                spec[k] = {'$in': list(v)}
        return self.find_one(spec)

    def get_count(self, **spec):
        return self.count(**spec)

    def get_all(self, **spec):
        return list(self.collection.find(spec))

    def iter_all(self, spec, fields=None, sort=None, **kwargs):
        for each in self.find(spec, fields=fields, sort=sort, **kwargs):
            yield each

    def get_in(self, **spec):
        field, array = spec.get('field'), spec.get('array')

        if field and array:
            spec = {field: {'$in': array}}
            return list(self.collection.find(spec))
        else:
            return []

    def count(self, *args, **spec):
        return self.collection.find(spec).count()

    def remove(self, **spec):
        return self.collection.remove(spec, **self._fsync_safe_w())

    def increment(self, spec, field, diff, verbose=False):
        if field is None:
            field = 'data'
        doc = {
               '$inc' : {
                         field : diff,
                         },
               }
        result = self.collection.update(spec, doc, **self._fsync_safe_w())
        if verbose and not result['updatedExisting']:
            return False
        else:
            return True

    def sum(self, field, group_by=None, match=None):
        """
        Summaries specified field values.

        :param field: Field name to summarize.
        :param group_by: Field name to group by. If specified, then method return [{<group_by>: <value>, 'sum': <sum value>}]
        :param match: Filter clause as dict {<field_name>: <value>, ...}
        :return: int | list of grouped sums.
        """
        # TODO: make group_by argument accept multiple fields.
        pipeline = []
        if match:
            pipeline.append({'$match': match})
        pipeline.append({'$group': {'_id': '$%s' % group_by if group_by else None, 'sum': {'$sum': '$%s' % field}}})
        if group_by:
            pipeline.append({'$project': {'_id': 0, '%s' % group_by: '$_id', 'sum': 1}})
        s = self.collection.aggregate(pipeline)
        result = s['result']
        if group_by:
            ret = result
        elif len(result) > 0:
            ret = result[0]['sum']
        else:
            # if result is empty then assume that there is no matched records and sum = 0
            ret = 0
        return ret

    def find(self, spec, **kwargs):
        return self.collection.find(spec, **kwargs)

    def find_one(self, spec):
        """get_one портит запросы с $in"""
        return self.collection.find_one(spec)

    def insert(self, docs, continue_on_error=False, **kwargs):
        return self.collection.insert(docs, continue_on_error=continue_on_error, **kwargs)

    def update(self, spec, doc, **kwargs):
        kwargs.update(self._fsync_safe_w())
        return self.collection.update(spec, doc, **kwargs)


class DirectRouteCollection(BaseCollection):
    """
    Коллекция с прямой логикой выбора коннекта/бд/коллекции.

    Не использует сложную логику выбора коннекта/базы/коллекции на основе
    флагов шардирования(is_sharded) и общих коллекций(is_common) и параметров запроса

    Настройки коннекта указываеются в конфиге `mongo->connections`.
    Имя коннекта указываются в атрибуте CONNECTION_NAME.
    Имя базы и коллекции указываются атрибутом DB_COLL_NAME.

    Внимание! При написании тестов разработчик должен сам заботится о инициализации и чистке БД
    """
    CONNECTION_NAME = None
    """Название коннекта в конфигах"""
    DB_COLL_NAME = None
    """
    Название бд и коллекции.

    Возможные варианты:
        * строка - если название БД и коллекции совпадают
        * tuple из двух элементов - (<БД>, <Коллекция>)
    """

    def __init__(self):
        self._connection = None
        self._database = None
        self._collection = None

        _, _, coll_name = self.get_and_validate_coll_attrs()
        self.name = coll_name

    def init_conn_db_coll(self):
        conn_name, db_name, coll_name = self.get_and_validate_coll_attrs()
        self._connection = mpfs.engine.process.dbctl().connection(conn_name)
        db_name = mpfs.common.util.dbnaming.dbname(db_name)
        self._database = getattr(self._connection, db_name)
        self._collection = self._database[coll_name]
        self.name = coll_name

    @property
    def connection(self):
        if self._connection is None:
            self.init_conn_db_coll()
        return self._connection

    @property
    def db(self):
        if self._database is None:
            self.init_conn_db_coll()
        return self._database

    @property
    def collection(self):
        if self._collection is None:
            self.init_conn_db_coll()
        return self._collection

    @classmethod
    def get_and_validate_coll_attrs(cls):
        db_coll_name = cls.DB_COLL_NAME
        if isinstance(db_coll_name, basestring):
            db_name = db_coll_name
            coll_name = db_coll_name
        elif (isinstance(db_coll_name, tuple) and
                len(db_coll_name) == 2 and
                isinstance(db_coll_name[0], basestring) and
                isinstance(db_coll_name[1], basestring)):
            db_name = db_coll_name[0]
            coll_name = db_coll_name[1]
        else:
            raise TypeError('"DB_COLL_NAME" should be: str or tuple.')

        if not isinstance(cls.CONNECTION_NAME, basestring):
            raise TypeError('Set "CONNECTION_NAME" class attr.')

        return cls.CONNECTION_NAME, db_name, coll_name


class UserIndexCollection(BaseCollection, Singleton, Cached):

    name = 'user_index'
    _values = defaultdict(dict)
    _content_loaded = False
    uid_field = '_id'
    is_sharded = True
    is_common = False

    @classmethod
    def reset(cls):
        cls._values = defaultdict(dict)
        cls._content_loaded = False

    def check_user(self, uid):
        def check_usr():
            spec = {"_id": propper_uid(uid), 'shard_key': shard_key(uid)}
            return self.db.user_index.find_one(spec)

        if not settings.mongo['options']['cache_user_index']:
            return check_usr()

        user_record = None

        if UserIndexCollection._content_loaded:
            user_record = UserIndexCollection._values.get(uid)
        if not user_record:
            user_record = check_usr()
            UserIndexCollection._values[uid] = user_record
            UserIndexCollection._content_loaded = True

        return user_record

    def update(self, spec, doc):
        self.reset()
        return super(UserIndexCollection, self).update(spec, doc)

    def remove(self, spec):
        self.reset()
        self.db.user_index.remove(spec, **self._fsync_safe_w())


class DiskCollection(BaseCollection):

    def get_version(self, uid):
        return ''

    def get_cached_version(self, uid):
        v = get_last_user_version()
        result = v if v else self.get_version(uid)
        return result

    def assert_user(self, uid):
        return
        #return assert_user(uid)

    def assert_parent(self, uid, key):
        parent_key = '/' + '/'.join(filter(None, key.split('/'))[:-1])
        if parent_key != '/':
            self.assert_key(uid, parent_key)

    def assert_key(self, uid, key):
        if is_new_fs_spec_required(self.name):
            query = {'uid': uid, 'path': key}
        else:
            query = {'_id': hashed(uid + ':' + key)}

        result = self.db[self.name].find_one(query)
        if not result:
            raise errors.StorageKeyNotFound('%s was not found, uid: %s' % (key, uid), data={'key': key})
        else:
            return result

    def _propper_key(self, key):
        return propper_key(key)

    def _put(self, uid, key, query, data, new_version, rawdata={}, *args, **kwargs):
        return ErrorHandler(self.db[self.name].update)(query, data, upsert=True, **self._fsync_safe_w())

    def data_for_mkdir(self, uid, key, rawdata, old_version, new_version):
        return self.data_for_save(uid, key, rawdata, old_version, new_version, 'dir')

    def data_for_mkfile(self, uid, key, rawdata, old_version, new_version):
        return self.data_for_save(uid, key, rawdata, old_version, new_version, 'file')

    def data_for_save(self, uid, key, rawdata, old_version, new_version, rtype):
        if is_new_fs_spec_required(self.name):
            data = {
                'path': key,
                'data': copy.deepcopy(rawdata),
                'type': rtype,
                'key': key,
                'uid': uid,
                'version': new_version,
            }
            query = {'uid': uid, 'path': key}
        else:
            _id = hashed(uid + ':' + key)
            data = {
                '_id': _id,
                'data': copy.deepcopy(rawdata),
                'type': rtype,
                'key': key,
                'uid': uid,
                'version': new_version,
            }
            query = {'_id': _id}

        if old_version and WITH_VERSION:
            query['version'] = int(old_version)
        return data, query

    def data_for_move(self, data):
        return data

    def put(self, uid, key, rawdata, old_version=None, **kwargs):
        uid = propper_uid(uid)
        key = self._propper_key(key)
        self.assert_user(uid)
        if 'skip_parents_check' not in kwargs or kwargs['skip_parents_check'] is False:
            self.assert_parent(uid, key)
        new_version = generate_version_number()
        data, query = self.data_for_mkfile(uid, key, rawdata, old_version, new_version)
        self._put(uid, key, query, data, new_version, rawdata, **kwargs)
        return MResponse(value=True, version=str(new_version))

    def show(self, uid, key, version):
        uid = propper_uid(uid)
        key = self._propper_key(key)
        self.assert_user(uid)
        ver = self.get_cached_version(uid)

        _id = hashed(uid + ':' + key)
        if is_new_fs_spec_required(self.name):
            query = {'uid': uid, 'path': key}
        else:
            query = {'_id': _id}
        resource = self.db[self.name].find_one(query)

        if not resource:
            return MResponse(value=None, version=None)
        else:
            result = {
                '_id'     : _id,
                'uid'     : uid,
                'data'    : resource['data'],
                'version' : str(resource.get('version', generate_version_number())),
                'key'     : key,
                'type' : resource.get('type'),
            }
            return MResponse(value=Document(**result), version=str(ver))

    def make_folder(self, uid, key, rawdata, old_version=None):
        '''
        Поместить данные
        '''
        uid = propper_uid(uid)
        key = self._propper_key(key)
        self.assert_user(uid)
        new_version = generate_version_number()
        data, query = self.data_for_mkdir(uid, key, rawdata, old_version, new_version)
        try:
            self._make_folder(uid, key, data, new_version)
        except (pymongo.errors.OperationFailure, UniqueConstraintViolationError):
            raise errors.StorageFolderAlreadyExist("Folder %s already exists" % key)
        else:
            return MResponse(value=True, version=str(new_version))

    def _make_folder(self, uid, key, data, new_version):
        self.assert_parent(uid, key)
        self.db[self.name].insert(data, **self._fsync_safe_w())

    def change_folder(self, uid, key, rawdata, old_version=None, **kwargs):
        '''
        Поместить данные
        '''
        uid = propper_uid(uid)
        key = self._propper_key(key)
        self.assert_user(uid)

        if is_new_fs_spec_required(self.name):
            query = {'path': key, 'uid': uid}
        else:
            _id = hashed(uid + ':' + key)
            query = {'_id': _id, 'uid': uid}

        result = self.db[self.name].find_one(query)
        if not result:
            raise errors.ServicesNotFound(key)
        new_version = generate_version_number()
        data, query = self.data_for_mkdir(uid, key, rawdata, old_version, new_version)
        self._change_folder(uid, key, query, data, new_version, **kwargs)
        return self.show(uid, key, new_version)

    def _change_folder(self, uid, key, data, query, new_version, **kwargs):
        self.db[self.name].update(query, data, **self._fsync_safe_w())

    def ls(self, uid, key):
        '''
        Листинг директории (выборка ключей без значения)
        '''
        uid = propper_uid(uid)
        key = self._propper_key(key)
        self.assert_user(uid)
        def _get_name(item):
            return name_for_key(item['key'])
        result_query = self.db[self.name].find(
            self.query_for_find(uid=uid, normal_key=key),
            fields=('key', )
        )
        result = sorted(imap(_get_name, ifilter(lambda x: key == parent_for_key(x['key']), result_query)))
        return MResponse(value=result, version=str(self.get_cached_version(uid)))

    def folder_content_count(self, uid, key):
        if is_new_fs_spec_required(self.name):
            query = {'uid': uid, 'parent': key}
        else:
            query = {'uid': uid, 'parent': id_for_key(uid, key)}
        return self.db[self.name].find(query).count()

    def folder_content(self, uid, key, version=None, filter_args={}, range_args={}):
        '''
        Листинг директории (выборка ключей и значений)
        '''
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)

        result = {}
        version = self.get_cached_version(uid)

        if is_new_fs_spec_required(self.name):
            parent = key
        else:
            parent = id_for_key(uid, key)
        query = self.query_for_find(uid=uid, parent=parent)

        all_resources = list(self.db[self.name].find(query))

        for item in all_resources:
            item_key = item['key']
            item_name = filter(None, item_key.split('/'))[-1]
            item_data = item['data']
            if 'meta' in item_data and isinstance(item_data.get('meta'), list):
                item_data['meta'] = dict(item_data['meta'])
            result[item_name] = Document(
                data=item_data,
                version=item.get('version'),
                key=item_key,
                type=item.get('type'),
            )
        return MResponse(value=result, version=str(version))

#    @log_time(log)
    def timeline(self, uid, key, version=None, filter_args={}, range_args={}, folders=False):
        '''
        Плоский список файлов по пользователю
        '''
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        self.assert_key(uid, key)
        result = []
        version = self.get_cached_version(uid)

        query = {
                 'uid': uid,
                 }
        if not folders:
            query['type'] ='file'
        query.update(filter_args)

        amounts = dict(filter(lambda (k,v): k in ('skip', 'limit'), range_args.iteritems()))

        if len(filter(None, key.split('/'))) == 1:
            amount_args = amounts
        else:
            amount_args = {}
        try:
            sort_args = (range_args['sort'], range_args.get('order', 1))
        except KeyError:
            all_resources = self.db[self.name].find(query, **amount_args)
        else:
            all_resources = self.db[self.name].find(query, **amount_args).sort(*sort_args)

        for item in ifilter(lambda x: x['key'].startswith(key + '/'), all_resources):
            item_key = item['key']
            item_data = item['data']

            if 'meta' in item_data and isinstance(item_data.get('meta'), list):
                item_data['meta'] = dict(item_data['meta'])

            result.append(Document(
                data=item_data,
                version=item.get('version'),
                key=item_key,
                type=item.get('type'),
            ))
        if amounts and not amount_args:
            skip = amounts.get('skip')
            limit = amounts.get('limit')
            if limit is not None and skip is not None:
                limit = skip + limit
            result = result[skip:limit]
        return MResponse(value=result, version=str(version))

    #@log_time(log)
    def tree(self, uid, key, level, version=None):
        '''
        Дерево директории до определенного уровня, без укладки в дерево
        '''
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        version = self.get_cached_version(uid)
        result = {}

        query = self.query_for_find(uid=uid, type='dir')
        all_resources = tuple(self.db[self.name].find(
            query,
            fields=('parent', 'key', 'data.meta', 'data.mtime', 'data.ctime', 'data.utime'))
        )
        grandparent_level = len(key.split('/'))
        max_grandparent_level = grandparent_level + level

        for item in all_resources:
            item_id   = item['_id']
            item_key  = item['key']
            item_data = item['data']
            item_data['type'] = 'dir'

            chunks = item_key.split('/')
            this_level = len(chunks)

            if max_grandparent_level >= this_level > grandparent_level:

                if 'meta' in item_data and isinstance(item_data.get('meta'), list):
                    item_data['meta'] = dict(item_data['meta'])

                parent_key = '/'.join(chunks[:-1])
                result[item_key] = Document(
                    data=item_data,
                    version=item.get('version'),
                    key=item_key,
                    parent_key=parent_key,
                    type=item.get('type'),
                )

        return MResponse(value=result, version=str(version))

    def has_subfolders(self, uid, key, *args, **kwargs):
        if is_new_fs_spec_required(self.name):
            spec = {'uid': uid, 'parent': key, 'type': 'dir'}
        else:
            spec = {'uid': uid, 'parent': id_for_key(uid, key), 'type': 'dir'}
        return bool(self.db[self.name].find_one(spec))

    def folder_counters(self, uid, key, version=None):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        version = self.get_cached_version(uid)
        if key == '/':
            RE_DIRECT_CHILD = "^/[^/]*$"
        else:
            RE_DIRECT_CHILD = '^%s/[^/]*$' % re.escape(key)
        result = {
            'numfiles': self.db[self.name].find(
                self.query_for_find(
                    uid=uid, type='file',normal_key=key, key=re.compile(RE_DIRECT_CHILD)
                )
            ).count(),
            'numfolders': self.db[self.name].find(
                self.query_for_find(
                    uid=uid, type='dir', normal_key=key, key=re.compile(RE_DIRECT_CHILD)
                )
            ).count()
        }
        return MResponse(value=result, version=str(version))

    def remove(self, uid, key, old_version=None, new_version=None, data=None):
        '''
        Удалить данные
        '''
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        _id = hashed(uid + ':' + key)
        result = self.assert_key(uid, key)
        query = self.query_for_update(_id=_id, uid=uid, key=key)

        if is_new_fs_spec_required(self.name):
            query.pop('_id', None)
            query['path'] = key

        if old_version and WITH_VERSION:
            query['version'] = old_version
            if self.db[self.name].find_and_modify(query, remove=True, **self._fsync_safe_w()) is None:
                raise errors.StorageError()
            if result['type'] == 'dir':
                self.remove_children(uid, key)
        else:
            if result['type'] == 'dir':
                self.remove_children(uid, key)
            self.db[self.name].remove(query, **self._fsync_safe_w())

        new_version = self._remove(uid, key, old_version, new_version, result['type'], data=data)
        return MResponse(value=True, version=str(new_version))

    def remove_folder_content(self, uid, key):
        '''
            Удаляется только содержимое каталога, в changelog изменения не проставляются.
        '''
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        return self.remove_children(uid, key)

    def remove_children(self, uid, key):
        find_result = self.db[self.name].find(self.query_for_find(uid=uid), fields=('key', 'version', 'data', 'type'))
        size = 0
        for element in ifilter(lambda x: x['key'].startswith(key + '/'), find_result):
            if element['type'] == 'file':
                size += element['data']['size']
            self.db[self.name].remove(element, **self._fsync_safe_w())
        return size

    def _remove(self, uid, key, old_version, new_version, rtype, data=None):
        return new_version

    def remove_single(self, uid, key, old_version=None, new_version=None, data=None):
        result, rtype = self._remove_single(uid, key, old_version, new_version)
        self._remove(uid, key, old_version, new_version, rtype)
        return result

    def remove_single_without_changelog(self, uid, key, old_version=None, new_version=None):
        return self.remove_single(uid, key, old_version, new_version)

    def _remove_single(self, uid, key, old_version=None, new_version=None):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        res = self.assert_key(uid, key)

        if is_new_fs_spec_required(self.name):
            query = {
                'path': key,
                'uid': uid,
            }
        else:
            query = {
                '_id': hashed(uid + ':' + key),
                'uid': uid,
            }

        if old_version:
            query['version'] = int(old_version)
        result = self.db[self.name].remove(query, **self._fsync_safe_w())
        return result, res['type']

    def show_single(self, uid, key, version):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)

        if is_new_fs_spec_required(self.name):
            query = {
                'path': key,
                'uid': uid,
            }
        else:
            query = {
                '_id': hashed(uid + ':' + key),
                'uid': uid,
            }

        if version:
            query['version'] = int(version)
        result = self.db[self.name].find_one(query)
        if result:
            try:
                result['data']['meta'] = dict(result['data']['meta'])
            except (KeyError, TypeError,), e:
                pass
            return result
        else:
            raise errors.StorageNotFound(key)

    def create(self, uid):
        '''
        Создать домен (коллекцию)
        '''
        uid = propper_uid(uid)

        user_index_data = UserIndexCollection().check_user(uid)
        cname_in_index = self.name in user_index_data['collections']

        if is_new_fs_spec_required(self.name):
            query = {'uid': uid, 'path': '/'}
        else:
            query = self.query_for_find(_id=hashed(uid + ':/'), uid=uid)

        root_folder_in_collection = self.db[self.name].find_one(query)

        if cname_in_index and root_folder_in_collection:
            raise errors.StorageDomainAlreadyExists('Domain %s already exists' % self.name)
        else:
            if not root_folder_in_collection:
                root_data = self.root_data(uid)
                self.db[self.name].update(query, root_data, upsert=True, **self._fsync_safe_w())

            if not cname_in_index:
                UserIndexCollection().update(
                    {
                        '_id': uid,
                        'shard_key': shard_key(uid)
                    }, {
                        '$push': {'collections': self.name}
                    })

        return True

    def remove_domain(self, uid):
        uid = propper_uid(uid)

        user_index_data = UserIndexCollection().check_user(uid)
        cname_in_index = self.name in user_index_data['collections']

        if cname_in_index:
            self.db[self.name].remove({'uid': uid}, **self._fsync_safe_w())
        return True

    def root_data(self, uid):
        if is_new_fs_spec_required(self.name):
            root_data = {
                'uid': uid,
                'type': 'dir',
                'data': {},
                'key': '/',
                'path': '/',
            }
        else:
            _id = hashed(uid + ':/')
            root_data = {
                '_id': _id,
                'uid': uid,
                'type': 'dir',
                'data': {},
                'key': '/',
            }

        return root_data

    def check(self, uid):
        '''
        Проверить домен (коллекцию)
        '''
        uid = propper_uid(uid)
        user_index_data = UserIndexCollection().check_user(uid)

        if not user_index_data:
            raise errors.StorageInitUser()

        cname_in_index = self.name in user_index_data['collections']

        if is_new_fs_spec_required(self.name):
            query = {'uid': uid, 'path': '/'}
        else:
            query = {'_id': hashed(uid + ':/')}

        root_folder_in_collection = self.db[self.name].find_one(query)

        return cname_in_index and root_folder_in_collection

    def move(self, uid, src, dst, old_version=None):
        if isinstance(src, Address) and isinstance(dst, Address):
            src_uid = src.uid
            dst_uid = dst.uid
            src = src.path
            dst = dst.path
        else:
            src_uid = dst_uid = propper_uid(uid)
        src = self._propper_key(src)
        dst = self._propper_key(dst)
        new_version = generate_version_number()

        if is_new_fs_spec_required(self.name):
            query = {'uid': src_uid, 'path': src}
        else:
            query = {'_id': hashed(src_uid + ':' + src)}
        src_data = self.db[self.name].find_one(query)

        try:
            self.remove(dst_uid, dst, old_version, new_version=new_version)
        except errors.StorageKeyNotFound:
            pass

        if is_new_fs_spec_required(self.name):
            dst_data = {
                'path': dst,
                'uid': dst_uid,
                'key': dst,
                'version': new_version,
            }
        else:
            dst_data = {
                '_id': hashed(dst_uid + ':' + dst),
                'uid': dst_uid,
                'key': dst,
                'version': new_version,
            }

        for k in ('data', 'type', 'shard_key'):
            try:
                dst_data[k] = src_data[k]
            except KeyError:
                pass
        dst_data = self.data_for_move(dst_data)
        self.db[self.name].insert(dst_data, **self._fsync_safe_w())

        self.insert_version(dst_uid, dst, 'new', new_version, data=dst_data)
        for each in ifilter(lambda x: x['key'].startswith(src + '/'), self.db[self.name].find({'uid' : src_uid})):
            old_key_split = filter(None, each['key'].split('/'))
            new_key_split = filter(None, dst.split('/')) + old_key_split[len(filter(None, src.split('/'))):]
            new_key = '/' + '/'.join(new_key_split)

            if is_new_fs_spec_required(self.name):
                new_data = {
                    'path': new_key,
                    'uid': dst_uid,
                    'key': new_key,
                    'version': new_version,
                }
            else:
                new_data = {
                    '_id': hashed(dst_uid + ':' + new_key),
                    'uid': dst_uid,
                    'key': new_key,
                    'version': new_version,
                }

            for k in ('data', 'type', 'shard_key'):
                try:
                    new_data[k] = each[k]
                except KeyError:
                    pass
            new_data = self.data_for_move(new_data)
            self.db[self.name].insert(new_data, **self._fsync_safe_w())
            self.insert_version(dst_uid, new_key, 'new', new_version, data=each)
        self.remove(src_uid, src, src_data['version'], new_version=new_version)
        return MResponse(value=True, version=str(new_version))

    def insert_version(self, uid, key, op, version, *args, **kwargs):
        pass

    def update_where(self, uid, key, new_value, old_value, old_version):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        _id = hashed(uid + ':' + key)
        new_version = generate_version_number()
        data, query = self.data_for_save(uid, key, {}, old_version, new_version, '')
        query["data"] = copy.deepcopy(old_value)
        update = {"$set" : {"data" : copy.deepcopy(new_value), "version" : new_version}}
        if isinstance(new_value, dict) and 'meta' in new_value:
            update['$set']['data']['meta'] = sorted(new_value['meta'].iteritems(), key=operator.itemgetter(0))
        if isinstance(old_value, dict) and 'meta' in old_value:
            query['data']['meta'] = sorted(old_value['meta'].iteritems(), key=operator.itemgetter(0))
        return self._update_where(uid, key, _id, query, update, old_version, new_version, new_value)

    def _update_where(self, uid, key, _id, query, update, old_version, new_version, rawdata={}):
        if self.db[self.name].find_and_modify(query=query, update=update, **self._fsync_safe_w()):
            return MResponse(value=True, version='')
        else:
            raise errors.StorageUpdateWhereFailed()

    def find_by_field(self, uid, args, sargs):
        #=======================================================================
        # Найти запись по полю
        #=======================================================================
        spec = {
                'uid' : propper_uid(uid),
                }
        spec.update(args)
        result = self.db[self.name].find(spec)
        if sargs:
            return result.sort(*sargs)
        else:
            return result

    def find_one_by_field(self, uid, args):
        #=======================================================================
        # Найти одну запись по полю
        #=======================================================================
        spec = {
                'uid' : propper_uid(uid),
                }
        spec.update(args)
        return self.db[self.name].find_one(spec)

    def raw_find(self, uid, spec, sort=None):
        find_spec = {'uid': propper_uid(uid)}
        if spec:
            find_spec.update(spec)
        return self.db[self.name].find(find_spec, sort=sort)

    def find(self, uid, parent, common_args, data_args, maxresults, sort_field):
        uid = propper_uid(uid)
        self.assert_user(uid)
        parent = self._propper_key(parent)
        if parent == '/':
            RE_CHILDREN = "^/[^/]+(/[^/]*)?$"
        else:
            RE_CHILDREN = "^%s/[^/]*(/[^/]*)?$" % re.escape(parent)
        query = self.query_for_find(uid=uid, normal_key=parent, key=re.compile(RE_CHILDREN))
        for k in common_args:
            if isinstance(common_args[k], (list, tuple)):
                common_args[k] = {'$in' : list(common_args[k])}
        for k in data_args:
            if isinstance(data_args[k], (list, tuple)):
                data_args[k] = {'$in' : list(data_args[k])}
        query.update(('data.%s' % k, v) for k, v in common_args.iteritems())
        query.update(('data.data.%s' % k, v) for k, v in data_args.iteritems())
        result = map(operator.itemgetter('data'), self.db[self.name].find(query))
        result.sort(key=operator.itemgetter(sort_field))
        if maxresults:
            return result
        else:
            return result[-1:]

    def find_one(self, uid, parent, common_args, data_args, maxresults, sort_field):
        result = self.find(uid, parent, common_args, data_args, maxresults, sort_field)[0]
        try:
            return result[0]
        except IndexError:
            raise errors.StorageKeyNotFound(data={'key': parent})

    def query_for_find(self, **kw):
        kw.pop('normal_key', '')
        return kw

    def query_for_update(self, **kw):
        kw.pop('normal_key', '')
        return kw

    def flat_index(self, uid, key):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        self.assert_key(uid, key)
        version = self.get_cached_version(uid)
        query = self.query_for_find(uid=uid, normal_key=key)
        result = {}
        for each in ifilter(lambda v: v['key'].startswith(key + '/'), self.db[self.name].find(query)):
            data = each['data']
            data['meta'] = dict(data.get('meta', {}))
            result[each['key']] = data
        return MResponse(value=result, version=str(version))

    def tree_index(self, uid, key):
        uid = propper_uid(uid)
        key = self._propper_key(key)

        def make_tree(parent, children):
            result = []
            children = filter(lambda x: x['key'].startswith(parent['key']), children)
            for element in ifilter(lambda v: parent['key'] == parent_for_key(v['key']), children):
                if element['type'] == 'dir':
                    children_list = make_tree(element, children)
                else:
                    children_list = []
                try:
                    element['data']['meta'] = dict(element['data']['meta'])
                except Exception:
                    pass
                result.append({'this' : element, 'list' : children_list})
            return result

        query = self.query_for_find(uid=uid, normal_key=key)
        result = make_tree({'key' : key}, self.db[self.name].find(query))
        return result

    def increment(self, uid, key, diff, field, **kw):
        uid = propper_uid(uid)
        key = self._propper_key(key)
        query = self.query_for_show(uid, key)
        spec = kw.get('spec', {})
        verbose = kw.get('verbose')
        query.update(spec)
        return super(DiskCollection, self).increment(query, field, diff, verbose=verbose)

    def find_by_hid(self, hid):
        '''
        Поиск по полю hid
        Работает только для юзерских коллекций
        '''
        raise errors.MPFSNotImplemented()

    def find_by_hid_all(self, hid):
        '''
        Поиск по полю hid
        Работает только для юзерских коллекций
        '''
        raise errors.MPFSNotImplemented()

    def find_by_stid(self, stid):
        '''
        Поиск по полю hid
        Работает только для юзерских коллекций
        '''
        raise errors.MPFSNotImplemented()


class UserCollection(DiskCollection):

    is_sharded = True
    is_common = False

    def _unzip_resource_data(self, rtype, data, zdata):
        return data

    def assert_key(self, uid, key):
        if key != '/':
            if is_new_fs_spec_required(self.name):
                query = {
                    'path': key,
                    'uid': uid,
                }
            else:
                query = {
                    '_id': hashed(uid + ':' + key),
                    'uid': uid,
                }

            result = self.db[self.name].find_one(query)
        else:
            result = True
        if not result:
            raise errors.StorageKeyNotFound('%s was not found, uid: %s' % (key, uid), data={'key': key})
        else:
            return result

    def resource_lock(self, uid, key, version=None):
        #print 'lock %s %s' % (uid, key)
        #self.is_resource_locked(uid, key, version)

        uid = propper_uid(uid)
        key = self._propper_key(key)
        cur_time = int(time.time())

        if is_new_fs_spec_required(self.name):
            update_spec = self.query_for_update(path=key, uid=uid)
        else:
            update_spec = self.query_for_update(_id=id_for_key(uid, key), uid=uid)

        if not self.db[self.name].find_one(update_spec):
            raise errors.ResourceNotFound('%s:%s' % (uid, key))

        lock_expiration_time = cur_time - RESOURCE_LOCK_DELAY
        unlocked_condition = {'$or' : [
                                       {'lock' : {'$exists' : 0}},
                                       {'lock' : {'$lte' : lock_expiration_time}},
                                       ]}

        if version:
            update_spec['version'] = int(version)

        update_spec.update(unlocked_condition)
        update_data = {'$set' : {'lock' : cur_time}}

        result = self.db[self.name].find_and_modify(update_spec, update=update_data, new=True, **self._fsync_safe_w())

        if not result or not result.get('lock') or result.get('lock') != cur_time:
            raise errors.ResourceLockFailed(key)

    def resource_unlock(self, uid, key, version=None):
        if is_new_fs_spec_required(self.name):
            update_spec = {'uid': uid, 'path': key}
        else:
            update_spec = {'uid': uid, '_id': id_for_key(uid, key)}
        update_data = {'$unset': {'lock': 1}}

        self.db[self.name].find_and_modify(update_spec, update=update_data, new=True, **self._fsync_safe_w())

    def is_resource_locked(self, uid, key, version=None):
        uid = propper_uid(uid)
        key = self._propper_key(key)

        lock_expiration_time = int(time.time()) - RESOURCE_LOCK_DELAY
        locked_condition = {'$and' : [
            {'lock' : {'$exists' : 1}},
            {'lock' : {'$gt' : lock_expiration_time}},
        ]}

        key_split = filter(None, key.split('/'))

        if is_new_fs_spec_required(self.name):
            paths = [key]

            for i in xrange(1, len(key_split)):
                pkey = '/' + '/'.join(key_split[:i])
                paths.append(pkey)

            spec = {'uid': uid, 'path': {'$in': paths}}
            spec.update(locked_condition)
        else:
            _ids = [id_for_key(uid, key)]

            for i in xrange(1, len(key_split)):
                pkey = '/' + '/'.join(key_split[:i])
                _ids.append(id_for_key(uid, pkey))

            spec = {'uid': uid, '_id': {'$in': _ids}}
            spec.update(locked_condition)

        if self.db[self.name].find(spec).count():
            raise errors.ResourceLocked()

    def show_locks(self, uid, version=None):
        uid = propper_uid(uid)

        lock_expiration_time = int(time.time()) - RESOURCE_LOCK_DELAY
        locked_condition = {'$and' : [
            {'lock' : {'$exists' : 1}},
            {'lock' : {'$gt' : lock_expiration_time}},
        ]}

        spec = { 'uid' : uid }
        spec.update(locked_condition)

        result = []
        for item in self.db[self.name].find(spec):
            result.append(item['key'])

        return result


    def resource_update(self, uid, key, to_set={}, to_unset=(), version=None):
        uid = propper_uid(uid)
        key = self._propper_key(key)
        if is_new_fs_spec_required(self.name):
            query_find = self.query_for_update(path=key, uid=uid)
        else:
            query_find = self.query_for_update(_id=id_for_key(uid, key), uid=uid)
        if version:
            query_find['version'] = int(version)
        query_update = defaultdict(dict)
        for k,v in to_set.iteritems():
            query_update['$set'][k] = v
        for k in to_unset:
            query_update['$unset'][k] = 1

        result = self.db[self.name].update(query_find, query_update, **self._fsync_safe_w())
        if not result['updatedExisting']:
            raise errors.StorageUpdateWhereFailed()
        return self.show(uid, key, None)

    def show(self, uid, key, version=None):
        uid = propper_uid(uid)
        key = self._propper_key(key)
        ver = self.get_cached_version(uid)
        query = self.query_for_show(uid, key)
        resource = self.db[self.name].find_one(query)
        if not resource:
            return MResponse(value=None, version=None)
        else:
            if isinstance(resource.get('data', {}), dict) and isinstance(resource.get('data', {}).get('meta'), list):
                resource['data']['meta'] = dict(resource['data']['meta'])

            if 'hid' in resource:
                resource['data']['hid'] = resource['hid']

            result = {
                'data'    : resource['data'],
                'version' : str(resource.get('version', generate_version_number())),
                'key'     : key,
                'type'    : resource.get('type'),
            }
            return MResponse(value=Document(**result), version=str(ver))

    def data_for_move(self, data):
        data = super(UserCollection, self).data_for_move(data)

        if is_new_fs_spec_required(self.name):
            data['parent'] = parent_for_key(data['key'])
        else:
            data['parent'] = parent_id_for_key(data['uid'], data['key'])

        return data

    def data_for_save(self, uid, key, rawdata, old_version, new_version, rtype):
        data, query = super(UserCollection, self).data_for_save(
            uid, key, rawdata, old_version, new_version, rtype)
        if isinstance(rawdata, dict) and 'meta' in rawdata:
            data['data']['meta'] = sorted(
                rawdata['meta'].iteritems(),
                key=operator.itemgetter(0)
            )
            hid = rawdata.get('hid')

            if hid is not None:
                data['hid'] = Binary(str(hid))

        data['uid'] = query['uid'] = uid

        if is_new_fs_spec_required(self.name):
            data['parent'] = parent_for_key(key)
        else:
            data['parent'] = parent_id_for_key(uid, key)

        return data, query

    def query_for_find(self, **kw):
        result = super(UserCollection, self).query_for_find(**kw)
        result['uid'] = kw['uid']
        return result

    def query_for_show(self, uid, key):
        if is_new_fs_spec_required(self.name):
            return {'path': key, 'uid': uid}
        else:
            return {'_id': hashed(uid + ':' + key), 'uid': uid}

    def query_for_update(self, **kw):
        result = super(UserCollection, self).query_for_update(**kw)
        result['uid'] = kw['uid']
        return result

    def root_data(self, uid):
        data = super(UserCollection, self).root_data(uid)
        return data

    def check(self, uid):
        '''
        Проверить домен (коллекцию)
        '''
        user_index_data = UserIndexCollection().check_user(uid)
        if not user_index_data:
            raise errors.StorageInitUser()
        cname_in_index = self.name in user_index_data['collections']

        if is_new_fs_spec_required(self.name):
            spec = {'uid': uid, 'path': '/'}
        else:
            spec = {'_id': hashed(uid + ':/'), 'uid': uid}

        root_folder_in_collection = self.db[self.name].find_one(spec)
        return cname_in_index and root_folder_in_collection

    def ls(self, uid, key):
        '''
        Листинг директории (выборка ключей без значения)
        '''
        uid = propper_uid(uid)
        key = self._propper_key(key)

        # self.assert_user(uid)

        def _get_name(item):
            try:
                return filter(None, item.get('key', '').split('/'))[-1]
            except Exception:
                return '/'

        if is_new_fs_spec_required(self.name):
            query = {
                'uid': uid,
                'parent': key
            }
        else:
            query = {
                'uid': uid,
                'parent': id_for_key(uid, key)
            }

        result = sorted(
            map(
                _get_name,
                self.db[self.name].find(query)
            )
        )
        return MResponse(value=result, version=str(self.get_cached_version(uid)))

    def _preprocess_key_for_uid(self, uid, key):
        if is_new_fs_spec_required(self.name):
            return key
        else:
            return id_for_key(uid, key)

    def _remove_file_in_folder(self, uid, element):
        size = element.get('data', {}).get('size', 0)
        if is_new_fs_spec_required(self.name):
            query = {'path': element['key'], 'uid': uid}
        else:
            query = {'_id': element['_id'], 'uid': uid}
        self.db[self.name].remove(query, **self._fsync_safe_w())
        return size

    def _remove_empty_folder(self, uid, key):
        if is_new_fs_spec_required(self.name):
            query = {'path': key, 'uid': uid}
        else:
            query = {'_id': key, 'uid': uid}
        self.db[self.name].remove(query, **self._fsync_safe_w())

    def remove_children(self, uid, key):
        size = 0
        root_id = self._preprocess_key_for_uid(uid, key)
        traversed_folders = [(root_id, False)]

        while traversed_folders:
            folder, is_all_subfolder_loaded = traversed_folders.pop()

            if is_all_subfolder_loaded:
                self._remove_empty_folder(uid, folder)
                continue

            subitems = self.db[self.name].find({'uid': uid, 'parent': folder}, limit=LIMIT)
            subfolders = []
            items_count = 0
            for item in subitems:
                if item['type'] == 'file':
                    if item['data'].get('is_live_photo'):
                        resource_address = Address.Make(item['uid'], item['key'])
                        LivePhotoFilesManager.remove_live_video(resource_address)
                    size += self._remove_file_in_folder(uid, item)
                else:
                    subfolders.append(item)
                items_count += 1

            traversed_folders.append((folder, items_count < LIMIT))
            for f in subfolders:
                key = f['key'] if is_new_fs_spec_required(self.name) else f['_id']
                traversed_folders.append((key, False))

        return size

    def folder_counters(self, uid, key, version=None):
        uid = propper_uid(uid)
        self.assert_user(uid)
        key = self._propper_key(key)
        version = self.get_cached_version(uid)

        if is_new_fs_spec_required(self.name):
            parent = key
        else:
            parent = id_for_key(uid, key)

        result = {
            'numfiles': self.db[self.name].find({'uid': uid, 'type': 'file', 'parent': parent}).count(),
            'numfolders': self.db[self.name].find({'uid': uid, 'type': 'dir', 'parent': parent}).count()
        }
        return MResponse(value=result, version=str(version))

    def find_by_hid_and_uid(self, uid, hid):
        query = {'hid': Binary(str(hid)), 'uid': uid}
        return self.db[self.name].find_one(query)

    def find_by_spec(self, spec, shard=None):
        """
        Костыль, чтобы можно было искать в коллекции по spec-е

        Используется при поиске файла для хардлинка.
        """
        spec = spec.copy()
        key, value = spec.popitem()

        if shard:
            spec['shard'] = shard

        return self.find_hardlink_by_field(key, value, **spec)

    def find_hardlink_by_field(self, field, value, shard=None, **kw):
        query = {field: value}
        query.update(kw)
        # FIXME: this request duplicated in self.show()
        if is_collection_uses_dao(self.name) and shard:
            res = self.db[self.name].find_one_on_shard(query, shard_name=shard)
        else:
            res = self.db[self.name].find_one(query)
        if not res:
            return MResponse(value=None, version=None)
        else:
            if hasattr(self, 'unpack_single_element'):
                unpacked = self.unpack_single_element(res)
                return MResponse(value=Document(**unpacked), version=str(res.get('version')))
            else:
                return self.show(res.get('uid'), res.get('key'), res.get('version'))

    def find_by_hid(self, hid, **kw):
        return self.find_hardlink_by_field('hid', Binary(str(hid), **kw))

    def find_by_hid_all(self, hid):
        for element in self.db[self.name].find({'hid': Binary(str(hid))}):
            yield element

    def find_by_stid(self, stid, shard=None):
        return self.find_hardlink_by_field('data.stids.stid', stid, shard=shard)

    def move(self, uid, src, dst, old_version=None):
        if isinstance(src, Address) and isinstance(dst, Address):
            src_uid = propper_uid(src.uid)
            dst_uid = propper_uid(dst.uid)
            src = src.path
            dst = dst.path
        else:
            src_uid = dst_uid = propper_uid(uid)
        src = self._propper_key(src)
        dst = self._propper_key(dst)
        self.assert_parent(dst_uid, dst)
        new_version = generate_version_number()

        if is_new_fs_spec_required(self.name):
            query = {'uid': src_uid, 'path': src}
        else:
            query = {'uid': src_uid, '_id': hashed(src_uid + ':' + src)}
        src_data = self.db[self.name].find_one(query)

        group_version = True
        if mpfs.engine.process.use_shared_folders():
            try:
                from mpfs.core.social.share import Group
                group = Group.find(uid=uid, path=src)
                group_version = group.path != src
            except GroupNotFound:
                pass

        try:
            self.remove(dst_uid, dst, old_version, new_version=new_version)
        except errors.StorageKeyNotFound:
            pass

        def get_new_data(new_key, old_data, new_ver=new_version):
            if is_new_fs_spec_required(self.name):
                new_data = {
                    'path': new_key,
                    'uid': dst_uid,
                    'key': new_key,
                    'version': new_ver,
                }
            else:
                new_data = {
                    '_id': hashed(dst_uid + ':' + new_key),
                    'uid': dst_uid,
                    'key': new_key,
                    'version': new_ver,
                }
            for k in ('data', 'type', 'zdata', 'hid'):
                try:
                    new_data[k] = old_data[k]
                except KeyError:
                    pass
            return new_data

        dst_data = self.data_for_move(get_new_data(dst, src_data))

        self.db[self.name].insert(dst_data, **self._fsync_safe_w())
        self.insert_version(dst_uid, dst, 'new', new_version, data=dst_data)

        def move_one_element(element):
            new_version = generate_version_number()
            old_key_split = filter(None, element['key'].split('/'))
            new_key_split = filter(None, dst.split('/')) + old_key_split[len(filter(None, src.split('/'))):]
            new_key = '/' + '/'.join(new_key_split)
            new_element_data = get_new_data(new_key, element, new_ver=new_version)

            self.db[self.name].insert(self.data_for_move(new_element_data), **self._fsync_safe_w())
            self.insert_version(dst_uid, new_key, 'new', new_version, data=new_element_data)
            if element['type'] == 'dir':
                if is_new_fs_spec_required(self.name):
                    query = self.query_for_find(parent=element['key'], uid=src_uid)
                else:
                    query = self.query_for_find(parent=element['_id'], uid=src_uid)

                for each in self.db[self.name].find(query):
                    move_one_element(each)

            if is_new_fs_spec_required(self.name):
                query = self.query_for_update(path=element['key'], uid=src_uid)
            else:
                query = self.query_for_update(_id=element['_id'], uid=src_uid)
            self.db[self.name].remove(query, **self._fsync_safe_w())

        # process folder content
        if dst_data['type'] == 'dir':
            if is_new_fs_spec_required(self.name):
                query = self.query_for_find(uid=src_uid, parent=src_data['key'])
            else:
                query = self.query_for_find(uid=src_uid, parent=src_data['_id'])

            for each in retry_expired_cursor_failure(lambda: self.db[self.name].find(query))():
                move_one_element(each)

        if is_new_fs_spec_required(self.name):
            query = self.query_for_update(path=src_data['key'], uid=src_uid, version=src_data['version'])
        else:
            query = self.query_for_update(_id=src_data['_id'], uid=src_uid, version=src_data['version'])
        self.db[self.name].remove(query)
        new_version = generate_version_number()
        self.insert_version(src_uid, src, 'deleted', new_version, data=src_data, group_version=group_version)
        return MResponse(value=True, version=str(new_version))

    def all_values_for_user(self, uid):
        return self.db[self.name].find({'uid' : propper_uid(uid)})

    def iter_subtree(self, uid, key, limit=None):
        """Получить все документы из поддерева дисковых ресурсов,
        начинающегося с parent.

        Обход делается в ширину. На каждую полученную папку делается запрос в базу.

        :type uid: str
        :type key: str
        :rtype: :class:`collections.Iterator`
        """
        if is_new_fs_spec_required(self.name):
            key_converter = lambda u, k: k
        else:
            key_converter = id_for_key

        queue = deque()
        queue.append((uid, key))
        count = 0
        while queue:
            uid, key = queue.popleft()
            cursor = self.collection.find(
                {'uid': uid, 'parent': key_converter(uid, key)},
                read_preference=pymongo.ReadPreference.SECONDARY_PREFERRED,
                limit=0 if limit is None else limit - count
            )
            for doc in cursor:
                yield doc
                count += 1
                if doc['type'] == 'dir':
                    queue.append((uid, doc['key']))

    def get_immediate_children(self, uid, parent_path):
        if is_new_fs_spec_required(self.name):
            key_converter = lambda u, k: k
        else:
            key_converter = id_for_key

        cursor = self.collection.find(
            {'uid': uid, 'parent': key_converter(uid, parent_path)},
            read_preference=pymongo.ReadPreference.SECONDARY_PREFERRED
        )
        return list(cursor)


class UserCollectionKeyValue(UserCollection):
    pass


class SystemCollection(DiskCollection):

    def data_for_save(self, uid, key, rawdata, old_version, new_version, rtype):
        data, query = super(SystemCollection, self).data_for_save(uid, key, rawdata, old_version, new_version, rtype)
        data.pop('uid', '')
        return data, query

    def query_for_find(self, **kw):
        result = super(SystemCollection, self).query_for_find(**kw)
        result.pop('uid', '')
        return result

    def query_for_update(self, **kw):
        result = super(SystemCollection, self).query_for_update(**kw)
        result.pop('uid', '')
        return result

    def root_data(self, uid):
        root_data = super(SystemCollection, self).root_data(uid)
        root_data.pop('uid')
        return root_data

    def remove_domain(self, uid):
        self.db[self.name].remove(**self._fsync_safe_w())


class ShardedByMainFolder(SystemCollection):

    def data_for_save(self, uid, key, rawdata, old_version, version, rtype):
        data, query = super(ShardedByMainFolder, self).data_for_save(uid, key, rawdata, old_version, version, rtype)
        data['shard_key'] = query['shard_key'] = shard_key((filter(None, key.split('/')) or ['/'])[0])
        return data, query

    def root_data(self, uid):
        data = super(ShardedByMainFolder, self).root_data(uid)
        data['shard_key'] = shard_key('/')
        return data

    def query_for_find(self, **kw):
        normal_key = kw.pop('normal_key', '')
        result = super(ShardedByMainFolder, self).query_for_find(**kw)
        return result

    def query_for_update(self, **kw):
        normal_key = kw.pop('normal_key', '')
        result = super(ShardedByMainFolder, self).query_for_update(**kw)
        try:
            if not normal_key:
                normal_key = kw['key']
            splitted = filter(None, normal_key.split('/')) or ['/']
            result['shard_key'] = shard_key(splitted[0])
        except KeyError:
            pass
        return result


class ShardedById(SystemCollection):
    pass


class KeyValue(BaseCollection):

    key_fields = ()
    doc_keys = ('_id', 'v')

    def insert(self, doc, **kwargs):
        doc['v'] = generate_version_number()
        result = self.db[self.name].insert(doc, **self._fsync_safe_w())
        if not result:
            raise pymongo.errors.DuplicateKeyError(doc)
        return result

    def put(self, doc, *args, **kwargs):
        spec = {}
        for k in self.doc_keys:
            try:
                spec[k] = doc[k]
            except KeyError:
                pass
        if 'v' in self.doc_keys:
            doc['v'] = generate_version_number()
        for k in self.key_fields:
            try:
                spec[k] = doc[k]
            except KeyError:
                pass
        if spec:
            doc = {'$set' : doc}
            return self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())
        else:
            self.db[self.name].insert(doc, **self._fsync_safe_w())

    def remove(self, **spec):
        return self.db[self.name].remove(spec, **self._fsync_safe_w())

    def increment(self, uid, key, diff, field, **kw):
        query = self.query_for_show(uid, key)
        return super(KeyValue, self).increment(query, field, diff, **kw)

    def remove_domain(self, fake_parameter=None):
        self.db[self.name].remove()


user_index = UserIndexCollection()
