# -*- coding: utf-8 -*-
"""

MPFS
CORE

Интерфейс к Биллинговым коллекциям

"""
import time
import pymongo
import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.errors import billing as billing_errors
from mpfs.metastorage.mongo.collections.base import KeyValue
from mpfs.metastorage.mongo.util import generate_version_number


class BillingKeyValue(KeyValue):
    is_sharded = False
    is_common = True

    # переопределение методов: `find`, `find_one`, `get_all` - попытка заставить
    # pymongo ходить запросами только на мастер.
    # Подробнее: https://st.yandex-team.ru/CHEMODAN-29942
    @staticmethod
    def _billing_kwargs(kwargs):
        if not kwargs:
            kwargs = {}

        global_read_preference = mpfs.engine.process.get_read_preference()
        if global_read_preference:
            kwargs['read_preference'] = global_read_preference
        else:
            kwargs['read_preference'] = pymongo.ReadPreference.PRIMARY
        return kwargs

    def find(self, spec, **kwargs):
        kwargs = self._billing_kwargs(kwargs)
        return self.collection.find(spec, **kwargs)

    def find_one(self, spec, **kwargs):
        kwargs = self._billing_kwargs(kwargs)
        return self.collection.find_one(spec, **kwargs)

    def get_all(self, **spec):
        kwargs = self._billing_kwargs({})
        return list(self.collection.find(spec, **kwargs))

    def put(self, doc, *args, **kwargs):
        """
        Копипаста из KeyValue с дополнением
        Не стал делать везде, потому что могут быть проблемы

        :param doc: данные для сохранения
        :param args: args
        :param kwargs: kwargs, в которых может быть set() remove с полями ддя удаления
        :return:
        """
        spec = {}
        unset = kwargs.pop('remove', {})

        for k in self.doc_keys:
            try:
                spec[k] = doc.pop(k)
            except KeyError:
                pass

        if 'v' in self.doc_keys:
            doc['v'] = generate_version_number()

        for k in self.key_fields:
            try:
                spec[k] = doc[k]
            except KeyError:
                pass

        if spec:
            doc = {'$set': doc}

            return self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())
        else:
            self.db[self.name].insert(doc, **self._fsync_safe_w())

    def insert(self, doc, **kwargs):
        try:
            return super(BillingKeyValue, self).insert(doc)
        except pymongo.errors.DuplicateKeyError, e:
            raise billing_errors.BillingEntityAlreadyExists(doc)


class BillingClientAttributes(BillingKeyValue):
    name = 'billing_client_attributes'
    key_fields = ('attr', 'value', 'uid')


class BillingServiceAttributes(BillingKeyValue):
    name = 'billing_service_attributes'
    key_fields = ('attr', 'value', 'sid')


class BillingServiceAttributesHistory(BillingKeyValue):
    name = 'billing_service_attributes_history'
    key_fields = ('attr', 'value', 'sid')


class BillingServices(BillingKeyValue):
    name = 'billing_services'
    key_fields = ('uid', '_id')


class BillingServicesHistory(BillingKeyValue):
    name = 'billing_services_history'
    key_fields = ('uid', '_id')


class BillingOrders(BillingKeyValue):
    name = 'billing_orders'
    key_fields = ('uid', '_id')


class BillingOrdersHistory(BillingKeyValue):
    name = 'billing_orders_history'
    key_fields = ('uid', '_id')


class BillingSubscriptions(BillingKeyValue):
    name = 'billing_subscriptions'
    key_fields = ('uid', '_id')


class BillingLocks(BillingKeyValue):
    name = 'billing_locks'
    key_fields = ('_id')

    def set_lock(self, key, hostname, lock_delay):
        update_spec = {'_id': key}
        if not self.db[self.name].find_one(update_spec):
            self.db[self.name].insert({'_id': key})

        cur_time = int(time.time())
        lock_expiration_time = cur_time - lock_delay
        unlocked_condition = {'$or': [
            {'lock': {'$exists': 0}},
            {'lock': {'$lte': lock_expiration_time}},
        ]}

        update_spec.update(unlocked_condition)
        update_data = {'$set': {'lock': cur_time, 'host': hostname}}

        result = self.db[self.name].update(update_spec, update_data, **self._fsync_safe_w())
        if not result or not result['updatedExisting']:
            raise errors.ResourceLocked(key)

    def release_lock(self, key, hostname):
        update_spec = {'_id': key, 'host': hostname}
        update_data = {'$unset': {'lock': 1, 'host': 1}}

        result = self.db[self.name].update(update_spec, update_data, **self._fsync_safe_w())
        if not result or not result['updatedExisting']:
            raise errors.ResourceLocked(key)
