# -*- coding: utf-8 -*-
from datetime import datetime, timedelta

import mpfs.engine.process

from mpfs.metastorage.mongo.util import suppress_exceptions
from mpfs.metastorage.mongo import collection


class BlockingsCollection(collection.MPFSMongoLazyCollection):
    is_sharded = False
    is_common = False

    def __init__(self, db=None, *args, **kwargs):
        db = mpfs.engine.process.dbctl().database('blockings')
        name = '' # we dynamically generate it in get_name()
        super(BlockingsCollection, self).__init__(db, name, *args, **kwargs)

    @property
    def name(self):
        return self.get_name()

    def get_actual(self):
        actual = super(BlockingsCollection, self).get_actual()
        actual.turn_off_autoretry = True
        return actual

    def get_name(self):
        """get_name() used to determine actual collection in __getattr__() so we will always have proper collection."""
        date_format = '%Y%m%d'
        now = datetime.now().strftime(date_format)
        tomorrow = (datetime.now() + timedelta(days=1)).strftime(date_format)
        return 'excess_%s_0000_%s_0000' % (now, tomorrow)

    @suppress_exceptions()
    def get(self, hid):
        """
        Get blocking for resource by its `hid`.
        
        :param hid:
        :return: dict
        """
        return self.format(self.find_one({'_id': hid}))

    @suppress_exceptions(default=[])
    def find(self, hids):
        """
        Get blockings for list of resources by their `hids`.
        
        :param hids: list of hids
        :return: list
        """
        cursor = self.get_actual().find({'_id': {'$in': hids}})
        return [self.format(b) for b in cursor]

    @classmethod
    def format(cls, raw_blocking_dict):
        """Recursively translate raw blocking dictionary to its MPFS representation."""
        if not isinstance(raw_blocking_dict, dict):
            return raw_blocking_dict
        ret = {}
        for k, v in raw_blocking_dict.iteritems():
            if isinstance(v, dict):
                v = cls.format(v) # recursively format dict
            elif k.startswith('p'):
                k = 'public'
            elif k.startswith('a'):
                k = 'auth'
            ret[k] = v
        return ret
