# -*- coding: utf-8 -*-
"""

MPFS
CORE

Интерфейс к Mongo для браузера

"""
from itertools import imap

import mpfs.engine.process

from mpfs.config import settings
from mpfs.metastorage.mongo.collections.base import BaseCollection, UserCollectionKeyValue
from mpfs.metastorage.mongo.util import *
from mpfs.metastorage import MResponse
from mpfs.common import errors
from mpfs.common.errors.browser import *


RE_REQUEST_ID = re.compile('^t(\d+)$')
RE_ID = re.compile('^(\d+)_(\w+)$')
RE_TYPE_ID = re.compile('^(\d+)_(\w+)$')

REQUEST_TYPES_IDS = settings.browser['request_types']
REQUEST_TYPES_NAMES = {v: k for k, v in REQUEST_TYPES_IDS.iteritems()}

BINDATA_SUBTYPE = settings.browser['bindata_subtype']
DATA_COMPRESSION_ON = settings.browser['use_data_compression']


def get_type_name(type_id):
    try:
        result = REQUEST_TYPES_IDS[type_id]
    except KeyError:
        raise BrowserUnknownDataType(type_id)
#        result = 't%s' % type_id
    return result


def get_type_id(type_name):
    try:
        result = REQUEST_TYPES_NAMES[type_name]
    except KeyError:
        raise BrowserUnknownDataType(type_name)
#        result = RE_REQUEST_ID.search(type_name).group(1)
    return result


def gen_abstract_mongo_id(*args):
    return Binary(hashed(':'.join(imap(str, args))), subtype=BINDATA_SUBTYPE)


def gen_mongo_id(type_id, uid, version):
    return gen_abstract_mongo_id(type_id, uid, version)


def gen_sync_index_id(type_id, uid):
    return Binary(hashed('%s:%s' % (type_id, uid)), subtype=BINDATA_SUBTYPE)


def parse_type_id(_id):
    try:
        type_id = RE_ID.split(_id)[1]
        type_id = int(type_id)
    except Exception:
        raise InvalidKey()

    if type_id in REQUEST_TYPES_IDS:
        return type_id
    else:
        raise BrowserUnknownDataType(type_id)


def get_mongo_id(_id):
    try:
        key = RE_ID.split(_id)[2]
        key = str(key)
    except Exception:
        is_valid_key = False
    else:
        try:
            is_valid_key = int(key) != 0
        except Exception:
            is_valid_key = True
    if is_valid_key:
        return Binary(key, subtype=BINDATA_SUBTYPE)
    else:
        raise InvalidKey()


def split_sync_id(sync_id):
    type_id, sync_id = RE_TYPE_ID.split(sync_id)[1:3]
    return get_type_name(int(type_id)), Binary(sync_id, subtype=BINDATA_SUBTYPE)


def get_sync_id(_id, type_id):
    return '%s_%s' % (type_id, _id)


class SyncIndex(BaseCollection):

    name = 'sync_index'
    is_sharded = False
    is_common = False
    uid_field = '_id'

    def update_version(self, uid, type_id, version):
        spec = {
                '_id' : uid,
                }
        doc = {
               '$set' : {
                         'types.%s' % get_type_name(type_id) : version,
                         }
               }
        self.db[self.name].update(spec, doc, multi=True, upsert=True, **self._fsync_safe_w())

    def update_versions(self, uid, data):
        spec = {
                '_id' : uid,
                }
        doc = {
               '$set' : {},
               }
        for t,v in data.iteritems():
            doc['$set']['types.%s' % get_type_name(t)] = v
        value = self.db[self.name].update(spec, doc, multi=True, upsert=True, **self._fsync_safe_w())
        try:
            zdata = value['states']
        except (TypeError, KeyError):
            result = {}
        else:
            result = decompress_data(zdata)
        return result

    def get_last_version(self, uid, type_id):
        spec = {
                '_id' : uid,
                }
        index_record = self.db[self.name].find_one(spec)
        try:
            current_version = index_record['types'][get_type_name(type_id)]
        except Exception:
            current_version = generate_version_number()
        return current_version

    def get_all_last_versions(self, uid, types):
        spec = {
                '_id' : uid,
                }
        found = self.db[self.name].find_one(spec)
        result = {}
        if found:
            for type_id in types:
                try:
                    result[type_id] = found['types'][get_type_name(type_id)]
                except KeyError:
                    result[type_id] = generate_version_number()
            try:
                zdata = found['states']
            except (TypeError, KeyError):
                states = {}
            else:
                states = decompress_data(zdata)
            drop_version = found.get('dv')
        else:
            states = {}
            for type_id in types:
                result[type_id] = generate_version_number()
            drop_version = None
        return result, states, drop_version

    def _check_state_version(self, is_version_specified, version, state):
        if not is_version_specified and 'version' in state:
            raise SyncStateVersionMismatch()  # если версия в документе есть, а в запросе не указана - кидаем ошибку
        elif is_version_specified and 'version' in state and version != state['version']:
            raise SyncStateVersionMismatch()  # если версия в документе и в запросе есть, но они не совпадают - ошибка
        elif is_version_specified and 'version' not in state and version is not None:
            raise SyncStateVersionMismatch()  # версии в документе нет, но в запросе она есть и не равна null - ошибка

    def set_state(self, uid, states, is_version_specified=False, version=None):
        spec = {'_id': uid}
        found_value = self.db[self.name].find_one(spec)
        if found_value is not None:
            self._check_state_version(is_version_specified, version, found_value)

        doc = {'$set': {'states': compress_data(states)}}
        if is_version_specified:
            version = generate_version_number()
            doc['$set']['version'] = version

        self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())

        if is_version_specified:
            return {'version': version}

    def get_state(self, uid, append_version=False):
        found_value = self.db[self.name].find_one({'_id': uid})

        try:
            zdata = found_value['states']
            version = found_value.get('version', None)
        except (TypeError, KeyError):
            raise SyncStateNotFound()

        data = decompress_data(zdata)
        if append_version:
            return {
                'state': data,
                'version': version,
            }
        else:
            return data

    def set_drop_version(self, uid):
        '''
            При dop выставляется версия, от которой впоследствии будут браться изменения.
        '''
        spec = {
                 '_id' : uid,
                 }
        drop_version = generate_version_number()
        doc = {
               '$set' : {
                         'dv' : drop_version,
                        }
               }
        types = self.db[self.name].find_one(spec, fields=('types', )).get('types', {})
        self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())
        result = {}
        for t in types:
            result[str(get_type_id(t))] = drop_version
        return result

    def user_info(self, uid):
        spec = {
                '_id' : uid,
                }
        found_value = self.db[self.name].find_one(spec)
        if found_value is None:
            raise SyncUserNotFound()
        else:
            result = {}
            drop_version = found_value.get('dv')
            for k,v in found_value.get('types', {}).iteritems():
                if not drop_version or drop_version < v:
                    result[str(REQUEST_TYPES_NAMES[k])] = v
            return result


sync_index = SyncIndex()


class SyncCollections(BaseCollection):

    name = 'browser'
    is_sharded = False
    is_common = False
    not_in_zdata = ('v', 'u', 't', 'id', 'p')

    def _unpack_entity(self, type_id, element):
        entity = {}
        for k in ('u', 'v'):
            entity[k] = element[k]
        try:
            parent_id = element['p']
        except KeyError:
            entity['p'] = '0'
        else:
            entity['p'] = get_sync_id(parent_id, type_id)
        if DATA_COMPRESSION_ON:
            data = decompress_data(element['zdata'])
        else:
            data = element['zdata']
        entity.update(data)
        entity['id'] = get_sync_id(element['_id'], type_id)
        entity['t'] = type_id
        return entity

    def _unpack_sequence(self, seq):
        result = []
        for type_id, type_values in seq.iteritems():
            for each in type_values:
                result.append(self._unpack_entity(type_id, each))
        return result

    def _delete_subtree(self, uid, key):
        coll, _id = split_sync_id(key)
        subtree = list(self.db[coll].find({'u': uid, 'p': _id}))

        i = 0
        while i < len(subtree):
            el = subtree[i]
            data = decompress_data(el['zdata'])
            if data.get('f', False):
                subtree.extend(list( \
                    self.db[coll].find({'u': uid, 'p': el['_id']})))
            i += 1

        ## assume DATA_COMPRESSION_ON is defined
        for el in reversed(subtree):
            data = decompress_data(el['zdata'])
            data['d'] = True
            version = generate_version_number()
            self.db[coll].update({'_id': el['_id'], 'u': uid}, \
                {'$set': {'v': version, 'zdata': compress_data(data)}}, \
                    multi=True, upsert=True, **self._fsync_safe_w())

    def delete_data(self, entities, uid):
        result = []

        for entity in entities:
            entity_id, version = entity['id'], entity['v']

            try:
                mongo_id = get_mongo_id(entity_id)
                type_id = parse_type_id(entity_id)

                spec = {
                    '_id': mongo_id,
                    'v': version,
                    'u': uid,
                }

                collection_name = get_type_name(type_id)
                remove_result = self.db[collection_name].remove(spec)
                if not remove_result or remove_result['n'] != 1:
                    spec = {'_id': mongo_id}
                    current_element = self.db[collection_name].find_one(spec)
                    if not current_element:
                        raise BrowserObjectNotFound()
                    else:
                        raise BrowserOutdatedVersion()
                result.append(entity)
            except (BrowserObjectNotFound, BrowserOutdatedVersion, BrowserUnknownDataType, InvalidKey), e:
                result.append({
                    'id': entity_id,
                    'error': e.repr(),
                })

        return MResponse(value=result)

    def put_data(self, rawdata):
        result = []
        versions = {}
        uid = None
        sync_is_outdated = False

        for element in rawdata:
            # version = generate_version_number()
            element_version = element.pop('v', '')
            uid = element.pop('u')
            version = None
            current_value = {}
            try:
                type_id = element.pop('t')
            except KeyError:
                raise BrowserNoDataType()
            else:
                try:
                    collection_name = get_type_name(type_id)
                    try:
                        key = element.pop('id')
                    except KeyError:

                        if sync_is_outdated:
                            _id = None
                            raise SyncOutdatedTypeVersion()

                        s_field = element.get('s')
                        if s_field:
                            _id = gen_mongo_id(type_id, uid, s_field)
                        else:
                            version = generate_version_number()
                            _id = gen_mongo_id(type_id, uid, version)
                        spec = {
                                '_id' : _id,
                                'u' : uid,
                               }
                    else:
                        _id = get_mongo_id(key)

                        if sync_is_outdated:
                            raise SyncOutdatedTypeVersion()

                        spec = {
                                '_id' : _id,
                                'u' : uid,
                                }

                        current_element = self.db[collection_name].find_one(spec)
                        if current_element:
                            current_value = self._unpack_entity(type_id, current_element)
                            for k in self.not_in_zdata:
                                current_value.pop(k, None)

                            if current_value.get('d', False) is not True \
                                and element_version \
                                and int(element_version) < int(current_element['v']):
                                raise SyncOutdatedTypeVersion()

                    if current_value:
                        current_value.update(element)
                    else:
                        current_value = element

                    ## For folder tombstone, delete subtree:
                    if current_value.get('f', False) and \
                        current_value.get('d', False):
                            self._delete_subtree(uid, get_sync_id(_id, type_id))

                    if version is None:
                        version = generate_version_number()
                    doc = {
                           '$set' : {
                                     'v' : version,
                                     }
                           }

                    parent_key = element.pop('p', None)
                    if parent_key is not None:
                        try:
                            doc['$set']['p'] = get_mongo_id(parent_key)
                        except InvalidKey:
                            pass

                    if DATA_COMPRESSION_ON:
                        doc['$set']['zdata'] = compress_data(current_value)
                    else:
                        doc['$set']['zdata'] = current_value

                    self.db[collection_name].update(spec, doc, multi=True, upsert=True, **self._fsync_safe_w())
                    result.append({
                                   'id' : get_sync_id(_id, type_id),
                                   'v': version,
                                  })
                except SyncOutdatedTypeVersion, e:
                    sync_is_outdated = True
                    error_result = {
                                    'error': e.repr(),
                                    }
                    if _id:
                        error_result['id'] = get_sync_id(_id, type_id)
                    result.append(error_result)
                else:
                    versions[type_id] = version
        if versions:
            sync_index.update_versions(uid, versions)
        return MResponse(uid=uid, value=result, versions=versions)

    def diff(self, uid, types, version, *args, **kwargs):
        spec = {
                'u' : uid,
                }
        limit = {}
        if kwargs['limit'] >= 0:
            limit['limit'] = kwargs['limit']
        result = {}
        current_versions, states, drop_version = sync_index.get_all_last_versions(uid, map(lambda x: x[0], types))
        for type_id, type_version in types:
            type_id = int(type_id)
            type_version = int(type_version)
            current_version = current_versions[type_id]
            if current_version == type_version:
                continue
            else:
                selected_version = type_version if type_version > drop_version else drop_version
                spec['v'] = {'$gt' : selected_version}
                collection_name = get_type_name(type_id)
                if (settings.feature_toggles['crop_history_segment'] and
                        collection_name == 'history_segment' and
                        generate_version_number(-91 * 24 * 60 * 60) > selected_version):
                    spec['v']['$gt'] = generate_version_number(-91 * 24 * 60 * 60)
                found = list(self.db[collection_name].find(spec, **limit).sort('v', 1))
                found_length = len(found)
                result[type_id] = found
                if limit.get('limit'):
                    if found_length < limit.get('limit'):
                        limit['limit'] -= found_length
                    else:
                        break
        result = self._unpack_sequence(result)
        return MResponse(value=result, states=states)

    def find(self, uid, spec, *args, **kwargs):
        collections = {}
        try:
            type_id = spec.pop('t')
        except KeyError:
            raise BrowserNoDataType()
        else:
            collections[type_id] = get_type_name(type_id)
        result = {}
        try:
            _id = spec.pop('id')
        except KeyError:
            pass
        else:
            spec['_id'] = get_mongo_id(_id)
        spec = dict(map(lambda (k,v): ('data.%s' % k, v), spec.iteritems()))
        spec['u'] = uid
        for type_id, collection_name in collections.iteritems():
            result[type_id] = list(self.db[collection_name].find(spec))
        return sorted(self._unpack_sequence(result), key=lambda d: (d['t'], d['v']))

    def find_one(self, uid, spec, *args, **kwargs):
        collections = {}
        try:
            type_id = spec.pop('t')
        except KeyError:
            raise BrowserNoDataType()
        else:
            collections[type_id] = get_type_name(type_id)
        result = None
        spec = dict(map(lambda (k,v): ('data.%s' % k, v), spec.iteritems()))
        spec['u'] = uid
        try:
            spec['_id'] = get_mongo_id(kwargs['id'])
        except KeyError:
            pass
        for type_id, collection_name in collections.iteritems():
            result = self.db[collection_name].find_one(spec)
            if result:
                break
        if result:
            return self._unpack_entity(type_id, result)
        else:
            raise errors.StorageKeyNotFound(spec)

    def drop(self, uid):
        return sync_index.set_drop_version(uid)


class SyncSubscription(UserCollectionKeyValue):
    name = 'sync_subscription'
    is_sharded = False
    is_common = False
    
    def __init__(self, *args, **kwargs):
        self.db = mpfs.engine.process.dbctl().database('subscription')

    def put(self, uid, data, version=None, *args, **kwargs):
        client_id = data['options']['client_id']
        spec = {
                 '_id' : gen_abstract_mongo_id(uid, client_id),
                 'uid': uid,
                 }
        doc = {
               '$set' : {
                        'ctime' : data['ctime'],
                        'url'   : data['url'],
                        'ops'   : data['options'],
                        }
               }
        self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())

    def unsubscribe(self, uid, callback, client_id):
        spec = {
                '_id' : gen_abstract_mongo_id(uid, client_id),
                'uid' : uid,
                'url' : callback,
                }
        return self.db[self.name].remove(spec, **self._fsync_safe_w())

    def remove(self, uid, key, version=None, new_vesion=None):
        return self.db[self.name].remove({'uid': propper_uid(uid), 'url': key}, **self._fsync_safe_w())

    def remove_single(self, uid, key, version=None, new_vesion=None):
        return self.db[self.name].remove({'uid': propper_uid(uid), '_id': ObjectId(key)}, **self._fsync_safe_w())

    def folder_content(self, uid, key=None, version=None, new_vesion=None, filter_args={}, range_args={}):
        params = {}
        if uid:
            params.update({'uid': propper_uid(uid)})
        if key:
            params.update({'url': key})
        params.update(filter_args)
        return tuple(self.db[self.name].find(params))
