# -*- coding: utf-8 -*-
"""

Саппортовая коллекция

"""
from bson.objectid import ObjectId
from datetime import datetime
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors

from mpfs.metastorage.mongo.util import propper_uid
from mpfs.metastorage.mongo.binary import Binary
from mpfs.metastorage.mongo.collections.base import ShardedById, BaseCollection


SUPPORT_PROHIBITED_CLEANING_USERS_MAX_COUNT = settings.support['prohibited_cleaning_users_max_count']


class SupportMpfsCollection(ShardedById):
    name = 'support_mpfs'
    is_sharded = False
    is_common = True

    def put(self, uid, rawdata, version=None, **kwargs):
        self.db[self.name].insert(
            {'uid': uid, 'data': rawdata},
            **self._fsync_safe_w()
        )

    def update(self, uid, key, data, **kwargs):
        data = dict([('data.%s' % k, v) for (k, v) in data.iteritems()])
        query = {
            'uid': uid,
            'data.id': key['id'],
        }
        update = {
            '$set': data,
        }
        self.db.support_mpfs.update(query, update, **kwargs)

    def folder_content(self, uid, key, version=None, filter_args={}, range_args={}):
        result = []
        params = {}
        if uid:
            params.update({'uid': propper_uid(uid)})
        if key:
            for k, v in key.iteritems():
                params.update({'data.%s' % k: v})
        all_resources = tuple(self.db[self.name].find(params))
        for item in all_resources:
            data = item.get('data')
            data['uid'] = item.get('uid')
            result.append(data)
        return result


class SupportModerationQueueMpfsCollection(ShardedById):
    name = 'support_moderation_queue'
    is_sharded = False
    is_common = True

    def put_public_link(self, short_url, resource_type):
        moderation_item = {
            "created": datetime.utcnow(),
            "source": "public_views",
            "links": [{"type": resource_type, "url": short_url}],
            "status": "not-moderated",
        }
        self.db[self.name].insert(moderation_item)

    def moderation_queue(self, sort, order, bounds, filters):
        query = dict()
        for filter_, value in filters.iteritems():
            if filter_ in ("source", "status"):
                query[filter_] = value

        amount_args = dict()
        if sort:
            amount_args['sort'] = [(sort, order)]

        if bounds:
            if 'amount' in bounds:
                amount_args['limit'] = bounds['amount']
            if 'offset' in bounds:
                amount_args['skip'] = bounds['offset']

        result = list()
        all_resources = (self.db[self.name].find(query, **amount_args))
        for item in all_resources:
            item['created'] = str(item['created'])
            item['_id'] = str(item['_id'])
            result.append(item)
        return result

    def moderation_queue_count(self, spec):
        return self.db[self.name].find(spec).count()

    def set_status(self, _id, moderator, moderation_time, status):
        query = {
            '_id': ObjectId(_id),
        }
        update = {
            '$set': {'moderator': moderator,
                     'moderation_time': moderation_time,
                     'status': status},
        }
        self.db[self.name].update(query, update, **self._fsync_safe_w())
        return {}


class SupportBlockedHidsCollection(BaseCollection):
    name = 'support_blocked_hids'
    is_sharded = False
    is_common = True

    def put(self, hid, block_type):
        if not isinstance(hid, Binary):
            hid = Binary(str(hid), subtype=2)
        spec = {
            'hid': hid,
        }
        doc = {
            'hid': hid,
            'block_type': block_type,
            'ctime': datetime.now(),
        }
        self.db[self.name].update(spec, doc, upsert=True, **self._fsync_safe_w())

    def get(self, hids):
        spec = {'hid': {'$in': hids}}
        return list(self.db[self.name].find(spec, fields={'_id': False}, **self._fsync_safe_w()))


class SupportBlockHistoryCollection(BaseCollection):
    name = 'support_block_history'
    is_sharded = False
    is_common = True


class SupportProhibitedCleaningUsersCollection(BaseCollection):
    """Коллекция с пользователями, для которых запрещена чистка корзины и хидден даты.

    По замыслу создателей в этой коллекции должно быть не больше 15 записей (ограничение).
    """
    name = 'support_prohibited_cleaning_users'
    is_sharded = False
    is_common = True

    def put(self, uid, comment, moderator, check_limit=True):
        """Добавить одну запись в коллекцию.

        Если запись с таким `uid` уже есть, то она перезапишется.
        Максимальное количество записей в коллекции ограничено сеттингами.
        """
        count = self.db[self.name].find().count()
        if check_limit and count >= SUPPORT_PROHIBITED_CLEANING_USERS_MAX_COUNT:
            raise errors.SupportTooManyUsersWithProhibitedCleaningError()

        result = self.db[self.name].update({'uid': uid}, {
            'uid': uid,
            'ctime': int(time.time()),
            'comment': comment,
            'moderator': moderator
        }, upsert=True, **self._fsync_safe_w())
        return True

    def remove(self, uid):
        """Удалить запись из коллекции для пользователя с идентификатором `uid`."""
        self.db[self.name].remove({'uid': uid})

    def list_all(self):
        """Вернуть все записи из коллекции."""
        return list(self.db[self.name].find({}, {'_id': False}))
