# -*- coding: utf-8 -*-
"""

Разные системные коллекции

"""
import time
from datetime import datetime
from itertools import ifilter
from collections import defaultdict

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util import dbnaming
from mpfs.common.util import Singleton, Cached, name_from_path
from mpfs.metastorage import MResponse, Document
from mpfs.metastorage.mongo.util import *
from mpfs.metastorage.mongo.collections.base import UserCollectionKeyValue, BaseCollection, KeyValue, DirectRouteCollection
from mpfs.metastorage.mongo.binary import Binary


class RegerateVideoInfoFails(DirectRouteCollection):
    """Коллекция неудачных перегенераций video_info

    В эту коллекцию смотрим мы в скрипте `mpfs-admin-update-previews-on-demand.py` и логридер
    Она нужна, чтобы не напрягать кладун перегенерацией video_info
    """
    CONNECTION_NAME = 'meta_tmp'
    DB_COLL_NAME = ('mpfs_meta_tmp', 'regenerate')

    def _hid2bin(self, hid):
        if not isinstance(hid, Binary):
            hid = Binary(str(hid), subtype=2)
        return hid

    def set_failed(self, hid):
        """Пометить hid как неудачно перегенерирумый"""
        hid = self._hid2bin(hid)
        # порт из mpfs-admin-update-previews-on-demand.py, сохраняет документы вида:
        #{"_id": ObjectId("58235083a5f21ec651894dd4")
        # "hid": BinData(2,"IAAAADE2ZDIyZjkyYmVmNmRlMWFkNWFlNTVjMmNjMDQ1NWVm"),
        # "failed" : 1478709379.349983}
        doc = {'$set': {'failed': time.time()}}
        self.collection.update({'hid': hid}, doc, upsert=True)

    def is_already_failed(self, hid):
        hid = self._hid2bin(hid)
        return bool(self.collection.find_one({'hid': hid}))


class DiskInfoCollection(UserCollectionKeyValue, Singleton, Cached):
    name = 'disk_info'
    is_sharded = True
    is_common = False

    _values = defaultdict(dict)
    _content_loaded = False

    def _propper_key(self, key):
        return '/' + '/'.join(filter(None, key.split('/')))

    def query_for_find(self, **kw):
        if 'parent' in kw:
            result = {'parent': kw['parent']}
            if 'uid' in kw:
                result['uid'] = kw['uid']
            return result
        else:
            return super(DiskInfoCollection, self).query_for_find(**kw)

    @classmethod
    def reset(cls):
        cls._values = defaultdict(dict)
        cls._content_loaded = False

    def _get_value(self, uid, key, version=None, params_to_cache_values=None, is_all_content_loaded=True):
        key = '/' + '/'.join(filter(None, key.split('/')))
        try:
            stored_version = str(self._values[uid][key]['version'])
            if version and stored_version != version:
                raise KeyError()
        except KeyError:
            DiskInfoCollection.reset()
            for each in self.get_all(**params_to_cache_values):
                self._values[uid][each['key']] = each
            if not self._values[uid] and not mpfs.engine.process.dbctl().allow_empty_disk_info():
                raise errors.StorageEmptyDiskInfo(uid)
            DiskInfoCollection._content_loaded = is_all_content_loaded
        try:
            result = Document(**self._values[uid][key])
            version = str(self._values[uid][key].get('version', 0))
        except KeyError:
            result = None
            version = None
        return MResponse(value=result, version=version)

    def value(self, uid, key, version=None):
        return self._get_value(uid, key, version=version,
                               params_to_cache_values={'uid': uid})

    def value_for_quota(self, uid, key, version=None):
        """Делает легковесный запрос для получения значений по квоте Диска.

        В отличии от self.value(...) не получает лишних данных (не идет за списком девайсов и прочим).
        """
        return self._get_value(uid, key, version=version,
                               params_to_cache_values={'uid': uid,
                                                       'key': {'$in': ['/total_size', '/limit', '/trash_size']}},
                               is_all_content_loaded=False)

    def folder_content(self, uid, path, version=None, filter_args={}, range_args={}):
        # некэширующий запрос, при необходимости оставить его, остальное закомментировать
        # result = BaseController.folder_content(self, uid, path, version=version, filter_args=filter_args, range_args=range_args)

        uid = str(uid)
        path = '/' + '/'.join(filter(None, path.split('/')))

        if not DiskInfoCollection._content_loaded or not self._values.get(uid):
            DiskInfoCollection.reset()
            for each in self.get_all(uid=uid):
                self._values[uid][each['key']] = each
            if not self._values[uid] and not mpfs.engine.process.dbctl().allow_empty_disk_info():
                raise errors.StorageEmptyDiskInfo(uid)
            DiskInfoCollection._content_loaded = True

        result = {}
        for key, value in ifilter(lambda (k, v): k.startswith(path + '/'), self._values[uid].iteritems()):
            data = {
                'data': value['data'],
                'version': value['version'],
                'key': key,
                'type': value.get('type'),
            }
            result[name_from_path(key)] = Document(**data)
        result = MResponse(value=result, version=str(version))

        return result

    def put(self, uid, key, val, version=None, **kwargs):
        key = '/' + '/'.join(filter(None, key.split('/')))
        try:
            del self._values[uid][key]
        except KeyError:
            DiskInfoCollection.reset()
        else:
            DiskInfoCollection._content_loaded = False
        result = UserCollectionKeyValue.put(self, uid, key, val, version=version, **kwargs)
        return result

    def remove(self, uid, key, old_version=None, new_version=None, data=None):
        key = '/' + '/'.join(filter(None, key.split('/')))
        try:
            del self._values[uid][key]
        except KeyError:
            DiskInfoCollection.reset()
        else:
            DiskInfoCollection._content_loaded = False
        result = UserCollectionKeyValue.remove(self, uid, key)
        return result

    def increment(self, uid, path, diff, field=None, **kw):
        try:
            del self._values[uid][path]
        except KeyError:
            DiskInfoCollection.reset()
        else:
            DiskInfoCollection._content_loaded = False
        return super(DiskInfoCollection, self).increment(uid, path, diff, field, **kw)

    def cache_used_delta(self, uid, path, delta):
        key = '/' + '/'.join(filter(None, path.split('/')))
        self.value(uid, path)
        self.__class__._values[uid][key]['data'] += delta
        if key == '/total_size':
            value = self.value(uid, path).value

    def flush_used_delta(self, uid, path, delta):
        return super(DiskInfoCollection, self).increment(uid, path, delta, None)


class RecountCollection(BaseCollection, Singleton):

    is_sharded = False
    is_common = True
    name = 'recount'

    def add(self, uid):
        """
        Добавить пользователя в коллекцию на пересчёт счётчиков места
        """
        current_time = int(time.time())
        doc = {
            '$setOnInsert': {'ctime': current_time},
            '$set': {'mtime': current_time},
            '$inc': {'count': 1}
        }
        self.db[self.name].update({'_id': uid}, doc, upsert=True, **self._fsync_safe_w())


class LastFilesCacheCollection(BaseCollection):
    name = 'last_files_cache'
    is_sharded = True
    is_common = False
