# -*- coding: utf-8 -*-
from __future__ import absolute_import
from collections import defaultdict
from pymongo.errors import DuplicateKeyError

import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.util import dbnaming, Singleton, filter_uid_by_percentage
from mpfs.config import settings
from mpfs.metastorage.mongo import pool
from mpfs.metastorage.mongo.static import DiskStatus
from mpfs.metastorage.mongo.collections.meta import MPFSMongoCollectionsMeta
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter


POSTGRES_USER_INFO_ENTRY = 'pg'
POSTGRES_INIT_PERCENTAGE = settings.feature_toggles['postgres_init_percentage']


class MPFSMongoReplicaSetMapper(Singleton):
    __values = defaultdict(dict)
    __manual_route = None
    __sharded_collections = []
    __uid_field_mapper = {}

    def __init__(self):
        self.rspool = pool.MPFSMongoReplicaSetPool()
        self.collections_meta = MPFSMongoCollectionsMeta()
        self.collections_meta.setup()
        self.pg_query_executer = PGQueryExecuter()

    def setup(self):
        pass

    @classmethod
    def reset(cls):
        cls.__values = defaultdict(dict)

    def route_collection(self, collection, query):
        """
        Общий рутинг коллекции

        :param collection: объект имеющий атрибут name - имя коллекции
        :param query: dict запроса
        :return: сроутированный объект коллекции
        """
        err_msg = lambda: 'Collection type: %s. Collection name: "%s". Query: %s' % (type(collection), collection.name, query)

        if not self.collections_meta:
            raise errors.RoutingError(err_msg())

        # если общая коллекция - всегда в misсdb
        if self.collections_meta.is_common(collection.name):
            connection = mpfs.engine.process.dbctl().connection('common')
            dbname = dbnaming.dbname(collection.name)
            return connection[dbname][collection.name]

        if not len(query):
            raise errors.EmptyQueryRoutingError(err_msg())

        # если шардированная коллекция
        if self.collections_meta.is_sharded(collection.name):
            return self._route_sharded_collection(collection, query)

        # не надо роутить нешардированную коллекцию
        raise errors.NotShardedCollectionRoutingError(err_msg())

    def _route_sharded_collection(self, collection, query):
        """
        Роутинг шардированной коллекцци

        Если установлен ручной роутинг -
            вернет коллекцию с нужного rs

        Если ничего такого нет -
            проверит, что есть поле с uid в запросе и вернет сроутированный объект коллекции

        :param collection: объект коллекции
        :param query: dict запроса
        :return: сроутированный объект коллекции
        """
        if self.__manual_route:
            rs_name = self.__manual_route
        else:
            request_params = query[0]
            uid_field_name = self.collections_meta.get_uid_field(collection.name)

            rs_name = None
            uid = None
            if uid_field_name in request_params:
                uid = request_params[uid_field_name]
                rs_name = self.get_rsname_for_uid(uid)

            if not rs_name:
                msg = 'Can\'t get rs_name for request. Collection name: "%s". '\
                    'Uid value: "%s". Uid field: "%s". Request: "%s"' % (collection.name, uid_field_name, uid, request_params)
                raise errors.NoShardForUidRoutingError(msg)

        return self.get_collection_for_rs(collection.name, rs_name)

    def get_rsname_for_uid(self, uid):
        """
        Вернет имя шарда, где живет пользователь

        :param uid: uid
        :return: rs_name (string)
        """
        route_info = self.info_route(uid)
        return route_info['shard'] if route_info else None

    def get_collection_for_rs(self, cname, rs_name, **kwargs):
        """
        Вернет объект коллекции, направленный на переданный шард

        :param cname: имя коллекции
        :param rs_name: имя шарда
        :return: collection object
        """
        rs_conn = self.rspool.get_connection_for_rs_name(rs_name, **kwargs)
        db_name = dbnaming.dbname(cname)
        return rs_conn[db_name][cname]

    def info_route(self, uid):
        """
        Информация по маршруту пользователя из базы usermap

        :param uid: uid
        :return: dict с информацией
        """
        if uid not in self.__values:
            route_info = self.__db_find_route(uid)
            self.__values[uid] = route_info
        return self.__values[uid]

    def add_route(self, uid, shard=None):
        """
        Добавит маршрут в базу usermap
        Если shard ==  None, то выберет рандомный шард из разрешенных

        :param uid: uid
        :param shard: имя шарда, куда селим пользователя
        :return: None
        """
        self.pg_query_executer.create_user(uid)
        self.__values[uid] = self.__db_find_route(uid)

    def change_route(self, uid, shard=None, status=DiskStatus.RW):
        """
        Изменить маршрут в базу usermap
        Если shard ==  None, то выберет рандомный шард из разрешенных

        :param uid:
        :param shard: имя шарда, куда переключаем пользователя
        :param status: статус, который установится вместе с изменением шарда
        :return:
        """
        self.__values[uid] = self.__db_find_route(uid)

    def has_route(self, uid):
        """
        Проверить, что маршрут на шард у юзера существует

        :param uid: uid
        :return: False or True
        """
        try:
            return bool(self.info_route(uid))
        except errors.NoShardForUidRoutingError:
            return False

    def remove_route(self, uid):
        """
        Удалить запись о маршруте

        :param uid: uid
        :return: None
        """
        try:
            del (self.__values[uid])
        except KeyError:
            pass

    def get_manual_route_shard_name(self):
        return self.__manual_route

    def set_manual_route(self, rs_name):
        """
        Выставить ручную маршрутизацию на шард для шардированных

        :param rs_name: имя шарда
        """
        self.__manual_route = rs_name

    def drop_manual_route(self):
        """
        Сбросить ручную маршрутизацию для шардированных
        """
        self.__manual_route = None

    def __db_find_route(self, uid):
        if self.pg_query_executer.is_user_in_postgres(uid):
            return {'_id': uid, 'shard': 'pg'}
        raise errors.NoShardForUidRoutingError('Uid "%s" not found in `usermap`' % uid)
