# -*- coding: utf-8 -*-
import random
import traceback

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.common.util import Singleton
from mpfs.metastorage.mongo.util import get_connection_args
from mpfs.metastorage.mongo.rsclient import MPFSMongoReplicaSetClient

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class MPFSMongoReplicaSetPool(Singleton):
    """
    Пул прямых соединений с шардированными репликасетами.
    """
    rs_hosts_by_names = {}
    rs_names_by_hosts = {}
    sharded_collections = []
    unsharded_connection_pool = {}
    connection_pool = {}
    rs_allowed_to_register = []
    rs_names = []

    def __init__(self):
        connection_template = settings.mongo['connection_template']
        self.shard_connection_args = get_connection_args(connection_template)

        if not self.__class__.rs_hosts_by_names or not self.__class__.rs_names_by_hosts:
            self.__class__.rs_hosts_by_names = {}
            self.__class__.rs_names_by_hosts = {}
            self.__class__.sharded_collections = []
            self.__class__.unsharded_connection_pool = {}

    def get_all_shards_info(self):
        return []

    def get_shard_allowed_to_register(self, preferable_shard=None):
        """
        Выбираем произвольный шард, доступный на регистрацию
        Если шард недоступен на запись, делаем еще две попытки
        В конце концов бросим исключение

        :param preferable_shard: str имя предпочительного для регистрации шарда
        :return: str имя шарда
        """
        allowed_to_register = [preferable_shard] if preferable_shard else self.__class__.rs_allowed_to_register
        log.info('getting shard for reg from %s, preferrable %s' % (allowed_to_register, preferable_shard))

        failed_shards = set()
        for attempt in (1, 2, 3):
            rs_name = random.choice(allowed_to_register)
            if rs_name in failed_shards:
                continue
            else:
                if self.is_shard_writeable(rs_name):
                    log.info('got %s for register' % rs_name)
                    return rs_name
                else:
                    log.info('shard %s is not writable' % rs_name)
                    failed_shards.add(rs_name)

        log.info("couldn't find alive shard, tried these %s" % failed_shards)
        raise errors.StorageNoAliveShardsForRegister(failed_shards)

    def get_all_shards_names(self):
        return self.__class__.rs_names

    def get_connection_for_rs_name(self, rs_name, timeout=False, read_preference=None):
        if rs_name not in self.__class__.connection_pool:
            kwargs = {
                'replicaSet': rs_name,
                'name': rs_name,
            }
            kwargs.update(self.shard_connection_args)

            if timeout:
                kwargs.update({
                    'socketTimeoutMS': 120000,
                })

            if read_preference:
                kwargs.update({
                    'readPreference': read_preference,
                })

            client = MPFSMongoReplicaSetClient(
                self.__class__.rs_hosts_by_names[rs_name],
                **kwargs
            )
            client.setup_indexes()
            self.__class__.connection_pool[rs_name] = client
            log.debug('CREATE_CONNECTION to replicaset %s@%s' % (rs_name, self.__class__.rs_hosts_by_names[rs_name]))
        # log.debug('USE_CACHED_CONNECTION to replicaset %s@%s' % (rs_name, self.__class__.rs_hosts_by_names[rs_name]))
        return self.__class__.connection_pool[rs_name]

    def clean_connection_pool(self):
        self.__class__.connection_pool = {}

    def is_shard_writeable(self, rs_name):
        """
        Проверка, что шард (репликасет) доступен на запись

        Отдаст True только если:
        - соединение с шардом есть
        - команда isMaster сработала
        - в выдаче команды есть ключ primary

        :param rs_name: str имя шарда
        :return: boolean да/нет
        """
        try:
            connection = self.get_connection_for_rs_name(rs_name)
            state_info = connection.user_data.command({'isMaster': 1})
            if 'primary' in state_info:
                return True
            else:
                error_log.error('bad state for %s: %s' % (rs_name, state_info))
                return False
        except Exception, e:
            error_log.error('bad state for %s: %s' % (rs_name, e))
            error_log.error(traceback.format_exc())
            return False
