# -*- coding: utf-8 -*-
import pymongo
import threading

import mpfs.engine.process

from mpfs.metastorage.mongo import database
from mpfs.metastorage.mongo.util import collect_indexes_for_connection, restart_monitor
from mpfs.metastorage.mongo.logging import log_request


log = mpfs.engine.process.get_default_log()


class MPFSMongoReplicaSetClient(pymongo.MongoReplicaSetClient):
    database_class = database.MPFSMongoDatabase

    def __init__(self, *args, **kwargs):
        self.__name = kwargs.pop('name', None)
        self.__indexes = {}
        self._mpfs_monitor_lock = threading.Lock()
        super(MPFSMongoReplicaSetClient, self).__init__(*args, **kwargs)
        mpfs.engine.process.register_after_fork(self, restart_monitor)
        log.info("[MPFSCommonMongoReplicaSetClient] Initialized with: args=%r, kwargs=%r", args, kwargs)

    def setup_indexes(self):
        if not self.__indexes:
            self.__indexes = collect_indexes_for_connection(self)

    def get_indexes(self, dbname, collname):
        return self.__indexes.get(dbname, {}).get(collname, {})

    def __getattr__(self, name):
        return self.database_class(self, name)

    def drop_database(self, name_or_database):
        """Drop a database.

        Raises :class:`TypeError` if `name_or_database` is not an instance of
        :class:`basestring` (:class:`str` in python 3) or Database

        :Parameters:
          - `name_or_database`: the name of a database to drop, or a
            :class:`~pymongo.database.Database` instance representing the
            database to drop
        """
        name = name_or_database
        if isinstance(name, self.database_class):
            name = name.name

        if not isinstance(name, basestring):
            raise TypeError("name_or_database must be an instance of "
                            "%s or Database" % (basestring.__name__,))

        self._purge_index(name)
        self[name].command("dropDatabase")

    @log_request
    def _send_message(self, *args, **kwargs):
        return super(MPFSMongoReplicaSetClient, self)._send_message(*args, **kwargs)

    @log_request
    def _send_message_with_response(self, *args, **kwargs):
        return super(MPFSMongoReplicaSetClient, self)._send_message_with_response(*args, **kwargs)

    @property
    def name(self):
        return self.__name
