# -*- coding: utf-8 -*-
from __future__ import absolute_import

import sys
import pymongo

from operator import itemgetter
from collections import defaultdict

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.common.util import Singleton
from mpfs.common.util import dbnaming
from mpfs.metastorage.mongo import util
from mpfs.metastorage.mongo import cursor
from mpfs.metastorage.mongo import mapper
from mpfs.metastorage.mongo import database
from mpfs.metastorage.mongo import connection
from mpfs.metastorage.mongo import rsclient

pymongo.collection.Cursor = cursor.MPFSCommonCursor


class MongoSourceController(Singleton):
    """
    Слой абстракций для работы с монгами(всеми). Он же `mpfs.engine.process.dbctl()`

    На его внешний облик сильно повлияли исторические обстоятельства - иногда отсутсвуют здравая логика и разделение абстракций.

    Место этой сущности в иерархии приложения:
    1. Бизнес логика(через фреймворк имени @akinfold или напрямую) ->
        2. Устаревшая прокладка `mpfs.core.metastorage.control` ->
            3. Псевдоколлекции(не путать с `pymongo.Collection`) `mpfs.metastorage.mongo.collections.*` ->
                4. ЭТА ХУРМА(dbctl) ->
                    5.1. Прямые коннекты к некоторым БД
                    5.2. Коннекты к шардам через mapper
                    5.3. Резолвинг псевдоколлекций(4) в настоящие pymongo-ие(через `NotRoutedCollection`)

    Как пользоваться:
        1. В БИЗНЕС ЛОГИКЕ НЕ ИСПОЛЬЗОВАТЬ!!!
        2. В скриптах - лучше не использовать.
        3. В других случаях - не использовать.

    Если ты делаешь продуктовую задачу и читаешь этот тескт, то лучше беги отсюда
    https://www.youtube.com/watch?v=T1WGAOCfJuo&feature=youtu.be&t=15s
    """
    connections_config = settings.mongo['connections']

    def __init__(self):
        source_conf = settings.mongo['source_controller']
        self._mapper = None
        self.AUTORECONNECT_TIMEOUT = source_conf['autoreconnect_timeout']
        self.TRANSIENT_FAILURE_TIMEOUT = source_conf['transient_failure_timeout']
        self.AUTORETRY_COUNT_RECONNECT = source_conf['autoretry_count_reconnect']
        self.AUTORETRY_COUNT_TRANSIENT_FAILURE = source_conf['autoretry_count_transient_failure']

        self._var_allow_fsync_safe_w = True
        self._var_allow_empty_disk_info = False

        self.write_concern_default = settings.mongo['write_concern']['default']
        self.write_concern_map = settings.mongo['write_concern']['custom']

        self._connections = {
        }

        self._db = {
        }

    @property
    def mapper(self):
        if self._mapper is None:
            self._mapper = mapper.MPFSMongoReplicaSetMapper()
        return self._mapper

    def database(self, db_alias=None):
        """
        Получить базу по имени.

        Базы кешируются.
        Именa базы в монге и в коде может отличаться. Смотри маппинг имен в конфиге:
        "settings->mongo->databases_name_map"
        """
        if db_alias is None:
            # магия начинается здесь
            return NotRoutedDatabase(db_alias)

        if db_alias in ('legacy', 'legacy_operations', 'legacy_changelog', 'legacy_subscriptions'):
            errors.MongosNotSupportedError('`%s` db not supported' % db_alias)

        if db_alias not in self._db:
            real_db_name = settings.mongo['databases_name_map'].get(db_alias, db_alias)
            real_db_name = dbnaming.dbname(real_db_name)
            conn = self.connection(db_alias)
            conn.attribute_class = database.MPFSMongoLazyDatabase
            db = conn[real_db_name]
            self._db[db_alias] = db
        return self._db[db_alias]

    def connection(self, conn_name):
        """
        Получить коннект по имени.

        Коннекты кешируются.
        Должна быть секция, описывающая коннект в конфиге:
        "settings->mongo->connections"
        """
        if conn_name == 'mongos':
            errors.MongosNotSupportedError('`%s` connection not supported' % conn_name)

        if conn_name not in self._connections:
            conf = self.connections_config.get(conn_name)
            if conf is None:
                raise KeyError('No such connection("%s") in settings. Check "settings->mongo->connections".' % conn_name)

            connection_args = util.get_connection_args(conf)
            # почему то для старых коннектов использовался `MPFSMongoConnection`,
            # а для новый `MPFSMongoReplicaSetClient`. Портировано как было.
            if 'replicaSet' in connection_args:
                conn = rsclient.MPFSMongoReplicaSetClient(**connection_args)
            else:
                conn = connection.MPFSMongoLazyConnectionProxy(**connection_args)
            self._connections[conn_name] = conn
        return self._connections[conn_name]

    def setup(self):
        self.mapper.setup()

    # все методы ниже - устаревший код. 10 раз подумай, прежде чем использовать
    def allow_fsync_safe_w(self, *args):
        if args:
            self._var_allow_fsync_safe_w = args[0]
        else:
            return self._var_allow_fsync_safe_w

    def fsync_safe_w(self, db_name='legacy'):
        if self._var_allow_fsync_safe_w:
            return self.write_concern_map.get(db_name, self.write_concern_default)
        else:
            return {}

    def checkdb(self, db_name):
        try:
            self.database(db_name).command({"buildinfo": 1})
        except Exception, e:
            raise errors.StorageNoResponse, e, sys.exc_info()[2]

    def allow_empty_disk_info(self, *args):
        if args:
            self._var_allow_empty_disk_info = args[0]
        else:
            return self._var_allow_empty_disk_info


class NotRoutedCollection(object):
    """
    Класс немаршрутизированной коллекции

    Образовался путем удаления логики маршрутизации из `MPFSCommonMongoCollection`
    Определяет куда пойдет запрос на этапе выполнения запроса
    Поддерживает pymongo методы из множества `allowed_routing_methods` + 2 метода с "хитрой" маршрутизацией:
        * aggregate - автор @mafanasev
        * insert - автор @dedm

    Используется для маршрутизации шардированных коллекци `mongo.collections.*`
    Не рекомендуется использовать этот класс в других местах

    Eсли хотим поддержать еще один метод обычной `pymongo.Collection` нужно понять:
        1. Где в сигнатуре метода находится запрос.
        2. Как извлечь из запроса признак шарда(uid).
        3. Будут ли запросы включать несколько реальных запросов по разным шардам.
    """
    allowed_routing_methods = {
        'count',
        'find',
        'update',
        'remove',
        'find_and_modify',
        'find_one',
    }

    def __init__(self, name):
        self.name = name

    def __getattr__(self, attr_name):
        return self._default_request_method(attr_name)

    def _default_request_method(self, method_name):
        """
        Общий код для маршрутизации запросов для методов из `allowed_routing_methods`
        """
        if method_name not in self.allowed_routing_methods:
            raise errors.NotAllowedMethodRoutingError('Method "%s" does not support routing' % method_name)

        def route_and_call(*args, **kwargs):
            coll = mpfs.engine.process.dbctl().mapper.route_collection(self, args)
            return getattr(coll, method_name)(*args, **kwargs)
        return route_and_call

    def aggregate(self, *args, **kwargs):
        match_query = {}
        if args:
            pipeline = args[0]  # type: list | tuple | dict
            if isinstance(pipeline, dict):
                match_query = pipeline.get('$match', {})
            else:
                match_query = pipeline[0].get('$match', {})
        coll = mpfs.engine.process.dbctl().mapper.route_collection(self, (match_query,))
        return coll.aggregate(*args, **kwargs)

    def insert(self, *args, **kwargs):
        """
        MPFS-ый чудо `insert` от @dedm
        """
        doc_or_docs = args[0]  # в соответствии с докой первый аргумент insert'а -- это документ или список документов
        is_many = isinstance(doc_or_docs, (list, tuple))
        dbctl = mpfs.engine.process.dbctl()

        if not is_many:
            # Инсертить можно один документ или список документов, поэтому если передан один,
            # то используем старый добрый проверенный временем код вставки одного документа.
            coll = dbctl.mapper.route_collection(self, args)
            return coll.insert(*args, **kwargs)
        else:
            result = []
            # Если передано несколько документов, то определяем шард для каждого документа
            # и вставляем документы в коллекции на соответствующих шардах.
            grouped_by_collection = defaultdict(lambda: defaultdict(list))

            # Роутим каждый документ и группируем по коллекции.
            collections = {}
            for i, doc in enumerate(doc_or_docs):
                new_args = [doc]
                new_args.extend(args[1:])

                # Организуем хэш-таблицу коллекций, чтобы иметь возможность группировать документы по коллекции.
                coll = dbctl.mapper.route_collection(self, new_args)
                coll_key = (coll.database.connection.name, coll.database.name, coll.name)
                if coll_key not in collections:
                    collections[coll_key] = coll
                else:
                    coll = collections[coll_key]

                # Сохраняем номер элемента в оригинальной последовательности,
                # чтоб правильно распределить _id-шники в результате.
                grouped_by_collection[coll]['docs'].append(doc)
                grouped_by_collection[coll]['indexes'].append(i)

            # Делаем запросы на вставку документов в коллекции на шардах и собираем общий результат.
            result = []
            for coll, batch in grouped_by_collection.iteritems():
                indexes = batch['indexes']
                docs = batch['docs']
                new_args = [docs]
                new_args.extend(args[1:])

                # Непосредственно сама вставка.
                ret = coll.insert(*args, **kwargs)
                ret = ret if isinstance(ret, list) else [ret]
                result.extend(zip(indexes, ret))

            # Если в результате у нас не пустой список, то достаём из него id-шники документов.
            # Иначе возвращаем просто пустой список.
            if result:
                # Порядок id-шников результата должен соответствовать порядку элементов в doc_or_docs
                # поэтому сортируем результат по индексу
                result.sort(key=itemgetter(0))
                result = list(zip(*result)[1])  # раззиповываем _id-шники в отдельный список
            return result


class NotRoutedDatabase(object):
    """
    Класс немаршрутизированной базы
    """
    def __init__(self, name):
        self.name = name

    def __getattr__(self, attr_name):
        return NotRoutedCollection(attr_name)

    def __getitem__(self, item_name):
        return NotRoutedCollection(item_name)
