# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.core.filesystem.dao.legacy import CollectionRoutedDatabase, is_collection_uses_dao
from mpfs.core.user.dao.migration import PgMigrationLockDao
from mpfs.metastorage.mongo.mapper import POSTGRES_USER_INFO_ENTRY
from mpfs.metastorage.mongo.static import DiskStatus
from mpfs.metastorage.mongo.util import propper_uid, shard_key, generate_version_number
from mpfs.metastorage.mongo.collections.base import user_index as user_index_mpfs_collection
from mpfs.common import errors
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter

log = mpfs.engine.process.get_default_log()


class MongoUserController(object):
    """Сущность для проверки инициализированности uid-а в диске и управления локами на uid.

    Трактат "В чем разница между `user_index_mpfs_collection` и `user_index_collection`"

    `user_index_mpfs_collection` - это инстанс mpfs.metastorage.mongo.collections.base.UserIndexCollection()
    Т.е. это объект нашей прослойки, а не pymongo.collection
    `user_index_collection` - получаем из dbctl.database() значит это инстанс `NotRoutedCollection`, который
    на этапе выполнения запроса превращается MPFSMongoCollection, можно сказать, что это pymongo.collection

    Таким образом:
        * `user_index_collection` - несроученный эквавалент объекта pymongo.collection(disk-unit-N.user_index.user_index`)
        * `user_index_mpfs_collection` - порождение сумрачного гения MPFS

    Так зачем же нам нужен `user_index_mpfs_collection`?
    Ответ - он имеет логику кеширования запросов в user_index, поэтому пользуемся.
    """
    dbctl = mpfs.engine.process.dbctl()
    user_index_collection = CollectionRoutedDatabase().user_index
    pg_query_executer = PGQueryExecuter()

    def _has_usermap_record(self, uid):
        return self.pg_query_executer.is_user_in_postgres(uid)

    def _has_userindex_record(self, uid):
        return bool(user_index_mpfs_collection.check_user(propper_uid(uid)))

    def assert_user_init(self, uid):
        """Райзить ошибку, если пользователь не заведен

        Два условия:
            1. Есть запись в шарпее
            2. Есть запись в `user_index`
        """
        if not self._has_usermap_record(uid):
            raise errors.StorageInitUser('Not found "%s" in "user_map"' % uid)

        if not self._has_userindex_record(uid):
            raise errors.StorageInitUser('Not found "%s" in "user_index"' % uid)

    def is_user_init(self, uid):
        return self._has_usermap_record(uid) and self._has_userindex_record(uid)

    def is_user_completely_initialized(self, uid):
        if not self._has_usermap_record(uid):
            return False
        user_info = user_index_mpfs_collection.check_user(propper_uid(uid))
        return user_info and not user_info.get('lock_key')

    def user_init_in_progress(self, uid):
        if not self._has_usermap_record(uid):
            return False
        user_info = user_index_mpfs_collection.check_user(propper_uid(uid))
        return bool(user_info and user_info.get('lock_key'))

    def check(self, uid):
        """Старый странный метод. Оставлен для совместимости
        """
        if self.is_user_init(uid):
            return user_index_mpfs_collection.check_user(uid)
        return None

    def update(self, uid, type=None, locale=None, b2b_key=None, pdd=None, is_mailish=None, hancom_enabled=None,
               office_selection_strategy=None):
        self.assert_user_init(uid)

        uid = propper_uid(uid)
        spec = {'_id': uid, 'shard_key': shard_key(uid)}

        new_values = {'locale': locale, 'type': type}
        if b2b_key is not None:
            new_values['b2b_key'] = b2b_key
        if pdd is not None and isinstance(pdd, dict):
            new_values['pdd'] = pdd
        if is_mailish is not None and isinstance(is_mailish, bool):
            new_values['is_mailish'] = is_mailish
        if hancom_enabled is not None and isinstance(hancom_enabled, bool):
            new_values['hancom_enabled'] = hancom_enabled
        if office_selection_strategy is not None and isinstance(office_selection_strategy, basestring):
            new_values['office_selection_strategy'] = office_selection_strategy
        doc = {"$set": new_values}
        user_index_mpfs_collection.update(spec, doc)

    def reset_b2b(self, uid):
        self.assert_user_init(uid)

        uid = propper_uid(uid)
        spec = {'_id': uid, 'shard_key': shard_key(uid)}
        doc = {'$unset': {'b2b_key': ''}}
        user_index_mpfs_collection.update(spec, doc)

    def make_yateam(self, uid, yateam_uid):
        self.assert_user_init(uid)

        spec = {'_id': uid, 'shard_key': shard_key(uid)}
        doc = {'$set': {'yateam_uid': yateam_uid}}
        user_index_mpfs_collection.update(spec, doc)

    def reset_yateam(self, uid):
        self.assert_user_init(uid)

        spec = {'_id': uid, 'shard_key': shard_key(uid)}
        doc = {'$unset': {'yateam_uid': ''}}
        user_index_mpfs_collection.update(spec, doc)

    def set_last_quick_move_version(self, uid, version):
        """
        Костыль, нужный для поддержки работы синхронизации с ПО для прототипа первой версии быстрого перемещения и
        переименования папок.
        Работает следующим образом: когда выполняем перемещение папки - запоминаем сюда версию этого перемещения и
        далее, когда ПО приходит к нам в ручку diff с версией, мы сравниваем переданную в ручке версию с этой версией.
        Если она меньше или равна, то мы отдаем ошибку, которая сообщает ПО о том, что надо получить полный diff.
        Зачем это нужно: т.к. при быстром перемещении мы не записываем в коллекцию changelog изменения по всем файлам
        из этой папки, то ПО не сможет синхронизироваться "обычным старым способом". Нужно заставить его получить полный
        diff (листинг) всех ресурсов пользователя.
        """
        self.assert_user_init(uid)

        spec = {'_id': uid, 'shard_key': shard_key(uid)}
        doc = {'$set': {'last_quick_move_version': version}}
        user_index_mpfs_collection.update(spec, doc)

    def last_quick_move_version(self, uid):
        self.assert_user_init(uid)

        doc = user_index_mpfs_collection.check_user(uid)
        return doc.get('last_quick_move_version')

    def force_snapshot_version(self, uid):
        self.assert_user_init(uid)

        doc = user_index_mpfs_collection.check_user(uid)
        return doc.get('force_snapshot_version')

    def is_reindexed_for_quick_move(self, uid):
        self.assert_user_init(uid)

        doc = user_index_mpfs_collection.check_user(uid)
        return doc.get('is_reindexed_for_quick_move', False)

    def set_reindexed_for_quick_move(self, uid):
        self.assert_user_init(uid)

        spec = {'_id': uid, 'shard_key': shard_key(uid)}
        doc = {'$set': {'is_reindexed_for_quick_move': True}}
        user_index_mpfs_collection.update(spec, doc)

    def create_usermap_entry(self, uid, shard):
        try:
            self.dbctl.mapper.add_route(uid, shard=shard)
        except errors.UidRouteAlreadyCreatedError:
            log.info('User %s already has usermap entry, skipping' % uid)

    def create(self, uid, type=None, locale=None, shard=None, b2b_key=None, pdd=None, is_mailish=None,
               hancom_enabled=None, lock_key=None, enable_quick_move=False, is_paid=False, faces_indexing_state=None,
               office_selection_strategy=None, default_product_id=None):
        """Создать пользователя в базе
        """
        uid = propper_uid(uid)

        self.create_usermap_entry(uid, shard)

        self.user_index_collection.create_user(uid, type, locale, b2b_key, pdd, is_mailish, hancom_enabled, lock_key,
                                               enable_quick_move, is_paid, faces_indexing_state,
                                               office_selection_strategy=office_selection_strategy,
                                               default_product_id=default_product_id)

    def release_lock(self, uid):
        self.user_index_collection.release_lock(uid)

    def remove_own_lock_timestamp(self, uid, lock_key):
        self.user_index_collection.remove_own_lock_timestamp(uid, lock_key)

    def uid_version(self, uid):
        """Получить версию пользователя
        """
        self.assert_user_init(uid)
        doc = user_index_mpfs_collection.check_user(uid)
        return doc.get('version', generate_version_number())

    def version(self, uids):
        # наследние кровавого режима
        if isinstance(uids, (list, tuple)):
            result = {}
            for uid in uids:
                result[uid] = self.uid_version(uid)
            return result
        else:
            return self.uid_version(uids)

    def remove(self, uid, check_init=True):
        """
        Удалить пользователя (all collections assigned to him)
        """
        if check_init:
            self.assert_user_init(uid)
        uid = propper_uid(uid)
        shardkey = shard_key(uid)
        spec = {'_id': uid, 'shard_key': shardkey}
        user_record = self.user_index_collection.find_one(spec)
        db = CollectionRoutedDatabase()
        if user_record:
            for coll in user_record['collections']:
                uid_field_name = self.dbctl.mapper.collections_meta.get_uid_field(coll)
                db[coll].remove({uid_field_name: uid})
                doc = {'$pull': {'collections': coll}}
                self.user_index_collection.update(spec, doc)
            db.billing_services.remove({'uid': uid})
            db.billing_orders.remove({'uid': uid})
            # делаем через mpfs-коллекции т.к. там происходит reset кеша
            user_index_mpfs_collection.remove({'_id': uid})
        self.dbctl.mapper.remove_route(uid)

    def is_user_in_postgres(self, uid):
        """Определить, смигрирован ли ``uid`` в постгрес.
        """
        return self.pg_query_executer.is_user_in_postgres(uid)

    def is_collection_in_postgres(self, uid, coll_name):
        """Определить, смигрирована ли коллекция для пользователя в постгрес или она в монге.

        .. warning::
            **DEPRECATED:** Вместо этого метода используй :meth:`is_user_in_postgres`
        """
        return self.is_user_in_postgres(uid)

    # Ниже методы относящиеся к режиму RO

    def info(self, uid):
        info_route = self.dbctl.mapper.info_route(uid)
        return {
            'status': info_route.get('status', DiskStatus.RW),
            'shard': info_route['shard']
        }

    def assert_writable(self, uid):
        """Проверить, можно ли в юзера писать

        Нельзя писать только в 2-х случаях:
            * Стоит флаг RO
            * Есть лок миграции
        """
        self.assert_user_init(uid)
        try:
            user_index_writable = self.info(uid)['status']
        except errors.StorageInitUser:
            pass
        else:
            if user_index_writable == DiskStatus.RO:
                raise errors.UserIsReadOnly(uid)

        if PgMigrationLockDao().is_locked(uid):
            raise errors.UserIsReadOnly(uid)

    def is_writable(self, uid):
        try:
            self.assert_writable(uid)
        except errors.UserIsReadOnly:
            return False
        else:
            return True

    # Управление обычными статусами RO/RW.

    def set_readonly(self, uid):
        """Поставить лок на запись пользователю.
        """
        self.assert_user_init(uid)
        # Тут сделать реализацию ридонли

    def set_writeable(self, uid):
        """Снять лок на запись.
        """
        self.assert_user_init(uid)
        # Тут сделать реализацию ридонли
