# -*- coding: utf-8 -*-
import zlib
import re
import time
import traceback
import pprint
import pymongo

from contextlib import contextmanager
from pymongo.errors import AutoReconnect, OperationFailure, InvalidOperation
from pymongo.mongo_replica_set_client import register_monitor
from pymongo.read_preferences import modes

import mpfs.engine.process

from mpfs.config import settings
from mpfs.metastorage.mongo.binary import Binary
from mpfs.common.util import to_json, from_json, SuppressExceptions, chunks2
from mpfs.common.util import fulltime, hashed
from mpfs.common.util import filetypes


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

RE_SPLIT_PATH = re.compile('/')


def get_connection_args(config):
    """
    Получение из конфига настроек подключения к mongo.

    :param config: секция конфига с настройками коннекта
    :return:
    """
    config = config.copy()
    if 'readPreference' in config:
        config['readPreference'] = getattr(pymongo.ReadPreference, config['readPreference'])
    return config


def decompress_data(data):
    if isinstance(data, (str, Binary)):
        return from_json(zlib.decompress(data))
    else:
        return data


def generate_version_number(offset=0.0):
    return int(fulltime(offset))


def compress_data(data):
    return Binary(zlib.compress(to_json(data, internal=True), settings.mongo['options']['zlib_level']))


def shard_key(uid):
    return int(hashed(uid)[:4], 16)


def propper_uid(uid):
    return str(uid)


def propper_key(key):
    if '/' in key:
        return '/' + '/'.join(filter(None, RE_SPLIT_PATH.split(key)))
    else:
        return key


def is_subpath(subpath, path, use_regex=False):
    if use_regex:
        return bool(re.match('^%s($|/.*)' % re.escape(path), subpath))
    return subpath.startswith(path.rstrip('/')+'/')


def parent_for_key(key):
    return '/' + '/'.join(filter(None, RE_SPLIT_PATH.split(key))[:-1])


def name_for_key(key):
    try:
        return filter(None, RE_SPLIT_PATH.split(key))[-1]
    except IndexError:
        return '/'


def change_path_parent(path, old_parent, new_parent):
    if path.startswith(old_parent):
        return '%s%s' % (new_parent, path[len(old_parent):])
    else:
        raise Exception('Wrong old prefix. '
                        'Path: "%s", old prefix: "%s"' % (path, old_parent))


def id_for_key(*args):
    '''
        _id(uid, key) or _id(uid:key)
    '''
    if len(args) == 2:
        uid, key = args
        _group = '%s:%s' % (uid, key)
    else:
        _group = args[0]
    return hashed(_group)


def parent_id_for_key(*args):
    '''
        parent_id_for_key(uid, key) or parent_id_for_key(uid:parent_key)
    '''
    if len(args) == 2:
        uid, key = args
        _group = uid + ':' + parent_for_key(key)
    else:
        _group = args[0]
    return hashed(_group)


def map_kv_for_write(data):
    if 'media_type' in data:
        data['mt'] = filetypes.getGroupNumber(data['media_type'])
        del data['media_type']
    return data


def map_kv_for_read(data):
    if 'mt' in data:
        data['media_type'] = filetypes.getGroupByNumber(data['mt'])
        del data['mt']
    return data


def get_active_mode(rs_client, kwargs):
    """Вернуть активный режим запроса.

    Режим, переданный в запрос явно, имеет выше приоритет чем режим,
    установленный для клиента.

    :type rs_client: :class:`pymongo.mongo_replica_set_client.MongoReplicaSetClient`
    :type kwargs: dict
    :return: str
    """
    mode = kwargs.get('read_preference')
    if mode is None:
        mode = getattr(rs_client, 'read_preference')
    return modes.get(mode, 'UNKNOWN')


def wrap_if_hasattr(decorator, attr_name):
    """
    Декоратор, который влючает/выключает другой декоратор

    В зависимости от наличия атрибута в классе(равного True),
    возвращает либо сам метод, либо обернутый декоратором(первый аргумент)
    метод.
    """
    def conditional_decorator(fn):
        dec = decorator(fn)
        def wrapper(self, *args, **kw):
            if hasattr(self, attr_name) and getattr(self, attr_name) == True:
                return fn(self, *args, **kw)
            return dec(self, *args, **kw)
        return wrapper
    return conditional_decorator


def restart_monitor(rsc):
    """
    Перезапуск monitor MongoReplicaSetClient

    Связанные ссылки:
        * Recreate monitors: https://jira.mongodb.org/browse/PYTHON-549
        * Фикс "monitor is dead" от разрабов pymongo: https://github.com/mongodb/mongo-python-driver/commit/7e6878f9c92c79f99006291a56b2a4af055cb72a#diff-4052f7108918fbee25e6e4d31880033eR1088
        * Последствия replica_set_client.close(): https://st.yandex-team.ru/CHEMODAN-23119
        * Monitor thread is dead: https://st.yandex-team.ru/CHEMODAN-20677
    """
    log.info('[restart_monitor] Executed on: %s', rsc)

    from mpfs.metastorage.mongo.rsclient import MPFSMongoReplicaSetClient
    if not isinstance(rsc, MPFSMongoReplicaSetClient):
        return

    with rsc._mpfs_monitor_lock:
        if rsc._MongoReplicaSetClient__monitor.isAlive():
            log.info('[restart_monitor] Monitor is alive: (%r), nothing to be done', rsc._MongoReplicaSetClient__monitor)
            return
        rsc._MongoReplicaSetClient__monitor = type(rsc._MongoReplicaSetClient__monitor)(rsc)
        rsc._MongoReplicaSetClient__monitor.setDaemon(True)
        register_monitor(rsc._MongoReplicaSetClient__monitor)
        rsc._MongoReplicaSetClient__monitor.start_sync()  # блокируемся на 5 сек
        log.info('[restart_monitor] Restart done for: %s', rsc)


def autoretry(f):
    def is_transient_failure(e):
        s = str(e)
        return ('writeback' in str(s))

    def wrapper(*args, **kwargs):
        i_r = 1
        i_s = 1
        i_o = 1
        collection = args[0]
        while True:
            try:
                result = f(*args, **kwargs)
            except OperationFailure, ofe:
                if is_transient_failure(ofe) and i_s <= mpfs.engine.process.dbctl().AUTORETRY_COUNT_TRANSIENT_FAILURE:
                    i_s += 1
                    error_log.error('Got %s error on %s.%s.%s, will try again, %s attempt out of %s' % \
                                    (ofe.__class__.__name__, collection.database.name,
                                     collection.name, f.__name__, i_s,  mpfs.engine.process.dbctl().AUTORETRY_COUNT_TRANSIENT_FAILURE)
                                    )
                    time.sleep(mpfs.engine.process.dbctl().TRANSIENT_FAILURE_TIMEOUT)
                else:
                    raise
            except AutoReconnect, are:
                if i_r <= mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT:
                    i_r += 1
                    error_log.error('Got %s error on %s.%s.%s, will try again, %s attempt out of %s' % \
                                    (are.__class__.__name__, collection.database.name,
                                     collection.name, f.__name__, i_r,  mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT)
                                    )
                    time.sleep(mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT)
                else:
                    raise
            except InvalidOperation, ore:
                if i_o <= mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT:
                    i_o += 1
                    error_log.error('Got %s error on %s.%s.%s, will try again, %s attempt out of %s' % \
                                    (ore.__class__.__name__, collection.database.name,
                                     collection.name, f.__name__, i_r,  mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT)
                                    )

                    restart_monitor(collection.database.connection)
                else:
                    raise
            else:
                return result
    return wrapper


def extract_mulca_ids(element):
    """

    :param element: db.collection.findOne
    :return: {mid_name: mid_value}
    """
    file_stids = {}
    if 'stids' in element.get('data', {}):
        # Если мулечные id лежат в несжатом виде.
        for each in element['data']['stids']:
            if each['type'][-3:] == 'mid':
                file_stids[each['type']] = each['stid']
    else:
        unzipped = decompress_data(element['zdata'])
        file_stids = unzipped.get('setprop', {})
        previews = unzipped.get('mulca', {}).get('previews')
        if previews and isinstance(previews, dict):
            for pk in ('XXXL', 'XXL', 'XL', 'L', 'M', 'S', 'XS', 'XXS', 'XXXS'):
                pv = previews.get(pk)
                if pv:
                    file_stids[pk] = pv
    return file_stids


def collect_indexes_for_connection(connection):
    """
    Загружаем все индексы для каждой базы данных соединения

    :param connection: объект типа pymongo.Connection или pymongo.MongoReplicaSetClient
    :return: defaultdict

    Пример выдачи для шардированной схемы:

    {u'admin': {},
     u'album_items': {},
     u'albums': {},
     u'attach_data': {u'attach_data': {u'_id': [u'_id'],
                                       u'data.stids.stid': [u'data.stids.stid'],
                                       u'hid': [u'hid'],
                                       u'parent': [u'parent'],
                                       u'uid': [u'uid'],
                                       u'uid_data.file_id': [u'uid',
                                                             u'data.file_id'],
                                       u'uid_parent': [u'uid', u'parent']}},
     u'changelog': {u'changelog': {u'_id': [u'_id'],
                                   u'data.stids.stid': [u'data.stids.stid'],
                                   u'dtime': [u'dtime'],
                                   u'uid': [u'uid'],
                                   u'uid_gid_version': [u'uid',
                                                        u'gid',
                                                        u'version'],
                                   u'uid_version': [u'uid', u'version']}},
     u'dataapi_users_meta': {u'users_meta': {u'_id': [u'_id']}},
     u'disk_dataapi_bazinga_tasks': {u'cronJobs': {u'_id': [u'_id']},
                                     u'cronTaskStatistics': {u'_id': [u'_id']},
                                     u'cronTasks': {u'_id': [u'_id']},
                                     u'onetimeJobs': {u'_id': [u'_id']},
                                     u'onetimeTasks': {u'_id': [u'_id']}},
     u'disk_info': {u'disk_info': {u'_id': [u'_id'],
                                   u'uid': [u'uid'],
                                   u'uid_parent': [u'uid', u'parent']}},
     u'filesystem_locks': {u'filesystem_locks': {u'_id': [u'_id'],
                                                 u'dtime': [u'dtime'],
                                                 u'uid': [u'uid'],
                                                 u'uid_key': [u'uid', u'key']}},
     u'hidden_data': {u'hidden_data': {u'_id': [u'_id'],
                                       u'data.stids.stid': [u'data.stids.stid'],
                                       u'hid': [u'hid'],
                                       u'uid': [u'uid'],
                                       u'uid_data.file_id': [u'uid',
                                                             u'data.file_id'],
                                       u'uid_parent': [u'uid', u'parent']}},
     u'link_data': {u'link_data': {u'_id': [u'_id'],
                                   u'dtime': [u'dtime'],
                                   u'parent': [u'parent'],
                                   u'uid': [u'uid']}},
     u'misc_data': {u'misc_data': {u'_id': [u'_id'],
                                   u'data.stids.stid': [u'data.stids.stid'],
                                   u'hid': [u'hid'],
                                   u'uid': [u'uid'],
                                   u'uid_data.file_id': [u'uid',
                                                         u'data.file_id'],
                                   u'uid_data.mt': [u'uid', u'data.mt'],
                                   u'uid_parent': [u'uid', u'parent']}},
     u'narod_data': {},
     u'operations': {u'operations': {u'_id': [u'_id'],
                                     u'dtime': [u'dtime'],
                                     u'uid': [u'uid'],
                                     u'uid_md5': [u'uid', u'md5'],
                                     u'uid_state': [u'uid', u'state']}},
     u'tag_resource': {},
     u'tags': {},
     u'tags_images': {},
     u'trash': {u'trash': {u'_id': [u'_id'],
                           u'data.stids.stid': [u'data.stids.stid'],
                           u'hid': [u'hid'],
                           u'uid': [u'uid'],
                           u'uid_data.file_id': [u'uid', u'data.file_id'],
                           u'uid_parent': [u'uid', u'parent']}},
     u'user_data': {u'user_data': {u'_id': [u'_id'],
                                   u'data.stids.stid': [u'data.stids.stid'],
                                   u'hid': [u'hid'],
                                   u'uid': [u'uid'],
                                   u'uid_data.file_id': [u'uid',
                                                         u'data.file_id'],
                                   u'uid_data.mt': [u'uid', u'data.mt'],
                                   u'uid_parent': [u'uid', u'parent']}},
     u'user_index': {u'user_index': {u'_id': [u'_id'],
                                     u'data.stids.stid': [u'data.stids.stid'],
                                     u'hid': [u'hid'],
                                     u'uid': [u'uid'],
                                     u'uid_data.file_id': [u'uid',
                                                           u'data.file_id'],
                                     u'uid_parent': [u'uid', u'parent']}}}

    Пример выдачи для старой схемы с общей базой:

    {u'admin': {u'changelog': {u'_id': [u'_id'], u'version': [u'version']}},
     u'confif': {},
     u'config': {u'changelog': {u'_id': [u'_id']},
                 u'chunks': {u'_id': [u'_id'],
                             u'ns_lastmod': [u'ns', u'lastmod'],
                             u'ns_min': [u'ns', u'min'],
                             u'ns_shard_min': [u'ns', u'shard', u'min']},
                 u'chunks-backup-5345b5edec32800ba62228c7': {u'_id': [u'_id'],
                                                             u'ns_lastmod': [u'ns',
                                                                             u'lastmod'],
                                                             u'ns_min': [u'ns',
                                                                         u'min'],
                                                             u'ns_shard_min': [u'ns',
                                                                               u'shard',
                                                                               u'min']},
                 u'collections': {u'_id': [u'_id']},
                 u'collections-backup-5345b5edec32800ba62228c7': {u'_id': [u'_id']},
                 u'databases': {u'_id': [u'_id']},
                 u'lockpings': {u'_id': [u'_id'], u'ping': [u'ping']},
                 u'locks': {u'_id': [u'_id'],
                            u'state_process': [u'state', u'process'],
                            u'ts': [u'ts']},
                 u'mongos': {u'_id': [u'_id']},
                 u'settings': {u'_id': [u'_id']},
                 u'shards': {u'_id': [u'_id'], u'host': [u'host']},
                 u'tags': {u'_id': [u'_id'], u'ns_min': [u'ns', u'min']},
                 u'version': {u'_id': [u'_id']}},
     u'dataapi_users_meta': {u'users_meta': {u'_id': [u'_id']}},
     u'disk_bazinga_tasks': {},
     u'mpf': {},
     u'mpfs': {u'albums': {u'_id': [u'_id']},
               u'app': {u'_id': [u'_id']},
               u'app_notification': {u'_id': [u'_id']},
               u'app_setting': {u'_id': [u'_id']},
               u'attach_data': {u'_id': [u'_id'],
                                u'data.stids.stid': [u'data.stids.stid'],
                                u'hid': [u'hid'],
                                u'parent': [u'parent'],
                                u'uid': [u'uid']},
               u'autofill': {u'_id': [u'_id']},
               u'autofill_profile': {u'_id': [u'_id']},
               u'billing_locks': {u'_id': [u'_id']},
               u'billing_orders': {u'_id': [u'_id'],
                                   u'ctime': [u'ctime'],
                                   u'uid': [u'uid']},
               u'billing_orders_history': {u'_id': [u'_id'],
                                           u'ctime': [u'ctime'],
                                           u'uid': [u'uid']},
               u'billing_service_attributes': {u'_id': [u'_id'],
                                               u'uid': [u'uid']},
               u'billing_services': {u'_id': [u'_id'],
                                     u'btime_pid': [u'btime', u'pid'],
                                     u'btime_state': [u'btime', u'state'],
                                     u'uid': [u'uid']},
               u'billing_services_history': {u'_id': [u'_id'],
                                             u'sid': [u'sid'],
                                             u'uid': [u'uid']},
               u'billing_subscriptions': {u'_id': [u'_id'], u'sid': [u'sid']},
               u'bookmark': {u'_id': [u'_id']},
               u'browser': {u'_id': [u'_id']},
               u'changelog': {u'_id': [u'_id'],
                              u'dtime': [u'dtime'],
                              u'uid': [u'uid'],
                              u'uid_gid_version': [u'uid', u'gid', u'version'],
                              u'uid_version': [u'uid', u'version'],
                              u'version': [u'version']},
               u'device_info': {u'_id': [u'_id']},
               u'disk_info': {u'_id': [u'_id'],
                              u'parent': [u'parent'],
                              u'shard_key': [u'shard_key'],
                              u'uid': [u'uid']},
               u'dist_intel_201311': {u'_id': [u'_id'],
                                      u'id': [u'id'],
                                      u'install': [u'install'],
                                      u'serial': [u'serial']},
               u'dist_kingston_flash_drives': {u'_id': [u'_id'],
                                               u'serial': [u'serial']},
               u'dist_media_markt_tablets': {u'_id': [u'_id'],
                                             u'serial': [u'serial']},
               u'dist_samsung_201309': {u'_id': [u'_id'],
                                        u'id': [u'id'],
                                        u'install': [u'install'],
                                        u'serial': [u'serial']},
               u'dist_samsung_nb': {u'_id': [u'_id'],
                                    u'id': [u'id'],
                                    u'install': [u'install'],
                                    u'serial': [u'serial']},
               u'dist_samsung_ub': {u'_id': [u'_id'],
                                    u'id': [u'id'],
                                    u'install': [u'install'],
                                    u'serial': [u'serial']},
               u'dist_sony_nb': {u'_id': [u'_id'],
                                 u'id': [u'id'],
                                 u'install': [u'install'],
                                 u'serial': [u'serial']},
               u'dist_sony_tblt': {u'_id': [u'_id']},
               u'dist_yms': {u'_id': [u'_id'], u'serial': [u'serial']},
               u'extension': {u'_id': [u'_id']},
               u'extension_setting': {u'_id': [u'_id']},
               u'filesystem_locks': {u'_id': [u'_id'],
                                     u'dtime': [u'dtime'],
                                     u'uid': [u'uid'],
                                     u'uid_key': [u'uid', u'key']},
               u'group_invites': {u'_id': [u'_id'],
                                  u'ctime': [u'ctime'],
                                  u'gid': [u'gid'],
                                  u'uid': [u'uid']},
               u'group_links': {u'_id': [u'_id'],
                                u'gid': [u'gid'],
                                u'uid': [u'uid']},
               u'groups': {u'_id': [u'_id'],
                           u'gid': [u'gid'],
                           u'owner': [u'owner']},
               u'hardlinks': {u'_id': [u'_id']},
               u'hidden_data': {u'_id': [u'_id'],
                                u'data.stids.stid': [u'data.stids.stid'],
                                u'hid': [u'hid'],
                                u'uid': [u'uid']},
               u'history_delete_directive': {u'_id': [u'_id']},
               u'invite_mpfs': {u'_id': [u'_id']},
               u'invite_mpfs_codes': {u'_id': [u'_id'],
                                      u'data.meta.referral': [u'data.meta.referral']},
               u'invite_mpfs_contacts': {u'_id': [u'_id']},
               u'invite_mpfs_referrals': {u'_id': [u'_id'], u'ref': [u'ref']},
               u'invite_mpfs_sent': {u'_id': [u'_id'], u'uid': [u'uid']},
               u'link_data': {u'_id': [u'_id'],
                              u'data.dtime': [u'data.dtime'],
                              u'parent': [u'parent'],
                              u'uid': [u'uid']},
               u'misc_data': {u'_id': [u'_id'],
                              u'data.stids.stid': [u'data.stids.stid'],
                              u'hid': [u'hid'],
                              u'parent': [u'parent'],
                              u'uid': [u'uid'],
                              u'uid_data.mt': [u'uid', u'data.mt']},
               u'narod_data': {u'_id': [u'_id'],
                               u'data.stids.stid': [u'data.stids.stid'],
                               u'hid': [u'hid'],
                               u'parent': [u'parent'],
                               u'uid': [u'uid'],
                               u'uid_data.mt': [u'uid', u'data.mt']},
               u'narod_migrate_failed': {u'_id': [u'_id']},
               u'nigori': {u'_id': [u'_id']},
               u'operations': {u'_id': [u'_id'],
                               u'md5': [u'md5'],
                               u'mtime': [u'mtime'],
                               u'state': [u'state'],
                               u'uid': [u'uid'],
                               u'uid_state': [u'uid', u'state']},
               u'password': {u'_id': [u'_id']},
               u'preference': {u'_id': [u'_id']},
               u'public_albums': {u'_id': [u'_id']},
               u'queue': {u'_id': [u'_id'], u'chunk': [u'chunk']},
               u'recount': {u'_id': [u'_id']},
               u'rejected_mids': {u'_id': [u'_id'], u'data.date': [u'data.date']},
               u'search_engine': {u'_id': [u'_id']},
               u'session': {u'_id': [u'_id']},
               u'settings': {u'_id': [u'_id']},
               u'space_subscribes': {u'_id': [u'_id'], u'uid': [u'uid']},
               u'subscription': {u'_id': [u'_id'],
                                 u'ctime': [u'ctime'],
                                 u'uid': [u'uid']},
               u'support_moderation_queue': {u'_id': [u'_id']},
               u'support_mpfs': {u'_id': [u'_id'],
                                 u'data.hash': [u'data.hash'],
                                 u'data.stids': [u'data.stids'],
                                 u'uid': [u'uid']},
               u'sync_index': {u'_id': [u'_id']},
               u'synced_notification': {u'_id': [u'_id']},
               u'tags_images': {u'_id': [u'_id']},
               u'theme': {u'_id': [u'_id']},
               u'trash': {u'_id': [u'_id'],
                          u'data.stids.stid': [u'data.stids.stid'],
                          u'hid': [u'hid'],
                          u'parent': [u'parent'],
                          u'uid': [u'uid']},
               u'typed_url': {u'_id': [u'_id']},
               u'user_data': {u'_id': [u'_id'],
                              u'data.stids.stid': [u'data.stids.stid'],
                              u'hid': [u'hid'],
                              u'parent': [u'parent'],
                              u'uid': [u'uid'],
                              u'uid_data.mt': [u'uid', u'data.mt']},
               u'user_index': {u'_id': [u'_id'], u'shard_key': [u'shard_key']}},
     u'mpfs_changelog': {u'changelog': {u'_id': [u'_id'],
                                        u'dtime': [u'dtime'],
                                        u'uid': [u'uid'],
                                        u'uid_gid_version': [u'uid',
                                                             u'gid',
                                                             u'version'],
                                        u'uid_version': [u'uid', u'version']}},
     u'mpfs_dev': {u'changelog': {u'_id': [u'_id']},
                   u'disk_info': {u'_id': [u'_id']},
                   u'hidden_data': {u'_id': [u'_id']},
                   u'link_data': {u'_id': [u'_id']},
                   u'rejected_mids': {u'_id': [u'_id']},
                   u'trash': {u'_id': [u'_id']},
                   u'user_data': {u'_id': [u'_id']},
                   u'user_index': {u'_id': [u'_id']}},
     u'mpfs_operations': {u'operations': {u'_id': [u'_id'],
                                          u'dtime': [u'dtime'],
                                          u'md5': [u'md5'],
                                          u'state': [u'state'],
                                          u'uid': [u'uid'],
                                          u'uid_state': [u'uid', u'state']}},
     u'mpfs_operatopns': {},
     u'sync': {u'app': {u'_id': [u'_id']},
               u'app_notification': {u'_id': [u'_id']},
               u'app_setting': {u'_id': [u'_id']},
               u'autofill': {u'_id': [u'_id']},
               u'autofill_profile': {u'_id': [u'_id']},
               u'bookmark': {u'_id': [u'_id']},
               u'device_info': {u'_id': [u'_id']},
               u'dictionary': {u'_id': [u'_id']},
               u'experiments': {u'_id': [u'_id']},
               u'extension': {u'_id': [u'_id']},
               u'extension_setting': {u'_id': [u'_id']},
               u'favicon_image': {u'_id': [u'_id']},
               u'favicon_tracking': {u'_id': [u'_id']},
               u'history_delete_directive': {u'_id': [u'_id']},
               u'history_segment': {u'_id': [u'_id']},
               u'managed_user_setting': {u'_id': [u'_id']},
               u'nigori': {u'_id': [u'_id']},
               u'password': {u'_id': [u'_id']},
               u'preference': {u'_id': [u'_id']},
               u'priority_preference': {u'_id': [u'_id']},
               u'queue_sync': {u'_id': [u'_id']},
               u'search_engine': {u'_id': [u'_id']},
               u'session': {u'_id': [u'_id']},
               u'sync_index': {u'_id': [u'_id']},
               u'sync_subscription': {u'_id': [u'_id']},
               u'synced_notification': {u'_id': [u'_id']},
               u'theme': {u'_id': [u'_id']},
               u'typed_url': {u'_id': [u'_id']},
               u'yandex_elements': {u'_id': [u'_id']},
               u'yandex_global_setting': {u'_id': [u'_id']}},
     u'test': {u'settings': {u'_id': [u'_id']}},
     u'video-disk': {u'videoViews': {u'_id': [u'_id']}}}

    """
    result = {}

    # https://st.yandex-team.ru/CHEMODAN-21320
    return result

    try:
        for dbname in connection.database_names():
            if 'test' in dbname:
                continue
            db = connection[dbname]
            result[dbname] = {}
            for collname in db.collection_names(include_system_collections=False):
                collection = db[collname]
                result[dbname][collname] = {}
                for index_name, index_value in collection.index_information().iteritems():
                    # index_value example: {'key': [('uid', 1.0), ('parent', 1.0)], u'v': 1}}
                    index_fields = map(lambda x: x[0], index_value['key'])
                    index_simple_name = '_'.join(index_fields)
                    result[dbname][collname][index_simple_name] = index_fields
    except OperationFailure:
        if error_log:
            error_log.error('could not load indexes for %s, hints will not be used' % connection)
            error_log.error(traceback.format_exc())
    else:
        if log:
            log.info('loaded indexes for %s: %s' % (connection, pprint.pformat(result)))
    return result


class LazyDatabaseObject(object):
    actual_object = None
    actual_class = None
    attribute_class = None
    singleton = False

    def __init__(self, *args, **kwargs):
        self.args = args
        self.kwargs = kwargs

    def get_instance(self):
        """Used by get_actual() to get actual instance."""
        return self.actual_class(*self.args, **self.kwargs)

    def get_actual(self):
        """Use this method to get actual object wrapped by proxy."""
        if not self.actual_object:
            obj = self.get_instance()
            if self.singleton:
                self.__class__.actual_object = obj
            else:
                self.actual_object = obj
        return self.actual_object

    def __getattr__(self, item):
        return self.attribute_class(self, item)

    def __getitem__(self, name):
        return self.__getattr__(name)

    @property
    def name(self):
        return self.__name

# Алиас для обратной совместимости
suppress_exceptions = SuppressExceptions


@contextmanager
def disable_autoreconnect():
    """
    Контекстный менеджер, отключающий автореконнект
    """
    reconnect_count = mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT
    failure_count = mpfs.engine.process.dbctl().AUTORETRY_COUNT_TRANSIENT_FAILURE
    timeout = mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT
    try:
        mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT = 0
        mpfs.engine.process.dbctl().AUTORETRY_COUNT_TRANSIENT_FAILURE = 0
        mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT = 0.3
        yield
    finally:
        mpfs.engine.process.dbctl().AUTORETRY_COUNT_TRANSIENT_FAILURE = failure_count
        mpfs.engine.process.dbctl().AUTORETRY_COUNT_RECONNECT = reconnect_count
        mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT = timeout


@contextmanager
def custom_mongo_timeout(new_timeout):
    """
    Контекстный менеджер, позволяющий менять таймаут на автореконнект
    """
    old_timeout = mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT
    try:
        mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT = new_timeout
        yield
    finally:
        mpfs.engine.process.dbctl().AUTORECONNECT_TIMEOUT = old_timeout


@contextmanager
def manual_route(db_name):
    """
    Контекстный менеджер, позволяющий сходить на указанный шард независимо от всего
    """
    try:
        mpfs.engine.process.dbctl().mapper.set_manual_route(db_name)
        yield
    finally:
        mpfs.engine.process.dbctl().mapper.drop_manual_route()
