# -*- coding: utf-8 -*-
import re


class SqlTemplatedQuery(object):
    select_re = re.compile(r'^\s*select\s.+?\sfrom', re.IGNORECASE | re.MULTILINE | re.DOTALL)
    order_by_re = re.compile(r'order by .+? (asc|desc)( nulls first| nulls last)?', re.IGNORECASE | re.MULTILINE)

    templates = {}

    def __init__(self, select_query_text, count_query_text=None):
        self._select_text = select_query_text
        self._count_text = count_query_text

    def __str__(self):
        # заменяем проценты на два процента кроме тех случаев, когда за ним идет скобка - это уже похоже на template
        percent_escaped_string = self._select_text.replace('%', '%%').replace('%%(', '%(')
        return percent_escaped_string % self.templates

    def __iadd__(self, other):
        if isinstance(other, SqlTemplatedQuery):
            return SqlTemplatedQuery(self._select_text + other._select_text)
        return SqlTemplatedQuery(self._select_text + other)  # возвращаем копию, чтобы не менять текущий объект

    def __add__(self, other):
        if isinstance(other, SqlTemplatedQuery):
            return SqlTemplatedQuery(self._select_text + other._select_text)
        return SqlTemplatedQuery(self._select_text + other)

    def __contains__(self, item):
        return item in self._select_text

    def __mod__(self, other):
        return str(self) % other

    def get_replaced_select_with_count(self):
        """
        Deprecated функция, нужная только для работы класса PostgresCursor. Не используй ее в нормальном коде.
        """
        if self._count_text is not None:
            return SqlTemplatedQuery(self._count_text)

        # вот тут используется хак с регулярками, чтобы просто посчитать количество элементов, которое вернет запрос
        # по хорошему, надо бы, конечно, сделать query builder, но сейчас на такую переделку нету времени :(
        if 'UNION' in self._select_text:
            # только один запрос для мигратора (получение всех папок по uid'у), поэтому написано очень неоптимально!
            return SqlTemplatedQuery('SELECT COUNT(*) FROM (' + self._select_text + ') AS count_subquery')
        return SqlTemplatedQuery(self.select_re.sub('SELECT COUNT(*) FROM', self._select_text, 1))

    def get_removed_order_by(self):
        """
        Deprecated функция, нужная только для работы класса PostgresCursor. Не используй ее в нормальном коде.
        """
        return SqlTemplatedQuery(self.order_by_re.sub(' ', self._select_text, 1))

    def replace_param(self, old_label, new_label):
        return SqlTemplatedQuery(self._select_text.replace(':' + old_label, ':' + new_label))

    @staticmethod
    def join(subqueries):
        if not len(subqueries):
            return None
        query = subqueries[0]
        for subquery in subqueries[1:]:
            query += '\nUNION ALL\n'
            query += subquery
        return query


def sql_template(name, value):
    SqlTemplatedQuery.templates[name] = value % SqlTemplatedQuery.templates


def sql_query(text):
    return SqlTemplatedQuery(text)


sql_template('SELECT_RAW_FILE_FIELDS_TEMPLATE', """
    f.uid, f.modify_uid, f.fid, f.parent_fid, f.id, f.public_hash, f.date_uploaded, f.date_created,
    f.date_modified, f.date_removed, f.date_hidden_data, f.source_uid, f.name, f.visible, f.version, f.public,
    f.symlink, f.short_url, f.media_type, f.mime_type, f.source, f.path_before_remove, f.download_counter, f.folder_url,
    f.custom_properties, f.blocked, f.published, f.custom_setprop_fields, f.date_exif, f.is_live_photo, f.yarovaya_mark,
    f.ext_aesthetics, f.photoslice_album_type, f.albums_exclusions, f.ext_coordinates, f.office_access_state,
    f.office_doc_short_id,
    s.stid, s.digest_stid, s.preview_stid, s.date_origin, s.size, s.av_scan_status, s.video_data, s.storage_id,
    s.md5_sum, s.sha256_sum
""")

sql_template('SELECT_CONVERT_FILE_FIELDS_TEMPLATE', """
    uid::text,
    modify_uid::text,
    replace(fid::text, '-', '') as fid,
    replace(parent_fid::text, '-', '') as parent_fid,
    encode(id, 'hex') as id,
    encode(public_hash, 'escape') as public_hash,
    EXTRACT(EPOCH FROM date_uploaded)::bigint as date_uploaded,
    EXTRACT(EPOCH FROM date_created)::bigint as date_created,
    EXTRACT(EPOCH FROM date_modified)::bigint as date_modified,
    EXTRACT(EPOCH FROM date_removed)::bigint as date_removed,
    EXTRACT(EPOCH FROM date_hidden_data)::bigint as date_hidden_data,
    source_uid::text,
    visible::int,
    public::int,
    blocked::int,
    published::int,
    EXTRACT(EPOCH FROM date_exif)::bigint as date_exif,
    name, version, symlink, short_url, media_type, mime_type, source, path_before_remove, is_live_photo, yarovaya_mark,
    download_counter, folder_url, custom_properties, custom_setprop_fields,
    stid, digest_stid, preview_stid, date_origin, size, av_scan_status, video_data,
    replace(storage_id::text, '-', '') as storage_id,
    replace(md5_sum::text, '-', '') as md5_sum,
    encode(sha256_sum, 'hex') as sha256_sum,
    ext_aesthetics,
    photoslice_album_type,
    albums_exclusions,
    ext_coordinates,
    office_access_state,
    office_doc_short_id
""")

sql_template('SELECT_STORAGE_FILE_FIELDS_TEMPLATE', """
    s.stid, s.digest_stid, s.preview_stid, s.date_origin, s.size, s.av_scan_status, s.video_data,
    replace(s.storage_id::text, '-', '') as storage_id,
    replace(s.md5_sum::text, '-', '') as md5_sum,
    encode(s.sha256_sum, 'hex') as sha256_sum,
    s.width, s.height, s.angle
""")

sql_template('SELECT_FILE_FIELDS_TEMPLATE', """
    f.uid::text,
    f.modify_uid::text,
    replace(f.fid::text, '-', '') as fid,
    replace(f.parent_fid::text, '-', '') as parent_fid,
    encode(f.id, 'hex') as id,
    encode(f.public_hash, 'escape') as public_hash,
    EXTRACT(EPOCH FROM f.date_uploaded)::bigint as date_uploaded,
    EXTRACT(EPOCH FROM f.date_created)::bigint as date_created,
    EXTRACT(EPOCH FROM f.date_modified)::bigint as date_modified,
    EXTRACT(EPOCH FROM f.date_removed)::bigint as date_removed,
    EXTRACT(EPOCH FROM f.date_hidden_data)::bigint as date_hidden_data,
    f.source_uid::text,
    f.visible::int,
    f.public::int,
    f.blocked::int,
    f.published::int,
    EXTRACT(EPOCH FROM f.date_exif)::bigint as date_exif,
    f.name, f.version, f.symlink, f.short_url, f.media_type, f.mime_type, f.source, f.path_before_remove,
    f.download_counter, f.folder_url, f.custom_properties, f.custom_setprop_fields, f.is_live_photo, f.yarovaya_mark,
    f.ext_aesthetics,
    f.photoslice_album_type,
    f.albums_exclusions,
    f.ext_coordinates,
    f.office_access_state,
    f.office_doc_short_id,
    %(SELECT_STORAGE_FILE_FIELDS_TEMPLATE)s
""")

sql_template('SELECT_FOLDER_FIELDS_TEMPLATE', """
    d.uid::text,
    d.modify_uid::text,
    replace(d.fid::text, '-', '') as fid,
    replace(d.parent_fid::text, '-', '') as parent_fid,
    encode(d.id, 'hex') as id,
    encode(d.public_hash, 'escape') as public_hash,
    EXTRACT(EPOCH FROM d.date_uploaded)::bigint as date_uploaded,
    EXTRACT(EPOCH FROM d.date_created)::bigint as date_created,
    EXTRACT(EPOCH FROM d.date_modified)::bigint as date_modified,
    EXTRACT(EPOCH FROM d.date_removed)::bigint as date_removed,
    EXTRACT(EPOCH FROM d.date_hidden_data)::bigint as date_hidden_data,
    d.visible::int,
    d.public::int,
    d.published::int,
    d.blocked::int,
    d.name, d.version, d.symlink, d.short_url, d.files_count, d.folders_count, d.folder_type, d.path_before_remove,
    d.download_counter, d.custom_properties, d.folder_url, d.yarovaya_mark, d.custom_setprop_fields
""")

"""
Запрос на получение файла по пути.
Входные параметры:
    :uid
    :path
"""
SQL_FILE_BY_PATH = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    CAST(:path AS text) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.fid = (SELECT fid FROM code.path_to_fid(:path, :uid))
""")

"""
Запрос на получение файла по пути и версии.
Входные параметры:
    :uid
    :path
    :version
"""
SQL_FILE_BY_PATH_AND_VERSION = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    CAST(:path AS text) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.fid = (SELECT fid FROM code.path_to_fid(:path, :uid)) AND
    f.version = :version
""")

"""
Запрос на получение файлов по списку путей.
Входные параметры:
    :uid
    :paths - список!
"""
SQL_FILES_BY_PATHS = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    q.path as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
    JOIN (SELECT fid, path FROM code.paths_to_fids(:paths, :uid)) q USING (fid)
WHERE f.uid = :uid
""")

"""
Запрос на получение файлов по списку имен и по имени родительской папки.
Входные параметры:
    :uid
    :parent_path
    :names
"""
SQL_FILES_BY_PARENT_PATH_AND_NAMES = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    :parent_path || '/' || f.name as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.parent_fid=(SELECT fid FROM code.path_to_fid(:parent_path, :uid)) AND
    f.name IN :names
""")

"""
Запрос на получение папок по списку имен и по имени родительской папки.
Входные параметры:
    :uid
    :parent_path
    :names
"""
SQL_FOLDERS_BY_PARENT_PATH_AND_NAMES = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    :parent_path || '/' || d.name as path
FROM
    disk.folders d
WHERE
    d.uid = :uid AND
    d.parent_fid=(SELECT fid FROM code.path_to_fid(:parent_path, :uid)) AND
    d.name IN :names
""")

"""
Запрос на получение файлов по списку file_id.
Входные параметры:
    :uid
    :file_ids
    :root_folder
"""
SQL_FILES_BY_FILE_ID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.uid=:uid AND f.id IN :file_ids
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Запрос на получение всех файлов по uid'у.
Входные параметры:
    :uid
    :root_folder
"""
SQL_FILES_BY_UID = SqlTemplatedQuery("""
WITH RECURSIVE folders_subtree AS (
    SELECT
        fid, uid, :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
 UNION
    SELECT
        child.fid, child.uid, parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    folder.path || '/' || f.name as path
FROM
    disk.files f
INNER JOIN folders_subtree folder ON folder.uid = f.uid AND folder.fid = f.parent_fid JOIN disk.storage_files s USING (storage_id)
""", """
WITH RECURSIVE folders_subtree AS (
    SELECT
        fid, uid, :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
    UNION
    SELECT
        child.fid, child.uid, parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    COUNT(*)
FROM
    disk.files f
INNER JOIN folders_subtree folder ON folder.uid = f.uid AND folder.fid = f.parent_fid
""")

"""
Запрос на получение общего количества ресурсов в папке
Входные параметры:
    :uid
    :root_folder
"""
SQL_COUNT_RESOURCES_BY_UID = SqlTemplatedQuery("""
WITH RECURSIVE folders_subtree AS (
    SELECT
        fid, uid, :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
    UNION
    SELECT
        child.fid, child.uid, parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    SUM(count_resounces)
FROM (
    SELECT
        COUNT(*) count_resounces
    FROM
        disk.files f
    INNER JOIN folders_subtree folder ON folder.uid = f.uid AND folder.fid = f.parent_fid
    UNION ALL
    SELECT
        COUNT(*)
    FROM
        folders_subtree) resources
""")


"""
Запрос на получение всех файлов в папке, отсортированных по fid
Входные параметры:
    :uid
    :root_folder
    :limit
"""
SQL_FILES_BY_UID_DETERMINED = SqlTemplatedQuery("""
WITH RECURSIVE
recurse AS (
    SELECT
        dir.fid, dir.parent_fid, :root_folder as path
    FROM
        disk.folders dir
    WHERE
        dir.uid = :uid AND
        dir.fid = (select fid from code.path_to_fid(:root_folder, :uid))
UNION
    SELECT
        child.fid, child.parent_fid, path || '/' || child.name
    FROM
        recurse parent join disk.folders child on parent.fid = child.parent_fid
    WHERE
        child.uid = :uid
)
SELECT f.fid, folder.path || '/' || f.name FROM
disk.files f INNER JOIN recurse folder ON f.uid = :uid AND folder.fid = f.parent_fid
ORDER BY f.fid LIMIT :limit
""")


"""
Запрос на получение всех файлов в папке, отсортированных по path
Входные параметры:
    :uid
    :root_folder
    :limit
"""
SQL_FOLDERS_BY_UID_DETERMINED = SqlTemplatedQuery("""
WITH RECURSIVE
recurse AS (
    SELECT
        dir.fid, dir.parent_fid, :root_folder as path
    FROM
        disk.folders dir
    WHERE
        dir.uid = :uid AND
        dir.fid = (select fid from code.path_to_fid(:root_folder, :uid))
UNION
    SELECT
        child.fid, child.parent_fid, path || '/' || child.name
    FROM
        recurse parent join disk.folders child on parent.fid = child.parent_fid
    WHERE
        child.uid = :uid
)
SELECT recurse.fid, recurse.path FROM recurse ORDER BY recurse.path DESC LIMIT :limit
""")


"""
Запрос на получение всех файлов по uid'у и медиатипу
Входные параметры:
    :uid
    :root_folder
    :media_type_tuple
"""
SQL_FILES_BY_UID_AND_MEDIATYPE = SqlTemplatedQuery("""
WITH RECURSIVE folders_subtree AS (
    SELECT
        fid, uid, :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
 UNION
    SELECT
        child.fid, child.uid, parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    folder.path || '/' || f.name as path
FROM
    disk.files f
INNER JOIN folders_subtree folder ON folder.uid = f.uid AND folder.fid = f.parent_fid JOIN disk.storage_files s USING (storage_id)
WHERE f.media_type IN :media_type_tuple
""")

"""
Запрос на получение файлов, лежащих в папке, по пути родительской папки
Входные параметры:
    :uid
    :parent_path
"""
SQL_FILES_BY_PARENT_PATH = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    :parent_path || '/' || f.name as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid))
""", """
SELECT
    COUNT(*)
FROM
    disk.files f
WHERE
    f.uid = :uid AND
    f.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid))
""")

"""
Запрос на получение файлов, лежащих в папке, по пути родительской папки, пофильтрованный по медиатипу и времени
модификации
Входные параметры:
    :uid
    :parent_path
    :media_type_list
"""
SQL_FILES_BY_PARENT_PATH_AND_MEDIATYPE_AND_MTIME = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    :parent_path || '/' || f.name as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid)) AND
    f.media_type IN :media_type_list
""")

"""
Запрос на получение папки по пути.
Входные параметры:
    :uid
    :path
"""
SQL_FOLDER_BY_PATH = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    CAST(:path AS text) as path
FROM
    disk.folders d
WHERE
    d.uid = :uid AND
    d.fid = (SELECT fid FROM code.path_to_fid(:path, :uid))
""")

"""
Запрос на получение папки по пути и версии.
Входные параметры:
    :uid
    :path
    :version
"""
SQL_FOLDER_BY_PATH_AND_VERSION = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    CAST(:path AS text) as path
FROM
    disk.folders d
WHERE
    d.uid = :uid AND
    d.fid = (SELECT fid FROM code.path_to_fid(:path, :uid)) AND
    d.version = :version
""")

"""
Запрос на получение папкок по списку путей.
Входные параметры:
    :uid
    :paths - список!
"""
SQL_FOLDERS_BY_PATHS = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    q.path as path
FROM
    disk.folders d JOIN (SELECT fid, path FROM code.paths_to_fids(:paths, :uid)) q ON q.fid = d.fid
WHERE d.uid = :uid
""")

"""
Запрос на получение папок по списку file_id.
Входные параметры:
    :uid
    :file_ids
    :root_folder
"""
SQL_FOLDERS_BY_FILE_ID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
    FROM
        disk.folders d
    WHERE d.uid=:uid AND d.id IN :file_ids
) AS s WHERE (s.path LIKE :root_folder || '/%%') or s.path=:root_folder
""")

"""
Запрос на получение всех папок по uid'у.
Входные параметры:
    :uid
    :root_folder
"""
SQL_FOLDERS_BY_UID = SqlTemplatedQuery("""
(
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    '/' as path
FROM
    disk.folders d
WHERE d.uid=:uid AND d.name = ''
)
UNION
(
WITH RECURSIVE folders_subtree AS (
    SELECT
        *,
        :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
 UNION
    SELECT
        child.*,
        parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    path
FROM
    folders_subtree d
)
""", """
WITH RECURSIVE folders_subtree AS (
    SELECT
        *,
        :root_folder as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:root_folder, :uid))
 UNION
    SELECT
        child.*,
        parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    CASE WHEN COUNT(*) = 0 THEN 0 ELSE COUNT(*) + 1 END
FROM
    folders_subtree d
""")

"""
Запрос на получение папок, лежащих в папке, по пути родительской папки
Входные параметры:
    :uid
    :parent_path
"""
SQL_FOLDERS_BY_PARENT_PATH = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    :parent_path || '/' || d.name as path
FROM
    disk.folders d
WHERE
    d.uid = :uid AND
    d.parent_fid = (SELECT fid FROM code.path_to_fid(:parent_path, :uid))
""")

"""
Запрос на получение последних файлов для поддерева
Входные параметры:
    :uid
    :path
    :limit
"""
SQL_LAST_FILES_FOR_SUBTREE = SqlTemplatedQuery("""
SELECT
    %(SELECT_CONVERT_FILE_FIELDS_TEMPLATE)s,
    path
FROM
(
WITH RECURSIVE folders_subtree AS (
    SELECT
        fid, uid, CAST(:path AS text) as path
    FROM
        disk.folders
    WHERE
        uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
 UNION
    SELECT
        child.fid, child.uid, parent.path || '/' || child.name as path
    FROM
        disk.folders child INNER JOIN folders_subtree parent ON parent.fid = child.parent_fid
    WHERE child.uid=:uid
)
SELECT
    %(SELECT_RAW_FILE_FIELDS_TEMPLATE)s,
    folder.path || '/' || f.name as path
FROM
    disk.files f INNER JOIN folders_subtree folder ON folder.uid = f.uid AND folder.fid = f.parent_fid JOIN disk.storage_files s USING (storage_id)
ORDER BY
	date_modified DESC
LIMIT :limit
) AS result
""")

"""
Запрос на получение последних файлов.
Работает быстрей чем SQL_LAST_FILES_FOR_SUBTREE
Используется костыль:
 * Cначала сортируем все времена по date_modified
 * Оставляем :service_limit записей
 * Для них генерируем путь
 * Дофильтровываем по пути
Входные параметры:
    :uid
    :limit
    :service_limit
"""
SQL_LAST_FILES_BY_UID = SqlTemplatedQuery("""
SELECT
    %(SELECT_CONVERT_FILE_FIELDS_TEMPLATE)s,
    path
FROM (
    SELECT
        *,
        (SELECT path FROM code.fid_to_path(fid, uid)) as path
    FROM disk.files
    WHERE uid = :uid
    ORDER BY date_modified DESC
    LIMIT :service_limit
) sub_q
JOIN disk.storage_files USING (storage_id)
WHERE path ~ '^/(disk|photounlim)/'
ORDER BY sub_q.date_modified DESC
LIMIT :limit
""")

"""
Запрос на получение файлов по хиду. Нужно для поиска хардлинка. Ищет целиком файл и двух таблицах, возвращает все файлы.
Входные параметры:
    :hid
    :root_folder
"""
SQL_FILES_BY_HID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.storage_id = :hid
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")


"""
Запрос на получение файлов по хиду у определенного пользователя по определенной папке
Входные параметры:
    :uid
    :hid
    :root_folder
"""
SQL_FILES_BY_UID_HID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.uid = :uid AND f.storage_id = :hid
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")


"""
Запрос на получение файлов по office_doc_short_id у определенного пользователя по определенной папке
Входные параметры:
    :uid
    :office_doc_short_id
    :root_folder
"""
SQL_FILES_BY_UID_OFFICE_DOC_SHORT_ID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE uid = :uid AND office_doc_short_id = :office_doc_short_id
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")


"""
Запрос на получение файлов по стиду. Ищет целиком файл и двух таблицах, возвращает все файлы.
Входные параметры:
    :stid
"""
SQL_FILES_BY_STID = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE s.stid = :stid OR s.digest_stid = :stid OR s.preview_stid = :stid
""")

"""
Удаление файла по пути.
Входные параметры:
    :uid
    :path
"""
SQL_DELETE_FILE_BY_PATH = """
DELETE FROM disk.files WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Удаление папки по пути.
Входные параметры:
    :uid
    :path
"""
SQL_DELETE_FOLDER_BY_PATH = """
DELETE FROM disk.folders WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Удаление файлов по списку путей.
Входные параметры:
    :uid
    :paths
"""
SQL_DELETE_FILES_BY_PATHS = """
DELETE FROM disk.files WHERE uid=:uid AND fid IN (SELECT fid FROM code.paths_to_fids(:paths, :uid))
"""

"""
Удаление папок по списку путей.
Входные параметры:
    :uid
    :paths
"""
SQL_DELETE_FOLDERS_BY_PATHS = """
DELETE FROM disk.folders WHERE uid=:uid AND fid IN (SELECT fid FROM code.paths_to_fids(:paths, :uid))
"""

"""
Удаление файла по пути и версии.
Входные параметры:
    :uid
    :path
    :version
"""
SQL_DELETE_FILE_BY_PATH_AND_VERSION = """
DELETE FROM disk.files WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid)) AND version=:version
"""

"""
Удаление папки по пути и версии.
Входные параметры:
    :uid
    :path
    :version
"""
SQL_DELETE_FOLDER_BY_PATH_AND_VERSION = """
DELETE FROM disk.folders WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid)) AND version=:version
"""

"""
Удаление всех файлов пользователя (используется только для тестов, не удаляет записи из storage_files).
Входные параметры:
    :uid
"""
SQL_DELETE_FILES_BY_UID = """
DELETE FROM disk.files WHERE uid=:uid
"""

"""
Удаление всех папок пользователя (используется только для тестов).
Входные параметры:
    :uid
"""
SQL_DELETE_FOLDERS_BY_UID = """
DELETE FROM disk.folders WHERE uid=:uid
"""

"""
Удаление file_id у папки или файла по пути.
Входные параметры:
    :uid
    :path
"""
SQL_REMOVE_FILE_ID_BY_PATH = """
UPDATE disk.files SET id=NULL WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid));
UPDATE disk.folders SET id=NULL WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Удаление файла по fid'у.
Входные параметры:
    :uid
    :fid
"""
SQL_DELETE_FILE_BY_FID = """
DELETE FROM disk.files WHERE uid=:uid AND fid=:fid
"""

"""
Удаление бинарей файла по storage_id.
Входные параметры:
    :storage_id
"""
SQL_DELETE_STORAGE_FILE_BY_STORAGE_ID = """
DELETE FROM disk.storage_files WHERE storage_id=:storage_id
"""

"""
Найти бинарь файла по storage_id.
Входные параметры:
    :storage_id
"""
SQL_FIND_STORAGE_FILE_BY_STORAGE_ID = SqlTemplatedQuery("""
SELECT
    %(SELECT_STORAGE_FILE_FIELDS_TEMPLATE)s
FROM disk.storage_files AS s WHERE storage_id=:storage_id
""")

"""
Удаление папки по fid'у.
Входные параметры:
    :uid
    :fid
"""
SQL_DELETE_FOLDER_BY_FID = """
DELETE FROM disk.folders WHERE uid=:uid AND fid=:fid
"""

"""
Получение чанка снепшота из таблицы папок.
Входные параметры:
    :uid
    :file_id
    :last_ids
    :root_folder
"""
SQL_SNAPSHOT_FOLDERS_CHUNK = SqlTemplatedQuery("""
SELECT * FROM
    (SELECT
        %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
    FROM
        disk.folders d
    WHERE
        d.uid=:uid AND
        d.id >= (:file_id)::bytea AND
        NOT (d.fid = ANY ((:last_ids)::uuid[])) AND
        d.date_hidden_data IS NULL
    ORDER BY d.id ASC) as s
WHERE
    s.path LIKE :root_folder
LIMIT :limit""")

"""
Получение чанка снепшота из таблицы файлов.
Входные параметры:
    :uid
    :file_id
    :last_ids
    :root_folder
"""
SQL_SNAPSHOT_FILES_CHUNK = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE
        f.uid=:uid AND
        f.id >= (:file_id)::bytea AND
        NOT (f.fid = ANY ((:last_ids)::uuid[])) AND
        f.date_hidden_data IS NULL
    ORDER BY f.id ASC) as s
WHERE
    s.path LIKE :root_folder
LIMIT :limit""")

"""
Запрос на изменение числового значния поля `data`.
"""
SQL_DISK_INFO_INCEREMNT_DATA = """
UPDATE disk.disk_info SET data = to_jsonb(data::text::bigint + :value) WHERE id=:id AND uid=:uid
"""

"""
Запрос на изменение числового значния поля `data.bytes`.
"""
SQL_DISK_INFO_INCEREMNT_DATA_BYTES = """
UPDATE disk.disk_info SET data = data || concat('{"bytes":', coalesce(data->>'bytes', '0')::bigint + :value, '}')::jsonb WHERE id=:id AND uid=:uid
"""

"""
Запрос на изменение времени лока
"""
SQL_FILESYSTEM_LOCK_UPDATE = """
UPDATE disk.filesystem_locks SET dtime = :dtime WHERE id=:id AND uid=:uid AND path=:path RETURNING *
"""

"""
Запрос на изменение времени лока
"""
SQL_FILESYSTEM_LOCK_WITH_DATA_UPDATE = """
UPDATE disk.filesystem_locks SET dtime = :dtime, data = :data WHERE id=:id AND uid=:uid AND path=:path RETURNING *
"""

"""
Запрос на обнуление поля file_id у файла или папки - нужен для того, чтобы в транзакции делать доставание ресурса,
обнуление file_id, вставку ресурса, удаление исходного при включенном констреинте на уникальность file_id.
"""
SQL_SET_FILE_ID_TO_NULL_BY_PATH = """
UPDATE disk.files SET id=NULL WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid));
UPDATE disk.folders SET id=NULL WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid));
"""

"""
Запрос на установку поля file_id у файла.
"""
SQL_SET_FILE_ID_FOR_FILE_BY_PATH = """
UPDATE disk.files SET id=:file_id WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Запрос на инкремент счетчика загрузки файла
"""
SQL_FILES_INCREMENT_DOWNLOAD_COUNTER = """
UPDATE disk.files SET download_counter = download_counter + 1 WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Запрос на инкремент счетчика загрузки папки
"""
SQL_FOLDERS_INCREMENT_DOWNLOAD_COUNTER = """
UPDATE disk.folders SET download_counter = download_counter + 1 WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""

"""
Обновление офисных полей disk.files из disk.link_data
"""
SQL_FILES_UPDATE_OFFICE_FIELDS = """
UPDATE
  disk.files f
SET
  office_access_state=la.office_access_state,
  office_doc_short_id=la.office_doc_short_id
FROM
  disk.link_data la
WHERE
  f.uid=:uid AND f.id=:file_id AND
  la.uid=:uid AND la.file_id=:file_id
"""


"""
Запрос, в который можно подставить подзапрос с select'ом и который вернет подходящие файлы.
"""
SQL_FILES_WITH_SELECT_TEMPLATE = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
    JOIN (%s) q USING (fid)
WHERE f.uid = :uid
""")

"""
Запрос, в который можно подставить подзапрос с select'ом и который вернет подходящие папки.
"""
SQL_FOLDERS_WITH_SELECT_TEMPLATE = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
FROM
    disk.folders d JOIN (%s) q ON q.fid = d.fid
WHERE d.uid = :uid
""")

"""
Ищет файлы, содержащие переданный параметр в имени.
Входные параметры:
    :uid
    :name
    :root_folder
"""
SQL_SEARCH_FILES_BY_NAME = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.uid = :uid AND f.name LIKE '%%' || :name || '%%'
) AS s WHERE s.path LIKE :root_folder || '/%%'

""")

"""
Ищет папки, содержащие переданный параметр в имени.
Входные параметры:
    :uid
    :name
    :root_folder
"""
SQL_SEARCH_FOLDERS_BY_NAME = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
    FROM
        disk.folders d
    WHERE d.uid=:uid AND d.name LIKE '%%' || :name || '%%'
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Ищет пользователя по yateam_uid.
Входные параметры:
    :yateam_uid
"""
SQL_SEARCH_USERS_BY_YATEAM_UID = """
SELECT uid, version, blocked, deleted, user_type, reg_time, locale, shard_key, b2b_key, pdd, yateam_uid, collections
FROM disk.user_index WHERE yateam_uid=:yateam_uid
"""

"""
Ищет пользователя по b2b_key.
Входные параметры:
    :b2b_key
"""
SQL_SEARCH_USERS_BY_B2B_KEY = """
SELECT uid, version, blocked, deleted, user_type, reg_time, locale, shard_key, b2b_key, pdd, yateam_uid, collections
FROM disk.user_index WHERE b2b_key=:b2b_key
"""

"""
Ищет пользователей, у которых есть yateam_uid.
"""
SQL_SEARCH_YATEAM_USERS = """
SELECT uid, version, blocked, deleted, user_type, reg_time, locale, shard_key, b2b_key, pdd, yateam_uid, collections
FROM disk.user_index WHERE yateam_uid IS NOT NULL
"""

"""
Меняет флаг hancom_enabled
Входные параметры:
    :hancom_status
"""
SQL_CHANGE_HANCOM_STATUS = """
UPDATE disk.user_index SET hancom_enabled=:hancom_status WHERE uid=:uid
"""

"""
Меняет флаг unlimited_autouploading_enabled
Входные параметры:
    :unlimited_autouploading_status
"""
SQL_CHANGE_UNLIMITED_AUTOUPLOADING_STATUS = """
UPDATE disk.user_index SET unlimited_autouploading_enabled=:unlimited_autouploading_status WHERE uid=:uid
"""

"""
Меняет флаг unlimited_video_autouploading_allowed
Входные параметры:
    :unlimited_video_autoupload_allowed
    :uid
"""
SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_ALLOWED = """
UPDATE disk.user_index
SET unlimited_video_autouploading_allowed=:unlimited_video_autoupload_allowed
WHERE uid=:uid
"""

"""
Меняет флаг unlimited_video_autoupload_enabled и колонку unlimited_video_autoupload_reason
Входные параметры:
    :unlimited_video_autoupload_status
    :unlimited_video_autoupload_reason_text
    :uid
"""
SQL_CHANGE_UNLIMITED_VIDEO_AUTOUPLOADING_STATUS = """
UPDATE disk.user_index
SET unlimited_video_autouploading_enabled=:unlimited_video_autoupload_status,
    unlimited_video_autouploading_reason=:unlimited_video_autoupload_reason_text
WHERE uid=:uid
"""

"""
Меняет флаг unlimited_photo_autouploading_allowed
Входные параметры:
    :unlimited_photo_autoupload_allowed
    :uid
"""
SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_ALLOWED = """
UPDATE disk.user_index
SET unlimited_photo_autouploading_allowed=:unlimited_photo_autoupload_allowed
WHERE uid=:uid
"""

"""
Меняет флаг unlimited_photo_autoupload_enabled
Входные параметры:
    :unlimited_photo_autoupload_status
    :uid
"""
SQL_CHANGE_UNLIMITED_PHOTO_AUTOUPLOADING_STATUS = """
UPDATE disk.user_index
SET unlimited_photo_autouploading_enabled=:unlimited_photo_autoupload_status
WHERE uid=:uid
"""

"""
Меняет флаг only_office_enabled
Входные параметры:
    :only_office_enabled
    :uid
"""
SQL_CHANGE_ONLY_OFFICE_STATUS = """
UPDATE disk.user_index
SET only_office_enabled=:only_office_enabled
WHERE uid=:uid
"""

"""
Меняет флаг online_editor_enabled
Входные параметры:
    :only_office_enabled
    :uid
"""
SQL_CHANGE_ONLINE_EDITOR_STATUS = """
UPDATE disk.user_index
SET online_editor_enabled=:online_editor_enabled
WHERE uid=:uid
"""

"""
Меняет флаг advertising_enabled
Входные параметры:
    :advertising_enabled
    :uid
"""
SQL_CHANGE_ADVERTISING_STATUS = """
UPDATE disk.user_index
SET advertising_enabled=:advertising_enabled
WHERE uid=:uid
"""

"""
Меняет флаг public_settings_enabled
Входные параметры:
    :public_settings_enabled
    :uid
"""
SQL_CHANGE_PUBLIC_SETTINGS_STATUS = """
UPDATE disk.user_index
SET public_settings_enabled=:public_settings_enabled
WHERE uid=:uid
"""

"""
возвращает колонки unlimited_video_autoupload_enabled и unlimited_video_autoupload_reason и unlimited_photo_autoupload_enabled и unlimited_video_autouploading_allowed и unlimited_photo_autouploading_allowed
Входные параметры:
    :uid
"""
SQL_GET_UNLIMITED_AUTOUPLOADING_STATUS = """
SELECT  unlimited_video_autouploading_enabled, unlimited_video_autouploading_reason, unlimited_photo_autouploading_enabled, unlimited_video_autouploading_allowed, unlimited_photo_autouploading_allowed
FROM disk.user_index
WHERE uid=:uid
"""

"""
возвращает колонку faces_indexing_state
Входные параметры:
    :uid
"""
SQL_GET_FACES_INDEXING_STATE = """
SELECT faces_indexing_state
FROM disk.user_index
WHERE uid=:uid
"""

"""
Ищет b2b организацию по id.
Входные параметры:
    :id
"""
SQL_SEARCH_ORGANIZATION_BY_ID = """
SELECT id, is_paid, quota_limit, quota_free, quota_used_by_disk
FROM disk.organizations WHERE id=:id
"""

"""
Сохраняет использованное b2b организацией место в диске.
Входные параметры:
    :id
    :quota_limit
    :quota_free
    :quota_used_by_disk
"""
SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_USED = """
INSERT INTO disk.organizations (id, quota_limit, quota_free, quota_used_by_disk) VALUES(:id, :quota_limit, :quota_free, :quota_used_by_disk)
ON CONFLICT (id) DO UPDATE SET quota_limit=:quota_limit, quota_free=:quota_free, quota_used_by_disk=:quota_used_by_disk
"""

"""
Сохраняет квоту b2b организации и признак платности.
Входные параметры:
    :id
    :quota_limit
    :quota_free
    :is_paid
"""
SQL_UPSERT_ORGANIZATION_QUOTA_LIMITS_AND_PAID = """
INSERT INTO disk.organizations (id, quota_limit, quota_free, is_paid) VALUES(:id, :quota_limit, :quota_free, :is_paid)
ON CONFLICT (id) DO UPDATE SET quota_limit=:quota_limit, quota_free=:quota_free, is_paid=:is_paid
"""

"""
Ищем стиды по списку (для чистки) в таблице misc_data.
Входные параметры:
    :stids
"""
SQL_STIDS_BY_STIDS_IN_MISC_DATA = """
SELECT file_stid,digest_stid,preview_stid FROM disk.misc_data WHERE file_stid IN :stids OR digest_stid IN :stids OR preview_stid IN :stids
"""

"""
Ищем стиды по списку (для чистки) в таблице storage_files.
Входные параметры:
    :stids
"""
SQL_STIDS_BY_STIDS_IN_FILES = """
SELECT stid,digest_stid,preview_stid FROM disk.storage_files WHERE stid IN :stids OR digest_stid IN :stids OR preview_stid IN :stids
"""

"""
Ищем стиды по списку (для чистки) в таблице storage_duplicates_files.
Входные параметры:
:stids
"""
SQL_STIDS_BY_STIDS_IN_DUPLICATED_STORAGE_FILES = """
SELECT storage_id, stid FROM disk.duplicated_storage_files WHERE stid IN :stids
"""

"""
Запрос на получение файлов по списку стидов. Ищет целиком файл и двух таблицах, возвращает все файлы.
Входные параметры:
    :stids
    :root_folder
"""
SQL_FILES_BY_STIDS = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE s.stid IN :stids OR s.digest_stid IN :stids OR s.preview_stid IN :stids
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Запрос на получение данных из misc_data по списку стидов.
Входные параметры:
    :stids
"""
SQL_FILES_BY_STIDS_IN_MISC_DATA = """
SELECT
    id, uid, path, type, parent, version, zdata, file_id, file_id_zipped, hid, mimetype, mediatype, visible, size,
    file_stid, digest_stid, preview_stid, date_modified, date_uploaded, date_exif
FROM
    disk.misc_data
WHERE file_stid IN :stids OR digest_stid IN :stids OR preview_stid IN :stids
"""

"""
Ищем файлы для чистки по размеру и времени удаления в hidden.
Входные параметры:
    :dtime_lte
    :size_gte
    :limit
"""
SQL_FILES_BY_SIZE_AND_DTIME = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE f.date_hidden_data <= :dtime_lte
LIMIT :limit
""")

"""
Ищем файлы для чистки по размеру и времени удаления в hidden. При этом передаем непустой список uid'ов, по которым
не нужно искать.
Входные параметры:
    :dtime_lte
    :size_gte
    :uids_nin
    :limit
"""
SQL_FILES_BY_SIZE_AND_DTIME_AND_NOT_IN_UIDS = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE f.date_hidden_data <= :dtime_lte AND f.uid NOT IN :uids_nin
LIMIT :limit
""")

"""
Ищем дубликаты, которые больше не связанны с storage_file.
"""
SQL_HANGING_STORAGE_FILE_DUPLICATES = SqlTemplatedQuery("""
SELECT dsf.storage_id, dsf.stid
FROM disk.duplicated_storage_files dsf
    LEFT JOIN disk.storage_files sf ON dsf.storage_id = sf.storage_id
WHERE sf.storage_id ISNULL
""")

"""
Удаляем дубликаты по storage_id.
Входные параметры:
    :storage_ids_in
"""
SQL_DELETE_STORAGE_FILE_DUPLICATES_BY_STORAGE_IDS = SqlTemplatedQuery("""
DELETE FROM disk.duplicated_storage_files WHERE storage_id in :storage_ids_in
""")

"""
Удаляем файлы в таблице files по списку идентификаторов.
Входные параметры:
    :uid_id_pairs
"""
SQL_REMOVE_FILES_BY_UID_FID = """
DELETE FROM disk.files WHERE (uid, fid) IN :uid_id_pairs
"""

"""
Ищем файлы в таблице files по списку идентификаторов.
Входные параметры:
    :uid_id_pairs
"""
SQL_FIND_FILES_BY_UID_FID = """
SELECT * FROM disk.files WHERE (uid, fid) IN :uid_id_pairs
"""

"""
Удаляем файлы в таблице storage_files, которые больше не ссылаются ни на какие файлы из таблицы files version_data.
Входные параметры:
    :storage_ids - хиды удаляемых файлов
"""
SQL_REMOVE_HANGING_STORAGE_FILES = """
DELETE FROM disk.storage_files WHERE storage_id IN (
  SELECT
    s.storage_id
  FROM
    disk.storage_files as s
    LEFT OUTER JOIN disk.files as f ON s.storage_id = f.storage_id
    LEFT OUTER JOIN disk.version_data as v ON s.storage_id = v.storage_id
  WHERE
    s.storage_id IN :storage_ids AND
    f.storage_id IS NULL AND
    v.storage_id IS NULL
) RETURNING replace(storage_id::text, '-', '') as storage_id
"""

"""
Запрос на получение всех файлов по коллекции.
Входные параметры:
    :root_folder
"""
SQL_FILES_BY_COLLECTION = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Запрос на получение всех папок по коллекции.
Входные параметры:
    :root_folder
"""
SQL_FOLDERS_BY_COLLECTION = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
    FROM
        disk.folders d
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Запрос на получение всех файлов по коллекции, у которых mtime больше чем `mtime_gte`.
Входные параметры:
    :uid
    :mtime_gte
    :root_folder
"""
SQL_FILES_BY_UID_MTIME_COLLECTION = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE uid = :uid AND date_modified >= :mtime_gte
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Получаем записи кеша последних файлов.
Входные параметры:
    :uid
    :limit
"""
SQL_GET_LAST_FILES_CACHE = """
SELECT * FROM disk.last_files_cache WHERE uid = :uid ORDER BY file_date_modified DESC LIMIT :limit
"""

"""
Получаем записи кеша по id группы и uid.
Входные параметры:
    :gid
    :uid
"""
SQL_GET_CACHE_BY_GID_AND_UID = """
SELECT * FROM disk.last_files_cache WHERE gid = :gid AND uid = :uid
"""

"""
Удаляем записи кеша по id группы.
Входные параметры:
    :gid
"""
SQL_DELETE_CACHE_BY_GID = """
DELETE FROM disk.last_files_cache WHERE gid = :gid
"""

"""
Удаляем записи кеша по id группы и с игнорированием некоторых uid-ов.
Входные параметры:
    :gid
    :ignore_uids
"""
SQL_DELETE_CACHE_BY_GID_AND_IGNORE_UIDS = """
DELETE FROM disk.last_files_cache WHERE gid = :gid AND uid NOT IN :ignore_uids
"""

"""
Удаляем записи кеша по id группы и uid.
Входные параметры:
    :gid
    :uid
"""
SQL_DELETE_CACHE_BY_GID_AND_UID = """
DELETE FROM disk.last_files_cache WHERE gid = :gid AND uid = :uid
"""

"""
Удаляем записи кеша по списку id записей
Входные параметры:
    :ids
"""
SQL_DELETE_CACHE_BY_IDS = """
DELETE FROM disk.last_files_cache WHERE id IN :ids
"""

"""
Запрос на получение одного файла по hid'у.
Входные параметры:
    :hid
"""
SQL_ONE_FILE_BY_HID = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE f.storage_id = :hid
LIMIT 1
""")

"""
Запрос на получение файлов по uid'у и hid'ам.
Входные параметры:
    :uid
    :hids
    :limit
    :offset
"""
SQL_FIND_BY_UID_AND_HIDS = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.storage_id in :hids AND f.uid = :uid
    LIMIT :limit OFFSET :offset
) AS s WHERE s.path LIKE :root_folder || '/%%'
""")

"""
Запрос на получение одного файла по hid'у у пользователя.
Нужен для восстановления битых файлов
Входные параметры:
    :uid
    :hid
"""
SQL_ONE_FILE_BY_UID_AND_HID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.storage_id = :hid AND f.uid = :uid
) AS s WHERE s.path LIKE '/disk/%'
LIMIT 1
""")

"""
Запрос на получение одного файла по hid'у для другого пользователя.
Нужен для восстановления битых файлов
Входные параметры:
    :uid
    :hid
"""
SQL_ONE_FILE_BY_HID_FOR_ANOTHER_UID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.storage_id = :hid AND f.uid != :uid
) AS s WHERE s.path LIKE '/disk/%'
LIMIT 1
""")

"""
Посчитать, сколько у пользователя файлов с определенным hid'ом в :root_folder.
Входные параметры:
    :uid
    :hid
    :root_folder
"""
SQL_FILE_COUNT_BY_HID_FOR_UID = SqlTemplatedQuery("""
SELECT COUNT(*) FROM (
    SELECT
        f.uid, f.storage_id,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f
    WHERE f.storage_id = :hid AND f.uid = :uid
) AS s WHERE s.path LIKE :root_folder || '/%'
""")

"""
Запрос на поиск файлов по hid'у в :root_folder.
Входные параметры:
    :uid
    :hid
    :root_folder
"""
SQL_FILES_BY_HID_FOR_UID = SqlTemplatedQuery("""
SELECT * FROM (
    SELECT
        %(SELECT_FILE_FIELDS_TEMPLATE)s,
        (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
    FROM
        disk.files f JOIN disk.storage_files s USING (storage_id)
    WHERE f.storage_id = :hid AND f.uid = :uid
) AS s WHERE s.path LIKE :root_folder || '/%'
""")

SQL_DOES_FILE_EXIST_BY_UID_HID_BASE = """
SELECT 1 as "found"
FROM disk.files
WHERE uid = :uid AND storage_id = :storage_id AND %s
LIMIT 1
"""

SQL_DOES_FILE_EXIST_BY_UID_HID = SQL_DOES_FILE_EXIST_BY_UID_HID_BASE % 'is_live_photo IS NOT true'
SQL_DOES_FILE_EXIST_BY_UID_HID_LIVE_PHOTO = SQL_DOES_FILE_EXIST_BY_UID_HID_BASE % 'is_live_photo'

"""
Запрос на удаление заэкспайренных записей в changelog'е.
Входные параметры:
    :batch_size
"""
SQL_REMOVE_EXPIRED_ENTRIES_CHANGELOG = """
DELETE FROM disk.changelog WHERE id IN (
    SELECT
        id
    FROM
        disk.changelog
    WHERE
        dtime IS NOT NULL AND dtime <= NOW() - INTERVAL '1 second' * 2592000 - INTERVAL '3 hours'
    LIMIT :batch_size
)
"""

"""
Запрос на удаление заэкспайренных записей в filesystem_lock'ах.
Входные параметры:
    :batch_size
"""
SQL_REMOVE_EXPIRED_ENTRIES_FILESYSTEM_LOCKS = """
DELETE FROM disk.filesystem_locks WHERE id IN (
    SELECT
        id
    FROM
        disk.filesystem_locks
    WHERE
        dtime IS NOT NULL AND dtime <= NOW() - INTERVAL '1 second' * 3600
    LIMIT :batch_size
)
"""

"""
Запрос на удаление заэкспайренных записей в operations.
Входные параметры:
    :batch_size
"""
SQL_REMOVE_EXPIRED_ENTRIES_OPERATIONS = """
DELETE FROM disk.operations WHERE id IN (
    SELECT
        id
    FROM
        disk.operations
    WHERE
        dtime IS NOT NULL AND dtime <= NOW() - INTERVAL '1 second' * 2592000
    LIMIT :batch_size
)
"""

"""
Запрос на получение операций за период.
"""
SQL_OPERATIONS_GET_BY_DATE_PERIOD_AND_STATE = """
SELECT
    *
FROM
    disk.operations
WHERE
    dtime IS NOT NULL AND
    dtime >= :from_dt AND
    dtime <= :to_dt AND
    state IN :states
"""

"""
Запрос на получение количества операций по типу, статусу.
Выбирает только среди операций, которые обновлялись в течение суток.
"""
SQL_OPERATIONS_COUNT_BY_STATE_TYPES_WITH_LIMIT = """
SELECT
    COUNT(*)
FROM
    disk.operations
WHERE
    uid = :uid AND
    dtime >= NOW() - INTERVAL '1 day' AND
    state IN :states AND
    type IN :types
LIMIT :limit
"""

"""
Запрос обновление dtime операции.
"""
SQL_OPERATION_UPDATE_DTIME = """
UPDATE
    disk.operations
SET
    dtime = :dtime
WHERE
    id = :id AND
    uid = :uid
"""

SQL_GET_FILE_STID_BY_STORAGE_ID = 'SELECT stid FROM disk.storage_files WHERE storage_id = :storage_id'

"""
Создание связи между основным файлом в таблице files и дополнительным файлом в storage_files
"""
SQL_INSERT_ADDITIONAL_STORAGE_FILE_LINK = """
INSERT INTO disk.additional_file_links (uid,main_file_fid,additional_file_fid,type)
VALUES (:uid,:main_file_fid,:additional_file_fid,:type)
"""

"""
Удалить связь между основным файлом в таблице files и дополнительным файлом в storage_files
"""
SQL_REMOVE_ADDITIONAL_STORAGE_FILE_LINK = """
DELETE FROM disk.additional_file_links
WHERE
    uid=:uid AND
    main_file_fid=(SELECT fid FROM code.path_to_fid(:main_file_path, :uid)) AND
    additional_file_fid=(SELECT fid FROM code.path_to_fid(:additional_file_path, :uid)) AND
    type=:type
"""

"""
Удаление связи между основным файлом в таблице files и дополнительным файлом в storage_files
"""
SQL_UNSET_LIVE_PHOTO_FILE_FLAG = """
UPDATE disk.files SET is_live_photo=NULL WHERE uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:photo_path, :uid))
"""


"""
Удаление связи между основным файлом в таблице files и дополнительным файлом в storage_files
Входные параметры:
    :uid_id_pairs
"""
SQL_UNSET_LIVE_PHOTO_FILE_FLAGS = """
UPDATE disk.files SET is_live_photo=NULL WHERE (uid, fid) IN :uid_id_pairs
"""


SQL_GET_ADDITIONAL_FILE_BY_MAIN_FILE_PATH_AND_TYPE = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.fid = (SELECT a.additional_file_fid FROM code.path_to_fid(:path, :uid) AS p JOIN disk.additional_file_links AS a ON p.fid=a.main_file_fid WHERE a.type=:type)
""")

"""
Запрос на получение файла по пути. Лочит файл для обновления.
Входные параметры:
    :uid
    :path
"""
SQL_SELECT_FILE_BY_PATH_FOR_UPDATE = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    CAST(:path AS text) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid = :uid AND
    f.fid = (SELECT fid FROM code.path_to_fid(:path, :uid))
FOR UPDATE
""")

"""
Создание связи между основным файлом в таблице files и дополнительным файлом в storage_files
"""
SQL_SET_LIVE_PHOTO_FILE_FLAG = """
UPDATE disk.files SET is_live_photo=true WHERE uid=:uid AND fid=:fid
"""

"""
Получить список пар вида (uid, fid) от дополнительных файлов по списку (uid, fid) от их соответствующих основных частей
"""
SQL_GET_ADDITIONAL_FILE_UID_FID_BY_MAIN_UID_FID = SqlTemplatedQuery("""
SELECT
    uid, additional_file_fid
FROM
    disk.additional_file_links
WHERE (uid, main_file_fid) IN :uid_id_pairs
""")

"""
Удалить линки дополнительных файлов по списку пар вида (uid, fid), где fid - это fid дополнительного файла
"""
SQL_REMOVE_ADDITIONAL_FILE_LINK_BY_UID_FID = SqlTemplatedQuery("""
DELETE FROM
    disk.additional_file_links
WHERE (uid, additional_file_fid) IN :uid_id_pairs
""")

"""
Ищем файлы по списку идентификаторов вида (uid, fid).
Входные параметры:
    :uid_id_pairs
"""
SQL_GET_FILES_BY_UID_FID = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    (f.uid, f.fid) IN :uid_id_pairs
""")


"""
Запрос для обновления информации о превьюшке
"""
SQL_UPDATE_PREVIEW = '''
UPDATE disk.storage_files SET
    %s
WHERE preview_stid = :old_preview_stid
RETURNING 1
'''

"""
Запрос создающий запись в user_index
"""
SQL_UPSERT_USER_INDEX = '''
    INSERT INTO disk.user_index AS ui (
        uid,
        version,
        blocked,
        deleted,
        user_type,
        reg_time,
        locale,
        shard_key,
        b2b_key,
        pdd,
        yateam_uid,
        collections,
        is_mailish,
        hancom_enabled,
        unlimited_autouploading_enabled,
        is_reindexed_for_quick_move,
        office_selection_strategy,
        faces_indexing_state,
        faces_indexing_state_time,
        default_product_id
    )
    VALUES (
        :uid,
        :version,
        :blocked,
        :deleted,
        :user_type,
        :reg_time,
        :locale,
        :shard_key,
        :b2b_key,
        :pdd,
        :yateam_uid,
        :collections,
        :is_mailish,
        :hancom_enabled,
        :unlimited_autouploading_enabled,
        :is_reindexed_for_quick_move,
        :office_selection_strategy,
        :faces_indexing_state,
        :faces_indexing_state_time,
        :default_product_id
    )
    ON CONFLICT (uid) DO UPDATE SET
        %s
    WHERE
        ui.uid = :uid
'''

"""
Запрос создающий пользователя с локом в user_index
"""
SQL_UPSERT_USER_INDEX_WITH_LOCK = '''
    INSERT INTO disk.user_index AS ui (
        uid,
        version,
        blocked,
        deleted,
        user_type,
        reg_time,
        locale,
        shard_key,
        b2b_key,
        pdd,
        yateam_uid,
        collections,
        is_mailish,
        hancom_enabled,
        unlimited_autouploading_enabled,
        lock_key,
        lock_timestamp,
        is_reindexed_for_quick_move,
        office_selection_strategy,
        faces_indexing_state,
        faces_indexing_state_time,
        default_product_id
    )
    VALUES (
        :uid,
        :version,
        :blocked,
        :deleted,
        :user_type,
        :reg_time,
        :locale,
        :shard_key,
        :b2b_key,
        :pdd,
        :yateam_uid,
        :collections,
        :is_mailish,
        :hancom_enabled,
        :unlimited_autouploading_enabled,
        :lock_key,
        :lock_timestamp,
        :is_reindexed_for_quick_move,
        :office_selection_strategy,
        :faces_indexing_state,
        :faces_indexing_state_time,
        :default_product_id
    )
    ON CONFLICT (uid) DO UPDATE SET
        %s
    WHERE
        ui.uid = :uid
        AND (ui.lock_key IS NULL OR (ui.lock_key IS NOT NULL AND (ui.lock_timestamp IS NULL OR ui.lock_timestamp < :ttl_threshold)))
    RETURNING 1
'''

"""
Удаляем лок у пользователя
"""
SQL_RELEASE_USER_INIT_LOCK = '''
UPDATE disk.user_index SET lock_key = NULL, lock_timestamp = NULL WHERE uid = :uid
'''

"""
Удаляем lock_timestamp у пользователя, не трогая lock_key
"""
SQL_REMOVE_OWN_USER_INIT_LOCK_TIMESTAMP = '''
UPDATE disk.user_index SET lock_timestamp = NULL WHERE uid = :uid AND lock_key = :lock_key
'''


"""
Переменование папки и запись новой версии по uid, path
"""
SQL_UPDATE_FOLDER_NAME_AND_VERSION = """
UPDATE
  disk.folders
SET
  name=:new_name,
  version=:version
WHERE
  uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:path, :uid))
"""


"""
Перемещение папки и запись новой версии по uid, src_path и target_parent_path
"""
SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION = """
UPDATE
  disk.folders
SET
  parent_fid=(SELECT fid FROM code.path_to_fid(:target_parent_path, :uid)),
  name=:target_name,
  version=:version
WHERE
  uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:src_path, :uid))
"""

"""
Перемещение папки и запись новой версии по uid, src_path и target_parent_path, а также проставление флага Яровой
"""
SQL_UPDATE_FOLDER_PARENT_NAME_AND_VERSION_WITH_YAROVAYA_MARK_SET = """
UPDATE
  disk.folders
SET
  parent_fid=(SELECT fid FROM code.path_to_fid(:target_parent_path, :uid)),
  name=:target_name,
  version=:version,
  yarovaya_mark=true
WHERE
  uid=:uid AND fid=(SELECT fid FROM code.path_to_fid(:src_path, :uid))
"""

"""
Создать запись в link_data
"""
SQL_CREATE_LINK_DATA_ITEM = """
INSERT INTO disk.link_data (
    id,
    uid,
    path,
    version,
    user_ip,
    public_uid,
    target,
    type,
    date_created,
    date_modified,
    file_id,
    resource_id_uid
)
VALUES (
    :id,
    :uid,
    :path,
    :version,
    NULLIF(:user_ip, '')::inet,
    :public_uid,
    :target,
    :type,
    :date_created,
    :date_modified,
    :file_id,
    :resource_id_uid
)
"""


"""
Получение прошлого office_doc_short_id из link_data
"""
SQL_GET_PREVIOUS_OFFICE_DOC_SHORT_ID = """
SELECT office_access_state, office_doc_short_id
FROM
  disk.link_data
WHERE
  uid=:uid AND file_id=:file_id AND date_deleted IS NOT NULL
ORDER BY version DESC
LIMIT 1
"""


"""
Проставление офисных полей в link_data
"""
SQL_LINK_DATA_UPDATE_OFFICE_FIELDS = """
UPDATE
  disk.link_data
SET
  office_access_state=:office_access_state,
  office_doc_short_id=:office_doc_short_id
WHERE
  uid=:uid AND path=:path
"""

"""
Проставление office_access_state в link_data
"""
SQL_LINK_DATA_UPDATE_OFFICE_ACCESS_STATE = """
UPDATE
  disk.link_data
SET
  office_access_state=:office_access_state
WHERE
  uid=:uid AND path=:path
"""

"""
Проставление office_doc_short_id в link_data
"""
SQL_LINK_DATA_UPDATE_OFFICE_DOC_SHORT_ID = """
UPDATE
  disk.link_data
SET
  office_doc_short_id=:office_doc_short_id
WHERE
  uid=:uid AND path=:path
"""

"""
Проверить, есть ли родитель с меткой Яровой
"""
SQL_HAS_PARENTS_WITH_YAROVAYA_MARK_BY_PATH = """
WITH RECURSIVE recurse AS (
  SELECT
    1 AS idx,
    dir.name,
    dir.parent_fid,
    dir.fid,
    dir.yarovaya_mark
  FROM
    disk.folders dir
  WHERE
    dir.parent_fid IS NULL
    AND dir.name = (:all_parent_names)[1]
    AND dir.uid = :uid
  UNION
  SELECT
    idx + 1,
    child.name,
    child.parent_fid,
    child.fid,
    child.yarovaya_mark
  FROM
    recurse parent
    JOIN disk.folders child ON parent.fid = child.parent_fid
  WHERE
    uid = :uid
    AND
    child.name = (:all_parent_names)[idx + 1]
)
SELECT 1 as "found"
FROM recurse
WHERE yarovaya_mark is TRUE
LIMIT 1
;
"""

"""
Запрос на получение файла по file_id. Получает самый свежий (старший по версии) файл из всех файлов с одинаковыми
file_id. Ищет по всем коллекциям (во всех корневых папках), но фильтрует файлы из trash и hidden по date_removed и
date_hidden_data.
Входные параметры:
    :uid
    :file_id
"""
SQL_FILE_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED = SqlTemplatedQuery("""
SELECT
    %(SELECT_FILE_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(f.fid, f.uid)) as path
FROM
    disk.files f JOIN disk.storage_files s USING (storage_id)
WHERE
    f.uid=:uid AND
    f.fid=(
        SELECT fid
        FROM disk.files
        WHERE uid=:uid AND id=:file_id AND date_hidden_data IS NULL AND date_removed IS NULL
        ORDER BY version DESC LIMIT 1
    )
""")

"""
Запрос на получение папки по file_id. Получает самую свежую (старшую по версии) папку из всех файлов с одинаковыми
file_id. Ищет по всем коллекциям (во всех корневых папках), но фильтрует папки из trash и hidden по date_removed и
date_hidden_data.
Входные параметры:
    :uid
    :file_id
"""
SQL_FOLDER_BY_FILE_ID_WITH_OLDEST_VERSION_NOT_REMOVED = SqlTemplatedQuery("""
SELECT
    %(SELECT_FOLDER_FIELDS_TEMPLATE)s,
    (SELECT path FROM code.fid_to_path(d.fid, d.uid)) as path
FROM
    disk.folders d
WHERE
    d.uid=:uid AND
    d.fid=(
        SELECT fid
        FROM disk.folders
        WHERE uid=:uid AND id=:file_id AND date_hidden_data IS NULL AND date_removed IS NULL
        ORDER BY version DESC LIMIT 1
    )
""")

"""
Запрос на получение операций с определенными типами и подтипами,
dtime которых больше определенного времени, с указанием state.
"""
SQL_OPERATIONS_GET_BY_STATE_TYPES_SUBTYPES_AND_DTIME_AGE = """
SELECT
    id, uid, ctime, dtime, mtime, state, version, type, subtype, md5, uniq_id, data, ycrid, lock_id, date_lock_acquired
FROM
    disk.operations
WHERE
    dtime < :maxdtime AND
    (type, subtype) IN :types_subtypes AND
    state IN :states
;
"""

"""
Получение числа операций с определенными типами и подтипами,
dtime которых больше определенного времени, с указанием state.
"""
SQL_OPERATIONS_COUNT_BY_STATE_TYPES_SUBTYPES_AND_DTIME_AGE = """
SELECT
    COUNT(*)
FROM
    disk.operations
WHERE
    dtime < :maxdtime AND
    (type, subtype) IN :types_subtypes AND
    state IN :states
;
"""

SQL_GET_USER_ACTIVITY_INFO = """
SELECT
    uid,
    platform_type,
    first_activity,
    last_activity
FROM disk.user_activity_info
WHERE
    uid = :uid
"""

SQL_GET_USER_PLATFORM_LAST_ACTIVITY_DATE = """
SELECT
    last_activity
FROM disk.user_activity_info
WHERE
    uid = :uid AND platform_type = :platform_type
"""

SQL_CLEAR_USER_ACTIVITY_INFO = """
DELETE FROM disk.user_activity_info WHERE uid = :uid
"""

SQL_BULK_UPSERT_USER_ACTIVITY_INFO = """
WITH raw_values (uid, platform_type, first_activity, last_activity) AS (
    VALUES
        %(values)s

), modified AS (
    INSERT INTO disk.user_activity_info AS uai(uid, platform_type, first_activity, last_activity)
    SELECT
        rv.uid::BIGINT,
        rv.platform_type::disk.platform_type,
        rv.first_activity::DATE,
        rv.last_activity::DATE
    FROM raw_values AS rv
    JOIN disk.user_index AS ui ON rv.uid::BIGINT = ui.uid
    ON CONFLICT (uid, platform_type) DO UPDATE
    SET
        first_activity = least(uai.first_activity, EXCLUDED.first_activity),
        last_activity = greatest(uai.last_activity, EXCLUDED.last_activity)
    RETURNING uid
)
SELECT DISTINCT
    rv.uid::TEXT
FROM raw_values AS rv
WHERE rv.uid::BIGINT NOT IN (SELECT * FROM modified)
"""

SQL_BULK_GET_LATEST_ACTIVITY = """
SELECT uid, max(last_activity) as activity_before_update
FROM disk.user_activity_info
WHERE uid = ANY( :uids )
GROUP BY uid
"""

SQL_BULK_UPSERT_USER_ACTIVITY_INFO_AND_GET_EARLIEST_ACTIVITY_AFTER = """
WITH raw_values (uid, platform_type, first_activity, last_activity) AS (
    VALUES
        %(values)s
), modified AS (
    INSERT INTO disk.user_activity_info AS uai(uid, platform_type, first_activity, last_activity)
    SELECT
        rv.uid::BIGINT,
        rv.platform_type::disk.platform_type,
        rv.first_activity::DATE,
        rv.last_activity::DATE
    FROM raw_values AS rv
    JOIN disk.user_index AS ui ON rv.uid::BIGINT = ui.uid
    ON CONFLICT (uid, platform_type) DO UPDATE
    SET
        first_activity = least(uai.first_activity, EXCLUDED.first_activity),
        last_activity = greatest(uai.last_activity, EXCLUDED.last_activity)
    RETURNING uid, platform_type, first_activity, last_activity
)
SELECT uid, min(last_activity) as activity_after_update
FROM modified
GROUP BY uid
"""

"""
Получить fid'ы всех папок пользователя, сгруппированных по parent_fid'у
"""
SQL_GET_FIDS_BY_PARENT_FID_FOR_USER = """
SELECT
    parent_fid,
    array_agg(fid) AS children
FROM disk.folders
WHERE
    uid=:uid AND
    parent_fid IS NOT NULL
GROUP BY uid,parent_fid
"""

"""
Получить fid корня (/), элемента, у которого нет родителя
"""
SQL_GET_ROOT_FID = """
    SELECT fid FROM disk.folders WHERE uid=:uid AND parent_fid IS NULL
"""

SQL_INCREMENT_VERSION_AND_SET_FORCE_SNAPSHOT_VERSION = """
    UPDATE disk.user_index
    SET version = version + 1, force_snapshot_version = version + 1
    WHERE uid = :uid
"""

SQL_SET_IS_PAID = r"UPDATE disk.user_index SET is_paid = :is_paid WHERE uid = :uid"

SQL_SET_ONLINE_EDITOR = r"UPDATE disk.user_index SET office_online_editor_type = :online_editor WHERE uid = :uid"

SQL_SET_OFFICE_SELECTION_STRATEGY = """
UPDATE disk.user_index
SET
    office_selection_strategy = :office_selection_strategy
WHERE
    uid = :uid
"""

"""
Найти промокод
"""
SQL_FIND_PROMO_CODE = """
SELECT
    *
FROM disk.promo_codes
WHERE
    id = :id
"""

"""
Посчитать размер таблицы промокодов
"""
SQL_COUNT_PROMO_CODE = "SELECT count(*) FROM disk.promo_codes"

"""
Найти доступный промокод
"""
SQL_FIND_AVAILABLE_PROMO_CODE = """
SELECT
    *
FROM disk.promo_codes
WHERE
    id = :id AND
    count > 0 AND
    begin_datetime <= :now AND
    end_datetime >= :now
"""

"""
Уменьшить число доступных промокодов
"""
SQL_DECREMENT_PROMO_CODE_COUNT = """
UPDATE disk.promo_codes
    SET count = count - 1
WHERE
    id = :id
"""

"""
Найти промокод в архиве
"""
SQL_FIND_PROMO_CODE_ARCHIVE = """
SELECT
    *
FROM disk.promo_codes_archive
WHERE
    promo_code = :promo_code
"""

"""
Обновить промокод в архиве
"""
SQL_UPSERT_PROMO_CODE_ARCHIVE = """
INSERT INTO disk.promo_codes_archive (
        id,
        promo_code,
        pid,
        sid,
        discount_template_id,
        activation_datetime,
        uid,
        status
    )
    VALUES (
        :id,
        :promo_code,
        :pid,
        :sid,
        :discount_template_id,
        :activation_datetime,
        :uid,
        :status
    )
    ON CONFLICT (id) DO UPDATE SET
        pid = :pid,
        sid = :sid,
        discount_template_id = :discount_template_id,
        activation_datetime = :activation_datetime,
        uid = :uid,
        status = :status
    RETURNING 1
"""

"""
Посчитать размер таблицы архивных промокодов
"""
SQL_COUNT_PROMO_CODE_ARCHIVE = "SELECT count(*) FROM disk.promo_codes_archive"

"""
Вставить или обновить лок миграции пользователя
"""
SQL_ACQUIRE_MIGRATION_LOCK = """
WITH ins AS (
     INSERT INTO disk.migration_lock(uid, lock_until, owner_mark)
             VALUES (:uid, :until, :owner_mark)
         ON CONFLICT (uid)
     DO UPDATE SET lock_until = :until, owner_mark = :owner_mark
         WHERE migration_lock.lock_until < now()
         RETURNING lock_until
)
SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from ins) as inserted
"""
"""
Обновить срок годности лока миграции пользователя
"""
SQL_SET_UNTIL_PG_MIGRATION_LOCK = """
WITH upd(uid) AS (
     UPDATE disk.migration_lock SET lock_until = :until
         WHERE uid = :uid AND owner_mark = :owner_mark AND lock_until >= now()
     RETURNING uid
)
SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from upd) as updated
"""

"""
Удалить лок миграции пользователя
"""
SQL_RELEASE_PG_MIGRATION_LOCK = """
WITH del(uid) AS (
     DELETE FROM disk.migration_lock
         WHERE uid = :uid AND owner_mark = :owner_mark AND lock_until >= now()
     RETURNING uid
)
SELECT disk.get_migration_lock(:uid), exists(SELECT 1 from del) as deleted
"""

"""
Проверить лок миграции пользователя
"""
SQL_CHECK_PG_MIGRATION_LOCK = """
SELECT disk.is_locked_for_migration(:uid, :now)
"""

"""
Получить resource_id, которые есть в альбоме и проверяемом списке
"""
SQL_INTERSECT_RESOURCE_ID_WITH_ALBUM_ITEMS = """
SELECT
    obj_id, id, uid, obj_type, group_id
FROM disk.album_items
WHERE
    uid = :uid AND
    album_id = :album_id AND
    obj_id = ANY(:file_ids)
"""

"""
Инициализировать ревизию альбомов
"""
SQL_INITIALIZE_ALBUMS_REVISION = """
INSERT INTO disk.albums_info (uid, revision) VALUES (:uid, 1) ON CONFLICT ON CONSTRAINT pk_albums_info DO NOTHING
"""

"""
Вернуть ревизию альбомов для пользователя, если она есть
"""
SQL_GET_ALBUMS_REVISION = """
SELECT revision FROM disk.albums_info WHERE uid=:uid
"""

"""
Запрос на получение настроек.
Входные параметры:
    :uid
"""
SQL_SELECT_SETTING_BY_UID = SqlTemplatedQuery("""
SELECT
    *
FROM
    disk.disk_info
WHERE
    uid = :uid
""")


"""
Запрос на получение записи в disk_info.
Входные параметры:
    :uid
    :path
"""
SQL_SELECT_DISK_INFO_BY_UID_AND_PATH = SqlTemplatedQuery("""
SELECT
    *
FROM
    disk.disk_info
WHERE
    uid = :uid AND
    path = :path
""")


"""
Запрос на получение настроек. Лочит для обновления.
Входные параметры:
    :uid
    :id
"""
SQL_SELECT_SETTING_BY_ID_FOR_UPDATE = SqlTemplatedQuery("""
SELECT
    *
FROM
    disk.disk_info
WHERE
    uid = :uid AND
    id = :id
FOR UPDATE
""")


"""
Запрос на добавление настроек.
"""
SQL_INSERT_SETTING = SqlTemplatedQuery("""
INSERT INTO disk.disk_info
    (id, uid, version, path, type, data)
VALUES
    (:id, :uid, :version, :path, :type, :data)
RETURNING *
""")


"""
Запрос на обновление настроек.
"""
SQL_UPDATE_SETTING = SqlTemplatedQuery("""
UPDATE disk.disk_info
SET
    version = :version,
    data = :data
WHERE
    id = :id AND uid = :uid
RETURNING *
""")


SQL_COUNT_SUBFOLDERS_BY_PATH = r"""
SELECT COUNT(*)
FROM disk.folders
WHERE
    uid = :uid AND
    parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))
"""


SQL_COUNT_SUBFILES_BY_PATH = r"""
SELECT COUNT(*)
FROM disk.files
WHERE
    uid = :uid AND
    parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid))
"""


SQL_GET_ANOTHER_MEDIA_TYPE_FILE_FROM_FOLDER = r"""
SELECT *
FROM disk.files
WHERE
    uid = :uid AND
    parent_fid = (SELECT fid FROM code.path_to_fid(:path, :uid)) AND
    media_type NOT IN :media_types
LIMIT 1
"""

"""
Найти таски на чистку корзины
"""
SQL_FIND_EXPIRED_TRASH_CLEANER_TASKS = """
SELECT
    *
FROM disk.trash_cleaner_queue
WHERE
    bound_date < :bound_date
"""

"""
Найти шаблон скидки
"""
SQL_FIND_DISCOUNT_TEMPLATE = """
SELECT
    *
FROM disk.discount_templates
WHERE
    id = :id
"""

"""
Обновить скидку
"""
SQL_UPSERT_DISCOUNT_TEMPLATE = """
INSERT INTO disk.discount_templates (
        id,
        description,
        provided_line,
        disposable,
        creation_datetime,
        period_timedelta,
        end_datetime
    )
    VALUES (
        :id,
        :description,
        :provided_line,
        :disposable,
        :creation_datetime,
        :period_timedelta,
        :end_datetime
    )
    ON CONFLICT (id) DO UPDATE SET
        description = :description,
        provided_line = :provided_line,
        disposable = :disposable,
        creation_datetime = :creation_datetime,
        period_timedelta = :period_timedelta,
        end_datetime = :end_datetime
    RETURNING 1
"""

"""
Найти скидку в архиве
"""
SQL_FIND_DISCOUNTS_ARCHIVE = """
SELECT
    *
FROM disk.discounts_archive
WHERE
    uid = :uid AND
    discount_template_id = :discount_template_id
LIMIT 1
"""

"""
Обновить скидку в архиве
"""
SQL_UPSERT_DISCOUNTS_ARCHIVE = """
INSERT INTO disk.discounts_archive (
        id,
        uid,
        discount_template_id
    )
    VALUES (
        :id,
        :uid,
        :discount_template_id
    )
    ON CONFLICT (id) DO UPDATE SET
        uid = :uid,
        discount_template_id = :discount_template_id
    RETURNING 1
"""

"""
Удалить скидку из архива
"""
SQL_DELETE_DISCOUNTS_ARCHIVE = """
DELETE FROM disk.discounts_archive
WHERE
    id = :id
"""

"""
Поставить таск на чистку корзины
"""
SQL_INSERT_TRASH_CLEANER_QUEUE = """
INSERT INTO disk.trash_cleaner_queue (id, bound_date, uid) VALUES(:id, :bound_date, :uid)
ON CONFLICT (id) DO NOTHING
"""

"""
Добавить пользователя в список запрещенных к чистке
"""
SQL_UPSERT_SUPPORT_PROHIBITED_CLEANING_USERS = """
INSERT INTO disk.support_prohibited_cleaning_users (id, comment, ctime, moderator, uid) VALUES (:id, :comment, :ctime, :moderator, :uid)
ON CONFLICT (uid) DO UPDATE SET comment=:comment, ctime=:ctime, moderator=:moderator
"""

"""
Добавить hid в список заблокированных
"""
SQL_UPSERT_SUPPORT_BLOCKED_HIDS = """
INSERT INTO disk.support_blocked_hids (id, block_type, ctime, storage_id) VALUES (:id, :block_type, :ctime, :storage_id)
ON CONFLICT (storage_id) DO UPDATE SET block_type=:block_type, ctime=:ctime
"""

"""
Добавить данные в саппортилку
"""
SQL_UPDATE_SUPPORT_MPFS = """
UPDATE disk.support_mpfs SET data = data || :data
WHERE uid=:uid AND data_id=:data_id
"""

"""
Найти данные в саппортилке
"""
SQL_FIND_SUPPORT_MPFS = """
SELECT * FROM disk.support_mpfs
"""

"""
Выбрать альбомы пользователя с сортировкой по кол-ву элементов в нем
"""
SQL_SELECT_ALBUMS_WITH_ITEMS_COUNT_SORT = """
SELECT
    a.*, (SELECT COUNT(*) FROM disk.album_items AS ai WHERE ai.album_id = a.id) AS items_count
FROM disk.albums AS a
WHERE uid = :uid
    AND album_type = :album_type
    AND hidden is not True
ORDER BY items_count DESC, id
LIMIT :limit OFFSET :offset;
"""

"""
Добавить пользователя в таблицу пересчета места
"""
SQL_ADD_USER_TO_RECOUNT = """
INSERT INTO disk.recount (uid, ctime, mtime, count) VALUES (:uid, :timestamp, :timestamp, 1)
ON CONFLICT (uid) DO UPDATE SET mtime=:timestamp, count = excluded.count + 1
"""

"""
Добавить необработанный счет
"""
SQL_INSERT_UNPROCESSED_RECEIPT = """
INSERT INTO disk.unprocessed_receipts (id, syncronization_datetime, receipt, traceback, uid)
VALUES (:id, :syncronization_datetime, :receipt, :traceback, :uid)
ON CONFLICT (id) DO NOTHING
"""

"""
Добавить или изменить даты овердрафта
"""
SQL_UPDATE_OR_CREATE_OVERDRAFT_INFO = """
INSERT INTO disk.overdraft (uid, overdraft_date, last_email_date, last_push_date) VALUES (:uid, :overdraft_date, :last_email_date, :last_push_date)
ON CONFLICT (uid) DO UPDATE SET overdraft_date=:overdraft_date, last_email_date=:last_email_date, last_push_date=:last_push_date
"""

"""
Выбрать даты овердрафта
"""
SQL_SELECT_OVERDRAFT_INFO = """
SELECT uid, overdraft_date, last_email_date, last_push_date from disk.overdraft WHERE uid=:uid
"""

"""
Выбрать всех овердрафтников
"""
SQL_SELECT_All_OVERDRAFT_INFO = """
SELECT uid from disk.overdraft
WHERE %s <= NOW() - INTERVAL '12 hours' OR %s IS NULL LIMIT :limit
"""

"""
Обновить дату для итерации
"""
SQL_UPDATE_ITERATE_FIELD = """
UPDATE disk.overdraft SET %s = NOW() WHERE uid IN :uids
"""

"""
Удалить пользователя из таблицы овердрафта
"""
SQL_DELETE_USER_FROM_OVERDRAFT = """
DELETE from disk.overdraft WHERE uid=:uid
"""

"""
Найти протухшие ссылки
"""
SQL_FIND_EXPIRED_LINKS = """
SELECT
    *
FROM disk.link_data
WHERE
    date_deleted IS NULL AND
    available_until IS NOT NULL AND
    available_until <= :now
"""
