# -*- coding: utf-8 -*-

import enum
from sqlalchemy import (
    func,
    Table,
    Column,
    MetaData,
    Integer,
    String,
    Enum,
    BigInteger,
    Boolean,
    Date,
    DateTime,
    ForeignKeyConstraint,
    PrimaryKeyConstraint,
    UniqueConstraint,
    CheckConstraint,
    ForeignKey,
    Interval,
    TIMESTAMP,
)
from sqlalchemy.dialects.postgresql import UUID, BYTEA, JSONB, ARRAY, INET, REAL

from mpfs.common.static.tags import (
    promo_codes,
    billing,
)
from mpfs.core.office.static import OfficeAccessStateConst
from mpfs.core.photoslice.albums.static import PhotosliceAlbumTypeConst

metadata = MetaData(schema='disk')


class AntiVirusScanStatus(enum.Enum):
    new = 'NEW'
    in_progress = 'IN PROGRESS'
    clean = 'CLEAN'
    infected = 'INFECTED'
    error = 'ERROR'


class MediaType(enum.Enum):
    audio = 'audio'
    backup = 'backup'
    compressed = 'compressed'
    data = 'data'
    development = 'development'
    diskimage = 'diskimage'
    document = 'document'
    encoded = 'encoded'
    font = 'font'
    image = 'image'
    settings = 'settings'
    text = 'text'
    video = 'video'
    web = 'web'
    executable = 'executable'
    spreadsheet = 'spreadsheet'
    flash = 'flash'
    book = 'book'
    unknown = 'unknown'


class FolderType(enum.Enum):
    downloads = 'downloads'
    photostream = 'photostream'


class AlbumType(enum.Enum):
    GEO = 'geo'
    PERSONAL = 'personal'
    FAVORITES = 'favorites'
    FACES = 'faces'


class ItemsSortingMethod():
    BY_PHOTOSLICE_TIME = 'photoslice_time'
    BY_DATE_CREATED = 'date_created'


class ResourceType(enum.Enum):
    FILE = 'file'
    DIR = 'dir'


class AlbumLayout(enum.Enum):
    SQUARES = 'squares'
    ROWS = 'rows'
    WATERFALL = 'waterfall'
    FIT_WIDTH = 'fit_width'


class AlbumItemType(enum.Enum):
    RESOURCE = 'resource'
    ALBUM = 'album'
    SHARED_RESOURCE = 'shared_resource'


class ChangelogOpType(enum.Enum):
    NEW = 'new'
    CHANGED = 'changed'
    DELETED = 'deleted'
    QUICK_MOVE = 'moved'


class ChangelogSharedType(enum.Enum):
    OWNER = 'owner'
    GROUP = 'group'
    UNSHARED = 'unshared'


class PlatformType(enum.Enum):
    ANDROID = 'android'
    IOS = 'ios'
    MAC = 'mac'
    WEB = 'web'
    WINDOWS = 'windows'
    SEARCH_APP = 'search_app'


class PhotosliceAlbumType(enum.Enum):
    CAMERA = PhotosliceAlbumTypeConst.CAMERA
    SCREENSHOTS = PhotosliceAlbumTypeConst.SCREENSHOTS


class PromoCodeArchiveStatusType(enum.Enum):
    ACTIVATED = promo_codes.ACTIVATED


class OfficeActionState(enum.Enum):
    ALL = OfficeAccessStateConst.ALL
    DISABLED = OfficeAccessStateConst.DISABLED


class DiscountProvidableLinesType(enum.Enum):
    PRIMARY_2018_DISCOUNT_10 = billing.PRIMARY_2018_DISCOUNT_10
    PRIMARY_2018_DISCOUNT_20 = billing.PRIMARY_2018_DISCOUNT_20
    PRIMARY_2018_DISCOUNT_30 = billing.PRIMARY_2018_DISCOUNT_30
    PRIMARY_2019_DISCOUNT_10 = billing.PRIMARY_2019_DISCOUNT_10
    PRIMARY_2019_DISCOUNT_20 = billing.PRIMARY_2019_DISCOUNT_20
    PRIMARY_2019_DISCOUNT_30 = billing.PRIMARY_2019_DISCOUNT_30
    PRIMARY_2019_V2_DISCOUNT_10 = billing.PRIMARY_2019_V2_DISCOUNT_10
    PRIMARY_2019_V2_DISCOUNT_20 = billing.PRIMARY_2019_V2_DISCOUNT_20
    PRIMARY_2019_V2_DISCOUNT_30 = billing.PRIMARY_2019_V2_DISCOUNT_30
    PRIMARY_2019_V3_DISCOUNT_10 = billing.PRIMARY_2019_V3_DISCOUNT_10
    PRIMARY_2019_V3_DISCOUNT_20 = billing.PRIMARY_2019_V3_DISCOUNT_20
    PRIMARY_2019_V3_DISCOUNT_30 = billing.PRIMARY_2019_V3_DISCOUNT_30
    PRIMARY_2019_V4_DISCOUNT_10 = billing.PRIMARY_2019_V4_DISCOUNT_10
    PRIMARY_2019_V4_DISCOUNT_20 = billing.PRIMARY_2019_V4_DISCOUNT_20
    PRIMARY_2019_V4_DISCOUNT_30 = billing.PRIMARY_2019_V4_DISCOUNT_30
    THIRD_TARIFF_2019_DISCOUNT_10 = billing.THIRD_TARIFF_2019_DISCOUNT_10
    THIRD_TARIFF_2019_DISCOUNT_20 = billing.THIRD_TARIFF_2019_DISCOUNT_20
    THIRD_TARIFF_2019_DISCOUNT_30 = billing.THIRD_TARIFF_2019_DISCOUNT_30


class InactiveUsersFlowState(enum.Enum):
    new = 'new'
    excluded = 'excluded'
    deleted = 'deleted'
    # inactive flow
    block_warning_1 = 'block_warning_1'
    block_warning_2 = 'block_warning_2'
    blocked = 'blocked'
    deletion_warning_1 = 'deletion_warning_1'
    deletion_warning_2 = 'deletion_warning_2'
    # no_email flow
    no_email_blocked = 'no_email_blocked'
    no_email_deletion_warning_1 = 'no_email_deletion_warning_1'
    no_email_deletion_warning_2 = 'no_email_deletion_warning_2'
    # blocked_flow
    blocked_flow_blocked = 'blocked_flow_blocked'
    blocked_flow_deletion_warning_1 = 'blocked_flow_deletion_warning_1'
    blocked_flow_deletion_warning_2 = 'blocked_flow_deletion_warning_2'


folders = Table(
    'folders', metadata,
    Column('uid', BigInteger, nullable=False),  # идентификатор пользователя
    Column('fid', UUID(as_uuid=True), nullable=False),  # постгресовый идентификатор папки
    Column('parent_fid', UUID(as_uuid=True), nullable=True),  # fid родительской папки
    Column('id', BYTEA(), nullable=True),  # старый монговый file_id

    Column('name', String, nullable=False),  # имя папки (именно имя этой папки в родительской папке, без слешей)
    Column('visible', Boolean, nullable=False, default=True),

    Column('version', BigInteger, nullable=False, default=0),
    Column('public', Boolean, nullable=False, default=False),
    Column('public_hash', BYTEA(), nullable=True),
    Column('symlink', String, nullable=True),
    Column('short_url', String, nullable=True),
    Column('blocked', Boolean, nullable=False, default=False),
    Column('published', Boolean, nullable=False, default=False),

    Column('files_count', Integer, nullable=False),
    Column('folders_count', Integer, nullable=False),

    Column('folder_type', Enum(FolderType), nullable=True, default=None),

    Column('date_uploaded', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_created', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_modified', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_removed', DateTime(timezone=True), nullable=True),
    Column('date_hidden_data', DateTime(timezone=True), nullable=True),

    Column('path_before_remove', String, nullable=True),
    Column('modify_uid', BigInteger, nullable=True),

    Column('folder_url', String, nullable=True),
    Column('download_counter', BigInteger, nullable=True),
    Column('custom_properties', String, nullable=True),

    Column('yarovaya_mark', Boolean, nullable=False, default=False),

    Column('custom_setprop_fields', JSONB(), nullable=True),

    PrimaryKeyConstraint('uid', 'fid', name='pk_folders'),
    UniqueConstraint('uid', 'id', name='uk_folders_id'),
    CheckConstraint('uid >= 0', name='check_uid'),
    CheckConstraint('version >= 0', name='check_version'),
    CheckConstraint("name != '' OR parent_fid IS NULL", name='check_empty_name'),
    CheckConstraint('character_length(name) <= 1024', name='check_name_length'),
)

files = Table(
    'files', metadata,
    Column('uid', BigInteger, nullable=False),  # идентификатор пользователя
    Column('fid', UUID(as_uuid=True), nullable=False),  # постгресовый идентификатор файла
    Column('parent_fid', UUID(as_uuid=True), nullable=True),  # fid родительской папки
    Column('storage_id', UUID(as_uuid=True), nullable=False),
    Column('id', BYTEA(), nullable=True),  # старый монговый file_id

    Column('name', String, nullable=False),  # имя файла (именно имя файла в родительской папке, без слешей)
    Column('visible', Boolean, nullable=False, default=True),

    Column('version', BigInteger, nullable=False, default=0),
    Column('public', Boolean, nullable=False, default=False),
    Column('public_hash', BYTEA(), nullable=True),
    Column('symlink', String, nullable=True),
    Column('short_url', String, nullable=True),
    Column('office_access_state', Enum(OfficeActionState), nullable=True, default=None),
    Column('office_doc_short_id', String, nullable=True),
    Column('download_counter', BigInteger, nullable=True),
    Column('folder_url', String, nullable=True),
    Column('custom_properties', String, nullable=True),
    Column('blocked', Boolean, nullable=False, default=False),
    Column('published', Boolean, nullable=False, default=False),

    Column('date_uploaded', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_created', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_modified', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_removed', DateTime(timezone=True), nullable=True),
    Column('date_hidden_data', DateTime(timezone=True), nullable=True),
    Column('date_exif', DateTime(timezone=True), nullable=True),

    Column('path_before_remove', String, nullable=True),
    Column('modify_uid', BigInteger, nullable=True),

    Column('media_type', Enum(MediaType)),  # монговый mt
    Column('mime_type', String),

    Column('source', String),  # источник файла (какой клиент его загрузил)
    Column('source_uid', BigInteger, nullable=True),

    Column('is_live_photo', Boolean, nullable=False, default=False),
    Column('yarovaya_mark', Boolean, nullable=False, default=False),

    Column('ext_aesthetics', REAL(), default=None, nullable=True),
    Column('photoslice_album_type', Enum(PhotosliceAlbumType), nullable=True, default=None),
    Column('albums_exclusions', ARRAY(String), nullable=True, default=None),

    Column('custom_setprop_fields', JSONB(), nullable=True),

    Column('ext_coordinates', String, nullable=True, default=None),

    PrimaryKeyConstraint('uid', 'fid', name='pk_files'),
    UniqueConstraint('uid', 'parent_fid', 'name', name='uk_files_uid_parent_fid'),
    UniqueConstraint('uid', 'id', name='uk_files_id'),
    CheckConstraint('uid >= 0', name='check_uid'),
    CheckConstraint('version >= 0', name='check_version'),
    CheckConstraint("name != ''", name='check_empty_name'),
    CheckConstraint('character_length(name) <= 1024', name='check_name_length'),
)

storage_files = Table(
    'storage_files', metadata,
    Column('storage_id', UUID(as_uuid=True), nullable=False),  # постгресовый идентификатор бинарного файла (hid)

    Column('stid', String, nullable=False),  # стид (идентификатор файла в сторадже Mulca/MDS)
    Column('digest_stid', String, nullable=False),  # стид файла с хешами
    Column('preview_stid', String, nullable=True),  # стид превьюшки

    Column('date_origin', DateTime(timezone=True), nullable=False, default=func.now()),

    Column('size', BigInteger, nullable=False, default=0),

    Column('md5_sum', UUID(as_uuid=True), nullable=False),
    Column('sha256_sum', BYTEA(), nullable=False),

    Column('av_scan_status', Enum(AntiVirusScanStatus), nullable=False, default=AntiVirusScanStatus.new),
    Column('video_data', JSONB()),

    Column('width', Integer, nullable=True),
    Column('height', Integer, nullable=True),
    Column('angle', Integer, nullable=True),

    PrimaryKeyConstraint('storage_id', name='pk_storage_files'),
    CheckConstraint('size >= 0', name='check_size'),
)


class AdditionalFileTypes(enum.Enum):
    live_video = 'live_video'


additional_file_links = Table(
    'additional_file_links', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('main_file_fid', UUID(as_uuid=True), nullable=False),
    Column('additional_file_fid', UUID(as_uuid=True), nullable=False),
    Column('type', Enum(AdditionalFileTypes), nullable=False),
)


class UserType(enum.Enum):
    STANDARD = "standart"
    ATTACH = "attach"


class LocaleType(enum.Enum):
    RU = "ru"
    EN = "en"
    TR = "tr"
    UK = "uk"


user_index = Table(
    'user_index', metadata,
    Column('uid', BigInteger, nullable=False),
    # TODO Constraints >= 0 as Folder does ??
    Column('version', BigInteger, nullable=False, default=0),
    Column('blocked', Boolean, nullable=False, default=False),
    Column('deleted', DateTime(timezone=True), nullable=True),

    Column('user_type', Enum("standart", "attach", name="user_type"), nullable=False, default=UserType.STANDARD),

    # TODO Why nullable=True ?
    Column('reg_time', DateTime(timezone=True)),
    # TODO Why not locale_type
    Column('locale', Enum("ru", "en", "tr" "uk", name="locale_type"), nullable=False, default=LocaleType.RU),

    # TODO nullable ?
    Column('shard_key', BigInteger),
    Column('b2b_key', String),
    Column('pdd', JSONB()),
    Column('yateam_uid', BigInteger, nullable=True),
    Column('last_quick_move_version', BigInteger, nullable=True),
    Column('is_reindexed_for_quick_move', Boolean, nullable=True),
    Column('force_snapshot_version', BigInteger, nullable=True),
    Column('is_mailish', Boolean, nullable=True, default=False),
    Column('hancom_enabled', Boolean, nullable=False, default=False),
    Column('unlimited_autouploading_enabled', Boolean, nullable=False, default=False),
    Column('unlimited_photo_autouploading_enabled', Boolean, nullable=True),
    Column('unlimited_video_autouploading_enabled', Boolean, nullable=True),
    Column('unlimited_video_autouploading_reason', String, nullable=True),
    Column('unlimited_photo_autouploading_allowed', Boolean, nullable=True),
    Column('unlimited_video_autouploading_allowed', Boolean, nullable=True),
    Column('office_selection_strategy', String, nullable=True),
    Column('is_paid', Boolean, nullable=True, default=False),
    Column('office_online_editor_type', String, nullable=True),
    Column('only_office_enabled', Boolean, nullable=True),
    Column('online_editor_enabled', Boolean, nullable=True),
    Column('advertising_enabled', Boolean, nullable=True),
    Column('public_settings_enabled', Boolean, nullable=True),
    Column('faces_indexing_state', String, nullable=True),
    Column('faces_indexing_state_time', DateTime(timezone=True), nullable=True),

    Column('collections', ARRAY(String), nullable=False),
    Column('lock_key', UUID(as_uuid=True), nullable=True),
    Column('lock_timestamp', DateTime(timezone=True), nullable=True),
    Column('default_product_id', String, nullable=True),

    PrimaryKeyConstraint('uid', name='pk_user_index'),
    CheckConstraint('uid >= 0', name='check_uid'),
)


link_data = Table(
    'link_data', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('path', String, nullable=False),
    Column('version', BigInteger, default=None, nullable=True),
    Column('user_ip', INET, default=None, nullable=True),
    Column('public_uid', BigInteger, default=None, nullable=True),

    Column('target', String, default=None, nullable=True),
    Column('file_id', BYTEA(), nullable=True),
    Column('resource_id_uid', BigInteger, nullable=True),

    Column('type',  Enum(*[x.value for x in ResourceType], name='resource_type'), nullable=False),
    Column('parent', UUID(as_uuid=True), default=None, nullable=True),

    Column('date_created', DateTime(timezone=True), default=func.now(), nullable=True),
    Column('date_modified', DateTime(timezone=True), default=func.now(), nullable=True),
    Column('date_deleted', DateTime(timezone=True), default=None, nullable=True),

    Column('office_access_state', Enum(OfficeActionState), nullable=True, default=None),
    Column('office_doc_short_id', String, nullable=True),
    Column('password', String, nullable=True),
    Column('read_only', Boolean, nullable=True),
    Column('available_until', DateTime(timezone=True), nullable=True),
    Column('external_organization_ids', ARRAY(BigInteger), nullable=True),



    PrimaryKeyConstraint('id', name='pk_link_data'),
    ForeignKeyConstraint(['parent'], ['link_data.id'], ondelete="CASCADE", deferrable=True, initially="DEFERRED",
                         name="fk_link_data_parent"),
    CheckConstraint('uid >= 0', name='check_uid'),
)

disk_info = Table(
    'disk_info', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('parent', UUID(as_uuid=True), nullable=True, default=None),
    Column('version', BigInteger, nullable=False, default=None),
    Column('path', String, nullable=False, default=False),
    Column('type', Enum("file", "dir", name="resource_type"), nullable=False),
    Column('data', JSONB()),

    PrimaryKeyConstraint('uid', 'id', name='pk_disk_info'),
    UniqueConstraint('id', name='uk_disk_info_id'),
    ForeignKeyConstraint(['parent'], ['disk_info.id'], ondelete="CASCADE",
                         deferrable=True, initially="DEFERRED", name="fk_disk_info_id"),
    CheckConstraint('uid >= 0', name='check_uid')
)

misc_data = Table(
    'misc_data', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('path', String, nullable=False),
    Column('type',  Enum(*[x.value for x in ResourceType], name='resource_type'), nullable=False),

    Column('parent', UUID(as_uuid=True), default=None, nullable=True),
    Column('version', BigInteger, default=None, nullable=True),
    Column('zdata', BYTEA(), default=None, nullable=True),

    Column('file_id', BYTEA(), default=None, nullable=True),  # todo: type?
    Column('file_id_zipped', Boolean, default=None, nullable=True),

    Column('hid', UUID(as_uuid=True), default=None, nullable=True),
    Column('mimetype', String, default=None, nullable=True),
    Column('mediatype', Enum(*[x.value for x in MediaType], name='media_type'), default=None, nullable=True),
    Column('visible', Boolean, default=None, nullable=True),
    Column('size', Integer, default=None, nullable=True),

    Column('file_stid', String, default=None, nullable=True),
    Column('digest_stid', String, default=None, nullable=True),
    Column('preview_stid', String, default=None, nullable=True),

    Column('date_modified', DateTime(timezone=True), default=None, nullable=True),
    Column('date_uploaded', DateTime(timezone=True), default=None, nullable=True),
    Column('date_exif', DateTime(timezone=True), default=None, nullable=True),

    PrimaryKeyConstraint('id', name='pk_misc_data'),
    ForeignKeyConstraint(['parent'], ['misc_data.id'], ondelete="CASCADE", deferrable=True, initially="DEFERRED",
                         name="fk_misc_data_parent"),
    CheckConstraint('uid >= 0', name='check_uid'),
)


albums = Table(
    'albums', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('title', String, nullable=False),

    Column('cover_id', BYTEA(), default=None, nullable=True),
    Column('cover_offset_y', REAL(), default=None, nullable=True),

    Column('description', String, default=None, nullable=True),
    Column('public_key', String, default=None, nullable=True),
    Column('public_url', String, default=None, nullable=True),
    Column('short_url', String, default=None, nullable=True),
    Column('is_public', Boolean, default=False, nullable=False),
    Column('is_blocked', Boolean, default=False, nullable=False),
    Column('is_desc_sorting', Boolean, default=False, nullable=False),
    Column('block_reason', String, default=None, nullable=True),
    Column('album_items_sorting', String, default=None, nullable=True),
    Column('flags', ARRAY(String), default=None, nullable=True),
    Column('layout', Enum(*[x.value for x in AlbumLayout], name='album_layout'), default=None, nullable=True),
    Column('date_created', DateTime(timezone=True), default=func.now(), nullable=True),
    Column('date_modified', DateTime(timezone=True), default=func.now(), nullable=True),
    Column('social_cover_stid', String, default=None, nullable=True),
    Column('fotki_album_id', BigInteger, default=None, nullable=True),
    Column('album_type', String, default=None, nullable=True),
    PrimaryKeyConstraint('uid', 'id', name='pk_albums'),
    UniqueConstraint('id', name='uk_albums_id'),
    # ForeignKeyConstraint(['cover_id'], ['album_items.id'], ondelete='SET NULL', deferrable=True, initially='DEFERRED',
    #                      name='fk_albums_cover_id'),
    CheckConstraint('uid >= 0', name='check_uid'),
    Column('hidden', Boolean, default=None, nullable=True),
)


album_items = Table(
    'album_items', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('album_id', BYTEA(), default=None),
    Column('description', String, default=None, nullable=True),
    Column('group_id', UUID(as_uuid=True), default=None, nullable=True),
    Column('order_index', REAL(), nullable=False),
    Column('obj_id', String, nullable=False),
    Column('obj_type', Enum(*[x.value for x in AlbumItemType], name='album_item_type'), nullable=False),
    Column('date_created', DateTime(timezone=True), default=func.now(), nullable=True),
    Column('face_info', JSONB(), default=None, nullable=True),

    PrimaryKeyConstraint('uid', 'id', name='pk_album_items'),
    UniqueConstraint('id', name='uk_album_items_id'),
    ForeignKeyConstraint(['album_id'], ['albums.id'], ondelete='CASCADE', deferrable=True, initially='DEFERRED',
                         name='fk_album_items_album_id'),
    CheckConstraint('uid >= 0', name='check_uid'),
)


albums_info = Table(
    'albums_info', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('revision', BigInteger, nullable=False),

    PrimaryKeyConstraint('uid', name='pk_albums_info'),
)


filesystem_locks = Table(
    'filesystem_locks', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('path', String, nullable=False),
    Column('dtime', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('data', JSONB(), nullable=True, default=None),

    PrimaryKeyConstraint('uid', 'path', name='pk_filesystem_locks'),
    UniqueConstraint('id', name='uk_filesystem_locks_id'),
    CheckConstraint('uid >= 0', name='check_uid'),
)

migration_lock = Table(
    'migration_lock', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('lock_until', DateTime(timezone=True), nullable=False),
    Column('owner_mark', String(), nullable=False),

    PrimaryKeyConstraint('uid', name='pk_migration_lock'),
)


operations = Table(
    'operations', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('uid', BigInteger, nullable=False),

    Column('ctime', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('dtime', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('mtime', DateTime(timezone=True), nullable=False, default=func.now()),

    Column('state', Integer, nullable=False),
    Column('version', BigInteger, nullable=False),

    Column('type', String, nullable=True, default=None),
    Column('subtype', String, nullable=True, default=None),
    Column('ycrid', String, nullable=True, default=None),
    Column('md5', UUID(as_uuid=True), nullable=True, default=None),
    Column('uniq_id', UUID(as_uuid=True), nullable=True, default=None),

    Column('data', JSONB(), nullable=True, default=None),

    PrimaryKeyConstraint('id', name='pk_operations'),
    CheckConstraint('uid >= 0', name='check_uid'),
)


async_tasks_data = Table(
    'async_tasks_data', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('data', BYTEA(), default=None, nullable=True),
    Column('date_created', DateTime(timezone=True), nullable=False, default=func.now()),

    PrimaryKeyConstraint('id', name='pk_async_tasks_data'),
    CheckConstraint('uid >= 0', name='check_uid'),
)


version_links = Table(
    'version_links', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('file_id', BYTEA(), nullable=False),
    Column('disk_path', String, nullable=True, default=None),
    Column('disk_path_hash', UUID(as_uuid=True), nullable=True, default=None),
    Column('date_created', DateTime(timezone=True), nullable=False, default=func.now()),

    PrimaryKeyConstraint('id', name='pk_version_links'),
)

version_data = Table(
    'version_data', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('version_link_id', UUID(as_uuid=True), nullable=False),
    Column('parent_version_id', UUID(as_uuid=True), nullable=True),
    Column('type', String, nullable=False),
    Column('date_created', DateTime(timezone=True), nullable=False, default=func.now()),
    Column('date_to_remove', DateTime(timezone=True), nullable=False),
    Column('record_date_created', DateTime(timezone=True), nullable=True, default=func.now()),
    Column('is_checkpoint', Boolean, nullable=False, default=False),
    Column('folded_counter', BigInteger, nullable=False, default=0),
    Column('uid_created', BigInteger, nullable=False),
    Column('platform_created', String, nullable=False),
    Column('storage_id', UUID(as_uuid=True), ForeignKey(storage_files.c.storage_id), nullable=True),
    Column('date_exif', DateTime(timezone=True), nullable=True),

    PrimaryKeyConstraint('id', name='pk_version_data'),
)

changelog = Table(
    'changelog', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('path', String, nullable=False),
    Column('type', Enum(*[x.value for x in ResourceType], name='resource_type'), nullable=False),
    Column('op', Enum(*[x.value for x in ChangelogOpType], name='changelog_op_type'), nullable=False),
    Column('version', BigInteger, nullable=False),

    Column('zdata', JSONB(), nullable=True, default=None),

    Column('gid', UUID(as_uuid=True), nullable=True, default=None),
    Column('group_path', String, nullable=True, default=None),
    Column('dtime', DateTime(timezone=True), nullable=True, default=func.now()),

    Column('shared', Enum(*[x.value for x in ChangelogSharedType], name='changelog_shared_type'), nullable=True, default=None),
    Column('rights', Integer, nullable=True, default=None),

    PrimaryKeyConstraint('id', name='pk_changelog'),
    CheckConstraint('uid >= 0', name='check_uid'),
)


organizations = Table(
    'organizations', metadata,
    Column('id', String, nullable=False),
    Column('is_paid', Boolean, nullable=False, default=False),
    Column('quota_limit', BigInteger, nullable=True, default=None),
    Column('quota_free', BigInteger, nullable=True, default=None),
    Column('quota_used_by_disk', BigInteger, nullable=True, default=None),

    PrimaryKeyConstraint('id', name='pk_organizations'),
)


last_files_cache = Table(
    'last_files_cache', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('gid', UUID(as_uuid=True), nullable=False),
    Column('owner_uid', BigInteger, nullable=False),
    Column('file_id', BYTEA(), nullable=False),
    Column('file_date_modified', DateTime(timezone=True), nullable=False),
    Column('date_created', DateTime(timezone=True), default=func.now(), nullable=False),
)


trash_cleaner_queue = Table(
    'trash_cleaner_queue', metadata,
    Column('id', String, nullable=False),
    Column('bound_date', Date, nullable=False),
    Column('uid', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_trash_cleaner_queue'),
)


promo_codes = Table(
    'promo_codes', metadata,
    Column('id', String, nullable=False),
    Column('pid', String, nullable=True, default=None),
    Column('discount_template_id', UUID(as_uuid=True), nullable=True, default=None),
    Column('begin_datetime', DateTime(timezone=True), nullable=False),
    Column('end_datetime', DateTime(timezone=True), nullable=False),
    Column('count', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_promo_codes'),
)


promo_codes_archive = Table(
    'promo_codes_archive', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('promo_code', String, nullable=False),
    Column('pid', String, nullable=True, default=None),
    Column('sid', String, nullable=True, default=None),
    Column('discount_template_id', UUID(as_uuid=True), nullable=True, default=None),
    Column('uid', BigInteger, nullable=False),
    Column('activation_datetime', DateTime(timezone=True), nullable=False),
    Column('status', String, nullable=False),
    PrimaryKeyConstraint('id', name='pk_promo_codes_archive'),
)


discount_templates = Table(
    'discount_templates', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('description', String, nullable=False),
    Column('provided_line', String, nullable=False),
    Column('disposable', Boolean, nullable=False, default=True),
    Column('creation_datetime', DateTime(timezone=True), nullable=False),
    Column('period_timedelta', Interval, nullable=True, default=None),
    Column('end_datetime', DateTime(timezone=True), nullable=True, default=None),
    PrimaryKeyConstraint('id', name='pk_discount_templates'),
)


discounts_archive = Table(
    'discounts_archive', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger, nullable=False),
    Column('discount_template_id', UUID(as_uuid=True), nullable=False),
    PrimaryKeyConstraint('id', name='pk_discounts_archive'),
)

billing_orders = Table(
    'billing_orders', metadata,
    Column('id', String, nullable=False),
    Column('auto', Boolean, nullable=True),
    Column('bb_pid', String, nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('currency', String, nullable=True),
    Column('discount_template_id', UUID(as_uuid=True), nullable=True),
    Column('group_name', String, nullable=True),
    Column('group_uids', ARRAY(BigInteger), nullable=True),
    Column('locale', String, nullable=True),
    Column('market', String, nullable=True),
    Column('mtime', DateTime(timezone=True), nullable=True),
    Column('otype', String, nullable=True),
    Column('payment_id', String, nullable=True),
    Column('payment_method', String, nullable=True),
    Column('pid', String, nullable=True),
    Column('price', BigInteger, nullable=True),
    Column('sid', String, nullable=True),
    Column('state', String, nullable=True),
    Column('uid', BigInteger, nullable=False),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_orders'),
)

billing_orders_history = Table(
    'billing_orders_history', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('moved', Boolean, nullable=True),
    Column('auto', Boolean, nullable=True),
    Column('bb_pid', String, nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('currency', String, nullable=True),
    Column('description', String, nullable=True),
    Column('discount_template_id', UUID(as_uuid=True), nullable=True),
    Column('fuckup_period', String, nullable=True),
    Column('locale', String, nullable=True),
    Column('market', String, nullable=True),
    Column('mtime', DateTime(timezone=True), nullable=True),
    Column('number', String, nullable=True),
    Column('otype', String, nullable=True),
    Column('payment_id', String, nullable=True),
    Column('payment_method', String, nullable=True),
    Column('pid', String, nullable=True),
    Column('price', BigInteger, nullable=True),
    Column('receipt', String, nullable=True),
    Column('refund_status', String, nullable=True),
    Column('sid', String, nullable=True),
    Column('state', String, nullable=True),
    Column('status_code', String, nullable=True),
    Column('trust_refund_id', String, nullable=True),
    Column('uid', BigInteger, nullable=False),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_orders_history'),
)

billing_services = Table(
    'billing_services', metadata,
    Column('id', String, nullable=False),
    Column('moved', Boolean, nullable=True),
    Column('auto', Boolean, nullable=True),
    Column('btime', DateTime(timezone=True), nullable=True),
    Column('buyer_uid', BigInteger, nullable=False),
    Column('child_sids', ARRAY(String), nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('description', String, nullable=True),
    Column('dtime', DateTime(timezone=True), nullable=True),
    Column('enabled', Boolean, nullable=True),
    Column('first_prolongation_fail', Integer, nullable=True),
    Column('is_group', Boolean, nullable=True),
    Column('group_name', String, nullable=True),
    Column('group_uids', ARRAY(BigInteger), nullable=True),
    Column('lbtime', DateTime(timezone=True), nullable=True),
    Column('mtime', DateTime(timezone=True), nullable=True),
    Column('notified', Integer, nullable=True),
    Column('number', String, nullable=True),
    Column('order_id', String, nullable=True),
    Column('original_transaction_id', BigInteger, nullable=True),
    Column('parent_sid', String, nullable=True),
    Column('paid_for_other', Boolean, nullable=True),
    Column('pid', String, nullable=True),
    Column('state', String, nullable=True),
    Column('uid', BigInteger, nullable=False),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_services'),
)

billing_services_history = Table(
    'billing_services_history', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('moved', Boolean, nullable=True),
    Column('auto', Boolean, nullable=True),
    Column('btime', DateTime(timezone=True), nullable=True),
    Column('buyer_uid', BigInteger, nullable=False),
    Column('child_sids', ARRAY(String), nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('description', String, nullable=True),
    Column('dtime', DateTime(timezone=True), nullable=True),
    Column('enabled', Boolean, nullable=True),
    Column('first_prolongation_fail', Integer, nullable=True),
    Column('free', Boolean, nullable=True),
    Column('is_group', Boolean, nullable=True),
    Column('group_name', String, nullable=True),
    Column('group_uids', ARRAY(BigInteger), nullable=True),
    Column('lbtime', DateTime(timezone=True), nullable=True),
    Column('mtime', DateTime(timezone=True), nullable=True),
    Column('notified', Integer, nullable=True),
    Column('number', String, nullable=True),
    Column('order_id', String, nullable=True),
    Column('original_transaction_id', BigInteger, nullable=True),
    Column('paid_for_other', Boolean, nullable=True),
    Column('parent_sid', String, nullable=True),
    Column('pid', String, nullable=True),
    Column('refund_status', String, nullable=True),
    Column('sid', String, nullable=True),
    Column('state', String, nullable=True),
    Column('uid', BigInteger, nullable=False),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_services_history'),
)

billing_service_attributes = Table(
    'billing_service_attributes', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('amount', BigInteger, nullable=True),
    Column('uid', BigInteger, nullable=True),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_service_attributes'),
)

billing_service_attributes_history = Table(
    'billing_service_attributes_history', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('amount', BigInteger, nullable=True),
    Column('uid', BigInteger, nullable=True),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_service_attributes_history'),
)

billing_subscriptions = Table(
    'billing_subscriptions', metadata,
    Column('id', String, nullable=False),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('description', String, nullable=True),
    Column('sid', String, nullable=True),
    Column('uid', BigInteger, nullable=False),
    Column('v', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_billing_subscriptions'),
)

billing_locks = Table(
    'billing_locks', metadata,
    Column('id', String, nullable=False),
    Column('host', String, nullable=True),
    Column('lock', Integer, nullable=True),
    PrimaryKeyConstraint('id', name='pk_billing_locks'),
)

user_activity_info = Table(
    'user_activity_info', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('platform_type', Enum(PlatformType), nullable=False),
    Column('first_activity', Date, nullable=False),
    Column('last_activity', Date, nullable=False),
    PrimaryKeyConstraint('uid', 'platform_type', name='pk_user_activity_info'),
    CheckConstraint('last_datetime >= first_datetime', name='check_activity_values'),
    ForeignKeyConstraint(['uid'], ['user_index.uid'], ondelete='CASCADE', name='fk_user_activity_info_uid',
                         deferrable=True, initially='DEFERRED'),
)


source_ids = Table(
    'source_ids', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('source_id', String, nullable=False),
    Column('storage_id', UUID(as_uuid=True), nullable=False),
    Column('is_live_photo', Boolean, nullable=False, default=False),
)


deletion_log = Table(
    'deletion_log', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('storage_id', UUID(as_uuid=True), nullable=False),
    Column('file_id', BYTEA(), nullable=False),
    Column('deletion_log_revision', TIMESTAMP(timezone=True), nullable=False),
    Column('is_live_photo', Boolean, nullable=False, default=False),
    PrimaryKeyConstraint('uid', 'deletion_log_revision', name='pk_deletion_log'),
)


unprocessed_receipts = Table(
    'unprocessed_receipts', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('syncronization_datetime', DateTime, nullable=False),
    Column('receipt', String, nullable=False),
    Column('traceback', String, nullable=False),
    Column('uid', BigInteger, nullable=False),
    PrimaryKeyConstraint('id', name='pk_unprocessed_receipts'),
)

duplicated_storage_files = Table(
    'duplicated_storage_files', metadata,
    Column('storage_id', UUID(as_uuid=True), nullable=False),
    Column('stid', String, nullable=False),
)

support_blocked_hids = Table(
    'support_blocked_hids', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('block_type', String, nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=False),
    Column('storage_id', UUID(as_uuid=True), nullable=False),
)

support_mpfs = Table(
    'support_mpfs', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('data', JSONB()),
    Column('data_id', String, nullable=True),
    Column('data_hash', String, nullable=True),
    Column('data_stids', ARRAY(String), nullable=True),
    Column('uid', BigInteger, nullable=False),
)

support_prohibited_cleaning_users = Table(
    'support_prohibited_cleaning_users', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('comment', String, nullable=True),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('moderator', String, nullable=False),
    Column('uid', BigInteger, nullable=False),
)

support_moderation_queue = Table(
    'support_moderation_queue', metadata,
    Column('id', BYTEA(), nullable=False),
    Column('created', DateTime(timezone=True), nullable=True),
    Column('description', String, nullable=True),
    Column('storage_id', UUID(as_uuid=True), nullable=True),
    Column('links', JSONB(), nullable=True),
    Column('moderation_time', DateTime(timezone=True), nullable=True),
    Column('moderator', String, nullable=True),
    Column('source', String, nullable=True),
    Column('status', String, nullable=True),
)

support_block_history = Table(
    'support_block_history', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('ctime', DateTime(timezone=True), nullable=True),
    Column('hids', String, nullable=True),
    Column('moderator', String, nullable=False),
    Column('public_hashes', String, nullable=True),
    Column('type', String, nullable=True),
    Column('uids', String, nullable=True),
)

groups = Table(
    'groups', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('owner', BigInteger, nullable=False),
    Column('path', String),
    Column('size', BigInteger, nullable=False),
    Column('version', BigInteger),
    PrimaryKeyConstraint('id', name='pk_groups'),
)

group_links = Table(
    'group_links', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('gid', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger),
    Column('path', String),
    Column('version', BigInteger),
    Column('rights', Integer, nullable=False),
    Column('b2b_key', String),
    Column('universe_login', String),
    Column('universe_service', String),
    Column('date_created', DateTime(timezone=True), nullable=False),
    PrimaryKeyConstraint('id', name='pk_group_links'),
)

group_invites = Table(
    'group_invites', metadata,
    Column('id', UUID(as_uuid=True), nullable=False),
    Column('gid', UUID(as_uuid=True), nullable=False),
    Column('uid', BigInteger),
    Column('name', String),
    Column('version', BigInteger),
    Column('rights', Integer, nullable=False),
    Column('universe_login', String),
    Column('universe_service', String),
    Column('avatar', String),
    Column('status', String),
    Column('date_created', DateTime(timezone=True), nullable=False),
    PrimaryKeyConstraint('id', name='pk_group_invites'),
)

office_allowed_pdd_domains = Table(
    'office_allowed_pdd_domains', metadata,
    Column('id', String, nullable=False),
    PrimaryKeyConstraint('id', name='pk_office_allowed_pdd_domains'),
)

recount = Table(
    'recount', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('ctime', DateTime(timezone=True)),
    Column('mtime', DateTime(timezone=True)),
    Column('count', Integer),
    PrimaryKeyConstraint('uid', name='pk_recount'),
)

overdraft = Table(
    'overdraft', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('overdraft_date', String, nullable=False),
    Column('last_email_date', DateTime(timezone=True)),
    Column('last_push_date', DateTime(timezone=True)),
    PrimaryKeyConstraint('uid', name='pk_overdraft'),
)

inactive_users_flow = Table(
    'inactive_users_flow', metadata,
    Column('uid', BigInteger, nullable=False),
    Column('state', Enum(InactiveUsersFlowState), nullable=False),
    Column('start_time', DateTime(timezone=True), nullable=True),
    Column('update_time', DateTime(timezone=True), nullable=True),
    Column('debug_data', JSONB(), nullable=True),
    PrimaryKeyConstraint('uid', name='pk_inactive_users_flow'),
)
