# -*- coding: utf-8 -*-

import random
import re

from enum import Enum
from simplejson import JSONDecodeError as SimpleJSONDecodeError
from demjson import JSONDecodeError as DemJSONDecodeError
from urllib2 import HTTPError

from mpfs.common.util import from_json
from mpfs.core.services.common_service import RequestsPoweredServiceBase
from mpfs.config import settings


POSTGRES_USER = settings.postgres['user']
POSTGRES_PASSWORD = settings.postgres['password']
POSTGRES_SPECIAL_UID_FOR_COMMON_SHARD = settings.postgres['common_uid']
CONNECTION_STRING_TEMPLATE = 'postgresql://%(user)s:%(password)s@%(host)s:%(port)s/%(dbname)s'
USER_SHARD_RE = re.compile(settings.postgres['user_shard_re'])


class SharperError(Exception):
    pass


class SharpeiIncompleteResponseError(SharperError):
    pass


class SharpeiJsonResponseError(SharperError):
    pass


class SharpeiUserNotFoundError(SharperError):
    pass


class SharpeiInvalidUidError(SharpeiUserNotFoundError):
    pass


class SharpeiShardNotFoundError(SharperError):
    pass


class SharpeiUserRegisterFailedError(SharperError):
    pass


class SharpeiUserAlreadyRegisteredError(SharperError):
    pass


class SharpeiUserRegistrationAlreadyInProcessError(SharperError):
    pass


class Sharpei(RequestsPoweredServiceBase):
    name = 'sharpei'

    def get_shard(self, uid):
        shard_data_dict = self._get_data_by_uid(uid)
        return Shard(shard_data_dict['shard'])

    def get_common_shard(self):
        return self.get_shard(POSTGRES_SPECIAL_UID_FOR_COMMON_SHARD)

    def get_shard_by_id(self, shard_id):
        for shard in self._get_stat_shards():
            if shard.get_id() == shard_id:
                return shard
        raise SharpeiShardNotFoundError('shard with id %s not found' % shard_id)

    def get_shard_by_name(self, shard_name):
        for shard in self._get_stat_shards():
            if shard.get_name() == shard_name:
                return shard
        raise SharpeiShardNotFoundError('shard with name %s not found' % shard_name)

    def get_all_shards(self):
        return [s for s in self._get_stat_shards() if USER_SHARD_RE.match(s.get_name())]

    def get_all_shard_ids(self):
        return [s.get_id() for s in self.get_all_shards()]

    def create_user(self, uid, shard_id=None):
        resp = self._create_user(uid, shard_id=shard_id)
        user_shard = Shard(resp)
        if shard_id is not None and user_shard.get_id() != shard_id:
            raise SharpeiUserAlreadyRegisteredError('Uid `%s` is already registered on shard `%s`' %
                                                    (uid, user_shard.get_id()))
        return user_shard

    def update_user(self, uid, new_shard_id, create_if_not_exists=False):
        try:
            current_shard = self.get_shard(uid)
        except SharpeiUserNotFoundError:
            if not create_if_not_exists:
                raise
            self.create_user(uid, new_shard_id)
        else:
            if new_shard_id != current_shard.get_id():
                self._update_user(uid, current_shard.get_id(), new_shard_id)

    @staticmethod
    def _resp_to_data(resp):
        try:
            return from_json(resp.content)
        except (DemJSONDecodeError, SimpleJSONDecodeError):
            raise SharpeiJsonResponseError()

    def _update_user(self, uid, shard_id, new_shard_id):
        params = {
            'uid': uid,
            'shard_id': shard_id,
            'new_shard_id': new_shard_id
        }
        try:
            self.request('POST', '/update_user', params=params)
        except HTTPError as e:
            if e.code == 500:
                raise SharpeiUserRegisterFailedError('uid %s' % uid)
            raise

    def _create_user(self, uid, shard_id=None):
        params = {'uid': uid}
        if shard_id is not None:
            params['shard_id'] = shard_id
        try:
            resp = self.request('POST', '/create_user', params=params)
        except HTTPError as e:
            if e.code == 500:
                raise SharpeiUserRegisterFailedError('uid %s' % uid)
            raise
        return self._resp_to_data(resp)

    def _get_stat_data(self):
        resp = self.request('GET', '/stat')
        return self._resp_to_data(resp)

    def _get_stat_shards(self):
        stat_data = self._get_stat_data()
        for shard_values in stat_data.itervalues():
            yield Shard(shard_values)

    def _get_data_by_uid(self, uid):
        params = {'uid': uid, 'mode': 'all'}
        try:
            resp = self.request('GET', '/get_user', params=params)
        except HTTPError as e:
            if e.code == 404:
                raise SharpeiUserNotFoundError('user %s not found' % uid)
            if e.code == 400:
                raise SharpeiInvalidUidError('invalid uid %s' % uid)
            raise
        return self._resp_to_data(resp)


class HostRole(Enum):
    unknown = ''
    master = 'master'
    slave = 'replica'


class HostStatus(Enum):
    unknown = ''
    alive = 'alive'
    dead = 'dead'


class ShardHost(object):
    """
    :type _role: HostRole
    :type _status: HostStatus
    :type _lag: int
    :type _host: str
    :type _port: str
    :type _dbname: str
    :type _datacenter: str
    """

    _role = None
    _status = None
    _lag = None
    _host = None
    _port = None
    _dbname = None
    _datacenter = None

    def __init__(self, json_dict):
        try:
            self._host = json_dict['address']['host']
            self._port = json_dict['address']['port']
            self._dbname = json_dict['address']['dbname']
            self._datacenter = json_dict['address']['dataCenter']

            try:
                self._status = HostStatus(json_dict['status'])
            except ValueError:
                self._status = HostStatus.unknown

            try:
                self._role = HostRole(json_dict['role'])
            except ValueError:
                self._role = HostRole.unknown

            self._lag = json_dict['state']['lag']
        except KeyError:
            raise SharpeiIncompleteResponseError()

    def __str__(self):
        return '<DB host: %s:%s/%s [%s]>' % (self._host, self._port, self._dbname, self._role)

    def is_master(self):
        return self.role == HostRole.master

    def get_connection_string(self):
        return CONNECTION_STRING_TEMPLATE % {
            'user': POSTGRES_USER,
            'password': POSTGRES_PASSWORD,
            'host': self._host,
            'port': self._port,
            'dbname': self._dbname
        }

    @property
    def role(self):
        return self._role

    @property
    def status(self):
        return self._status

    @property
    def lag(self):
        return self._lag

    @property
    def host(self):
        return self._host

    @property
    def port(self):
        return self._port

    @property
    def dbname(self):
        return self._dbname

    @property
    def datacenter(self):
        return self._datacenter


class MasterNotFoundError(Exception):
    pass


class MasterIsDeadError(Exception):
    pass


class SlavesNotFoundError(Exception):
    pass


class Shard(object):
    """
    :type _hosts: list[ShardHost]
    :type _name: str
    :type _id: str
    """

    _hosts = None
    _name = None
    _id = None

    def __init__(self, json_dict):
        try:
            self._id = str(json_dict['id'])
            self._name = json_dict['name']
            self._hosts = []
            for db in json_dict['databases']:
                try:
                    host = ShardHost(db)
                    self._hosts.append(host)
                except SharpeiIncompleteResponseError:
                    raise  # не обрабатываем эту ошибку, оставил обработчик, чтобы сделать ее тут, если понадобится
        except KeyError:
            raise SharpeiIncompleteResponseError()

    def get_id(self):
        return self._id

    def get_name(self):
        return self._name

    def get_master(self):
        for host in self._hosts:
            if host.is_master():
                if host.status == HostStatus.dead:
                    raise MasterIsDeadError(host.host)
                return host
        raise MasterNotFoundError()

    def get_random_slave(self, filter_dc=None, replication_lag_theshold=None):
        slaves = []
        for host in self._hosts:
            if host.is_master():
                continue
            if filter_dc is not None and host.datacenter in filter_dc:
                continue
            if replication_lag_theshold is not None and host.lag > replication_lag_theshold:
                continue
            if host.status != HostStatus.alive:
                continue
            slaves.append(host)

        if not slaves:
            raise SlavesNotFoundError()
        return random.choice(slaves)

    def __str__(self):
        return 'Shard id=`%s` (%s)' % (self._id, ','.join([str(h) for h in self._hosts]))
