# -*- coding: utf-8 -*-
"""
Набор механизмов авторизации.
"""
import re
import urllib
import authparser

from urlparse import urlparse
from functools import partial, wraps
from ticket_parser2.api.v1.exceptions import TvmException

from mpfs.common.errors import PassportCookieInvalid
from mpfs.common.errors import PassportTokenExpired
from mpfs.common.static.tags import PLATFORM_NETWORK_AUTHORIZATION_SILENT_MODE_LOG_PREFIX
from mpfs.common.static.tags import conf_sections
from mpfs.common.static.tags.conf_sections import (
    NETWORK_AUTHORIZATION,
    OAUTH_CLIENT_ID,
)
from mpfs.common.util.credential_sanitizer import CredentialSanitizer
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.iptools import IPRangeList
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.config import settings
from mpfs.common.static import tags
from mpfs.core.services.passport_service import (
    yateam_blackbox,
    blackbox,
    PASSPORT_NAMES_TO_ATTRIBUTES,
    PlatformUserDetailsHelper,
)
from mpfs.core.services.tvm_service import tvm, TVMEnv
from mpfs.core.services.tvm_service import TVMTicket
from mpfs.core.services.tvm_2_0_service import tvm2, TVM2Ticket
from mpfs.core.services.conductor_service import ConductorService
from mpfs.engine import process
from mpfs.platform.common import PlatformClient, PlatformUser, PlatformConfigClientInfo, GroupRateLimit
from mpfs.platform.exceptions import TVM2AuthorizationUidMismatchError, ServiceUnavailableError
from mpfs.platform.utils import parse_cookie, unquote
from mpfs.platform.credentials import (
    OAuthAuthCredentials,
    PassportCookieCredentials,
    BaseCredentials,
)
from mpfs.platform.dispatchers import InternalDispatcher
from mpfs.platform.events import SettingsChangeEvent, EventListenerMixin

from mpfs.platform.common import logger
from mpfs.platform.utils.uid_formatters import to_yateam_uid, to_device_uid


PLATFORM_TLDS = settings.platform['tlds']
FEATURE_TOGGLES_TVM2_MULTIAUTH = settings.feature_toggles['tvm2_multiauth']


class BaseAuthMetaclass(type):
    """
    Метакласс, который реализует механизм, аналогичный login в django
    https://github.com/django/django/blob/master/django/contrib/auth/__init__.py#L155
    Фактически он оборачивает функцию authotize в классах-наследниках BaseAuth и вызывает метод on_success в случае,
    если authotize вернула True.
    """

    def __new__(mcs, name, parents, attrs):
        if 'authorize' in attrs:
            attrs['authorize'] = mcs._wrap_authorize_method(attrs['authorize'])
        return super(BaseAuthMetaclass, mcs).__new__(mcs, name, parents, attrs)

    @staticmethod
    def _wrap_authorize_method(auth_func):
        @wraps(auth_func)
        def wrapper(self, request):
            is_authorized = auth_func(self, request)
            if is_authorized:
                self.on_success(request)
            return is_authorized
        return wrapper


class BaseAuth(object):
    """Базовый класс для бэкэндов авторизации"""
    __metaclass__ = BaseAuthMetaclass

    def on_success(self, request):
        """
        Функция, которая будет вызвана после обработки функции `authorize` в случае, если она вернет True
        """
        request.auth_method = self
        uid = request.user.uid if request.user is not None else None
        experiment_manager.update_context(uid=uid)


class BaseAuthWithCredentials(BaseAuth):
    """
    Базовый класс авторизации

    Процесс авторизации разбит на 2 метода: _get_credentials и _validate_credentials. При попытке авторизировать,
    сначала запускается метод _get_credentials. Если в запросе передана вся необходимая информация для авторизации
    данным способом, то метод вернет соответствующего наследника BaseCredentials, иначе вернется None и попытки
    авторизации не произойдет. После вызывается _validate_credentials с только что полученными credential,
    авторизирующий запрос по полученным данным.
    """
    credential_class = BaseCredentials
    """Класс, который возвращается методом get_credentials"""

    def authorize(self, request):
        credentials = self._get_credentials(request)
        if credentials is None:
            return False
        authorization_succeeded = self._validate_credentials(request, credentials)
        if not authorization_succeeded:
            if credentials:
                logger.error_log.error('Authorization Failed. Authorization method: %s; Credential: %s' %
                                       (self.__class__.__name__, CredentialSanitizer.hash_credential(str(credentials))))
            return False
        return True

    def _get_credentials(self, request):
        """
        Возвращает объект класса BaseCredentials, которые затем используются для авторизации запроса

        Если информации в запросе недостаточно для авторизации, то вернется None.
        :param PlatfromRequest request: Запрос, который необходимо авторизовать.
        :return: Credentials
        :rtype: BaseCredentials or None
        """
        raise NotImplementedError()

    def _validate_credentials(self, request, credentials):
        """
        Пытается авторизировать запрос по переданным креденшилам, устанавливая в нём необходимые атрибуты, как правило
        `client` и `user`.

        :param PlatfromRequest request: Запрос, который необходимо авторизовать.
        :param BaseCredentials credentials: credentials
        :return: Возвращает True в случае успешной авторизации, иначе возвращает False.
        :rtype: bool
        """
        raise NotImplementedError()


class InternalAuth(BaseAuth):
    """
    Внутренняя авторизация. Используется только во внутреннем API.

    Идентификация клиентов во внутреннем API происходит с помощью двух **обязательных для любого запроса** параметров:
        * client_id -- идентификатор клиента,
        * client_name -- отображаемое имя клиента.

    Параметры для идентификации клиентов можно передавать 2мя способами:
        1. В заголовке Authorization в формате Authorization: Internal client_id=<client_id>;client_name=<client_name>
        2. В query string запроса.

    Примеры:
        $ curl -H 'Authorization: Internal client_id=ya-home;client_name=Yandex Home' 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses'
        $ curl 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses?client_id=ya-home&client_name=Yandex Home'
    """
    INTERNAL_AUTH_HEADER_REGEX = re.compile(
        r'^Internal\s+((;?\s*client_id=(?P<client_id>[^;$]+))|(;?\s*client_name=(?P<client_name>[^;$]+)))+', re.I)

    def authorize(self, request):
        if request.mode != tags.platform.INTERNAL:
            return False

        auth_header = request.raw_headers.get('Authorization', '')
        client_id = None
        client_name = None
        if auth_header:
            match = self.INTERNAL_AUTH_HEADER_REGEX.match(auth_header)
            if match:
                client_id = match.groupdict().get('client_id', '')
                client_name = match.groupdict().get('client_name', '')
        else:
            client_id = request.args.get('client_id', '')
            client_name = request.args.get('client_name', '')

        from mpfs.platform.dispatchers import InternalDispatcher
        _, uid, _, _ = InternalDispatcher.split_path(request._raw_path)

        if uid is None:
            return False

        if client_id and client_name:
            client = PlatformClient()
            client.ip = request.remote_addr
            client.id = unquote(client_id)
            client.name = unquote(client_name)
            client.is_yandex = True
            request.client = client

            user = PlatformUser()
            user.uid = uid
            request.user = user

            logger.default_log.info(
                'Successfully authorized with InternalAuth; client_id=%s client_name=%s' % (client_id, client_name))

            return True

        return False


class InternalTokenAuth(EventListenerMixin, BaseAuth):
    """
    Авторизация для сервисов по токену. Используется только во внутреннем API.

    Происходит с помощью обязательного параметра:
        * token
    И необязательного:
        * uid

    Если uid нет, то будем искать его в url. Если его и в url нет, то дальнейшая авторизация будет проводиться в
    заисимости от значения атрибута auth_user_required в хэндлере.

    Параметры для идентификации клиентов можно передавать 2мя способами:
        1. В заголовке Authorization в формате Authorization: ClientToken token=<client_token>[;uid=<uid>]
        2. В query string запроса.

    Примеры:
        $ curl -H 'Authorization: ClientToken token=a2f313419bd193e79cad;uid=123456789' 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses'
        $ curl 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses?token=a2f313419bd193e79cad&uid=123456789'
    """
    INTERNAL_AUTH_HEADER_REGEX = re.compile(
        r'^ClientToken\s+((;?\s*token=(?P<token>[^;$]+))|(;?\s*uid=(?P<uid>[^;$]+)))+', re.I)

    event = SettingsChangeEvent

    def __init__(self, *args, **kwargs):
        super(InternalTokenAuth, self).__init__(*args, **kwargs)
        self.client_info_by_token = self._get_client_info_by_token()

    @staticmethod
    def _get_client_info_by_token():
        config_data = settings.platform['auth']
        config_data = filter(lambda i: 'token' in i['auth_methods'], config_data)

        data = {}
        for props in config_data:
            if not props['enabled']:
                continue

            if 'token' not in props:
                raise KeyError('Token listed in auth methods but no token value was found')
            token = str(props['token'])

            if token in data:
                raise KeyError('All auth tokens should be unique')

            data[token] = PlatformConfigClientInfo.from_client_info_dict(props)

        return data

    def handle_event(self, event):
        self.client_info_by_token = self._get_client_info_by_token()

    def authorize(self, request):
        if request.mode != tags.platform.INTERNAL:
            return False

        auth_header = request.raw_headers.get('Authorization', '')
        token = None
        uid = None
        if auth_header:
            match = self.INTERNAL_AUTH_HEADER_REGEX.match(auth_header)
            if match:
                token = match.groupdict().get('token')
                uid = match.groupdict().get('uid')

            if uid is None:
                _, uid, _, _ = InternalDispatcher.split_path(request._raw_path)
            if uid is None:
                uid = request.args.get('uid')
        else:
            _, uid, _, _ = InternalDispatcher.split_path(request._raw_path)
            if uid is None:
                token = request.args.get('token')
                uid = request.args.get('uid')

        if token is not None:
            if token not in self.client_info_by_token:
                return False
            client_info = self.client_info_by_token[token]

            client = PlatformClient()
            client.ip = request.remote_addr
            client.id = client_info.oauth_client_id
            client.name = client_info.oauth_client_name
            client.scopes = client_info.scopes
            client.limits = client_info.limits
            client.token = token
            client.is_yandex = True
            request.client = client

            user = None
            if uid is not None:
                user = PlatformUser()
                user.uid = uid
            request.user = user

            return True

        return False


class ExternalTokenAuth(EventListenerMixin, BaseAuth):
    """
    Авторизация для сервисов по токену. Используется во внешнем API для ручек Data API. Нужна для того, чтобы
    авторизовывать сервис для незалогиненых пользователей

    Происходит с помощью обязательных параметров:
        * token
        * device

    Параметры для идентификации клиентов нужно передавать в заголовке:
        Authorization: ExternalToken token=<client_token>;device=<value>

    Примеры:
        $ curl -H 'Authorization: ExternalToken token=a2f313419bd193e79cad;device=123456789' 'https://extapi.disk.yandex.net/v1/data/app/databases'
    """
    AUTH_HEADER_REGEX = re.compile(
        r'^ExternalToken\s+((;?\s*token=(?P<token>[^;$]+))|(;?\s*device=(?P<device>[^;$]+)))+', re.I)

    event = SettingsChangeEvent

    def __init__(self, *args, **kwargs):
        super(ExternalTokenAuth, self).__init__(*args, **kwargs)
        self.client_info_by_token = self._build_token_to_client_info_map()

    @staticmethod
    def _build_token_to_client_info_map():
        config_data = settings.platform['auth']
        config_data = filter(lambda i: 'ext_tokens' in i['auth_methods'], config_data)

        data = {}
        for props in config_data:
            if not props['enabled']:
                continue

            if 'ext_tokens' not in props:
                raise KeyError('Token listed in auth methods but no token values were found')

            ext_tokens = props['ext_tokens']
            if not isinstance(ext_tokens, list):
                raise TypeError('ext_tokens expected to be list, got `%s` instead' % type(ext_tokens))

            client_data = PlatformConfigClientInfo.from_client_info_dict(props)

            for token in ext_tokens:
                token = str(token)

                if token in data:
                    raise KeyError('All auth tokens should be unique')

                data[token] = client_data
        return data

    def handle_event(self, event):
        self.client_info_by_token = self._build_token_to_client_info_map()

    def authorize(self, request):
        if request.mode != tags.platform.EXTERNAL:
            return False

        auth_header = request.raw_headers.get('Authorization')
        if not auth_header:
            return False

        match = self.AUTH_HEADER_REGEX.match(auth_header)
        if not match:
            return False

        token = match.groupdict().get('token')
        device_id = match.groupdict().get('device')

        if device_id is None or token is None:
            return False

        if token not in self.client_info_by_token:
            return False
        client_info = self.client_info_by_token[token]

        client = PlatformClient()
        client.ip = request.remote_addr
        client.id = client_info.oauth_client_id
        client.name = client_info.oauth_client_name
        client.scopes = client_info.scopes
        client.limits = client_info.limits
        client.token = token
        client.is_yandex = True
        request.client = client

        user = PlatformUser()
        # тут важно, что добавляем в переданный параметр device_id, иначе дыра в безопасности :)
        user.uid = to_device_uid(device_id)
        request.user = user

        return True


class InternalConductorAuth(EventListenerMixin, BaseAuth):
    """
    Авторизация для сервисов по IP. Используется только во внутреннем API.

    Происходит с помощью обязательного параметра:
        * IP
    И необязательного:
        * uid

    Если uid нет, то будем искать его в url. Если его и в url нет, то авторизация не пройдет.

    Параметры для идентификации клиентов можно передавать 2мя способами:
        1. В заголовке Authorization в формате Authorization: IP uid=<uid>;
        2. В query string запроса.
        3. Если заголовка Authorization нет, то пробуем авторизовать по IP, при этом uid будем искать в url или в
           query string

    Примеры:
        $ curl -H 'Authorization: IP uid=123456789' 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses'
        $ curl 'http://intapi.disk.yandex.net:8080/v1/personality/profile/addresses?uid=123456789'
    """
    INTERNAL_AUTH_HEADER_REGEX = re.compile(r'^IP\s+(;?\s*uid=(?P<uid>[^;$]+))+', re.I)

    event = SettingsChangeEvent

    def __init__(self, *args, **kwargs):
        super(InternalConductorAuth, self).__init__(*args, **kwargs)
        self.client_info_by_item = self._get_client_info_by_item()
        self.conductor_service = ConductorService()

    @staticmethod
    def _get_client_info_by_item():
        config_data = settings.platform['auth']
        config_data = filter(lambda i: 'conductor' in i['auth_methods'], config_data)

        data = {}
        for props in config_data:
            if not props['enabled']:
                continue

            if 'conductor_items' not in props:
                raise KeyError('Conductor listed in auth methods but no conductor items were found')

            items = props['conductor_items']

            for item in items:
                if item in data:
                    raise KeyError('All conductor items should be unique')

                data[item] = PlatformConfigClientInfo.from_client_info_dict(props)
        return data

    def handle_event(self, event):
        self.client_info_by_item = self._get_client_info_by_item()

    @staticmethod
    def fill_request_client(request, client_id, client_name, scopes, uid, limits):
        client = PlatformClient()
        client.ip = request.remote_addr
        client.id = client_id
        client.name = client_name
        client.scopes = scopes
        client.limits = limits
        client.is_yandex = True
        request.client = client

        user = PlatformUser()
        user.uid = uid
        request.user = user

    def authorize(self, request):
        if request.mode != tags.platform.INTERNAL:
            return False

        auth_header = request.raw_headers.get('Authorization', '')
        uid = None
        if auth_header:
            match = self.INTERNAL_AUTH_HEADER_REGEX.match(auth_header)
            if match:
                uid = match.groupdict().get('uid')

        if uid is None:
            _, uid, _, _ = InternalDispatcher.split_path(request._raw_path)
        if uid is None:
            uid = request.args.get('uid')
        if uid is None:
            return False

        ip = request.remote_addr
        if not ip:
            return False

        if self.conductor_service.is_conductor_auth_fallback_mode(request):
            self.fill_request_client(request, 'internal', 'internal', [], uid, [])
            return True

        item = self.conductor_service.get_conductor_item_by_ip(ip)
        if item is not None:
            if item not in self.client_info_by_item:
                return False
            client_info = self.client_info_by_item[item]
            self.fill_request_client(request,
                                     client_info.oauth_client_id,
                                     client_info.oauth_client_name,
                                     client_info.scopes,
                                     uid,
                                     client_info.limits)
            return True

        return False


class TVMAuthBase(BaseAuth):
    """
    Общий класс аутентификации TVM.

    Содержит логику конфигурации аутентификации
    """
    envs_auth_names = None
    auth_tvm_client_ids_name = None

    def __init__(self, envs_auth_names, auth_tvm_client_ids_name, *args, **kwargs):
        super(TVMAuthBase, self).__init__(*args, **kwargs)
        self.__class__.envs_auth_names = envs_auth_names
        self.__class__.auth_tvm_client_ids_name = auth_tvm_client_ids_name
        self.client_info_by_client_id = self._get_client_info_by_client_id()

    @classmethod
    def _get_client_info_by_client_id(cls):
        data = {}
        for env, auth_name in cls.envs_auth_names.items():
            config_data = settings.platform['auth']
            config_data = filter(lambda i: auth_name in i['auth_methods'], config_data)

            data[env] = {}
            for props in config_data:
                if not props['enabled']:
                    continue

                if cls.auth_tvm_client_ids_name not in props:
                    raise KeyError('TVM listed in auth methods but no %s value was found' % cls.auth_tvm_client_ids_name)
                client_ids = props[cls.auth_tvm_client_ids_name]

                for client_id in client_ids:
                    if client_id in data[env]:
                        raise KeyError('All client ids should be unique')

                    data[env][client_id] = PlatformConfigClientInfo.from_client_info_dict(props)
        return data


class TVMAuth(EventListenerMixin, TVMAuthBase):
    """Авторизация для сервисов по TVM тикету.

    Происходит с помощью заголовков:
        * Ticket: <TVM_ticket>                              (обязательный)
        * Authorization: TVM uid=<uid>|yateam-<yateam-uid>  (опциональный)

    Авторизация работает как обычными тикетами, так и с тикетами Yateam окружения.

    TVM тикеты с Yateam окружением: тикеты полученные в yandex-team blackbox'е

    **Определение uid'а**:

    Если uid не передан в тикете, то попытаемся взять его из заголовков.
    Если тикет содержит uid, то указание uid'а в заголовоке будет игнорированно.

    Примеры:

    1) Обычный uid:

    $ curl -H 'Authorization: TVM uid=<uid>' \
           -H 'Ticket: <ticket>' \
           'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses'

    2) yandex-team uid:

      $ curl -H 'Authorization: TVM uid=yateam-<uid>' \
             -H 'Ticket: <ticket>' \
             'http://intapi.disk.yandex.net:8080/v1/personality/profile/addresses'
    """
    TVM_AUTH_HEADER_REGEX = {TVMEnv.External: re.compile(r'^TVM\s+(;?\s*uid=(?P<uid>[^;$]+))+', re.I),
                             TVMEnv.YaTeam: re.compile(r'^TVM\s+(;?\s*uid=yateam-(?P<uid>[^;$]+))+', re.I)}
    event = SettingsChangeEvent

    def __init__(self, *args, **kwargs):
        envs_auth_names = {TVMEnv.External: conf_sections.TVM,
                           TVMEnv.YaTeam: conf_sections.YATEAM_TVM}
        super(TVMAuth, self).__init__(envs_auth_names, conf_sections.TVM_CLIENT_IDS, *args, **kwargs)

    def handle_event(self, event):
        self.client_info_by_client_id = self._get_client_info_by_client_id()
        self.register_tvm_clients()

    def authorize(self, request):
        if request.mode != tags.platform.INTERNAL:
            return False

        tvm_ticket = request.raw_headers.get('Ticket')
        if not tvm_ticket:
            return False
        ticket_info = tvm.validate_ticket(TVMTicket.build_tvm_ticket(tvm_ticket, is_external=True))
        if ticket_info is None:
            return False
        if not ticket_info.has_client_id:
            return False

        client_id = ticket_info.client_ids[0]
        for env in TVMEnv:
            if client_id in self.client_info_by_client_id[env]:
                ticket_env = env
                break
        else:
            # client_id не зарегистрирова как клиент Диска
            return False

        uid = None
        if ticket_info.default_uid:
            uid = str(ticket_info.default_uid[0])

        auth_header = request.raw_headers.get('Authorization', '')
        if uid is None and auth_header:
            match = self.TVM_AUTH_HEADER_REGEX[ticket_env].match(auth_header)
            if match:
                uid = match.groupdict().get('uid')

        client_info = self.client_info_by_client_id[ticket_env][client_id]

        client = PlatformClient()
        client.ip = request.remote_addr
        client.id = client_info.oauth_client_id
        client.name = client_info.oauth_client_name
        client.scopes = client_info.scopes
        client.limits = client_info.limits
        client.is_yandex = True
        request.client = client

        user = None
        if uid is not None:
            user = PlatformUser()
            if ticket_env == TVMEnv.YaTeam:
                user.uid = to_yateam_uid(uid)
            else:
                user.uid = uid
        request.user = user

        return True

    @staticmethod
    def register_tvm_clients():
        """Составить список client_id сервисов с TVM авторизацией.

        Для которых в последствии будем получать публичные ключи.
        В списке будут client IDs всех окружений: Prod, YateamProd...
        """
        client_ids = []
        for auth_name in [conf_sections.TVM, conf_sections.YATEAM_TVM]:
            tvm_services = [service
                            for service in settings.platform['auth']
                            if auth_name in service['auth_methods']]
            for service in tvm_services:
                if conf_sections.TVM_CLIENT_IDS not in service:
                    logger.error_log.error('Service "%s" is missing %s' % (service['name'],
                                                                           conf_sections.TVM_CLIENT_IDS))
                    continue
                client_ids.extend(service[conf_sections.TVM_CLIENT_IDS])
        process.set_tvm_clients(client_ids)


class TVM2TicketsOnlyAuth(EventListenerMixin, TVMAuthBase):
    """
    Авторизация для севрисов по TVM 2.0 тикету

    Происходит с помощью обязательноего параметра в хэдере.
        X-Ya-Service-Ticket: <TVM_service_ticket>

    Можно авторизироваться без уида. Передать уид можно через url или через пользовательский
    тикет в хедере 'X-Ya-User-Ticket: <TVM_user_ticket>'. Если пользовательский тикет передан, то будет произведена
    его валидация.

    Примеры:
      $ curl -H 'X-Ya-Service-Ticket: <TVM_service_ticket>' 'http://intapi.disk.yandex.net:8080/v1/12345/personality/profile/addresses'
      $ curl -H 'X-Ya-Service-Ticket: <TVM_service_ticket>' -H 'X-Ya-User-Ticket: <TVM_user_ticket>' 'http://intapi.disk.yandex.net:8080/v1/personality/profile/addresses'
      $ curl -H 'X-Ya-Service-Ticket: <TVM_service_ticket>' 'http://intapi.disk.yandex.net:8080/v1/data/users'
    """
    allow_yateam = False
    event = SettingsChangeEvent
    conf_auth_name = conf_sections.TVM2_TICKETS_ONLY
    """соответствующий данному классу идентификатор авторизации в конфиге авторизаций"""

    def __init__(self, allow_yateam=False, *args, **kwargs):
        self.allow_yateam = allow_yateam
        super(TVM2TicketsOnlyAuth, self).__init__(envs_auth_names={TVMEnv.External: self.conf_auth_name},
                                                  auth_tvm_client_ids_name='tvm_2_0_client_ids',
                                                  *args, **kwargs)

    def authorize(self, request):
        if request.mode != tags.platform.INTERNAL:
            return False

        raw_tvm_service_ticket = request.raw_headers.get('X-Ya-Service-Ticket')
        if not raw_tvm_service_ticket:
            return False

        try:
            ticket_info = tvm2.validate_service_ticket(TVM2Ticket.build_tvm_ticket(raw_tvm_service_ticket))
        except TvmException:
            return False
        client_id = ticket_info.src

        if client_id not in self.client_info_by_client_id[TVMEnv.External]:
            return False

        uid = None
        try:
            uid = self.extract_uid(request)
        except TvmException:
            return False

        if uid is None:
            _, uid, _, _ = InternalDispatcher.split_path(request._raw_path)

        client_info = self.client_info_by_client_id[TVMEnv.External][client_id]

        client = PlatformClient()
        client.ip = request.remote_addr
        client.id = client_info.oauth_client_id
        client.name = client_info.oauth_client_name
        client.scopes = client_info.scopes
        client.limits = client_info.limits
        client.is_yandex = True
        request.client = client

        user = None
        if uid is not None:
            user = PlatformUser()
            user.uid = uid
        request.user = user

        return True

    def extract_uid(self, request):
        user_ticket, is_yateam = self._get_user_ticket()

        if user_ticket is None:
            return None

        return to_yateam_uid(str(user_ticket.default_uid)) if is_yateam else str(user_ticket.default_uid)

    def handle_event(self, event):
        self.client_info_by_client_id = self._get_client_info_by_client_id()

    def _get_user_ticket(self):
        user_ticket = process.get_tvm_2_0_user_ticket()
        if user_ticket is None:
            return None, None

        try:
            user_ticket = tvm2.validate_user_ticket(user_ticket)
            return user_ticket, False
        except TvmException:  # failsafe, try extract ya-team uid if allowed
            if self.allow_yateam:
                yateam_user_ticket = tvm2.validate_yateam_user_ticket(user_ticket)
                return yateam_user_ticket, True
            else:
                raise


class TVM2Auth(TVM2TicketsOnlyAuth):
    """
    Расширяет оригинальный класс, добавляя возможность передачи uid через хедер 'X-Uid: <uid>'

    Примеры:
      $ curl -H 'X-Ya-Service-Ticket: <TVM_service_ticket>' -H 'X-Uid: 12345' 'http://intapi.disk.yandex.net:8080/v1/personality/profile/addresses'
    """

    conf_auth_name = conf_sections.TVM2

    def extract_uid(self, request):
        uid_from_header = request.raw_headers.get('X-Uid')
        user_ticket, is_yateam = self._get_user_ticket()

        if uid_from_header is None and user_ticket is None:
            return None

        uid = None

        if user_ticket is None:
            uid = uid_from_header

        if uid_from_header is None:
            uid = user_ticket.default_uid

        if user_ticket and uid_from_header:
            uids_from_user_ticket = [user_ticket.default_uid]
            uids_from_user_ticket.extend(user_ticket.uids)

            if uid_from_header not in map(str, uids_from_user_ticket):
                logger.default_log.info('uid mismatch: x-uid %s != user_ticket_uids %s; access: %s' %
                                        (uid_from_header, uids_from_user_ticket, False))
                raise TVM2AuthorizationUidMismatchError(x_uid=uid_from_header, user_ticket_uids=uids_from_user_ticket,
                                                        user_ticket=user_ticket)

            if not FEATURE_TOGGLES_TVM2_MULTIAUTH and uid_from_header != str(user_ticket.default_uid):
                logger.default_log.info('uid mismatch: x-uid %s != user_ticket_uids %s; access: %s' %
                                        (uid_from_header, uids_from_user_ticket, True))
                raise TVM2AuthorizationUidMismatchError(x_uid=uid_from_header, user_ticket_uids=uids_from_user_ticket,
                                                        user_ticket=user_ticket)

            uid = uid_from_header

        return to_yateam_uid(str(uid)) if is_yateam else str(uid)


class PassportCookieAuth(EventListenerMixin, BaseAuthWithCredentials):
    """
    Авторизация по паспортным кукам

    Парсер кук: https://wiki.yandex-team.ru/passport/libauth-client-parser#pythonbinding
    Проверка кук: http://doc.yandex-team.ru/blackbox/reference/MethodSessionID.xml
    """

    HTTP_SESSION_COOKIE_NAME = 'Session_id'
    HTTPS_SESSION_COOKIE_NAME = 'sessionid2'

    good_passport_statuses = ('NEED_RESET', 'VALID')
    passport_client = blackbox

    event = SettingsChangeEvent
    credential_class = PassportCookieCredentials

    def __init__(self, *args, **kwargs):
        super(PassportCookieAuth, self).__init__(*args, **kwargs)
        self.client_info_by_host, self.client_info_by_client_id = self._get_client_info_by_host_or_client_id()

    def _get_client_info_by_host_or_client_id(self):
        # преобразование конфига в нужную структуру
        client_info_by_host = {}
        client_info_by_client_id = {}
        tlds = '(%s)' % '|'.join(re.escape(x) for x in settings.platform['tlds'])
        config_data = settings.platform['auth']
        config_data = filter(lambda i: 'cookie' in i['auth_methods'], config_data)

        for props in config_data:
            if not props['enabled']:
                continue

            client_id = props.get('cookie_auth_client_id')
            if client_id in client_info_by_client_id:
                raise KeyError('All client ids should be uniq')

            if client_id is not None:
                client_info_by_client_id[client_id] = {
                    'scopes': props.get('oauth_scopes', []),
                    'id': props['oauth_client_id'],
                    'name': props['oauth_client_name'],
                    'cookie_auth_client_id': client_id,
                    'validators': [],
                    'limits': GroupRateLimit.get_limit_groups_from_config(props.get('limit_groups', [])),
                }

            for origin_host in props['allowed_origin_hosts']:
                origin_host %= {'tlds': tlds}

                if client_id is None:
                    if origin_host in client_info_by_host:
                        raise KeyError('All origin hosts should be uniq')

                if origin_host.startswith('^') and origin_host.endswith('$'):
                    regexp = re.compile(origin_host)
                    validator = partial(self.host_regexp_validator, regexp)
                else:
                    validator = partial(self.host_string_validator, origin_host)

                if client_id is None:
                    client_info_by_host[origin_host] = {
                        'scopes': props.get('oauth_scopes', []),
                        'id': props['oauth_client_id'],
                        'name': props['oauth_client_name'],
                        'validator': validator,
                        'limits': GroupRateLimit.get_limit_groups_from_config(props.get('limit_groups', [])),
                    }
                else:
                    client_info_by_client_id[client_id]['validators'].append(validator)

        return client_info_by_host, client_info_by_client_id

    def handle_event(self, event):
        self.client_info_by_host, self.client_info_by_client_id = self._get_client_info_by_host_or_client_id()

    @staticmethod
    def host_regexp_validator(original_host_regexp, host):
        return original_host_regexp.match(host) is not None

    @staticmethod
    def host_string_validator(original_host, host):
        return original_host == host

    def validate_session_cookie(self, http_session_cookie, https_session_cookie, ip, host, multi_session=False):
        if isinstance(http_session_cookie, unicode):
            http_session_cookie = http_session_cookie.encode('utf-8')
        if isinstance(https_session_cookie, unicode):
            https_session_cookie = https_session_cookie.encode('utf-8')
        session = authparser.Session()
        # Первичная локальная проверка. 1 - кука валидна и обычного типа
        if session.parse(http_session_cookie) == 1:
            if multi_session is False:
                data = self.passport_client.check_session_id(http_session_cookie, https_session_cookie, ip, host)
                if data['status'] in self.good_passport_statuses:
                    return data
            else:
                data = self.passport_client.check_multi_session_id(http_session_cookie, https_session_cookie, ip, host)
                return [u for u in data if u['status'] in self.good_passport_statuses]
        raise ValueError('Bad cookie')

    @staticmethod
    def _process_passport_cookie(cookie):
        """
        Костыль для неправильно сформированных кук паспорта
        """
        if cookie:
            if isinstance(cookie, unicode):
                cookie = cookie.encode('utf-8')
            return urllib.unquote(cookie).strip('\'"')
        return None

    def _get_credentials(self, request):
        if request.mode != tags.platform.EXTERNAL:
            return None

        raw_host = request.raw_headers.get('Host', None)
        if raw_host:
            raw_host = raw_host.split(':', 1)[0]
        raw_origin = request.raw_headers.get('Origin', None)
        raw_cookie = request.raw_headers.get('Cookie', None)
        if not all((raw_host, raw_origin, raw_cookie)):
            return None

        # проверка пользовательской куки и получение иформации о пользователе
        parsed_cookies = parse_cookie(raw_cookie)

        http_session_cookie = self._process_passport_cookie(parsed_cookies.get(self.HTTP_SESSION_COOKIE_NAME, None))
        if not http_session_cookie:
            return None
        https_session_cookie = self._process_passport_cookie(
            parsed_cookies.get(self.HTTPS_SESSION_COOKIE_NAME, None))

        ip = request.remote_addr
        x_uid = request.raw_headers.get('X-Uid')

        return self.credential_class(raw_host, raw_origin, http_session_cookie, https_session_cookie,
                                     ip, request.cookie_auth_client_id, x_uid)

    def _validate_credentials(self, request, credentials):
        # обработка header-а Origin
        parsed_origin = urlparse(credentials.origin)
        if parsed_origin.scheme != 'https':
            logger.error_log.error('Origin header must have https scheme')
            return False
        origin_host = parsed_origin.hostname

        client_info = None
        if credentials.cookie_auth_client_id is not None:
            if credentials.cookie_auth_client_id not in self.client_info_by_client_id:
                logger.error_log.error(
                    'Don\'t know client with cookie_auth_client_id `%s`' % credentials.cookie_auth_client_id)
                return False
            info = self.client_info_by_client_id[credentials.cookie_auth_client_id]
            for validator in info['validators']:
                if validator(origin_host):
                    client_info = info
                    break
        else:
            for info in self.client_info_by_host.values():
                validator = info['validator']
                if validator(origin_host):
                    client_info = info
                    break

        if client_info is None:
            logger.error_log.error('Could\'t match any client. Probably wrong origin was sent')
            return False

        multi_session = credentials.x_uid is not None
        try:
            userinfo = self.validate_session_cookie(credentials.http_session_cookie, credentials.https_session_cookie,
                                                    credentials.ip, credentials.host, multi_session=multi_session)
        except (PassportCookieInvalid, ValueError):
            logger.error_log.error('Failed to validate cookie', exc_info=True)
            return False
        except Exception:
            logger.error_log.error('Got exception while processing Blackbox request', exc_info=True)
            raise ServiceUnavailableError()

        if multi_session:
            # при multi_session выбираем пользователя из списка
            for user in userinfo:
                if user['uid'] == credentials.x_uid:
                    userinfo = user
                    break
            else:
                logger.error_log.info('Uid "%s" not found in valid user_infos: %s', credentials.x_uid, userinfo)
                return False

        # field 'ticket' used in tvm1, but not tvm2
        if userinfo.get('ticket'):
            try:
                tvm_ticket = TVMTicket.build_tvm_ticket(userinfo['ticket'], is_external=False)
                process.set_external_tvm_ticket(tvm_ticket)
            except Exception:
                logger.error_log.error('Failed to fetch TVM ticket', exc_info=True)
                return False

        # проставляем клиента и пользователя в реквест
        client = PlatformClient()
        client.scopes = client_info['scopes']
        client.id = client_info['id']
        client.name = client_info['name']
        client.limits = client_info['limits']
        client.ip = credentials.ip
        client.is_yandex = True
        request.client = client

        request.user = self._construct_user_for_request(userinfo)
        return True

    @classmethod
    def _construct_user_for_request(cls, userinfo):
        user = PlatformUser()
        user.uid = userinfo['uid']
        user.login = userinfo['login']
        user.karma = get_karma(userinfo)
        user.user_ticket = userinfo.get('user_ticket')
        return user


class YaTeamCookieAuth(PassportCookieAuth):
    """
    Авторизация по YaTeam кукам
    """
    passport_client = yateam_blackbox

    @classmethod
    def _construct_user_for_request(cls, userinfo):
        user = super(YaTeamCookieAuth, cls)._construct_user_for_request(userinfo)
        user.uid = to_yateam_uid(userinfo['uid'])
        return user


class OAuthAuth(EventListenerMixin, BaseAuthWithCredentials):
    """
    OAuth авторизация.

    Поддерживает как стандартную, так и 2-Legged авторизацию.
    При 2-Legged авторизации атрибут `request.user` будет равен None.
    """

    auth_method = 'oauth'
    event = SettingsChangeEvent
    credential_class = OAuthAuthCredentials
    blackbox_client = blackbox
    with_user_info_details = False

    def __init__(self, *args, **kwargs):
        super(OAuthAuth, self).__init__(*args, **kwargs)
        self.client_info_by_client_id = self._build_client_info_by_client_id_map()

    def _get_credentials(self, request):
        if request.mode != tags.platform.EXTERNAL:
            return None

        token = request.raw_headers.get('Authorization', '')
        # Возвращаем баг во имя людей на него завязавшихся https://st.yandex-team.ru/CHEMODAN-24308
        token = token.replace('OAuth ', '')
        if not token:
            return None
        return self.credential_class(token)

    def _validate_credentials(self, request, credentials):
        client_ip = request.get_real_remote_addr()

        try:
            data = self.blackbox_client.check_oauth_token(credentials.token, client_ip,
                                                          with_details=self.with_user_info_details)
        except PassportTokenExpired:
            logger.error_log.error('Failed to validate OAuth token', exc_info=True)
            return False
        except Exception:
            logger.error_log.error('Got exception while processing Blackbox request', exc_info=True)
            raise ServiceUnavailableError()

        if not self._check_accepted_agreement(data):
            logger.error_log.error('Unsuccessful agreement check')
            return False

        if data.get('ticket'):
            try:
                tvm_ticket = TVMTicket.build_tvm_ticket(data['ticket'], is_external=False)
                process.set_external_tvm_ticket(tvm_ticket)
            except Exception:
                logger.error_log.error('Failed to fetch TVM ticket', exc_info=True)
                return False

        client = PlatformClient()
        client.scopes = set(data['oauth']['scope'].split(' '))
        client.id = unicode(data['oauth']['client_id'])
        client.ip = client_ip
        client.name = unicode(data['oauth']['client_name'])
        client.token = credentials.token
        client.is_yandex = data['oauth']['client_is_yandex']
        self._merge_client_info_from_config(client)
        request.client = client

        # При 2-Legged OAuth uid'а в ответе паспорта не будет.
        if 'uid' in data:
            request.user = self._construct_user_for_request(data)

        return True

    def _merge_client_info_from_config(self, client):
        client_info = self.client_info_by_client_id.get(client.id)
        if not client_info:
            return

        client.scopes.update(client_info.scopes)

        client.limits = client_info.limits

    @classmethod
    def _construct_user_for_request(cls, blackbox_data):
        user = PlatformUser()
        user.uid = blackbox_data['uid']['value']
        user.login = blackbox_data['login']
        user.karma = get_karma(blackbox_data)
        user.user_ticket = blackbox_data.get('user_ticket')
        if cls.with_user_info_details:
            user.details = PlatformUserDetailsHelper.construct_details_from(blackbox_data)
        return user

    @staticmethod
    def _check_accepted_agreement(bb_response):
        if not settings.feature_toggles['check_accepted_agreement_for_pdd']:
            return True

        if not bb_response.get('uid', {}).get('hosted'):
            return True

        if 'attributes' not in bb_response:
            return False
        if bb_response.get('attributes', {}).get(PASSPORT_NAMES_TO_ATTRIBUTES['pdd_accepted_agreement']) == '1':
            return True
        return False

    def handle_event(self, event):
        self.client_info_by_client_id = self._build_client_info_by_client_id_map()

    def _build_client_info_by_client_id_map(self):
        data = {}
        config_data = settings.platform['auth']
        config_data = filter(lambda i: self.auth_method in i['auth_methods'], config_data)

        for client_info in config_data:
            if not client_info['enabled']:
                continue

            client_id = client_info['oauth_client_id']

            data[client_id] = PlatformConfigClientInfo.from_client_info_dict(client_info)
        return data


class YaTeamOAuthAuth(OAuthAuth):
    """
    Авторизация по YaTeam OAuth-токену
    """
    auth_method = 'yateam_oauth'
    blackbox_client = yateam_blackbox
    with_user_info_details = True

    @classmethod
    def _construct_user_for_request(cls, blackbox_data):
        user = super(YaTeamOAuthAuth, cls)._construct_user_for_request(blackbox_data)
        user.uid = to_yateam_uid(user.uid)
        return user


class UserAgentAuth(BaseAuth):
    """Авторизация по User-Agent. Костыль для https://st.yandex-team.ru/CHEMODAN-37945"""

    def authorize(self, request):
        ua = request.raw_headers.get('user-agent')
        if UserAgentParser.is_yandex_disk_mobile(ua):
            client = PlatformClient()
            client.ip = request.remote_addr
            client.id = 'Yandex.Disk ' + UserAgentParser.get_os(ua)
            client.name = ua
            client.is_yandex = False
            request.client = client
            return True
        return False


class OriginAuth(BaseAuth):
    """Авторизация по Origin. Костыль для https://st.yandex-team.ru/CHEMODAN-37945"""
    allowed_origin_regex = re.compile(r'^https://(.*\.)?(yandex\.(%s)|yadi\.sk)$' % str.join('|', [re.escape(x) for x in PLATFORM_TLDS]))

    def authorize(self, request):
        origin = request.raw_headers.get('origin')
        if origin and self.allowed_origin_regex.match(origin):
            client = PlatformClient()
            client.ip = request.remote_addr
            client.id = origin
            client.name = origin
            client.is_yandex = False
            request.client = client
            return True
        return False


def get_karma(userinfo):
    """
    Правильно для Платформы извлекает карму из паспортного userinfo.

    :param dict userinfo: Данные о пользователе полученные из паспорта методами `check_oauth_token` или `check_session_id`.
    :return: Значение кармы пользователя или None в любой непонятной ситуации.
    :rtype: int | NoneType
    """
    karma = userinfo.get('karma', {}).get('value', None)
    # Параноидальная защита от недокументированных значений кармы.
    # Всегда когда карма не целое число, Платформа считает что карма не задана.
    return karma if isinstance(karma, (int, long)) else None


class ClientNetworks(EventListenerMixin):
    """Клиентские сети с авторизацией.

    **Важно**:
    Использует EventListener (для обновления информации о клиентах из конфига, который может меняться).
    Поэтому нужно один раз создавать инстанс объекта, чтобы подписаться на обновления конфига.
    """

    event = SettingsChangeEvent

    client_networks = {}
    """Глобальный стейт информации о разрешенных сетях для клиентов.

    Обновляется на собитие изменения конфига.
    """

    ALLOWED_NETWORK_ADDRESSES = 'allowed_network_addresses'

    def __init__(self):
        super(ClientNetworks, self).__init__()
        self._update_client_networks()

    @staticmethod
    def _update_client_networks():
        if settings.auth[NETWORK_AUTHORIZATION]['enabled']:
            # Сохраняем данные о клиентах в глобальный стейт

            # Сначала обнуляем все известные данные, которые могут быть не актуальны
            ClientNetworks.client_networks = {}
            # затем - добавляем по актуальному конфигу
            for client in settings.platform['auth']:
                if (ClientNetworks.ALLOWED_NETWORK_ADDRESSES in client and
                        client[ClientNetworks.ALLOWED_NETWORK_ADDRESSES]):
                    ClientNetworks.client_networks[client[OAUTH_CLIENT_ID]] = IPRangeList(
                        *client[ClientNetworks.ALLOWED_NETWORK_ADDRESSES]
                    )

    @staticmethod
    def is_client_authorized_from_request_ip(client):
        """Проверяет может ли клиент делать запросы с текущего IP.

        Разрешенные сети указываются для каждого клиента в конфиге.
        Если сети не заданы, то запросы разрешены с любых адресов.
        """
        if not settings.auth[NETWORK_AUTHORIZATION]['enabled']:
            return True

        if client.id not in ClientNetworks.client_networks:
            return True

        return client.ip in ClientNetworks.client_networks[client.id]

    @staticmethod
    def prepare_error_message(client):
        """Добавляем префикс silent-mode, если в режиме silent-mode"""
        error_msg = 'Client (id=%s) is not authorized for requests from address %s' % (client.id,
                                                                                       client.ip)
        if settings.auth['network_authorization']['silent_mode']:
            error_msg = '%s: %s' % (PLATFORM_NETWORK_AUTHORIZATION_SILENT_MODE_LOG_PREFIX,
                                    error_msg)
        return error_msg

    def handle_event(self, event):
        """Перечитываем данные о клиентах из конфига."""
        ClientNetworks._update_client_networks()
