# -*- coding: utf-8 -*-
from random import randint

import mpfs.engine.process

from mpfs.common.util import logger as mpfs_logger
from mpfs.common import errors
from mpfs.common.static import tags
from mpfs.config import settings
from mpfs.platform.utils import CaseInsensitiveDict


PLATFORM_DISK_APPS_IDS = set(settings.platform['disk_apps_ids'])


class _Logger(object):
    @property
    def default_log(self):
        return mpfs.engine.process.get_default_log()

    @property
    def access_log(self):
        return mpfs.engine.process.get_access_log()

    @property
    def error_log(self):
        return mpfs.engine.process.get_error_log()

    @property
    def stat_api_log(self):
        return mpfs.engine.process.get_stat_log('api-access')


logger = _Logger()
"""Объект через который следует обращаться к логам в платформе."""


HTTP_VERBS = ['POST', 'PUT', 'PATCH', 'GET', 'DELETE', 'OPTIONS', 'HEAD']
TEXT_CONTENT_TYPES = ['application/json', 'application/hal+json', 'application/xml', 'text/html', 'text/plain',
                      'application/x-yaml', 'text/yaml']


class PlatformUser(object):
    uid = None
    login = None
    details = None
    """Детали о пользователе из Паспорта (если запросили)"""
    karma = None
    """
    Карма пользователя в Паспорте.
    Может быть None, если пользователь авторизован не через Паспорт или Паспорт не вернул значение.
    Проверка и реакция на различные значения кармы -- дело хэндлеров.
    О том, что такое карма: https://doc.yandex-team.ru/Passport/AuthDevGuide/concepts/accounts-attributes.xml#karma
    О значениях кармы: https://doc.yandex-team.ru/blackbox/reference/method-oauth-response-json.xml
    """
    user_ticket = None

    def __repr__(self):
        context = {'class': self.__class__.__name__}
        context.update(self.__class__.__dict__)
        context.update(self.__dict__)
        return '%(class)s(uid=%(uid)r, login=%(login)r)' % context


class PlatformClient(object):
    id = None
    ip = None
    version = None
    area = None
    scopes = []
    name = None
    token = None
    user = None
    is_yandex = False
    limits = []
    """Список объектов класса GroupRateLimit"""

    def __init__(self):
        self.user = PlatformUser()

    @property
    def uid(self):
        return self.user.uid

    def __repr__(self):
        context = {'class': self.__class__.__name__}
        context.update(self.__class__.__dict__)
        context.update(self.__dict__)
        return '%(class)s(id=%(id)r, token=%(token)r, name=%(name)r, scopes=%(scopes)r)' % context


class PlatformConfigClientInfo(object):
    def __init__(self, name, oauth_client_id, oauth_client_name, scopes, limits):
        self.name = name
        self.oauth_client_id = oauth_client_id
        self.oauth_client_name = oauth_client_name
        self.scopes = scopes
        self.limits = limits

    @staticmethod
    def from_client_info_dict(client_info):
        return PlatformConfigClientInfo(
            client_info['name'],
            client_info['oauth_client_id'],
            client_info['oauth_client_name'],
            client_info.get('oauth_scopes', []),
            GroupRateLimit.get_limit_groups_from_config(client_info.get('limit_groups', []))
        )


class GroupRateLimit(object):
    name = None
    """Имя группы в рэйтлимитере"""

    multiplier = None
    """Множитель группы. Ненулевой множитель означает, что будут использовать группы name_{1..multiplier}.
    Если multiplier равен 0 или None, то будет использоваться группа name"""

    def __init__(self, name, multiplier=None):
        self.name = name
        self.multiplier = multiplier

    def get_full_group_name(self):
        group = self.name
        if self.multiplier and self.multiplier > 0:
            group = "%s_%i" % (group, randint(1, self.multiplier))

        return group

    @staticmethod
    def from_config(limit_group):
        return GroupRateLimit(limit_group['name'], limit_group.get('multiplier'))

    @staticmethod
    def get_limit_groups_from_config(limit_groups):
        return [GroupRateLimit.from_config(limit_group) for limit_group in limit_groups]


class PlatformRequest(object):
    headers = None
    raw_headers = None
    remote_addr = None
    method = None
    request_uri = None
    server_protocol = None
    url = None
    mode = tags.platform.EXTERNAL
    client = None
    """Экземпляр класса `PlatformClient`"""
    user = None
    """Экземпляр класса `PlatformUser`"""
    path = None
    language = None
    query = None
    """Проверенные и очищенные querystring параметры запроса"""
    args = None
    """Сырые querystring параметры запроса"""
    body = None
    """Проверенное и очищенное тело запроса"""
    data = None
    """Сырое тело запроса"""
    kwargs = None
    """Проверенные и очищенные path-параметры запроса"""
    raw_kwargs = None
    """Сырые path-параметры запроса"""
    accept_mimetypes = None
    user_initialized = False
    """Инициализировал ли данный запрос пользователя"""
    dispatcher = None
    """Диспатчер принявший запрос"""
    router = None
    """Роутер использованный для получения обработчика запроса"""
    start_time = None
    """Время начала обработки запроса"""
    cookie_auth_client_id = None
    """Идентификатор клиента для авторизации по куке, если есть"""
    auth_method = None
    """Класс, который авторизовал запрос (если один запрос может быть авторизован несколькими методами,
     то здесь будет только первый, который успешно авторизовал пользователя)"""
    check_rate_limit = True
    """Нужно ли делать походы в рл для запроса. Используется только внутри классов рейт лимитера для совершения
    минимального числа запросов"""

    # Внутренние атрибуты.
    # Пользоваться ими в хэндлерах не стоит, хотя бы потому, что они не защищены ни от каких атак снаружи.

    _raw_path = None
    """Путь, полученный из оригинального запроса."""

    def get_real_remote_addr(self, prefer_x_forwarded_for=True):
        """
        Содержит настоящий IP клиента (который изначально инициировал запрос)

        Для внешнего API не доверяем хедерам и пробрасываем IP клиента. Для внутреннего API IP вытаскиваем из хедеров
        X-Forwarded-For и X-Real-Ip в порядке приоритета.
        """
        real_remote_addr = self.remote_addr
        if self.mode == tags.platform.INTERNAL:
            # Запрос может содержать оба заголовка X-Forwarded-For и X-Real-Ip.
            # В этом случае X-Real-Ip скорее всего будет содержать IP хоста,
            # который непосредственно сделал запрос в API, т.е. это может быть прокся.
            # В то время как первый элемент списка X-Forwarded-For, если задан,
            # будет содержать IP источника запроса.
            # Поэтому X-Forwarded-For имеет более высокий приоритет нежели X-Real-Ip.
            x_forwarded_for_addr = None
            if self.raw_headers.getlist('X-Forwarded-For'):
                x_forwarded_for_addr = self.raw_headers.getlist('X-Forwarded-For')[0]

            x_real_ip_addr = self.raw_headers.get('X-Real-Ip', None)

            if prefer_x_forwarded_for:
                real_remote_addr = x_forwarded_for_addr or x_real_ip_addr or real_remote_addr
            else:
                real_remote_addr = x_real_ip_addr or x_forwarded_for_addr or real_remote_addr

        return real_remote_addr

    def is_disk_native_client(self):
        return self.router.request.client and self.router.request.client.id in PLATFORM_DISK_APPS_IDS


class PlatformResponse(object):
    DEFAULT_ALLOW_HEADERS = ['Accept', 'Accept-Language', 'Authorization', 'Content-Type', 'X-HTTP-Method',
                             'X-Requested-With', 'X-Uid']

    def __init__(self, status_code=200, content=None, headers=None):
        if not isinstance(headers, CaseInsensitiveDict):
            headers = CaseInsensitiveDict(headers or {})
        raw_allow_headers = headers.pop('Access-Control-Allow-Headers', None)
        allow_headers = []
        if raw_allow_headers:
            allow_headers += filter(None, [v.strip() for v in raw_allow_headers.split(',')])
        allow_headers += self.DEFAULT_ALLOW_HEADERS
        allow_headers = set(allow_headers)
        self.headers = {
            # CORS
            'Access-Control-Allow-Origin': '*',  # https://jira.yandex-team.ru/browse/CHEMODAN-16619
            'Access-Control-Allow-Headers': ', '.join(allow_headers),
            'Access-Control-Allow-Credentials': 'true',  # https://st.yandex-team.ru/CHEMODAN-25427
            'Cache-Control': 'no-cache',  # https://st.yandex-team.ru/CHEMODAN-64262
        }
        self.headers.update(headers)
        self.status_code = status_code
        self.content = content

    def get_result(self):
        return self.content

    def set_result(self, value):
        self.content = value

    result = property(get_result, set_result)

    def get_status(self):
        return self.status_code

    def set_status(self, value):
        self.status_code = value

    status = property(get_status, set_status)

    def __unicode__(self):
        return '%s %s' % (self.status_code, self.content)


def authenticated_clients_only(func):
    """Декоратор проверяющий, что запрос сделан авторизованным клиентом"""
    def wrapper(self, request, *args, **kwargs):
        if not request.client or not request.user:
            raise errors.platform.PlatformNotAuthorized()
        return func(self, request, *args, **kwargs)
    return wrapper


class ResponseObject(object):
    def __init__(self, response_cls=None, reason=None, status_code=None, *args, **kwargs):
        """
        Объект представляющий описение ответа API.

        :param response_cls: `CloudApiException` или `Serializer`.
        :param reason: Причина ответа.
        :param status_code: Статус-код ответа, для `CloudApiException` можно не указывать.
        :param args:
        :param kwargs:
        :return:
        """
        from mpfs.platform.exceptions import CloudApiError
        super(ResponseObject, self).__init__()
        self.reason = reason or getattr(response_cls, 'message', '')
        if response_cls and issubclass(response_cls, CloudApiError):
            self.response_cls = response_cls.serializer_cls
        else:
            self.response_cls = response_cls
        self.status_code = status_code or getattr(response_cls, 'status_code', 200)

    def __call__(self, *args, **kwargs):
        return self.response_cls(*args, **kwargs)

    def __dir__(self):
        return self.__dict__.keys() + dir(self.response_cls)

    def __getattr__(self, item):
        return getattr(self.response_cls, item)

    def __repr__(self):
        return '%s(%s)' % (type(self).__name__, self.response_cls)
