# -*- coding: utf-8 -*-
"""

MPFS
PLATFORM
API

Dispatcher

"""
from collections import OrderedDict
import os
import re
import traceback
import time
from itertools import cycle
import sys
import urlparse

import mpfs.engine.process
from mpfs.common.static import tags
from mpfs.common.util.credential_sanitizer import CredentialSanitizer
from mpfs.config import settings
from mpfs.core.services import passport_service
from mpfs.core.services import tvm_service
from mpfs.core.services import tvm_2_0_service
from mpfs.common.util.translation import parse_accept_lang_header
from mpfs.platform.async_mode import gevent_local_variables
from mpfs.platform.common import PlatformResponse, PlatformRequest, logger
from mpfs.platform.dynamic_settings.hooks import platform_before_handle_request_hook
from mpfs.platform.events import dispatcher
from mpfs.platform.exceptions import CloudApiError, InternalServerError, NotFoundError
from mpfs.platform.formatters import JsonFormatter, ProtobufFormatter, BaseFormatter
from mpfs.platform.utils import (
    is_valid_uid,
    split_cookie_based_path,
    unquote,
)
from mpfs.platform.v1.data import data_pb2

passport = passport_service.Passport()

ids = cycle(xrange(100000))

NATIVE_USER_AGENT_PATTERN = re.compile('Yandex.Disk.*? {.*?"os":"([^"]+)".*?}')
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']

YCRID_PREFIX_TO_OS_PATTERN = {
    'ios': re.compile('^ios'),
    'andr': re.compile('^android'),
    'wp': re.compile('^wp'),
    'win': re.compile('^win'),
    'mac': re.compile('^mac'),  # {'mac', 'macstore'}
    'lnx': re.compile('^cli'),
}


@gevent_local_variables('_request')
class BaseDispatcher(object):
    """
    Общий класс приема запросов Плафтормы

    Смысл: настроить запрос, найти хэндлер, передать request на обработку
    Результаты возвращаем в качестве результата хэндлера.
    Все данные запроса ищем в request
    """
    mode = None
    router = None
    default_lang = 'ru'
    """Предпочитаемый клиентом язык по умолчанию."""
    default_error_content_type = 'application/json'
    error_content_types = OrderedDict([
        ('application/json', JsonFormatter()),
        ('application/hal+json', JsonFormatter()),
        ('application/protobuf', ProtobufFormatter(data_pb2.Error)),
    ])

    def __init__(self, router):
        """
        Create new dispatcher.
        :param api_packages: List of packages containing PlatformHandlers.
        """
        super(BaseDispatcher, self).__init__()
        self._request = None
        self.router = router

    @property
    def request(self):
        return self._request

    @request.setter
    def request(self, value):
        self._request = value
        self.router.set_request(value)

    def dispatch(self, raw_request, *args, **kwargs):
        passport_service.Passport.reset()
        # не генерируем новые rid и crid если запрос уже содержит их
        rid = getattr(raw_request, 'rid', None) or '%s_%s' % (os.getpid(), ids.next())
        mpfs.engine.process.set_req_id(rid)
        crid = getattr(raw_request, 'crid', None) or \
            '%s-%s-%s' % (
                self._get_ycrid_prefix(raw_request.headers.get('User-Agent', '')),
                raw_request.headers.get('Yandex-Cloud-Request-ID'),
                mpfs.engine.process.hostname().split('.')[0]
            )
        mpfs.engine.process.set_cloud_req_id(crid)

        mpfs.engine.process.set_tvm_2_0_user_ticket(None)
        if SERVICES_TVM_2_0_ENABLED and raw_request.headers.get('X-Ya-User-Ticket'):
            raw_tvm_ticket = raw_request.headers.get('X-Ya-User-Ticket')
            tvm_ticket = tvm_2_0_service.TVM2Ticket.build_tvm_ticket(raw_tvm_ticket)
            mpfs.engine.process.set_tvm_2_0_user_ticket(tvm_ticket)

        # Реинициализируем внешний TVM тикет при получении запроса
        if raw_request.headers.get('Ticket'):
            raw_tvm_ticket = raw_request.headers.get('Ticket')
            tvm_ticket = tvm_service.TVMTicket.build_tvm_ticket(raw_tvm_ticket)
            mpfs.engine.process.set_external_tvm_ticket(tvm_ticket)
        else:
            mpfs.engine.process.set_external_tvm_ticket(None)

        request = None
        response = PlatformResponse()
        try:
            request = self._build_request(raw_request)

            request.rid = mpfs.engine.process.get_req_id()
            request.crid = mpfs.engine.process.get_cloud_req_id()
            request.check_rate_limit = kwargs.get('check_rate_limit', True)

            self.request = request
            match = self.router.resolve(request)
            if not match:
                raise NotFoundError()
            self.before_handler(request)
            match_kwargs = dict([(k, unquote(v)) for k, v in match.kwargs.iteritems()])
            response = match.handler.dispatch(request, *match.args, **match_kwargs)
            self.after_handler(request, response)
        except Exception, e:
            response = self.handle_error(e)
        finally:
            try:
                self.log_request(request, response)
            except Exception:
                logger.error_log.error('Error while logging request', exc_info=True)

        origin = self.request.raw_headers.get('Origin')
        if origin:
            response.headers['Access-Control-Allow-Origin'] = origin

        response.headers['Yandex-Cloud-Request-ID'] = crid

        return response

    @classmethod
    def _get_ycrid_prefix(cls, user_agent):
        user_agent_prefix = cls._extract_ycrid_prefix(user_agent)
        return 'rest' + ('_' + user_agent_prefix if user_agent_prefix else '')

    # https://st.yandex-team.ru/CHEMODAN-28071
    @staticmethod
    def _extract_ycrid_prefix(user_agent):
        m = NATIVE_USER_AGENT_PATTERN.match(user_agent or '')
        if m:
            os = m.group(1).lower()
            for ycrid_prefix, os_pattern in YCRID_PREFIX_TO_OS_PATTERN.iteritems():
                if os_pattern.match(os):
                    return ycrid_prefix

        return None

    def _build_request(self, raw_request):
        r = PlatformRequest()
        r.start_time = time.time()
        r.mode = self.mode
        r.raw_headers = raw_request.headers
        r.remote_addr = raw_request.remote_addr
        r.method = self.get_http_method(raw_request)
        request_uri = raw_request.environ.get('REQUEST_URI')
        if request_uri is None:
            request_uri = '%s?%s' % (raw_request.path, raw_request.query_string.decode('utf-8'))
        else:
            request_uri = request_uri.decode('utf-8')
        r.request_uri = request_uri
        r.server_protocol = raw_request.environ.get('SERVER_PROTOCOL')
        r.url = raw_request.url

        raw_path = self.parse_path(raw_request)
        cookie_auth_client_id, raw_path = split_cookie_based_path(raw_path)

        r._raw_path = raw_path
        r.path = self.get_platform_path(raw_path)
        r.cookie_auth_client_id = cookie_auth_client_id

        r.language = self.get_language(raw_request)
        r.args = raw_request.args

        # грязный хак позволяющий отключить во flask принудительный прогон тела запроса через unquote,
        # если Content-Type == application/x-www-form-urlencoded
        content_type = raw_request.headers['Content-Type']
        raw_request.environ['CONTENT_TYPE'] = 'SOME UNSUPPORTED BY FLASK CONTENT-TYPE'
        r.data = raw_request.data
        raw_request.environ['CONTENT_TYPE'] = content_type

        r.accept_mimetypes = raw_request.accept_mimetypes
        r.dispatcher = self
        r.router = self.router
        return r

    def before_handler(self, request):
        platform_before_handle_request_hook(dispatcher)

    def after_handler(self, request, response):
        pass

    def get_language(self, raw_request):
        """Возвращает предпочитаемый клиентом язык"""
        query_lang = raw_request.args.get('lang', None)
        if query_lang:
            return query_lang

        header_langs = parse_accept_lang_header(raw_request.headers.get('Accept-Language', ''))
        if header_langs:
            return header_langs[0][0]

        return self.default_lang

    def get_platform_path(self, path):
        version, uid, section, resource = self.split_path(path)
        return '/%s' % '/'.join(filter(None, (version, section, resource)))

    def get_http_method(self, raw_request):
        """
        Get method from X-HTTP-Method header
        http://wiki.yandex-team.ru/disk/platform/projects/HTTPRESTAPI/spec#http-metody
        :param raw_request:
        :return:
        """
        x_http_method = raw_request.headers.get('X-HTTP-Method')
        if x_http_method and x_http_method in ('GET', 'POST', 'PUT', 'PATCH', 'OPTIONS', 'HEAD', 'DELETE'):
            return x_http_method
        else:
            return raw_request.method

    @staticmethod
    def parse_path(raw_request):
        """Извлекает из запроса path."""
        request_uri = raw_request.environ.get('REQUEST_URI') or raw_request.environ.get('PATH_INFO')
        if request_uri is None:
            path = raw_request.path
        else:
            request_uri = request_uri.decode('utf-8')
            path = urlparse.urlsplit(request_uri).path
        return path

    @classmethod
    def split_path(cls, path):
        """
        In descendants parse path and return tuple (version, uid, section, resource).
        If one of the components doesn't exists then its defaults to None.
        :return: tuple
        """
        raise NotImplementedError()

    def handle_error(self, e, log_trace=True):
        if log_trace:
            trace = ''.join(traceback.format_exception(
                type(getattr(e, 'inner_exception', e)),
                getattr(e, 'inner_exception', e),
                sys.exc_info()[2]
            ))
            logger.error_log.error(trace)

        if not isinstance(e, CloudApiError):
            e = InternalServerError(inner_exception=e)
        return e

    def log_request(self, request, response):
        uid = None
        if request.user:
            uid = request.user.uid

        client_id = None
        if request.client:
            client_id = request.client.id

        headers_list = CredentialSanitizer.get_headers_list_with_sanitized_credentials(request.raw_headers)
        s = ' '.join(['"%s: %s"' % (k, v) for k, v in headers_list])
        logger.access_log.info('', extra={
            'remote_addr': request.remote_addr,
            'uid': uid,
            'client_id': client_id,
            'method': request.method,
            'uri': request.request_uri,
            'proto': request.server_protocol,
            'status': response.status_code,
            'request_time': time.time() - request.start_time,
            'headers': s})

        data = {
            'method': request.method,
            'request_uri': request.request_uri,
            'status_code': response.status_code,
            'puid': request.user.uid if request.user else None,
            'login': request.user.login if request.user else None,
            'new_user': getattr(request, 'user_initialized', False),
            'oauth_id': request.client.id if request.client else None,
            'oauth_name': request.client.name if request.client else None,
            'client_is_yandex': request.client.is_yandex if request.client else None,
            'client_impl': 'REST',
            'sdk_platform': 'platform',
        }
        logger.stat_api_log.info(data)

    def get_error_content_types(self):
        return self.error_content_types

    def get_error_content_type(self, request=None):
        """Возвращает Content-Type ответа"""
        if request is None:
            request = self.request
        if 'Accept' in request.raw_headers:
            ct = request.accept_mimetypes.best_match(self.get_error_content_types().keys())
        else:
            ct = self.default_error_content_type
        return ct

    def get_error_formatter(self):
        """Возвращает форматтер для форматирования тела ответа"""
        formatter = self.get_error_content_types().get(self.get_error_content_type())
        assert isinstance(formatter, BaseFormatter)
        return formatter


class InternalDispatcher(BaseDispatcher):
    mode = tags.platform.INTERNAL

    @classmethod
    def split_path(cls, path):
        """
        /v1/<uid>/disk/resource?path=... -> ("v1", "24123948", "disk", "resource")
        :param path:
        :return: tuple(version, uid, section, resource)
        """
        chunks = filter(None, path.split('/', 3))
        if len(chunks) == 4:
            return tuple(chunks)
        elif len(chunks) == 3:
            # проверяем, что chunks[1] похож на uid и если да, то запихиваем его в uid, если нет, то это часть path
            if is_valid_uid(chunks[1]):
                return chunks[0], chunks[1], chunks[2], None
            else:
                return chunks[0], None, chunks[1], chunks[2]
        else:
            return None, None, None, '/'.join(chunks)

    def handle_error(self, e, log_trace=True):
        e = super(InternalDispatcher, self).handle_error(e, log_trace)
        status_code, content, headers = e.as_response_tuple(stacktrace=True)
        headers['Content-Type'] = self.get_error_content_type()
        formatter = self.get_error_formatter()
        return PlatformResponse(status_code, formatter.format(content), headers)


class ExternalDispatcher(BaseDispatcher):
    mode = tags.platform.EXTERNAL

    @classmethod
    def split_path(cls, path):
        """
        /v1/disk/resource?path=... -> ("v1", None, "disk", "resource")
        :param path:
        :return: tuple(version, uid, section, resource)
        """
        chunks = filter(None, path.split('/', 2))
        if len(chunks) == 3:
            return chunks[0], None, chunks[1], chunks[2]
        elif len(chunks) == 2:
            return chunks[0], None, chunks[1], None
        else:
            return None, None, None, '/'.join(chunks)

    def handle_error(self, e, log_trace=True):
        e = super(ExternalDispatcher, self).handle_error(e, log_trace)
        status_code, content, headers = e.as_response_tuple(stacktrace=False)
        headers['Content-Type'] = self.get_error_content_type()
        formatter = self.get_error_formatter()
        return PlatformResponse(status_code, formatter.format(content), headers)
