# -*- coding: utf-8 -*-

from __future__ import unicode_literals

"""
ATTENTION! Данный модуль будет работать только внутри uwsgi процесса.
http://uwsgi-docs.readthedocs.io/en/latest/PythonDecorators.html
Его следует импортировать ТОЛЬКО в точке входа в приложение.
"""

import posixpath
import socket
import time
import uwsgi
from uwsgidecorators import postfork, thread

from kazoo.exceptions import NodeExistsError
from kazoo.client import KazooState

from mpfs.core.cache import cache_set
from mpfs.platform.dynamic_settings import env
from mpfs.platform.common import logger
from mpfs.core.zookeeper.manager import ZookeeperClientManager
from mpfs.core.zookeeper.shortcuts import get_checksum
from mpfs.core.zookeeper.thread import ZookeeperConnectingThread
from mpfs.platform.dynamic_settings.cache_reader import get_cached_auth_settings
from mpfs.platform.dynamic_settings.sync_primitives import auth_settings_queue, auth_settings_updated_after_fork_event
from mpfs.common.util import from_json, to_json
from mpfs.config import settings


def _update_latest_settings(value):
    """Обновить переменную, хранящую последние полученные настройки доступов.

    :param value: Строка полученная в результате запроса данных из ноды зукипера.
    :type value: str | unicode
    :return: Чексумма данных.
    """
    data = from_json(value)
    checksum = get_checksum(data)
    env.set_platform_auth_settings_latest(data)
    logger.default_log.info('Worker %s updated latest settings to version %s.' % (uwsgi.worker_id(), checksum))
    return checksum


def _get_settings_from_zk(path):
    """Загружает сеттинги из ZK и устанавливает watch."""
    zk = ZookeeperClientManager.get_client(base_path=settings.platform['zookeeper']['base_path'])
    logger.default_log.info('Getting settings from zk on worker %s.' % uwsgi.worker_id())
    value, stats = zk.get(path, watch=watch_callback)
    auth_settings_queue.put(value)
    logger.default_log.info('Enqueued new settings on worker %s.' % uwsgi.worker_id())


def watch_callback(event):
    """Колбэк в воркере, вызывающийся при поступлении события изменения ноды.

    Получает последние данные ноды от которой пришло событие и снова на нее подписывается,
    указывая себя как получателя. Изменяет глобальную переменную в памяти воркера, храняшую *последние полученные* (но не факт что уже примененные)
    настройки доступов в API. Данные из нее будут перемещены в основную переменную в момент между запросами,
    чтобы не было ситуации когда пол запроса работает с одними настройками,
    а другая половина запроса - с другими.
    """
    worker_id = uwsgi.worker_id()
    path = event.path
    logger.default_log.info('Got watch event for path "%s" on worker %s.' % (path, worker_id))
    _get_settings_from_zk(path)


@postfork
def preload_auth_settings_and_start_zk_init():
    """Прочитать файл кэша и запустить тред с подключением к зукиперу

    Если загрузить настройки из кэша не удалось, то будем ждать подключения к зукиперу. Выставляет глобальный объект
    клиента и запускает подклчюение к зукиперу. Также устанавливает наблюдение за нодой настроек сразу и после каждого
    события изменения.
    """
    logger.default_log.info('Worker %s initializes connection to Zookeeper.' % uwsgi.worker_id())

    def state_listener(state):
        # листенер который в случае потери связи будет переподписываться
        # на изменения и получать последние изменения
        logger.default_log.info(
            'Worker %s changed state to %s' % (
                uwsgi.worker_id(), state
            )
        )
        if state == KazooState.CONNECTED:
            ZookeeperClientManager.get_client(base_path=settings.platform['zookeeper']['base_path']).handler.spawn(
                _get_settings_from_zk,
                settings.platform['dynamic_settings']['auth_path']
            )

    try:
        value = get_cached_auth_settings()
    except Exception:
        logger.error_log.error('Can\'t use settings from cache, waiting for Zookeeper connection...')
    else:
        auth_settings_queue.put(value)

    client = ZookeeperClientManager.create_client(base_path=settings.platform['zookeeper']['base_path'])
    ZookeeperClientManager.set_client(client, base_path=settings.platform['zookeeper']['base_path'])
    client.add_listener(listener=state_listener)
    t = ZookeeperConnectingThread(base_path=settings.platform['zookeeper']['base_path'])
    t.start()


@postfork
@thread
def worker_thread_auth_settings_applier():
    """Поток который получает данные из очереди настроек и
    применяет их для переменной последних настроек.

    После первого обновления настроек он выставляет флаг, сообщающий потокам,
    что теперь можно начать обрабатывать запросы пользователей, тк мы получили правильные настройки.
    Это критически необходимо после форка воркера от мастера, тк мастер не держит
    коннект к зукиперу и у него в памяти могут быть уже 1000 лет как устаревшие настройки.
    В таком случае перед обработкой реквеста поток заблокируется в ожидании выставления этого флага.
    """
    logger.default_log.info(
        'Settings applier thread has been started on worker %s.' % uwsgi.worker_id()
    )
    while True:
        value = auth_settings_queue.get(block=True, timeout=None)
        _update_latest_settings(value)

        if not auth_settings_updated_after_fork_event.is_set():
            # выставляем флаг, что сеттинги обновлены и теперь можно
            # обслуживать запросы
            auth_settings_updated_after_fork_event.set()
            logger.default_log.info(
                'One time settings updated event has been set on worker %s.' % uwsgi.worker_id()
            )


def _get_worker_unique_identifier():
    """Получить уникальный идентификатор воркера."""
    mode = uwsgi.opt['api_configuration']
    return '%s:%s:%s' % (mode, socket.gethostname(), str(uwsgi.worker_id()))


def _get_worker_applied_state_uwsgi_cache_key():
    """Получить ключ в кеше uWSGI по которому хранится чексумма последних примененных настроек."""
    return '%s:%s' % (_get_worker_unique_identifier(), 'applied')


def _get_worker_latest_state_uwsgi_cache_key():
    """Получить ключ в кеше uWSGI по которому хранится чексумма последних полученных настроек."""
    return '%s:%s' % (_get_worker_unique_identifier(), 'latest')


def write_worker_settings_checksums_to_cache():
    """Записать чексуммы настроек воркера в кеш.

    Записываются 2 чексуммы: одна для примененных активных настроек, а вторая для последних полученных.
    """
    latest_settings = env.get_platform_auth_settings_latest() or {}
    applied_settings = settings.platform['auth']
    try:
        # примененные настройки
        cache_set(
            _get_worker_applied_state_uwsgi_cache_key(),
            get_checksum(applied_settings),
            settings.platform['dynamic_settings']['cache_workers_states_key_ttl'],
            settings.platform['dynamic_settings']['cache_workers_states_name']
        )
        # последние полученные, но не факт, что примененные
        cache_set(
            _get_worker_latest_state_uwsgi_cache_key(),
            get_checksum(latest_settings),
            settings.platform['dynamic_settings']['cache_workers_states_key_ttl'],
            settings.platform['dynamic_settings']['cache_workers_states_name']
        )
    except Exception as e:
        logger.error_log.exception(e.message)


@postfork
@thread
def worker_thread_auth_settings_reporter():
    """Поток в воркере, который раз в X минут записывает данные по настройкам из памяти воркера в эфемерные ноды
    для контроля.

    Данные отправляются из самого объекта конфига и из последних полученных (не факт что примененных) настроек.
    """
    logger.default_log.info('Settings reporter thread has been started on worker %s.' % uwsgi.worker_id())
    assert not settings.platform['dynamic_settings']['states_path'].startswith('/')
    base_config_path = settings.platform['dynamic_settings']['states_path']
    latest_config_path_base_path = posixpath.join(base_config_path, 'latest')
    applied_config_path_base_path = posixpath.join(base_config_path, 'config')

    worker_unique_id = _get_worker_unique_identifier()

    latest_config_path = posixpath.join(latest_config_path_base_path, worker_unique_id)
    applied_config_path = posixpath.join(applied_config_path_base_path, worker_unique_id)

    def do_work():
        latest_settings = to_json((env.get_platform_auth_settings_latest() or {})).encode('utf8')
        applied_settings = to_json(settings.platform['auth']).encode('utf8')

        def report_settings(zk_path, data, log_message_template_created_case, log_message_template_updated_case):
            """Записать данные `data` по пути `zk_path` и залогировать сие действие."""
            try:
                client.create(zk_path, ephemeral=True, makepath=True, value=data)
                logger.default_log.info(log_message_template_created_case % (uwsgi.worker_id(), get_checksum(from_json(data))))
            except NodeExistsError:
                client.set(zk_path, data)
                logger.default_log.info(log_message_template_updated_case % (uwsgi.worker_id(), get_checksum(from_json(data))))

        write_worker_settings_checksums_to_cache()

        client = None
        try:
            client = ZookeeperClientManager.create_client(base_path=settings.platform['zookeeper']['base_path'])
            client.start()
            client.ensure_path(client.chroot)
            report_settings(
                latest_config_path, latest_settings,
                'Latest settings created at ZK by worker %s for version %s.',
                'Latest settings updated at ZK by worker %s for version %s.'
            )
            report_settings(
                applied_config_path, applied_settings,
                'Config settings created at ZK by worker %s for version %s.',
                'Config settings updated at ZK by worker %s for version %s.'
            )
            logger.default_log.info('Settings reported on worker %s.' % uwsgi.worker_id())
        except Exception as e:
            logger.error_log.exception(e.message)
        finally:
            if client:
                client.stop()

    while True:
        # TODO: Не может ли здесь быть состояния гонки?
        do_work()
        time.sleep(settings.platform['dynamic_settings']['report_interval'])
