# -*- coding: utf-8 -*-
import re
from mpfs.platform.exceptions import EntityTagParsingError


class EntityTag(object):
    """
    Представление значения Entity Tag протокола HTTP 1.1.

    Используется для идентификации HTTP-ресурсов. Содержится в заголовке 'ETag' в качестве значения,
    в заголовках 'If-None-Match' и 'If-Match' в качестве списка значений.

    http://tools.ietf.org/html/rfc2616#section-3.11
    """
    _WEAK_PREFIX = 'W/'
    _PATTERN = re.compile('\s*(%s)?"(.*)"\s*' % _WEAK_PREFIX)

    def __init__(self, value, weak=False):
        self.value = value
        self.weak = weak

    @classmethod
    def parse(cls, raw_tag):
        m = cls._PATTERN.match(raw_tag)
        if not m:
            raise EntityTagParsingError('expected %s, got %s' % (cls._PATTERN, raw_tag))

        weak_str, value = m.groups()
        return EntityTag(value, weak_str == cls._WEAK_PREFIX)

    def matches(self, tag):
        return self._weakly_matches(tag) or self._strongly_matches(tag)

    def _weakly_matches(self, tag):
        return (self.weak or tag.weak) and self.value == tag.value

    def _strongly_matches(self, tag):
        return not (self.weak or tag.weak) and self.value == tag.value

    def __eq__(self, tag):
        return str(self) == str(tag)

    def __str__(self):
        weak_str = self._WEAK_PREFIX if self.weak else ''
        return '%s"%s"' % (weak_str, self.value)


class EntityTagMatcher(object):
    """
    Представление значения заголовка 'If-None-Match'.

    Используется для определения вхождения Entity Tag текущего ресурса в переданные клиентом заголовок 'If-None-Match'.
    """
    _SEPARATOR = ','
    _MATCH_ALL_PATTERN = re.compile('\s*\*\s*')

    def __init__(self, tags=None, match_all=False):
        self.tags = tags if tags is not None else []
        self.match_all = match_all

    @classmethod
    def parse(cls, value, ignore_errors=True):
        if cls._MATCH_ALL_PATTERN.match(value):
            return cls(match_all=True)

        raw_tags = value.split(cls._SEPARATOR)
        tags = [tag for tag in [cls._parse_tag(raw_tag, ignore_errors) for raw_tag in raw_tags] if tag]
        return cls(tags)

    @staticmethod
    def _parse_tag(raw_tag, ignore_errors):
        try:
            return EntityTag.parse(raw_tag)
        except EntityTagParsingError:
            if ignore_errors:
                return None
            else:
                raise

    def matches(self, request_tag):
        if self.match_all:
            return True

        for tag in self.tags:
            if tag.matches(request_tag):
                return True

        return False

    def __str__(self):
        if self.match_all:
            return '*'
        else:
            return self._SEPARATOR.join([str(tag) for tag in self.tags])
