# -*- coding: utf-8 -*-
from collections import defaultdict, Iterable
from copy import copy
import traceback


class CloudApiErrorMetaClass(type):
    """Метакласс обеспечивающий механизм шаблонизированных message и descroption"""
    def __new__(mcs, name, parents, attrs):
        # прячем шаблоны в приватные атрибуты и заменяем оригинальные атрибуты на property
        def gen_property(attr_name):
            return property(
                lambda self: getattr(self, attr_name, '') % self.context,
                lambda self, val: setattr(self, attr_name, val)
            )
        for attr in ('description', 'message'):
            private_attr = '_%s' % attr
            # вычисляем значение атрибута
            attr_val = attrs.get(attr, None)
            for parent in parents:
                if attr_val is not None:
                    break
                attr_val = getattr(parent, private_attr, None)
            attr_val = attr_val or ''
            attrs[private_attr] = attr_val
            attrs[attr] = gen_property(private_attr)
        return super(CloudApiErrorMetaClass, mcs).__new__(mcs, name, parents, attrs)

    @property
    def code(cls):
        """
        Текстовый код ошибки, к которому могут привязываться клиентские приложения.
        Если класс исключени изменился, но надо оставить код прежним, то можно в исключении задать атрибут `error_code`.
        """
        return getattr(cls, 'error_code', cls.__name__)

    @property
    def serializer_cls(cls):
        """
        Сериализатор используемый для сериализации ошибки при вызове метода `CloudApiError.as_response_tuple`.
        Если надо использовать другой сериализатор для ошибки, то можно задать атрибут `error_serializer_cls`.
        """
        from mpfs.platform.serializers import ErrorSerializer
        return getattr(cls, 'error_serializer_cls', ErrorSerializer)

    @property
    def message(cls):
        return getattr(cls, '_message', '')


class CloudApiError(Exception):
    __metaclass__ = CloudApiErrorMetaClass

    status_code = 500
    """HTTP-код соответствующий ошибке, который будет отдан клиенту"""

    description = 'Internal Server Error'
    """Техническое описание ошибки на английском.

    Для базовых ошибок API брать значения там http://www.iana.org/assignments/http-status-codes/http-status-codes.xhtml

    Можно задавать в виде шаблона испольщующего аргументы конструктора.
    """

    message = u'Ошибка сервера.'
    """Сообщение об ошибке на русском, которое клиент может вывести пользователю.

    Будет переводиться на английский, когда озадачимся локализацией сообщений об ошибках.

    Можно задавать в виде шаблона испольщующего аргументы конструктора.
    """

    inner_exception = None
    """Экземпляр исключения, вызвавший данное исключение.

    Как правило, это исключение сервиса, которое вызвало исключение платформы.
    """

    headers = {}
    """Заголовки возвращаемые ошибкой"""

    @property
    def code(self):
        """Текстовый код ошибки, к которому могут привязываться клиентские приложения"""
        return type(self).code

    def __init__(self, inner_exception=None, **kwargs):
        super(CloudApiError, self).__init__()
        self.context = defaultdict(str, kwargs)
        self.inner_exception = inner_exception
        self.headers = copy(type(self).headers)
        self.headers.update(kwargs.get('headers', {}))

    def __repr__(self):
        return '%s(%s, %s)' % (self.code, self.status_code, self.description)

    @property
    def serializer_cls(self):
        return type(self).serializer_cls

    def serialize(self):
        return self.serializer_cls(obj=self).data

    def as_response_tuple(self, stacktrace=False):
        """Возвращает кортеж (<HTTP-код>, <dict с телом ответа>, <dict с заголовками ответа>)"""
        content = self.serialize()
        if stacktrace:
            content['stacktrace'] = traceback.format_exc()

        return self.status_code, content, self.headers


class BadRequestError(CloudApiError):
    status_code = 400
    description = 'Bad Request'
    message = u'Некорректный запрос.'


class ValidationError(BadRequestError):
    """
    Ошибка валидации данных

    Это исключение выбрасывают филды при ошибке валидации входящих данных.
    """
    description = 'Invalid Data'
    message = u'Некорректные данные.'


class UnexpectedPayloadError(ValidationError):
    description = 'No payload expected.'
    message = u'Запрос не должен содержать данных.'


class NoDataProvidedError(ValidationError):
    description = 'No data provided.'
    message = u'Отсутствуют необходимые данные.'


class FieldValidationError(ValidationError):
    description = 'Error validating field "%(name)s": %(description)s'
    message = u'Ошибка проверки поля "%(name)s": %(message)s'


class FieldOneRequiredValidationError(ValidationError):
    """Ошибка возникающая, когда хотябы один из нескольких опциональных филдов должен быть задан."""
    description = 'Error validating fields %(fields_names)s: Value for one of the fields must be specified.'
    message = u'Ошибка проверки полей %(fields_names)s: Хотябы для одного поля должно быть задано значение.'

    def __init__(self, fields_names, *args, **kwargs):
        self.fields_names = fields_names
        # форматируем и пробрасываем в дефолтный конструктор,
        # чтоб description и messgae правильно отформатировались
        kwargs['fields_names'] = ', '.join(map(lambda s: '"%s"' % s, fields_names))
        super(FieldOneRequiredValidationError, self).__init__(*args, **kwargs)


class FieldsAllOrNoneValidationError(ValidationError):
    """Ошибка возникающая, когда существует группа сцепленных опциональных параметров.
     В этом случае должны быть переданы все либо ни одного."""
    description = 'Error validating fields %(fields_names)s: Value must be specified for all fields or none of them .'
    message = u'Ошибка проверки полей %(fields_names)s: Значение должно быть указано для всех полей или ни для одного.'

    def __init__(self, fields_names, *args, **kwargs):
        self.fields_names = fields_names
        # форматируем и пробрасываем в дефолтный конструктор,
        # чтоб description и messgae правильно отформатировались
        kwargs['fields_names'] = ', '.join(map(lambda s: '"%s"' % s, fields_names))
        super(FieldsAllOrNoneValidationError, self).__init__(*args, **kwargs)

class FieldRequiredError(ValidationError):
    description = 'This field is required.'
    message = u'Это поле является обязательным.'


class FieldWrongTypeError(ValidationError):
    description = 'Field contain wrong type value.'
    message = u'Поле содержит значение неверного типа.'


class FieldMustBeJsonObjectError(ValidationError):
    description = 'Field must be a valid JSON object.'
    message = u'Значение должно содержать правильный JSON-объект.'


class FieldWrongChoiceError(ValidationError):
    description = 'Wrong choice specified, available choices are %(choices)s.'
    message = u'Выбран не верный вариант. Список доступных вариантов: %(choices)s.'


class FieldMustBeWholeNumberError(ValidationError):
    description = 'Must be a whole number.'
    message = u'Значение должно быть целым числом.'


class FieldMustBeIso8601Error(ValidationError):
    description = 'Must be a date in ISO8601 format.'
    message = u'Значение должно представлять собой дату в формате ISO 8601.'


class FieldMustHaveTimeZoneError(ValidationError):
    description = 'Must have timezone data.'
    message = u'Занчение должно содержать данные о часовом поясе.'


class FieldMustBeNumberError(ValidationError):
    description = 'Must be a number.'
    message = u'Значение должно быть числом.'


class FieldMustBeListError(ValidationError, TypeError):
    description = 'Must be a list.'
    message = u'Значение должно представлять собой список.'


class FieldMustBeStringWithBase64EncodedDataError(ValidationError):
    description = 'Must be a string containing base64 encoded data.'
    message = u'Значение должно представлять собой строку, содержащую данные закодированные в base64.'


class FieldWrongTypeListItemError(ValidationError):
    description = 'Item #%(item_index)s has wrong type.'
    message = u'Элемент списка номер %(item_index)s содержит значение неверного типа.'


class FieldListAttributesForbiddenError(ValidationError):
    description = 'Object can\'t contain list attributes.'
    message = u'Объект не может содержать списки.'


class FieldMaxNestingLevelExceededError(ValidationError):
    description = 'Maximum nesting level for object is %(max_nesting_level)s.'
    message = u'Максимальный уровень вложенности объектов составляет %(max_nesting_level)s.'


class FieldObjectMaxSizeExceededError(ValidationError):
    description = 'Maximum allowed object size is %(max_size)s bytes.'
    message = u'Максимальный допустимый размер объекта составляет %(max_size)s байт.'


class FieldRelativeUrlError(ValidationError):
    description = 'Relative URLs are not acceptable.'
    message = u'Относительные URL не допустимы.'


class FieldUrlUnacceptableProtocolError(ValidationError):
    description = 'URL contains unacceptable protocol.'
    message = u'URL содержит недопустимый протокол.'


class UnauthorizedError(CloudApiError):
    status_code = 401
    description = 'Unauthorized'
    message = u'Не авторизован.'


class TVM2AuthorizationUidMismatchError(UnauthorizedError):
    response = 401
    description = 'Different uids were passed. X-Uid: %(x_uid)s; User-Ticket-Uid %(user_ticket_uid)s; User-Ticket %(user_ticket)s.'
    message = u'Были переданы разные уиды. X-Uid: %(x_uid)s; User-Ticket-Uid: %(user_ticket_uid)s; User-Ticket: %(user_ticket)s.'


class PaymentRequired(CloudApiError):
    status_code = 402
    description = 'Payment Required'
    message = u'Необходима оплата'


class ForbiddenError(CloudApiError):
    status_code = 403
    description = 'Forbidden'
    message = u'Доступ запрещён. Возможно, у приложения недостаточно прав для данного действия.'


class BadKarmaError(ForbiddenError):
    description = 'Bad karma'
    message = u'Доступ заблокирован. Для снятия блокировки необходимо перейти по ссылке https://passport.yandex.ru/passport?mode=userapprove.'


class NotFoundError(CloudApiError):
    status_code = 404
    description = 'Not Found'
    message = u'Ресурс не найден.'


class MethodNotAllowedError(CloudApiError):
    status_code = 405
    description = 'Method Not Allowed'
    message = u'Метод не поддерживается.'


class NotAcceptableError(CloudApiError):
    status_code = 406
    description = 'Not Acceptable'
    message = u'Ресурс не может быть представлен в запрошенном формате.'


class ConflictError(CloudApiError):
    status_code = 409
    description = 'Conflict'
    message = u'Конфликт'


class VersionNotFoundError(CloudApiError):
    status_code = 404
    description = 'Version Not Found'
    message = u'Версия не найдена'


class GoneError(CloudApiError):
    status_code = 410
    description = 'Gone'
    message = u'Ресурс больше недоступен.'


class PreconditionFailedError(CloudApiError):
    status_code = 412
    description = 'Precondition Failed'
    message = u'Предусловия не выполнены.'


class PayloadTooLargeError(CloudApiError):
    status_code = 413
    description = 'Payload Too Large'
    message = u'Размер данных для обработки слишком большой.'


class UnsupportedMediaTypeError(CloudApiError):
    status_code = 415
    description = 'Unsupported Media Type'
    message = u'Неподдерживаемый формат данных в теле запроса.'


class FormatterParseError(UnsupportedMediaTypeError):
    """Ошибка разбора данных."""
    description = 'Unable to parse data.'
    message = u'Не удалось разобрать данные.'


class UnprocessableEntityError(CloudApiError):
    status_code = 422
    description = 'Unprocessable Entity'
    message = u'Не удалось обработать тело запроса.'


class LockedError(CloudApiError):
    status_code = 423
    description = 'Locked'
    message = u'Ресурс заблокирован.'


class TooManyRequestsError(CloudApiError):
    status_code = 429
    description = 'Too Many Requests'
    message = u'Слишком много запросов.'

    def __init__(self, retry_after=None, *args, **kwargs):
        """
        Слишком много запросов.

        :param retry_after: Количестве секунд через которе клиенту следует повторить запрос.
        """
        if retry_after is not None:
            headers = kwargs.get('headers') or {}
            headers['Retry-After'] = retry_after
            kwargs['headers'] = headers
        super(TooManyRequestsError, self).__init__(*args, **kwargs)


class UnavailableForLegalReasons(CloudApiError):
    status_code = 451
    description = 'Unavailable For Legal Reasons'
    message = u'Недоступно по юридическим причинам.'


class InternalServerError(CloudApiError):
    status_code = 500
    description = 'Internal Server Error'
    message = u'Ошибка сервера.'


class FormatterFormatError(InternalServerError):
    """Ошибка форматирования данных."""
    description = 'Unable to format data.'
    message = u'Не удалось отформатировать данные.'


class NotImplementedError(CloudApiError):
    status_code = 501
    description = 'Not Implemented'
    message = u'Не поддерживается.'


class BadGatewayError(CloudApiError):
    status_code = 502
    description = 'Bad Gateway'
    message = ''


class ServiceUnavailableError(CloudApiError):
    status_code = 503
    description = 'Service Unavailable'
    message = u'Сервис временно недоступен.'


class InsufficientStorageError(CloudApiError):
    status_code = 507
    description = 'Insufficient Storage'
    message = u'Недостаточно места.'


class EntityTagParsingError(BadRequestError):
    """
    Ошибка парсинга Entity Tag. Может выбрасывать филдом IfNoneMatchField.
    """
    description = 'Unable to parse Entity Tag at If-None-Match header'
    message = u'Ошибка парсинга Entity Tag в заголовке If-None-Match'


class FieldKeyRequiredError(ValidationError):
    """
    У поля, которое представялет словарь, отсутствуют требуемые ключи.
    """
    description = 'Required keys is missing: %(keys)s'
    message = u'Отсутствуют обязательные ключи: %(keys)s'

    def __init__(self, keys, **kwargs):
        if isinstance(keys, Iterable):
            keys = ', '.join(keys)
        super(FieldKeyRequiredError, self).__init__(keys=keys, **kwargs)


class FailoverFileIsTooOld(CloudApiError):
    pass


class UserNotFoundInPassportError(UnauthorizedError):
    """
    Ошибка для внутреннего API. Выбрасывается, если не нашли пользователя в паспорте.
    """
    description = 'User not found in passport'
    message = u'Пользователь не найден в паспорте'


class FieldBadEmailError(ValidationError):
    description = 'Email doesn\'t match email pattern.'
    message = u'Введен некорректный email.'


class FieldWrongIntegerRangeError(ValidationError):
    description = 'The value isn\'t in available range: %(bottom_border)s %(upper_border)s'
    message = u'Значение не лежит в допустимых пределах: %(bottom_border)s %(upper_border)s'
