# -*- coding: utf-8 -*-
import base64
import calendar
from collections import OrderedDict
from copy import copy
from datetime import datetime
import re
import urlparse
from types import NoneType

from dateutil.tz import tzoffset
from decimal import Decimal
from inspect import isclass
import sys
import collections
import re
from dateutil import tz
import dateutil.parser
from mpfs.common.util import from_json, to_json
from mpfs.platform import validators
from mpfs.platform import exceptions
from mpfs.platform.etag import EntityTagMatcher
from mpfs.platform.exceptions import UnauthorizedError, ForbiddenError
from mpfs.platform.permissions import AllowAllPermission
from mpfs.platform.utils import parse_cookie, unquote
from mpfs.platform.common import logger


class BaseField(object):
    """
    Base class to handle API input and output data.

    If field doesn't validate it raises PlatformValidationError.

    Fields works as follows: You create instance of field, then call `clean` method which perform all needed tasks in
    right order to return valid and properly formatted value.

    Field doesn't store any state between `clean` calls, so you can reuse created instance for multiple values.

    Method `clean` call following methods:
      1. `to_python` - converts raw input value to its python representation.
      2. `validate` - check is value valid or not and is it required. You should call parent method when override it.
      3. `run_validators` - check all passed to constructor validators against value.
    """
    read_only = False
    name = None
    """Имя поля в FieldSet'е или Serializer'е."""

    parent = None
    """Родительский QueryDict или Serializer"""

    empty_values = list(validators.EMPTY_VALUES)
    default = None
    # To override this errors pass yours to constructor in error_messages parameter or override in subclasses.
    default_errors = {
        'required': exceptions.FieldRequiredError,
    }

    permissions = AllowAllPermission()
    """Объект типа 'BasePlatformPermission', который реализует механизм авторизации."""

    def __init__(self, required=False, validators=None, help_text='', errors=None, source=None,
                 pbid=None, permissions=AllowAllPermission(), **kwargs):
        """
        Create new instance.

        :param required: Field required or not.
        :param default: Default value for field. Works only if field is not required.
        :param validators: List of validators.
        :param help_text: Help text.
        :param errors: Dictionary of exceptions to override default ones.
        :param pbid: Номер поля в protobuf-сообщении, если не установлено, то поле не будет отображаться в protobuf.
        """
        self.required = required
        self.help_text = help_text
        self.validators = validators or []
        self.source = source
        self.pbid = pbid

        errs = {}
        for c in reversed(self.__class__.__mro__):
            errs.update(getattr(c, 'default_errors', {}))
        errs.update(errors or {})
        self.errors = errs
        self.default = kwargs.get('default', type(self).default)
        self.hidden = kwargs.get('hidden', False)  # Скрывает поле от схемы
        self.permissions = permissions

    def __str__(self):
        ref = '.'.join(['%s' % o for o in [self.parent, self.__class__.__name__] if o is not None])
        return u'%s(name="%s", help_text="%s")' % (ref, self.name, self.help_text)

    def to_native(self, value):
        """
        Converts input value to its internal representation.

        :param value: Input value, usually string.
        :return: Converted and validated value.
        """
        if self.read_only:
            raise NotImplementedError("This field doesn't support input.")
        return value

    def from_native(self, value):
        """
        Преобразует внутренне представление занчения во внешнее.

        :param value: Внутренее представление объекта.
        :return: Внешнее представление значения.
        """
        if value in self.empty_values:
            return self.default
        return value

    def initialize(self, parent, field_name=None):
        """Вызывается для установки родительского `FieldSet` перед вызовом :attr:`.from_native`."""
        self.parent = parent
        self.name = field_name or self.name

    def validate(self, value):
        """
        Perform field specific validation of value. Raise PlatformValidationError if value doesn't validate.

        :param value: Internal representation of input value.
        :return: None
        """
        if value in self.empty_values and self.required:
            raise self.errors['required']()

    def run_validators(self, value):
        """
        Run all validators for value and raise PlatformValidationError if value doesn't validate.

        :param value: Internal representation of input value.
        :return: None
        """
        for validator in self.validators:
            validator(value)

    def clean(self, value):
        """
        Return clean and validated internal representation of the value.

        Raise PlatformValidationError if value doesn't validate.
        :param value: Raw value.
        :return: object
        """
        value = self.to_native(value)
        self.validate(value)
        self.run_validators(value)
        return value

    def get_permissions(self):
        return self.permissions

    def check_permissions(self, request):
        if not request:
            return True
        try:
            return self.get_permissions().has_permission(request)
        except (UnauthorizedError, ForbiddenError):
            return False


class QueryDict(collections.Mapping):
    parent = None
    """
    Объект в котором живёт QueryDict

    Обычно это подкласс `BasePlatformHandler`.
    Установка этого атрибута лежит полностью на плечах объекта в котором живёт `QueryDict`.
    """

    def __init__(self, *args, **kwargs):
        self._fields = dict(*args, **kwargs)
        self.initialize_fields()
        self._data = {}

    def initialize_fields(self):
        for name, field in self._fields.iteritems():
            if isinstance(field, BaseField):
                field.parent = self
                field.name = name

    def __getitem__(self, key):
        return self._data[key]

    def __len__(self):
        return len(self._data)

    def __iter__(self):
        return iter(self._data)

    def __deepcopy__(self, memo):
        fields_copies = {}
        for name, field in self._fields.iteritems():
            fields_copies[name] = copy(field)
        return type(self)(**fields_copies)

    def __str__(self):
        return '.'.join(['%s' % o for o in [self.parent, self.__class__.__name__] if o is not None])

    def get_fields(self):
        return self._fields

    @property
    def request(self):
        if self.parent.router is not None and self.parent.router.request:
            return self.parent.router.request
        return None

    def clean(self, data):
        ret = {}
        for name, field in self.get_fields().iteritems():
            if not field.check_permissions(self.request):
                continue
            source = getattr(field, 'source', None) or name
            value = data.get(source)
            try:
                ret[name] = field.clean(value)
            except exceptions.ValidationError, e:
                raise (exceptions.FieldValidationError(inner_exception=e, name=source, description=e.description,
                                                       message=e.message),
                       None,
                       sys.exc_info()[2])
        return ret

    def load(self, data):
        self._data = dict([(name, value) for name, value in self.clean(data).iteritems()])


class StringField(BaseField):
    default = u''
    unquote = False
    """Разэкранировать процентноэкранированные строки или нет."""

    def __init__(self, unquote=None, *args, **kwargs):
        super(StringField, self).__init__(*args, **kwargs)
        if isinstance(unquote, bool):
            self.unquote = unquote

    def to_native(self, value):
        if value in self.empty_values:
            return self.default
        else:
            if self.unquote:
                return unquote(value)
            else:
                return value


class SortField(StringField):
    """Умеет обрабатывать строки вида `-key` и преобразовывать их в `{'field': 'key', 'order': -1}`."""
    ASCENDING = 1
    DESCENDING = -1

    def __init__(self, parse_order=True, asc=ASCENDING, desc=DESCENDING, *args, **kwargs):
        """
        :param bool parse_order: Доставить из поля направление сортировки или нет.
                                 Если True, то значение будет содержать dict вида {'field': 'key', 'order': 1},
                                 иначе -- строку вида `key`.
        :param asc: Значение возвращаемое в случае сортировки по возрастанию.
        :param desc: Значение возвращаемое в случае сортировки по убыванию (префикс `-`).
        """
        super(SortField, self).__init__(*args, **kwargs)
        self.parse_order = parse_order
        self.asc = asc
        self.desc = desc

    def to_native(self, value):
        result = {
            'field': self.default,
            'order': self.asc,
        }
        if value:
            sort_by = value.strip()
        else:
            sort_by = ''
        if sort_by:
            reverse = sort_by.startswith('-')
            if reverse:
                sort_by = sort_by[1:]
                result['order'] = self.desc
            result['field'] = sort_by
        if self.parse_order:
            return result
        else:
            return result['field']


class UrlField(StringField):
    DEFAULT_ALLOW_PROTOCOLS = ['http', 'https']

    default_errors = {
        'relative_url': exceptions.FieldRelativeUrlError,
        'protocol': exceptions.FieldUrlUnacceptableProtocolError,
    }

    @staticmethod
    def is_absolute_url(url):
        return bool(urlparse.urlparse(url).netloc)

    def __init__(self, allow_relative=False, allow_protocols=None, *args, **kwargs):
        super(UrlField, self).__init__(*args, **kwargs)
        self.allow_relative = allow_relative
        self.allow_protocols = allow_protocols or copy(self.DEFAULT_ALLOW_PROTOCOLS)

    def validate(self, value):
        super(UrlField, self).validate(value)

        if not self.allow_relative and not self.is_absolute_url(value):
            raise self.errors['relative_url']()

        url_chunks = urlparse.urlparse(value)
        if value and url_chunks.scheme.lower() not in self.allow_protocols:
            raise self.errors['protocol']()


class Base64ToBinaryField(StringField):
    """
    Принимает снаружи и отдаёт наружу строку представляющую собой base64-encoded массив байт,
    принимает изнутри и передаёт внутрь массив байт.
    """
    default = ''
    default_errors = {
        'invalid': exceptions.FieldMustBeStringWithBase64EncodedDataError,
    }

    def to_native(self, value):
        if self.is_raw_binary():
            return value
        else:
            # если надо то разэкранирует urlencoded символы
            value = super(Base64ToBinaryField, self).to_native(value)
            if isinstance(value, unicode):
                value = value.encode('utf-8')
            try:
                return base64.b64decode(value)
            except TypeError, e:
                raise self.errors['invalid'](inner_exception=e)

    def from_native(self, value):
        if self.is_raw_binary():
            return value
        return base64.b64encode(value)

    def is_raw_binary(self):
        from mpfs.platform.serializers import BaseSerializer
        parent = self.parent
        if isinstance(parent, BaseSerializer):
            return parent.format_options.raw_binary
        return False


class JsonObjectField(BaseField):
    default = None
    default_errors = {
        'invalid': exceptions.FieldMustBeJsonObjectError,
        'list_attributes': exceptions.FieldListAttributesForbiddenError,
        'max_nesting_level': exceptions.FieldMaxNestingLevelExceededError,
        'max_size': exceptions.FieldObjectMaxSizeExceededError,
    }

    def __init__(self, max_nesting_level=None, allow_list_attributes=True, max_size=None, *args, **kwargs):
        """
        :param max_nesting_level: Максимальная глубина вложенности объектов. 0 - значит объект должен быть плоским.
        :param max_size: Максимальный размер объекта в байтах.
        :param allow_list_attributes: Разрешено ли объекту и всем вложенным в него иметь списочные атрибуты.
        """
        super(JsonObjectField, self).__init__(*args, **kwargs)
        self.max_nesting_level = max_nesting_level
        self.allow_list_attributes = allow_list_attributes
        self.max_size = max_size

    def check_max_nesting_level(self, obj, remaining_levels):
        for k, v in obj.iteritems():
            if isinstance(v, dict):
                if remaining_levels > 0:
                    self.check_max_nesting_level(v, remaining_levels - 1)
                else:
                    raise self.errors['max_nesting_level'](max_nesting_level=self.max_nesting_level)

    def check_list_attributes_doesnt_exists(self, obj):
        for k, v in obj.iteritems():
            if isinstance(v, dict):
                self.check_list_attributes_doesnt_exists(v)
            elif isinstance(v, list):
                raise self.errors['list_attributes']()

    def validate(self, value):
        super(JsonObjectField, self).validate(value)
        if self.max_nesting_level is not None:
            self.check_max_nesting_level(value, self.max_nesting_level)
        if not self.allow_list_attributes:
            self.check_list_attributes_doesnt_exists(value)
        if self.max_size is not None and len(to_json(value)) > self.max_size:
            raise self.errors['max_size'](max_size=self.max_size)

    def to_native(self, value):
        e = None
        if isinstance(value, (str, unicode)):
            try:
                value = from_json(value)
            except Exception, e:
                pass
        if not isinstance(value, dict):
            raise self.errors['invalid'](inner_exception=e)
        return value


class JsonObjectToStrField(JsonObjectField):
    """Представляет значение внутри в виде строки, а снаружи в виде JSON-объекта."""
    def clean(self, value):
        value = super(JsonObjectToStrField, self).clean(value)
        return to_json(value)

    def from_native(self, value):
        value = from_json(value)
        return super(JsonObjectToStrField, self).from_native(value)


class ChoiceField(StringField):
    """Принимает снаружи строку или номер строки и может мапать их на указанное значение."""
    default_errors = {
        'wrong_choice': exceptions.FieldWrongChoiceError,
    }

    def __init__(self, choices=None, num_choices_start=0, *args, **kwargs):
        """
        Create new instance.

        :param choices: List of acceptable strings or OrderedDict mapping acceptable strings to values.
        :param num_choices_start: Число с которого следует начинать отсчёт индексов строк.
                                  Костыль ради Data API ибо @tolmalev хочет, чтоб бэкэндный протобуф и апишный протобуф
                                  были совместимы, а в бэкэндном протобуфе нумерация enum'ов почему-то с 1.
        """
        if not isinstance(choices, (tuple, list, OrderedDict, NoneType)):
            msg = 'Choices must be instance of tuple, list or OrderedDict, but %s found.' % (type(choices).__name__,)
            logger.error_log.error(msg)
            raise ValueError(msg)

        super(ChoiceField, self).__init__(*args, **kwargs)
        self.choices = choices
        self.num_choices_start = num_choices_start
        if isinstance(self.choices, dict):
            self.choices_reverse_lookup = OrderedDict([(v, k) for k, v in self.choices.iteritems()])

    def is_numeric_choices(self):
        from mpfs.platform.serializers import BaseSerializer
        parent = self.parent
        if isinstance(parent, BaseSerializer):
            return parent.format_options.numeric_choices
        return False

    def to_native(self, value):
        if self.is_numeric_choices() and isinstance(value, (int, long)):
            # если сериализатор требует поддержки числовых опций
            if value not in self.get_num_choices():
                context = self.get_context()
                context['choices'] = self.get_num_choices()
                raise self.errors['wrong_choice'](**context)
            choice = self.get_choices()[value - self.num_choices_start]
        else:
            choice = super(ChoiceField, self).to_native(value)

        if choice not in self.empty_values and choice not in self.choices:
            # Когда значение передано, но оно не верно.
            raise self.errors['wrong_choice'](**self.get_context())

        if isinstance(self.choices, OrderedDict):
            return self.choices.get(choice, self.default)
        else:
            return choice

    def from_native(self, value):
        if isinstance(self.choices, dict):
            value = self.choices_reverse_lookup.get(value, value)
        if self.is_numeric_choices():
            # если сериализатор требует поддержки числовых опций, то переводим выбраный вариант в его номер
            value = self.num_choices_start + self.get_choices().index(value)
        return value

    def get_context(self, context=None):
        ret = context or {}
        if self.is_numeric_choices():
            choices = range(self.num_choices_start, self.num_choices_start + len(self.get_choices()))
        else:
            choices = self.get_choices()
        ret['choices'] = ', '.join(choices)
        return ret

    def validate(self, value):
        super(ChoiceField, self).validate(value)
        if value or self.required:
            choices = self.choices_reverse_lookup if isinstance(self.choices, dict) else self.choices
            if value not in choices:
                context = self.get_context()
                raise self.errors['wrong_choice'](**context)

    def get_choices(self):
        """Возвращает список вариантов которые принимает"""
        if isinstance(self.choices, (tuple, list)):
            return self.choices
        elif self.choices is not None:
            return self.choices.keys()
        else:
            return []

    def get_num_choices(self):
        """Возвращает список доступных номеров вариентов"""
        return range(self.num_choices_start, self.num_choices_start + len(self.choices))


class BooleanField(BaseField):
    default = False

    def __init__(self, boolify=False, *args, **kwargs):
        super(BooleanField, self).__init__(*args, **kwargs)
        self.boolify = boolify

    def from_native(self, value):
        value = super(BooleanField, self).from_native(value)
        return self.to_bool(value) if self.boolify else value

    def to_native(self, value):
        value = super(BooleanField, self).to_native(value)
        return self.to_bool(value)

    def to_bool(self, value):
        if isinstance(value, (str, unicode)) and value.lower() in ('false', '0'):
            value = False
        elif value not in self.empty_values:
            value = bool(value)
        else:
            value = self.default
        return value

    def validate(self, value):
        if value is None and self.required:
            raise self.errors['required']()


class IntegerField(BaseField):
    default_errors = {
        'invalid': exceptions.FieldMustBeWholeNumberError,
    }

    def from_native(self, value):
        value = super(IntegerField, self).from_native(value)
        if value in self.empty_values:
            return self.default
        try:
            value = long(value)
        except (ValueError, TypeError) as e:
            raise self.errors['invalid'](inner_exception=e)
        return value

    def to_native(self, value):
        value = super(IntegerField, self).to_native(value)
        if value in self.empty_values:
            return self.default
        try:
            value = int(str(value))
        except (ValueError, TypeError) as e:
            raise self.errors['invalid'](inner_exception=e)
        return value


class DateField(BaseField):
    default_errors = {
        'invalid': exceptions.FieldMustBeIso8601Error,
    }

    def to_native(self, value):
        value = super(DateField, self).to_native(value)
        if value in self.empty_values:
            return self.default
        try:
            value = datetime.strptime(value, '%Y-%m-%d').date().isoformat()
        except (AttributeError, ValueError, TypeError) as e:
            raise self.errors['invalid'](inner_exception=e)
        return value

    def from_native(self, value):
        if value in self.empty_values:
            return self.default
        return datetime.strptime(value, '%Y-%m-%d').date().isoformat()


class DateTimeField(BaseField):
    """
    Представляет снаружи дату и время в виде строки в формате ISO8601, а внутри в виде объекта datetime.

    Если в родительском сериализаторе установлен аттрибут datetime_as_timestamp,
    то представляет снаружи дату и время в виде timestamp'а в милисекундах.
    """
    default_errors = {
        'invalid': exceptions.FieldMustBeIso8601Error,
        'tz_required': exceptions.FieldMustHaveTimeZoneError,
    }

    def is_datetime_as_timestamp(self):
        from mpfs.platform.serializers import BaseSerializer
        parent = self.parent
        if isinstance(parent, BaseSerializer):
            return parent.format_options.datetime_as_timestamp
        return False

    def __init__(self, tz_required=False, *args, **kwargs):
        """
        :param tz_required: Требовать наличия часового пояса в ISO8601 представлении даты.
        :param args:
        :param kwargs:
        :return:
        """
        super(DateTimeField, self).__init__(*args, **kwargs)
        self.tz_required = tz_required

    def to_native(self, value):
        value = super(DateTimeField, self).to_native(value)
        if value in self.empty_values:
            return self.default
        try:
            if self.is_datetime_as_timestamp():
                # если сериализатор требует поддержки дат в виде таймстэмпов
                value /= 1000.0
                value = datetime.fromtimestamp(value, tz=dateutil.tz.tzutc())
            else:
                value = dateutil.parser.parse(str(value))
        except (AttributeError, ValueError, TypeError) as e:
            raise self.errors['invalid'](inner_exception=e)
        if self.tz_required and not value.tzinfo:
            raise self.errors['tz_required']()
        return value

    def from_native(self, value):
        if self.is_datetime_as_timestamp():
            # если сериализатор требует поддержки дат в виде таймстэмпов
            ret = calendar.timegm(value.timetuple())
            mcs = value.microsecond
            ret = ret * 1000 + mcs / 1000
        else:
            ret = value.isoformat()
        return ret


class DateTimeToTSField(DateTimeField):
    """Принимает снаружи и отдаёт наружу дату в формате ISO8601, передаёт внутрь и принимает изнутри unix timestamp."""
    def __init__(self, milliseconds=False, *args, **kwargs):
        super(DateTimeToTSField, self).__init__(*args, **kwargs)
        self.milliseconds = milliseconds

    def from_native(self, value):
        if self.milliseconds:
            value /= 1000.0
        value = datetime.fromtimestamp(value, tz=dateutil.tz.tzutc())
        return super(DateTimeToTSField, self).from_native(value)

    def to_native(self, value):
        ret = super(DateTimeToTSField, self).to_native(value)
        if ret in self.empty_values:
            return self.default
        ts = calendar.timegm(ret.utctimetuple())
        if self.milliseconds:
            mcs = ret.microsecond
            ts = ts * 1000 + mcs / 1000
        return ts


class DateTimeToTSWithTimezoneField(DateTimeField):
    """Принимает снаружи и отдаёт наружу дату в формате ISO8601, передаёт внутрь и принимает изнутри unix timestamp и offset в секундах."""
    def __init__(self, milliseconds=False, *args, **kwargs):
        super(DateTimeToTSWithTimezoneField, self).__init__(*args, **kwargs)
        self.milliseconds = milliseconds

    def from_native(self, value):
        if value in self.empty_values:
            return self.default

        ts, tz_offset = value
        if self.milliseconds:
            ts /= 1000.0
        value = datetime.fromtimestamp(ts, tz=tzoffset(None, tz_offset))
        return value.isoformat()

    def to_native(self, value):
        ret = super(DateTimeToTSWithTimezoneField, self).to_native(value)
        if ret in self.empty_values:
            return self.default
        tz_offset = calendar.timegm(ret.replace(tzinfo=dateutil.tz.tzutc()).utctimetuple()) - calendar.timegm(ret.utctimetuple())

        ts = calendar.timegm(ret.utctimetuple())
        if self.milliseconds:
            mcs = ret.microsecond
            ts = ts * 1000 + mcs / 1000
        return ts, tz_offset


class LatitudeField(BaseField):
    read_only = True

    def from_native(self, value):
        re_result = re.search('\((\d+(\.\d+)?),(\d+(\.\d+)?)\)', value)
        return float(re_result.group(1)) if re_result else None


class LongitudeField(BaseField):
    read_only = True

    def from_native(self, value):
        re_result = re.search('\((\d+(\.\d+)?),(\d+(\.\d+)?)\)', value)
        return float(re_result.group(3)) if re_result else None


class FloatField(BaseField):
    default_errors = {
        'invalid': exceptions.FieldMustBeNumberError,
    }

    def to_native(self, value):
        value = super(FloatField, self).to_native(value)
        if value in self.empty_values:
            return self.default
        try:
            value = float(value)
        except (ValueError, TypeError) as e:
            raise self.errors['invalid'](inner_exception=e)
        return value

    def validate(self, value):
        super(FloatField, self).validate(value)
        # Check for NaN (which is the only thing not equal to itself) and +/- infinity
        if value != value or value in (Decimal('Inf'), Decimal('-Inf')):
            raise self.errors['invalid']()
        return value


class ListField(BaseField):
    separator = ','
    item_field_type = StringField
    default = []
    default_errors = {
        'invalid': exceptions.FieldMustBeListError,
        'bad_item': exceptions.FieldWrongTypeListItemError,
    }

    def __init__(self, item_field_type=None, separator=None, *args, **kwargs):
        super(ListField, self).__init__(*args, **kwargs)
        self.item_field_type = item_field_type or self.item_field_type
        self.item_field = self.item_field_type(*args, **kwargs)
        self.separator = separator or self.separator

    def to_native(self, value):
        if not isinstance(value, (list, tuple)):
            if isinstance(value, (str, unicode)):
                value = filter(None, map(lambda s: s.strip(), value.split(self.separator)))
            elif value:
                value = [value]
            else:
                value = []
        return map(self.item_field.to_native, value)

    def validate(self, value):
        if not isinstance(value, (list, tuple)):
            raise self.errors['invalid']()
        else:
            i = 0
            try:
                for i in range(len(value)):
                    self.item_field.validate(value[i])
            except exceptions.ValidationError, e:
                raise self.errors['bad_item'](inner_exception=e, item_index=i)


class MultipleChoicesField(ListField):
    """
    Принимает список параметров в виде строки со значениями разделёнными ```self.separator```.

    >>> COLOR_CHOICES = ['red', 'green', 'blue']
    >>> f = MultipleChoicesField(separator=':', choices=COLOR_CHOICES)
    """

    def __init__(self, separator=None, *args, **kwargs):
        super(ListField, self).__init__(*args, **kwargs)
        self.item_field_type = ChoiceField
        self.item_field = self.item_field_type(*args, **kwargs)
        self.separator = separator or self.separator


class SerializerField(BaseField):
    """Умеет применять переданный конструктору сериализатор к значению переданному в метод `from_native`."""
    default_errors = {
        'invalid': exceptions.FieldMustBeListError,
    }

    def __init__(self, serializer_cls, source=None, many=False, help_text='', init=None, **kwargs):
        """
        Создаёт новый экземпляр поля.

        :param serializer_cls: Класс или callable возвращающий класс сериализатора, который следует использовать.
        :param many: Если True, то поле будет ожидать список объектов, иначе только один объект.
        :param source: Указывает родительскому сериализатору из какого атрибута объекта брать значение для поля.
        :param dict init: Параметры которые будут переданы как kwargs в конструктор сериализатора.
        """
        super(SerializerField, self).__init__(source=source, help_text=help_text, **kwargs)
        self._serializer_cls = serializer_cls
        self.many = many
        self.init = init or {}

    @property
    def serializer_cls(self):
        return self._serializer_cls if isclass(self._serializer_cls) else self._serializer_cls()

    @property
    def serializer(self):
        if not hasattr(self.__class__, '_serializers_cache'):
            setattr(self.__class__, '_serializers_cache', {})
        s_cls = self.serializer_cls
        s = self.__class__._serializers_cache.get(s_cls, None)
        if s is None:
            s = s_cls()
            self.__class__._serializers_cache[s_cls] = s
        return s

    def from_native(self, value):
        visible_fields = None
        if not isclass(self.parent) and hasattr(self.parent, 'get_visible_fields'):
            parent_vfs = self.parent.get_visible_fields()
            vfs = []
            for f in parent_vfs:
                if '.' in f:
                    chunks = f.split('.', 1)
                else:
                    chunks = [f]
                if len(chunks) > 1 and chunks[0] == self.name:
                    vfs.append(chunks[1])
            visible_fields = vfs or visible_fields

        serializer = self.serializer
        if self.many:

            if not isinstance(value, collections.Iterable):
                raise self.errors['invalid']()

            serializer.initialize(router=self.parent.router, hal=self.parent.hal,
                                  format_options=self.parent.format_options,
                                  visible_fields=visible_fields,
                                  **self.init)
            ret = []
            for v in value:
                serializer._object = v
                d = serializer.data
                if d:
                    ret.append(d)
            return ret

        else:
            serializer.initialize(value, router=self.parent.router, hal=self.parent.hal,
                                             format_options=self.parent.format_options,
                                             visible_fields=visible_fields,
                                             **self.init)
            return serializer.data

    def to_native(self, value):
        value = super(SerializerField, self).to_native(value)
        if self.many:
            if isinstance(value, collections.Iterable):
                ret = []
                for v in value:
                    serializer = self.serializer_cls(data=v, router=self.parent.router, hal=self.parent.hal,
                                                     format_options=self.parent.format_options,
                                                     **self.init)
                    ret.append(serializer.object)
                return ret
            else:
                raise self.errors['invalid']()
        else:
            serializer = self.serializer_cls(data=value, router=self.parent.router, hal=self.parent.hal,
                                             format_options=self.parent.format_options,
                                             **self.init)
            return serializer.object


class ParentMethodField(BaseField):
    """Возвращает в качестве своего значения результат указанного метода родительского сериализатора."""
    read_only = True

    def __init__(self, method_name, required=False, source=None, field_type=None, help_text='', *args, **kwargs):
        """
        Создаёт новый экземпляр поля.

        :param method_name: Имя метода родительского сериализатора, который следует использовать.
        :param source: Указывает родительскому сериализатору из какого атрибута объекта брать исходное значение.
        :param field_type: Филд соответствующий результату метода. Используется при генерации JSON Schema.
        """
        super(ParentMethodField, self).__init__(source=source, help_text=help_text, *args, **kwargs)
        self.method_name = method_name
        self.field_type = field_type
        self.required = required

    def from_native(self, value):
        meth = getattr(self.parent, self.method_name, None)
        return meth(value)


class ParentAttrField(BaseField):
    """Возвращает в качестве своего значения значение указанного атрибута родительского сериализатора."""
    read_only = True

    def __init__(self, attribute_name, field_type=None, help_text='', *args, **kwargs):
        """
        Создаёт новый экземпляр поля.

        :param attribute_name: Имя атрибута родительского сериализатора, который следует использовать.
        :param field_type: Филд соответствующий типу атрибута. Используется при генерации JSON Schema.
        """
        super(ParentAttrField, self).__init__(help_text=help_text, *args, **kwargs)
        self.attribute_name = attribute_name
        self.field_type = field_type

    def from_native(self, value):
        return getattr(self.parent, self.attribute_name)


class HalLinkField(BaseField):
    """Формирует ссылку на указанный хэндлер."""
    read_only = True

    def __init__(self, handler, map=None, context=None, exclude=None, rfc6570=True):
        """
        Создаёт новый экземпляр.

        :param handler_class: Класс хэндлера на который ссылается поле.
        :param dict map: Маппинг имён параметров принимаемых хэндером на имена атрибутов представления.
        :param dict context: Указанный контекст будет добавлен к контексту поумолчанию.
        :param list exclude: Указанные атрибуты будут удалены из контекста перед формированием ссылки.
        :param bool rfc6570: Использовать RFC6570 для шаблонизированных URL или нет.

        Пример:
        >>> from mpfs.platform.serializers import BaseSerializer
        >>> from mpfs.platform.resources import res
        >>> from mpfs.platform.v1.disk.handlers import GetOperationStatusHandler
        >>> from mpfs.platform.routers import RegexRouter
        >>> router = RegexRouter(res(relations={'operations': res(relations={'GET': GetOperationStatusHandler})}))
        >>> class OperationSerializer(BaseSerializer):
        ...     def get_links(self):
        ...         return {'self': HalLinkField(GetOperationStatusHandler, map={'operation_id': 'id'})}
        ...
        >>>
        >>> operation = {'id': 'b686bf2fcaee63dafa', 'type': 'store'}
        >>> OperationSerializer(operation, router=router).data
        {'_links': {'self': {'href': '/operations?oid=b686bf2fcaee63dafa', 'method': 'GET'}}, 'id': 'b686bf2fcaee63dafa', 'type': 'store'}
        """
        super(HalLinkField, self).__init__()
        self.handler = handler
        """Хэндлер для которого будет сформирована ссылка."""

        self.map = map or {}
        """Маппинг имён параметров, принимаемых хэндлером, на имена атрибутов контекста."""

        self.rfc6570 = rfc6570
        """Использовать RFC6570 для шаблонизированных URL или нет."""

        self.context = context or {}

        self.exclude = exclude or []

    def from_native(self, value):
        """
        Возращает ссылку на переданный объект.

        :param value: Представление объекта, содержащее данные необходимые для формирования URL.
        :return: Объект HAL Link.
        :rtype: dict
        """
        # Создаём контекст для заполнения URL с учётом заданного маппинга параметров.
        context = copy(value)
        for target, source in self.map.iteritems():
            context[target] = value.get(source)
        context.update(self.context)
        for name in self.exclude:
            context.pop(name, None)
        # Получаем всё необходимое для формирования объекта HAL Link
        link = self.parent.router.get_link(self.handler, params=context, rfc6570=self.rfc6570)
        if link is None:
            return None
        else:
            method, url, templated = link
            ret = {'method': method.upper(), 'href': url}
            if templated:
                ret['templated'] = templated
            return ret


class IfNoneMatchField(BaseField):
    """
    Филд для работы со стандартным заголовком HTTP 1.1 'If-None-Match'.
    """
    default = EntityTagMatcher([])

    def to_native(self, value):
        if value in self.empty_values:
            return self.default

        return EntityTagMatcher.parse(value)


class CookieField(BaseField):
    """
    Филд для работы со стандартным заголовком HTTP 1.1 'Cookie'.
    """
    default = {}

    def to_native(self, value):
        if value in self.empty_values:
            return self.default

        return parse_cookie(value)

    def from_native(self, value):
        raise NotImplementedError()


class EmailField(StringField):
    """
    Филд для работы с email
    """
    default = None
    default_errors = {
        'bad_email': exceptions.FieldBadEmailError,
    }

    def validate(self, value):
        super(EmailField, self).validate(value)

        # TODO: Желательно взять честный способ проверки email из Почты
        regexp = re.compile(r'[\w.-]+@[\w.-]+\.[\w]+?')
        if (value or self.required) and not regexp.match(value):
            raise self.errors['bad_email']


class IntegerRangeField(IntegerField):
    """
    Диапазон целочисленных значений
    """
    default_errors = {
        'value_not_in_range': exceptions.FieldWrongIntegerRangeError,
    }

    bottom_border = None
    upper_border = None

    def __init__(self, bottom_border, upper_border, *args, **kwargs):
        """
        :param bottom_border: Нижняя доступная граница значений (включая).
        :param upper_border: Верхняя доступная граница значений (включая).
        """
        if not isinstance(bottom_border, int):
            msg = 'Bottom border should be int, but %s found.' % (type(bottom_border).__name__,)
            logger.error_log.error(msg)
            raise ValueError(msg)
        if not isinstance(upper_border, int):
            msg = 'Upper border should be int, but %s found.' % (type(upper_border).__name__,)
            logger.error_log.error(msg)
            raise ValueError(msg)

        super(IntegerRangeField, self).__init__(*args, **kwargs)
        self.bottom_border = bottom_border
        self.upper_border = upper_border

    def validate(self, value):
        super(IntegerRangeField, self).validate(value)
        if value is None:
            return
        if value < self.bottom_border or self.upper_border < value:
            raise self.errors['value_not_in_range'](bottom_border=self.bottom_border, upper_border=self.upper_border)
