# -*- coding: utf-8 -*-
"""
Форматтеры -- это классы которые умеют принимать снаружи и отдавать наружу данные в определённом формате,
например в виде XML, JSON, Protobuf, YAML, и умеют принимать изнутри и отдавать внутрь стандартные питонячьи структуры
данных: dict, list, string.

Форматтеры встраиваются в процесс обработки запроса перед десериализацией тела запроса от клиента и после сериализации
ответа клиенту.
Таким образом при спуске по стеку форматтеры преобразуют сырые данные в пригодное для скриализаторов представление,
а при подъёме по стеку преобразуют сериализованные представления в формат удобный клиенту.
"""
from google.protobuf.internal.containers import RepeatedCompositeFieldContainer, RepeatedScalarFieldContainer
from google.protobuf.message import Message
import sys
from mpfs.common.util import from_json, to_json
from mpfs.platform.exceptions import FormatterFormatError, FormatterParseError
from mpfs.platform.serializers import FormatOptions


class BaseFormatterMetaClass(type):
    """
    Метакласс оборачивающий методы `format` и `parse` в обработчики ошибок и таким образом обеспечивающий
    стандартный механизм обработки ошибок.
    """
    def __new__(mcs, name, parents, attrs):
        # прячем методы format и parse и заменяем их обёртками
        for attr in ('format', 'parse'):
            private_attr_name = '_%s' % attr
            if attr in attrs:
                # Если переопределён один из методов преобразования, то просто прячем его в приватное поле, так чтобы
                # на месте аттрибута с этим именем осталась родительская обёртка.
                attrs[private_attr_name] = attrs.pop(attr)

            wrapper_attr_name = '_%s_wrapper' % (attr,)
            if wrapper_attr_name in attrs:
                # Если переопределена обёртка методов format или parse, то помещаем её в атрибуты format или parse.
                attrs[attr] = attrs.pop(wrapper_attr_name)
        return super(BaseFormatterMetaClass, mcs).__new__(mcs, name, parents, attrs)


class BaseFormatter(object):
    __metaclass__ = BaseFormatterMetaClass

    FormatOptions = FormatOptions
    """Класс позволяющий передавать сериализаторам и филдам специфические параметры форматоа данных"""

    numeric_choices = False
    """Форматтеру работает с представлением значения ChoicesFields в виде номеров опций."""

    datetime_as_timestamp = False
    """Форматтеру работает с представлением значения DateTimeField в виде таймстэмпа."""

    parameters = {}
    """Параметры форматирования. Устанавливаются, как правило, автоматически хэндлером."""

    def format(self, data):
        """
        Форматирует данные обычно представленные в виде питонячьих dict или list в формат соответствующий форматтеру.

        :raise FormatterFormatError: При ошибке форматирования данных.

        :param data: Данные которые необходимо отформатировать. Обычно dict или list.
        :return: Данные в определённом формате.
        """
        raise NotImplementedError()

    def parse(self, raw_data):
        """
        Преобразует данные представленные в определённом формате в питонячью структуру данных, обычно dict или list.

        :raise FormatterParseError: При ошибке разбора данных.

        :param raw_data: Данные в определённом формате, обычно это bytes, str или unicode.
        :return: Питонячья структура данных представляющая исходные данные.
        :rtype: dict | list
        """
        raise NotImplementedError()

    def format_error_handler(self, exception):
        """Получает исключенеи возникшее в процессе форматирования данных и позволяет обработать его."""
        raise FormatterFormatError(inner_exception=exception), None, sys.exc_info()[2]
    
    def parse_error_handler(self, exception):
        """Получает исключенеи возникшее в процессе разбора данных и позволяет обработать его."""
        raise FormatterParseError(inner_exception=exception), None, sys.exc_info()[2]

    #
    # Internal methods
    #

    def _format_wrapper(self, data):
        try:
            # этот метод будет добавлен метаклассом
            return self._format(data)
        except Exception as e:
            return self.format_error_handler(e)

    def _parse_wrapper(self, data):
        try:
            # этот метод будет добавлен метаклассом
            return self._parse(data)
        except Exception as e:
            return self.parse_error_handler(e)


class JsonFormatter(BaseFormatter):
    DEFAULT_CHARSET = 'utf-8'

    def format(self, data):
        return to_json(data)

    def parse(self, raw_data):
        encoding = self.parameters.get('charset', self.DEFAULT_CHARSET)
        if isinstance(raw_data, (str, bytes)):
            raw_data = raw_data.decode(encoding)
        return from_json(raw_data)


class PlainTextFormatter(BaseFormatter):
    def format(self, data):
        ret = '%s' % (data,)
        return ret.encode(encoding='utf-8')

    def parse(self, raw_data):
        ret = '%s' % (raw_data,)
        return ret.decode(encoding='utf-8')


class UnicodeStringFormatter(BaseFormatter):
    def format(self, data):
        return data

    def parse(self, raw_data):
        encoding = self.parameters.get('charset', 'utf-8')
        if isinstance(raw_data, (str, bytes)):
            raw_data = raw_data.decode(encoding)
        return raw_data


class ProtobufFormatter(BaseFormatter):
    class FormatOptions(FormatOptions):
        numeric_choices = True
        datetime_as_timestamp = True
        raw_binary = True
        filter_fields = False

    def __init__(self, in_cls=None, out_cls=None, *args, **kwargs):
        super(ProtobufFormatter, self).__init__(*args, **kwargs)
        self.in_cls = in_cls
        self.out_cls = out_cls or in_cls

    def fill_object(self, obj, data):
        """Заполянет протобуфный объект данными из dict'а с поддержкой вложенных объектов и списков объектов"""
        for name, value in data.iteritems():
            if hasattr(obj, name):
                attr = getattr(obj, name)
                if isinstance(value, (list, tuple)):
                    if isinstance(attr, RepeatedCompositeFieldContainer):
                        if value:
                            for item_data in value:
                                item = attr.add()
                                self.fill_object(item, item_data)
                        else:
                            # Если список пустой, то просто инициализируем пустой список в протобуфном объекте,
                            # иначе протобуф думает, что мы не задали его значение.
                            # https://jira.yandex-team.ru/browse/CHEMODAN-19208
                            del attr[:]
                    else:
                        getattr(obj, name).extend(value)
                elif isinstance(getattr(obj, name, None), Message):
                    self.fill_object(attr, value)
                else:
                    setattr(obj, name, value)

    def format(self, data):
        obj = self.out_cls()
        self.fill_object(obj, data)
        return obj.SerializeToString()

    @staticmethod
    def get_dict(pb_obj):
        """Превращает protobuf'ный объект в питонячий dict."""
        return {descriptor.name: value for descriptor, value in pb_obj.ListFields()}

    def parse_object(self, obj):
        """Превращает протобуфный объект в dict с поддержкой вложенных объектов и списков объектов"""
        ret = {}

        for k, v in self.get_dict(obj).iteritems():
            if isinstance(v, RepeatedCompositeFieldContainer):
                l = []
                for item in v:
                    if isinstance(item, Message):
                        l.append(self.parse_object(item))
                if l:
                    ret[k] = l
            elif isinstance(v, RepeatedScalarFieldContainer):
                for item in v:
                    ret[k].append(item)
            else:
                try:
                    if obj.HasField(k):
                        if isinstance(v, Message):
                            ret[k] = self.parse_object(v)
                        else:
                            ret[k] = v
                except ValueError:
                    pass
        return ret

    def parse(self, data):
        obj = self.in_cls.FromString(data)
        obj_dict = self.parse_object(obj)
        return obj_dict
