# -*- coding: utf-8 -*-
import urllib
from collections import OrderedDict
from copy import deepcopy, copy
import sys
from types import NoneType
from functools import wraps
import hashlib

import urlparse
from mpfs.common.static import tags
from mpfs.common.util import from_json, to_json
from mpfs.config import settings
from mpfs.core.user.constants import (
    DISK_AREA_PATH,
    PHOTOUNLIM_AREA_PATH,
)
from mpfs.platform import fields
from mpfs.platform.async_mode import gevent_local_variables
from mpfs.platform.auth import (
    OAuthAuth,
    ClientNetworks,
    InternalAuth,
    InternalConductorAuth,
    InternalTokenAuth,
    TVM2Auth,
    TVM2TicketsOnlyAuth,
    TVMAuth,
)
from mpfs.platform.common import (
    PlatformResponse,
    HTTP_VERBS,
    ResponseObject,
    TEXT_CONTENT_TYPES,
    logger,
)
from mpfs.platform.exceptions import (InternalServerError, CloudApiError, UnauthorizedError, ValidationError,
                                      UnsupportedMediaTypeError, UnexpectedPayloadError, ForbiddenError,
                                      NotAcceptableError, BadKarmaError)
from mpfs.platform.v1.disk.exceptions import DiskNotFoundError
from mpfs.platform.fields import QueryDict, BaseField, IfNoneMatchField
from mpfs.platform.formatters import JsonFormatter, BaseFormatter
from mpfs.platform.rate_limiters import PerUserRateLimiter
from mpfs.platform.serializers import ApiInfoSerializer
from mpfs.platform.utils import (
    quote_string,
    CaseInsensitiveDict,
    parse_http_media_type_parameters,
    flatten_dict,
)
from mpfs.platform.permissions import BasePlatformPermission, DenyAllPermission, AllowAllPermission
from mpfs.platform.etag import EntityTag, EntityTagMatcher

PLATFORM_DISABLE_INTERNAL_AUTH = settings.platform['disable_internal_auth']
PLATFORM_KARMA_BAD_KARMA = settings.platform['karma']['bad_karma']
PLATFORM_KARMA_ENABLE_CHECK = settings.platform['karma']['enable_check']
PLATFORM_NOT_FORWARDABLE_HEADERS = settings.platform['not_forwardable_headers']
SERVICES_TVM_ENABLED = settings.services['tvm']['enabled']
SERVICES_TVM_2_0_ENABLED = settings.services['tvm_2_0']['enabled']


class PlatformHandlerMetaClass(type):
    """Метакласс обеспечивающий механизм наследования аттрибутов `query`, `kwargs` и `headers`."""
    def __new__(mcs, name, parents, attrs):
        for attr_name in ('query', 'kwargs', 'headers'):
            attr = {}
            for parent in parents:
                parent_attr = getattr(parent, attr_name, None)
                if parent_attr is not None:
                    # аккуратно копируем все филды, чтоб иметь собственные экземпляры филдов в создаваемом классе
                    for field_name, field in parent_attr.get_fields().iteritems():
                        attr[field_name] = copy(field)
            self_attr = attrs.get(attr_name, None)
            if self_attr is not None:
                # филды создаваемого класса не копируем, ибо без надобности
                attr.update(self_attr.get_fields())
            attrs[attr_name] = QueryDict({k: v for k, v in attr.iteritems() if isinstance(v, BaseField)})
        # наследуем `error_map`
        error_map = {}
        for parent in parents:
            error_map.update(getattr(parent, 'error_map', {}))
        error_map.update(attrs.get('error_map', {}))
        if error_map:
            attrs['error_map'] = error_map
        # Проверяем, что у хендлеров верный тип атрибута permissions
        if 'permissions' in attrs:
            perm = attrs['permissions']
            if not isinstance(perm, BasePlatformPermission):
                raise TypeError('Attribute "permissions" must be instance of "BasePlatformPermission". '\
                                'Handler "%s" got "%s".' % (name, perm))
        return super(PlatformHandlerMetaClass, mcs).__new__(mcs, name, parents, attrs)


@gevent_local_variables('request', 'args')
class BasePlatformHandler(object):

    __metaclass__ = PlatformHandlerMetaClass

    CONTENT_TYPE_HAL = 'application/hal+json'
    CONTENT_TYPE_JSON = 'application/json'

    DEFAULT_CONTENT_TYPES = OrderedDict([
        ('application/json', JsonFormatter()),
        ('application/hal+json', JsonFormatter()),
    ])
    """
    Список общих для платформы content-type'ов и соответствующих им форматтеров.

    Порядок имеет значение.
    При всех равных условиях (если Accept равен */* или несколько типов имеют одинаковое качество) выбран будет
    первый по порядку тип.
    """

    default_content_type = 'application/json'
    """Content-Type и Accept по умолчанию, если не задан заголовками запроса."""

    content_types = OrderedDict()
    """Список специфичных для хэндлера content-type'ов и соответствующих им форматтеров."""

    hidden = False
    """Скрывает хэндлер от схемы."""

    default_auth_methods = [InternalTokenAuth(), InternalConductorAuth(), OAuthAuth()]
    """Список механизмов авторизации поддерживаемых всеми хэндлерами."""
    if SERVICES_TVM_ENABLED:
        default_auth_methods.append(TVMAuth())
    if SERVICES_TVM_2_0_ENABLED:
        default_auth_methods.append(TVM2Auth())
        default_auth_methods.append(TVM2TicketsOnlyAuth())
    default_auth_methods.append(InternalAuth())

    auth_methods = []
    """Список механизмов авторизации поддерживаемых хэндлером."""

    auth_required = True
    """
    Имеют ли доступ к хэндлеру не авторизованные клиенты.
    Если False, то атрибуты `request.client` и `request.user` будут равны None и авторизация проверяться не будет.
    """

    auth_user_required = True
    """
    Требуется ли для доступа к хэндлеру авторизация пользователя вдобавок к авторизации клиента.
    Если False, то атрибут `request.user` может быть равен None.
    """

    permissions = DenyAllPermission()
    """Объект типа 'BasePlatformPermission', который реализует механизм авторизации."""

    args = ()
    kwargs = QueryDict()
    """
    Содержит объявление path-параметров принимаемых хэндлером

    Обработанные QueryDict'ом параметры при каждом запросе помещаются в `self.request.kwargs`,
    а не сырые параметры помещаются в `self.request.raw_kwargs`.
    """

    serializer_cls = None
    """
    Сериализатор используемый хэндлером для формирования конечного представления.
    Также используется генератором JSON Schema для определения типа результата возвращаемого хэндлером.
    """

    resp_status_code = 200
    """Код ответа поумолчанию."""

    @property
    def router(self):
        """Роутер в котором зарегистрирован хэндлер."""
        return getattr(self.parent, 'router', None)

    @property
    def method(self):
        if self.parent:
            for r, h in self.parent.relations.iteritems():
                if h == self:
                    if r in HTTP_VERBS:
                        return r
        return None

    @property
    def pattern(self):
        chunks = []
        handler = self
        parent = self.parent
        while parent:
            for r, h in parent.relations.iteritems():
                if h == handler:
                    if r not in HTTP_VERBS:
                        chunks.append(r)
                    break
            handler = parent
            parent = parent.parent
        if chunks:
            return '/%s' % '/'.join(reversed(chunks))
        else:
            return ''

    @property
    def path(self):
        from mpfs.platform.routers import URLPattern
        return URLPattern.expand_pattern(self.pattern, stub='{%s}')[0]

    resource = None
    """Ресурс в котором живёт хэндлер. Устанавливается ресурсом при инициализации хэндлера."""

    headers = QueryDict()
    """
    Содержит объявление заголовков принимаемых хэндлером

    Обработанные заголовки при каждом запросе помещаются в `self.request.headers`.
    """

    query = QueryDict({
        'fields': fields.ListField(help_text=u'Список возвращаемых атрибутов.'),
    })
    """
    Содержит объявление querystring параметров принимаемых хэндлером

    Обработанные QueryDict'ом параметры при каждом запросе помещаются в `self.request.query`.
    """

    body_serializer_cls = None
    """
    Класс сериализатора используемого для обработки тела запроса

    Обработанное сериализатором тело запроса помещается в `self.request.body`.

    Причина, по которой для обработки query string используется экземпляр QueryDict'а, объявляемый непосредственно в
    теле хэндлера и являющийся гораздо более наглядным, нежели сериализатор, описанный в другом модуле и просто
    подключённый к хэндлеру, заключается в том, что QueryDict - это плоский список не подразумевающий возможности
    вложения объектов и легко описываемый одноуровневой структурой такой как QueryDict,
    в то время, как тело запроса может содержать JSON достаточно сложной структуры, описывать которую
    сериализаторами в теле хэндлера было бы не красиво, и это сильно загромождало бы код хэндлера лишним кодом.
    """

    rate_limiter = PerUserRateLimiter('cloud_api_user')

    def __init__(self, parent=None, *args, **kwargs):
        """
        Создаёт экземпляр хэндлера и если router задан, то связывает хэндлер с роутером.

        :param parent: Ресурс в котором живёт хэндлер или ресурс.
        """
        super(BasePlatformHandler, self).__init__()
        self.request = None
        self.parent = parent
        self.query = deepcopy(self.__class__.query)
        self.query.parent = self
        self.kwargs = deepcopy(self.__class__.kwargs)
        self.kwargs.parent = self
        self.headers = deepcopy(self.__class__.headers)
        self.headers.parent = self
        self.content_types = copy(self.DEFAULT_CONTENT_TYPES)
        self.content_types.update(type(self).content_types)

    def __str__(self):
        return '%s.%s' % (self.__module__, self.__class__.__name__)

    def make_response(self, ret):
        """Properly setup self.response according to value returned by set.handler.

        :param ret: dict | list | tuple | PlatformResponse
        :return: PlatformResponse
        """
        if isinstance(ret, PlatformResponse):
            # если хэндлер возвращает готовый ответ, то пробрасываем его без изменений
            response = ret
        elif isinstance(ret, tuple):
            ret = list(ret)
            if len(ret) >= 2 and isinstance(ret[1], (dict, list)):
                ret[1] = self.format_response(ret[1])
            response = PlatformResponse(*ret)
        elif isinstance(ret, (dict, list,)):
            response = PlatformResponse(self.resp_status_code, self.format_response(ret))
        else:
            response = PlatformResponse(self.resp_status_code, ret)

        if 'Content-Type' not in response.headers:
            ct = self.get_response_content_type()
            if ct is not None:
                if ct in TEXT_CONTENT_TYPES:
                    # добавляем charset только к текстовым типам, например к протобуфу добавлять его не надо
                    ct = '%s; charset=utf-8' % (ct,)
                response.headers['Content-Type'] = ct

        return response

    def get_response_content_type(self, request=None):
        """Возвращает Content-Type ответа"""
        if request is None:
            request = self.request
        if 'Accept' in request.raw_headers:
            ct = request.accept_mimetypes.best_match(self.get_content_types().keys())
        else:
            ct = self.default_content_type
        return ct

    def get_response_formatter(self):
        """Возвращает форматтер для форматирования тела ответа"""
        formatter = self.get_content_types().get(self.get_response_content_type())
        assert isinstance(formatter, BaseFormatter)
        return formatter

    def get_request_content_type(self, request=None):
        """Возвращает Content-Type запроса"""
        if request is None:
            request = self.request
        return request.raw_headers.get('Content-Type') or self.default_content_type

    def get_request_formatter(self):
        """Возвращает форматтер для разбора тела запроса"""
        ct = self.get_request_content_type()
        formatter = self.get_content_types().get(ct.split(';')[0])
        if formatter:
            assert isinstance(formatter, BaseFormatter)
            formatter.parameters = parse_http_media_type_parameters(ct)
        return formatter

    def format_response(self, data):
        """
        Форматирует возвращаемые данные в запрошенный клиентом формат.
        """
        ct = self.get_response_content_type()
        formatter = self.get_content_types()[ct]
        assert isinstance(formatter, BaseFormatter)
        return formatter.format(data)

    def get_content_types(self):
        """
        Возвращает dict content type'ов и форматтеров соответствующих им для данного хэндлера.

        Петодо позволяет в дочерних хэндлерах динамически формаировать список доступных типов и форматтеров.
        """
        return self.content_types

    def get_permissions(self):
        """Returns permission that this handler requires."""
        return self.permissions

    def get_auth_methods(self):
        """Возвращает список механизмов авторизации поддерживаемых хэндлером."""
        result = list(self.auth_methods)

        for auth_method in self.default_auth_methods:
            already_added = any(type(auth_method) is type(added_auth_method)
                                for added_auth_method in result)
            if already_added:
                # не добавляем из default_auth_methods авторизации, которые добавили напрямую в handler'ы
                # 1. они могут быть кастомно сконфигурированы
                # 2. если конфигурация такая же, нет смысла дважды пытаться авторизовать по одной схеме
                continue

            result.append(auth_method)

        if PLATFORM_DISABLE_INTERNAL_AUTH:
            result = [auth_method
                      for auth_method in result
                      if not isinstance(auth_method, InternalAuth)]

        return result

    def permission_denied(self, request):
        """If request is not permitted, determine what kind of exception to raise."""
        raise ForbiddenError()

    def check_permissions(self, request):
        """Check if the request should be permitted.

        Raises an appropriate exception if the request is not permitted.
        """
        if not self.get_permissions().has_permission(request):
            self.permission_denied(request)

    def initialize(self, request, *args, **kwargs):
        """Инициализирует хэндлер перед обработкой запроса."""
        self.request = request
        self.args = args
        self.request.raw_kwargs = kwargs
        self.request.kwargs = self.clean_kwargs(kwargs)
        self.request.headers = self.clean_headers(self.request.raw_headers)
        self.request.query = self.clean_query(self.request.args)
        self.request.body = self._clean_body(self.request.data)

    def authorize(self, request, *args, **kwargs):
        """
        Проверяет авторизацию, если задан атрибут `self.auth_required`,
        и выставляет значения `request.client` и `request.user`.

        :exception UnauthorizedError: Если ни один из методов авторизации не смог авторизовать пользователя.
        :param PlatformRequest request: Запрос который требуется авторизовать.
        :return: Если авторизация прошла успешно, то True, если авторизация не проверялась, то False.
        :rtype: bool
        """
        disabled_auth_methods_names = self._get_disabled_auth_methods_names(request)
        for auth_method in self.get_auth_methods():
            if auth_method.__class__.__name__ in disabled_auth_methods_names:
                continue
            if auth_method.authorize(request):
                if self.auth_user_required and not request.user:
                    # если хэндлер требует авторизации и клиента и пользователя, а пользователя авторизовать не удалось,
                    # то считаем, что авторизация в целом не удалась
                    logger.error_log.error('User authorization is required')
                    raise UnauthorizedError()
                if not ClientNetworks.is_client_authorized_from_request_ip(request.client):
                    logger.error_log.error(ClientNetworks.prepare_error_message(request.client))
                    if not settings.auth['network_authorization']['silent_mode']:
                        continue
                return True

        # если ни один из механизмов авторизации не вернул True, значит авторизовать не шмагла
        if self.auth_required:
            raise UnauthorizedError()
        return False

    def check_accept_header(self, request):
        ct = self.get_response_content_type(request)
        if ct is None:
            raise NotAcceptableError()

    def dispatch(self, request, *args, **kwargs):
        """Dispatch request and return response. Play role of interface between main application and handler logic.

        :return: PlatformResponse
        """
        self.check_accept_header(request)
        self.authorize(request, *args, **kwargs)
        self.initialize(request, *args, **kwargs)
        self.check_permissions(self.request)
        self.check_rate_limit(request)
        try:
            ret = self.handle(self.request, *args, **kwargs)
        except Exception, e:
            try:
                ret = self.handle_exception(e)
            except CloudApiError, e:
                self.set_response_headers(e)
                raise
        ret = self.make_response(ret)
        self.set_response_headers(ret)
        return ret

    def set_response_headers(self, resp_or_exception):
        exposed_headers = self.get_exposed_headers(self.request)
        if exposed_headers:
            if 'Access-Control-Expose-Headers' in resp_or_exception.headers:
                headers = filter(None, resp_or_exception.headers['Access-Control-Expose-Headers'].split(','))
            else:
                headers = []
            headers.extend(exposed_headers)
            resp_or_exception.headers['Access-Control-Expose-Headers'] = ', '.join(set(headers))

        origin = self.request.raw_headers.get('Origin')
        if origin:
            resp_or_exception.headers['Access-Control-Allow-Origin'] = origin

    def check_rate_limit(self, request):
        """
        Проверяет не превышен ли допустимый лимит количества запросов.
        В случае превышения лимита, выбрасывает исключение.
        """
        if self.rate_limiter:
            self.rate_limiter.check(request)

    def clean_kwargs(self, raw_kwargs):
        """Обрабатывает параметры path и возвращает dict содержащий проверенные и преобразованные данные."""
        return self.kwargs.clean(raw_kwargs)

    def clean_headers(self, raw_headers):
        """
        Обрабатывает заголовки и возвращает dict содержащий проверенные и преобразованные данные.

        :rtype: CaseInsensitiveDict
        """
        return CaseInsensitiveDict(self.headers.clean(raw_headers))

    def clean_query(self, raw_query):
        """Обрабатывает параметры запроса и возвращает dict содержащий проверенные и преобразованные данные."""
        return self.query.clean(raw_query)

    def parse_body(self, raw_data):
        """
        Преобразует тело запроса из исходного формата в `dict` или `list` с помощью форматтера,
        соответствующего заголовку Content-Type.

        :rtype: dict, list
        """
        if raw_data:
            ct = self.get_request_content_type().split(';')[0]
            content_types = self.get_content_types()
            if ct not in content_types:
                raise UnsupportedMediaTypeError()

            formatter = self.get_request_formatter()
            return formatter.parse(raw_data)
        else:
            return {}

    def clean_body(self, data):
        """
        Обрабатывает тело запроса и возвращает объект содержащий проверенные и преобразованные данные.

        :return: Результат зависит от возвращаемого методом `self.body_serializer_cls.restore_object()`.
        """
        formatter = self.get_request_formatter()
        formatter_options = formatter.FormatOptions if formatter else None
        serializer = self.body_serializer_cls(data=data, router=self.router, format_options=formatter_options)
        return serializer.object

    def _clean_body(self, raw_data):
        if raw_data and self.body_serializer_cls is None:
            raise UnexpectedPayloadError()

        data = self.parse_body(raw_data)

        if self.body_serializer_cls:
            return self.clean_body(data)
        else:
            return None

    def handle(self, request, *args, **kwargs):
        """Contains all heavy lifting logic handling request.

        :param request:
        :param args:
        :param kwargs:
        :return: dict | list | tuple | PlatformResponse
        """
        raise NotImplementedError()

    def handle_exception(self, exception):
        """
        Gets all exceptions generated in self.handle and can suppress and return response or forward them to dispatcher.

        :param exception:
        :return: dict | list | tuple | PlatformResponse
        """
        if isinstance(exception, CloudApiError):
            raise
        else:
            raise InternalServerError(inner_exception=exception), None, sys.exc_info()[2]

    def is_hal_requested(self):
        return (self.CONTENT_TYPE_JSON not in self.request.accept_mimetypes) \
            and (self.CONTENT_TYPE_HAL in self.request.accept_mimetypes)

    def serialize(self, obj, *args, **kwargs):
        """Сериализует объект сериализатором хэндлера. Умеет включать/выключать HAL контролы в сериализаторе."""
        if self.serializer_cls:
            formatter = self.get_response_formatter()
            filtering_fields = self._get_fields_to_filter()
            serializer = self.serializer_cls(obj=obj, router=self.router, hal=self.is_hal_requested(),
                                             lang=self.request.language,
                                             format_options=formatter.FormatOptions,
                                             visible_fields=filtering_fields or None,
                                             *args, **kwargs)
            obj = serializer.data
        return obj

    response_objects = []
    """
    Список объектов `ResponseObjects` содержащих описание ответов хэндлера.

    Используется для вывода в полигоне списка возвращаемых хэндлером ответов.
    """

    def get_response_objects(self):
        """Возвращает список возможных ответов хэндлера."""
        ret = []
        ret += self.response_objects
        ret += [ResponseObject(NotAcceptableError)]
        permissions = self.get_permissions()
        if self.auth_required or not isinstance(permissions, AllowAllPermission):
            ret += [ResponseObject(UnauthorizedError)]
        if not isinstance(permissions, AllowAllPermission):
            ret += [ResponseObject(ForbiddenError)]
        if self.query.get_fields() or self.headers.get_fields() or self.kwargs.get_fields():
            ret += [ResponseObject(ValidationError)]
        if self.body_serializer_cls:
            ret += [ResponseObject(UnsupportedMediaTypeError)]
        if self.rate_limiter and self.rate_limiter.enabled:
            ret += [ResponseObject(self.rate_limiter.exceeded_limit_exception_cls)]
        if self.serializer_cls or self.resp_status_code == 204:
            ret += [ResponseObject(self.serializer_cls, 'OK', self.resp_status_code)]
        return ret

    def get_exposed_headers(self, request):
        """Возвращает список заголовков для CORS заголовка Access-Control-Expose-Headers"""
        return []

    @staticmethod
    def _get_disabled_auth_methods_names(request):
        """
        Возвращает список имен авторизаций, которые хотим пропустить.

        Предназначен только для целей разработчиков. X-Disable-Auth-Methods содержит имена классов авторизаций,
        перечисленных через точку с запятой. Для внешнего API список всегда возвращаем пустым. Подобная функциональность
        нужна для того, чтобы можно было отказаться, например, от Internal авторизации, поскольку данная авторизация
        всегда проходит при запросе с наших машин, что мешает тестированию других способов авторизаций.
        """
        if request.mode == tags.platform.EXTERNAL:
            return set()
        disabled_auth_methods_raw = request.raw_headers.get('X-Disable-Auth-Methods', '')
        if not disabled_auth_methods_raw:
            return set()
        disabled_auth_methods = {x.strip() for x in disabled_auth_methods_raw.split(';') if x.strip()}
        return disabled_auth_methods

    def _get_fields_to_filter(self):
        return self.request.query.get('fields', None)


class ServiceProxyHandler(BasePlatformHandler):
    error_map = {}
    """
    Services returns errors as exceptions containing status_code and message
    so we can map them by method `.get_platform_exception()` using `.error_map`.
    If you need more specific behavior on mapping service exceptions, then override `get_platform_exception` method.
    """
    service_base_exception = None
    """Базовое исключение сервиса, экземпляры которого следует ловить и преобразовывать в платформенные исключения."""
    service = None
    """Проксируемый сервис. Экземпляр mpfs.core.services.common_service.Service."""
    service_method = 'GET'
    """HTTP-метод используемый для запросов в сервис."""
    service_headers = {}
    """Заголовки отправляемые в сервис при любом запросе."""
    service_url = None
    """Шаблон URL-проксируемой хэндлером ручки."""
    service_timeout = None
    """Переопределение таймаута выполнения запроса в сервис."""
    DEFAULT_FORWARD_HEADERS_KEYS = ['x-forwarded-for']
    """Список заголовоков, которые пробрасывают все хэндлеры API."""
    forward_headers = False
    """Включена ли проброска заголовков в хэндлере."""
    forward_headers_keys = []
    """Список заголовков, которые пробрасывает конкретный хэндлер."""
    do_retries = False
    """Делать ли повторные попытки запросов"""

    def get_base_url(self):
        """Возвращает базовый URL сервиса"""
        return self.service.base_url

    def get_url(self, context=None, base_url=None):
        return self.build_url(self.service_url, context, base_url)

    def build_url(self, url, context=None, base_url=None):
        context = context or {}
        base_url = base_url or self.get_base_url()
        flat_context = flatten_dict(context)
        urlencoded_context = self.urlencode_context(flat_context)
        return '%s%s' % (base_url, url % urlencoded_context)

    def urlencode_context(self, context):
        ret = {}
        for k, v in context.iteritems():
            ret[k] = quote_string(v)
        return ret

    def get_forward_headers_keys(self):
        """
        Получить список имён заголовоков, которые пробрасывет хэндлер.

        :rtype: list
        """
        return self.DEFAULT_FORWARD_HEADERS_KEYS + self.forward_headers_keys

    def get_forward_headers(self, request):
        """
        Получить dict заголовков, которые пробросит хэндлер при получении запроса.

        :param request: Запрос для которого нужно собрать пробрасываемые заголовки.
        :rtype: CaseInsensitiveDict
        """
        if not self.forward_headers:
            # Если в хэндлере отключена проброска заголовков, то не пробрасываем.
            return CaseInsensitiveDict()

        raw_headers = request.raw_headers
        forward_headers_keys = {h.lower() for h in self.get_forward_headers_keys()}

        headers = CaseInsensitiveDict({k: v for k, v in raw_headers.iteritems() if k.lower() in forward_headers_keys})

        if 'x-forwarded-for' in forward_headers_keys:
            # Пробрасываем IP клиента сделавшего запрос.
            remote_ip = request.get_real_remote_addr()
            headers['X-Forwarded-For'] = remote_ip

        return headers

    def get_all_forwardable_headers(self, request):
        headers = CaseInsensitiveDict({k: v for k, v in request.raw_headers.iteritems()})
        for header in PLATFORM_NOT_FORWARDABLE_HEADERS:
            headers.pop(header, False)
        return headers

    def raw_request_service(self, url, method=None, headers=None, data=None, service=None):
        method = method or self.service_method
        if headers:
            headers.update(self.service_headers)
        else:
            headers = self.service_headers
        service = service or self.service

        # Пробрасываем заголовки если включён флаг `self.forward_headers`.
        req_headers = self.get_forward_headers(self.request)
        req_headers.update(headers or {})

        if data is not None and isinstance(data, (dict, list)):
            data = to_json(data)
        return service.open_url(url, method=method, headers=req_headers, pure_data=data,
                                retry=self.do_retries, return_status=True, timeout=self.service_timeout)

    def request_service(self, url, method=None, headers=None, data=None, service=None):
        status_code, resp, headers = self.raw_request_service(url, method=method, headers=headers, data=data, service=service)
        if resp and not isinstance(resp, (dict, list, NoneType)):
            resp = from_json(resp)
        return resp

    def get_context(self, context=None):
        c = context or {}
        c.update(self.request.kwargs)
        if self.request.user:
            c['uid'] = self.request.user.uid
        for k, v in self.request.query.iteritems():
            c[k] = v
        for k, v in self.request.headers.iteritems():
            c[k] = v
        return c

    def handle(self, request, *args, **kwargs):
        context = self.get_context()
        url = self.get_url(context)
        service = self.get_service(context)
        resp = self.request_service(url, service=service)
        resp = self.serialize(resp)
        return resp

    def get_service(self, context=None):
        return None

    def get_error_map(self, error_map=None):
        ret = self.error_map.copy()
        if error_map:
            ret.update(error_map)
        return ret

    def get_service_error_code(self, exception):
        """
        Возвращает код ошибки сервиса, являющийся ключём в `self.error_map`.

        Если маппинга ошибок нет, то должен возвращать None. В этом случае будет выбрасываться InternalServerError.
        """
        raise NotImplementedError()

    def get_exception_context(self, context=None, exception=None):
        c = context or {}
        c.update(self.request.args.iteritems())
        c.update(self.request.raw_kwargs.iteritems())
        return c

    def get_platform_exception(self, exception):
        """Возвращает платформенное исключение вместо переданного исключения сервиса из `self.error_map`."""
        error_code = self.get_service_error_code(exception)
        e_cls = self.get_error_map().get(error_code, InternalServerError)
        return e_cls(inner_exception=exception, **self.get_exception_context(exception=exception))

    def handle_exception(self, exception):
        if isinstance(exception, self.service_base_exception):
            platform_exception = self.get_platform_exception(exception)
            if platform_exception:
                # re-raise exception preserving original stack trace and value
                raise platform_exception, None, sys.exc_info()[2]
        return super(ServiceProxyHandler, self).handle_exception(exception)

    def get_response_objects(self):
        ret = super(ServiceProxyHandler, self).get_response_objects()
        ret += [e if isinstance(e, ResponseObject) else ResponseObject(e)
                for e in self.error_map.values()]
        return ret

    def patch_url_params(self, url, params):
        """
        Правильно добавляет или заменяет в URL qs-параметры.

        :param str url: Исходный URL который нужно подправить.
        :param dict params: Дикт с новыми параметрами.
        :return: URL в который добавлены параметры из params.
        :rtype: str
        """
        scheme, netloc, path, query_string, fragment = urlparse.urlsplit(url)
        query = urlparse.parse_qs(query_string.encode('ascii'))  # its safe because URL contains only ascii
        encoded_params = self.urlencode_context(params)
        for k, v in encoded_params.iteritems():
            query[k] = v
        query_string = urllib.urlencode(query, doseq=True)
        return urlparse.urlunsplit((scheme, netloc, path, query_string, fragment))


class GetApiInfoHandler(BasePlatformHandler):
    """Получить метаданные API"""
    auth_required = False
    serializer_cls = ApiInfoSerializer
    permissions = AllowAllPermission()
    rate_limiter = None

    def handle(self, request, *args, **kwargs):
        return self.serialize({})

    def authorize(self, request, *args, **kwargs):
        return False


class HeadApiInfoHandler(GetApiInfoHandler):
    """Получить хедеры API"""

    def handle(self, request, *args, **kwargs):
        return None


class ETagHandlerMixin(BasePlatformHandler):
    """
    Добавляет поддержку заголовков ETag/If-None-Match в соответствии с протоколом HTTP 1.1, основное назначение
    которых поддержка кеширования и условного выполнения HTTP-методов.

    Возвращает 304 код ответа, если ресурс не изменился.

    Реализация работает для любых HTTP-методов, которые возвращают контент. Для методов, не возвращающих контент,
    необходимо переопределить либо метод _calc_strong_etag.

    http://tools.ietf.org/html/rfc2616#section-3.11
    http://trac.tools.ietf.org/wg/httpbis/trac/ticket/116

    https://github.yandex-team.ru/MPFS/MPFS/pull/569#commitcomment-96766
    """
    headers = QueryDict({
        'If-None-Match': IfNoneMatchField(
                help_text=u'Значение заголовка ETag, полученное при предыдущем аналогичном запросе ресурса.'),
    })

    response_objects = [
        ResponseObject(None, u'Содержимое не изменилось.', 304),
    ]

    __MIXED_IN_ATTR = 'etag_handler_mixed_in'

    def __init__(self, *args, **kwargs):
        super(ETagHandlerMixin, self).__init__(*args, **kwargs)
        if not getattr(self, self.__MIXED_IN_ATTR, False):
            self.dispatch = self.__wrap_dispatch(self.dispatch)
            setattr(self, self.__MIXED_IN_ATTR, True)

    def __wrap_dispatch(self, dispatch):
        handler = self

        @wraps(dispatch)
        def dispatch_with_etag(request, *args, **kwargs):
            response = dispatch(request, *args, **kwargs)
            etag = handler.calc_etag(response)

            if etag is not None and response.get_status() == 200:
                etag_matcher = request.headers.get('If-None-Match', EntityTagMatcher([]))
                if etag_matcher.matches(etag):
                    return 304, None, {'ETag': etag}
                else:
                    response.headers['ETag'] = etag

            return response

        return dispatch_with_etag

    def get_exposed_headers(self, request):
        result = super(ETagHandlerMixin, self).get_exposed_headers(request)
        result.append('ETag')
        return result

    def calc_etag(self, response):
        weak_etag_value = self._calc_weak_etag(response)
        if weak_etag_value:
            return EntityTag(weak_etag_value, weak=True)

        strong_etag_value = self._calc_strong_etag(response)
        if strong_etag_value:
            return EntityTag(strong_etag_value, weak=False)

        return None

    def _calc_strong_etag(self, response):
        content = response.content
        if isinstance(content, unicode):
            content = content.encode('utf-8')
        return hashlib.md5(content).hexdigest()

    def _calc_weak_etag(self, response):
        return None


class CheckUserKarmaHandlerMixin(BasePlatformHandler):
    """
    Проверяет карму пользователя.

    Настраивается в конфиге в секции `platform.karma`.

    Добавляет в хэндлеры метод `self.bad_karma(request)`, который вызывается, если карма пользователя
    больше или равна порогу, заданному в конфиге `platform.karma.bad_karma`.
    Дефолтная реализация просто шврыяет исключение `BadKarmaError`, но хэндлеры могут переопределять это поведение
    по своему усмотрению.
    """
    def bad_karma(self, request):
        """
        Вызывается, когда карма больше или равна порогу, заданному в конфиге `platform.karma.bad_karma`.

        Может переопределять в хэндлерах. Дефолтная реализация просто выбрасывает `BadKarmaError`.
        """
        raise BadKarmaError()

    def _check_karma(self, request):
        if not PLATFORM_KARMA_ENABLE_CHECK:
            return

        if request.user and request.user.karma >= PLATFORM_KARMA_BAD_KARMA:
            self.bad_karma(request)

    def authorize(self, request, *args, **kwargs):
        super(CheckUserKarmaHandlerMixin, self).authorize(request, *args, **kwargs)
        self._check_karma(request)


class GetResouceInfoByIDMixin(BasePlatformHandler):
    """Получает информацию о ресурсе по идентификатору ресурса."""
    def get_resource_info(self, uid, resource_id, enabled_areas=(DISK_AREA_PATH, PHOTOUNLIM_AREA_PATH)):
        data = self.request_service(
            url=self.build_url(
                '/json/bulk_info_by_resource_ids?enable_service_ids=%s&uid=%s' % (
                    ','.join(enabled_areas),
                    uid
                )
            ),
            method='POST',
            data=[resource_id],
            headers={'Content-Type': 'application/json'}
        )
        if not data:
            raise DiskNotFoundError()
        if len(data) > 1:
            raise ValueError()

        return data[0]


class RequestsPoweredServiceProxyHandler(ServiceProxyHandler):
    service = None
    """Проксируемый сервис. Экземпляр mpfs.core.services.common_service.RequestsPoweredServiceBaseА."""

    def build_url(self, url, context=None, base_url=None):
        """Полный аналог базовый имплементации, но без хоста (только относительный URL)"""
        context = context or {}
        flat_context = flatten_dict(context)
        urlencoded_context = self.urlencode_context(flat_context)
        return url % urlencoded_context

    def raw_request_service(self, url, method=None, headers=None, data=None, service=None):
        """Сырой запрос в сервис проксирования.

        Один в один как базовый, но работает с сервисами написанными на новых http-клиентах.
        """
        method = method or self.service_method
        if headers:
            headers.update(self.service_headers)
        else:
            headers = self.service_headers
        service = service or self.service
        # Пробрасываем заголовки если включён флаг `self.forward_headers`.
        req_headers = self.get_forward_headers(self.request)
        req_headers.update(headers or {})

        resp = service.request(method, url, headers=req_headers, data=data,
                               timeout=self.service_timeout)

        return resp.status_code, resp.content, resp.headers
