# -*- coding: utf-8 -*-
from inspect import isclass
import operator

from mpfs.common.static import tags
from mpfs.platform.exceptions import UnauthorizedError, ForbiddenError
from mpfs.core.services.conductor_service import ConductorService


class PlatformPermissionBase(object):
    _operation = None
    _permissions = None

    def __and__(self, other):
        compound_permission = BasePlatformPermission()
        compound_permission._operation = operator.and_
        compound_permission._permissions = (self, other)
        return compound_permission

    def __or__(self, other):
        compound_permission = BasePlatformPermission()
        compound_permission._operation = operator.or_
        compound_permission._permissions = (self, other)
        return compound_permission


class BasePlatformPermission(PlatformPermissionBase):
    """
    Base class for platform permissions.
    Return False or raise PlatformPermissionDenied exception if request doesn't permit.
    """

    def has_permission(self, request):
        op = self._operation
        permissions = self._permissions
        if op and permissions:
            ret = None
            for operand in permissions:
                try:
                    result = operand.has_permission(request)
                except (UnauthorizedError, ForbiddenError):
                    result = False
                if ret is None:
                    ret = result
                    # Если и так уже всё ясно, то не вычисляем значения остальных операндов
                    if op == operator.or_ and ret is True:
                        break
                    elif op == operator.and_ and ret is False:
                        break
                else:
                    ret = op(ret, result)
            return ret
        return True

    @staticmethod
    def is_legacy_internal_auth(request):
        # метод нужен для одновременной поддержки старой авторизации во внутреннем API (Internal) и новых
        # способов авторизации по токену и ip адресу - ClientToken и IP, в будущем надо грамотно оторвать Internal
        if request.mode != tags.platform.INTERNAL:
            return False

        # Своим всё можно, только если это не кто-то с авторизацией по токену или по IP - у него есть свои скоупы,
        # и их надо проверить, а остальным да - все можно.. пока что :)
        auth_header = None
        if request.raw_headers:
            auth_header = request.raw_headers.get('Authorization', '')
            auth_header = auth_header.lower()

        if auth_header and auth_header.startswith('internal'):
            return True  # для обратной совместимости, потом надо оторвать
        elif request.args and request.args.get('client_id') is not None and request.args.get('client_name') is not None:
            return True  # для обратной совместимости, потом надо оторвать

        return False

    @staticmethod
    def is_conductor_auth_fallback_mode(request):
        return ConductorService.is_conductor_auth_fallback_mode(request)


class ClientHasScopesPermission(BasePlatformPermission):
    """
    Base permission for OAuth scope based permissions.
    """
    scopes = ()

    def has_permission(self, request):
        if request.mode == tags.platform.INTERNAL:
            if self.is_legacy_internal_auth(request):
                return True
            if self.is_conductor_auth_fallback_mode(request):
                return True
        if not super(ClientHasScopesPermission, self).has_permission(request):
            return False
        if not request.client:
            raise UnauthorizedError()

        scopes = self.scopes
        # не может быть разрешения без скоупов, ибо это означает доступ для всех,
        # а для этого есть спец. разрешения
        if not scopes:
            return False

        if not (set(scopes) <= set(request.client.scopes)):
            raise ForbiddenError()
        return True


class ConditionalPermission(BasePlatformPermission):
    """Проверяет то или иное разрешение в зависимости от результата метода `get_condition`"""
    conditions_permissions = {}
    """Маппинг значений возвращаемых методом `get_condition` на права которые будет проверяться"""

    def __init__(self, *args, **kwargs):
        super(ConditionalPermission, self).__init__()
        for k, v in self.conditions_permissions.iteritems():
            if isclass(v):
                self.conditions_permissions[k] = v(*args, **kwargs)

    def get_condition(self, request):
        """Возвращает ключ для выбора проверяемого разрешения из дикта `conditions_permissions`"""
        raise NotImplementedError()

    def has_permission(self, request):
        if request.mode == tags.platform.INTERNAL:
            if self.is_legacy_internal_auth(request):
                return True
            if self.is_conductor_auth_fallback_mode(request):
                return True
        if not super(ConditionalPermission, self).has_permission(request):
            return False
        condition = self.get_condition(request)
        permission = self.conditions_permissions.get(condition, None)
        if not permission or not permission.has_permission(request):
            raise ForbiddenError()
        return True


class DynamicScopesPermission(ClientHasScopesPermission):
    """Проверяет наличие у клиента скоупов вычисляемых динамически методом `get_required_scopes`"""

    def get_required_scopes(self, request):
        """
        Возвращает список скоупов, которыми должен обладать клиент, чтоб пройти авторизацию

        :param mpfs.platform.common.PlatformRequest request: Запрос для которого проверяются разрешения.
        :return: Список скоупов (строк) которыми должен обладать клиент, чтобы пройти авторизацию.
        :rtype: list | tuple
        """
        raise NotImplementedError()

    _request = None
    """
    Запрос для которого проверяются разрешения.

    Атрибут необходим для того чтобы в проперте scopes иметь доступ к параметрам запроса.
    """

    def has_permission(self, request):
        self._request = request
        return super(DynamicScopesPermission, self).has_permission(request)

    @property
    def scopes(self):
        scopes = self.get_required_scopes(self._request)
        return scopes


class DenyAllPermission(BasePlatformPermission):
    """Запрещает доступ всем"""

    def has_permission(self, request):
        return request.mode == tags.platform.INTERNAL


class AllowAllPermission(BasePlatformPermission):
    """Разрешает доступ всем"""

    def has_permission(self, request):
        return True


class AllowByClientIdPermission(BasePlatformPermission):

    def __init__(self, allowed_client_ids):
        super(AllowByClientIdPermission, self).__init__()
        self.allowed_client_ids = set(allowed_client_ids)

    def has_permission(self, request):
        if request.mode == tags.platform.INTERNAL:
            if self.is_legacy_internal_auth(request):
                return True
            if self.is_conductor_auth_fallback_mode(request):
                return True
        if not super(AllowByClientIdPermission, self).has_permission(request):
            return False
        elif request.client.id not in self.allowed_client_ids:
            raise ForbiddenError()
        return True